package CipUX::Trait;

use warnings;
use strict;
use Class::Std;
use English qw( -no_match_vars);
use Module::Pluggable
    search_path => ['CipUX::Trait'],
    instantiate => 'new',
    sub_name    => 'register';
use Readonly;
use base qw(CipUX);

{
    use version; our $VERSION = qv('3.4.0.13');
    use re 'taint';    # Keep data captured by parens tainted
    delete @ENV{qw(PATH IFS CDPATH ENV BASH_ENV)};    # Make %ENV safer

    # CONST
    Readonly::Scalar my $EMPTY_STRING => q{};

    # GLOBAL
    my $trait_name_register_hr = {};

    # METHOD
    sub init {

        my $self = shift;

        foreach my $trait ( $self->register ) {
            next until $trait->can('register');
            $trait->register();
        }

        return 1;

    }

    # set a feature name and class
    sub set_trait_name_register {

        my ( $self, $arg_r ) = @_;

        my $class
            = exists $arg_r->{class}
            ? $self->l( $arg_r->{class} )
            : $self->perr('class');

        my $name
            = exists $arg_r->{name}
            ? $self->l( $arg_r->{name} )
            : $self->perr('name');

        $trait_name_register_hr->{$name} = $class;

        return 1;
    }

    # returns Perl module name for registered names
    sub get_trait_name_register {

        if (wantarray) {
            return keys %{$trait_name_register_hr};
        }
        return $trait_name_register_hr;
    }

    sub get_trait {
        my @t = sort keys %{$trait_name_register_hr};
        return @t;
    }

}

1;

__END__

=pod

=for stopwords CipUX Kuelker perr VARNAME API exc STDERR config getopt homedir lf LDAP lp li latin UTF UTF-8 login STDOUT TODO cfg

=head1 NAME

CipUX::Trait - Common CipUX functions

=head1 VERSION

version 3.4.0.13

=head1 SYNOPSIS

  use CipUX::Trait;

  my $ trait = $CipUX::Trait->new;


=head1 DESCRIPTION

Provides feature plugins.

=head1 CONSTRUCTOR

=head2 new

Constructor

  my $trait = CipUX::Trait->new;

=head1 SUBROUTINES/METHODS

The following functions will be exported by CipUX::Trait.

=head2 init
=head2 set_trait_name_register
=head2 get_trait_name_register
=head2 get_trait

=head1 DIAGNOSTICS

This module generates no exception by itself.

=head1 CONFIGURATION AND ENVIRONMENT

Not applicable.

=head1 DEPENDENCIES

 CipUX
 Class::Std
 English
 Module::PLuggable;
 Readonly
 version

=head1 INCOMPATIBILITIES

Not known.

=head1 BUGS AND LIMITATIONS

Not known.

=head1 SEE ALSO

See the CipUX web page and the manual at L<http://www.cipux.org>

See the mailing list L<http://sympa.cipworx.org/wws/info/cipux-devel>

=head1 AUTHOR

Christian Kuelker  E<lt>christian.kuelker@cipworx.orgE<gt>

=head1 LICENSE AND COPYRIGHT

Copyright (C) 2010 by Christian Kuelker

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation; either version 2, or (at
your option) any later version.

This program is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
02111-1307 USA

=cut
