/* directory.cxx
     $Id: directory.cxx,v 1.14 2001/11/27 23:57:36 elf Exp $

   written by Marc Singer
   20 September 1996

   This file is part of the project CurVeS.  See the file README for
   more information.

   Copyright (C) 1996 Marc Singer

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   with your Debian GNU/Linux system, in
   /usr/share/common-licenses/GPL, or with the Debian GNU/Linux hello
   source package as the file COPYING. If not, write to the Free
   Software Foundation, Inc., 59 Temple Place -Suite 330, MA
   02111-1307, USA.

   -----------
   DESCRIPTION
   -----------

   Implementation of directory classes.

*/

#include "std.h"
#include <sys/stat.h>
#include <pwd.h>
#include <limits.h>
#include "directory.h"

#define C_FILES_STEP		64

/* LDirectory::add_file
   
   adds a file entry to an enumerated directory for cases when the
   file SHOULD be there, but is not.  CVS uses this to insert
   placeholders for files being added to the project by other
   developers.  The return value is the file index.

*/

int LDirectory::add_file (const char* sz, int cch)
{
  alloc_to (m_cFiles + 1);
  assert_ (m_cFiles + 1 < m_cFilesMax);
  char* szNew = (char*) malloc (cch + 1);
  m_rgInfo[m_cFiles].szPath = strncpy (szNew, sz, cch);
  szNew[cch] = 0;
  m_rgInfo[m_cFiles].mode = S_IFREG;
  ++m_cFiles;
  return m_cFiles - 1;
}


/* LDirectory::alloc_to

   allocates enough space in the m_rgInfo structure to hold the given
   number of files.

*/

int LDirectory::alloc_to (int cFiles)
{
  int cFilesTarget = cFiles + 2*C_FILES_STEP - 1;
  cFilesTarget -= cFilesTarget%C_FILES_STEP;

  if (cFiles >= m_cFilesMax) {
    if (!m_rgInfo) {
      int cb = sizeof (FILE_INFO)*cFilesTarget;
      m_rgInfo = (FILE_INFO*) malloc (cb);
      if (!m_rgInfo)
	return m_cFilesMax;
      memset (m_rgInfo, 0, cb);
    }
    else {
      int cb = sizeof (FILE_INFO)*cFilesTarget;
      void* pvNew = realloc (m_rgInfo, cb);
      if (!pvNew) {
	release_files ();
	return m_cFilesMax;
      }
      m_rgInfo = (FILE_INFO*) pvNew;
      int cbPrev = sizeof (FILE_INFO)*m_cFilesMax;
      memset ((char*) m_rgInfo + cbPrev, 0, cb - cbPrev);
    }
    m_cFilesMax = cFilesTarget;
  }
  return m_cFilesMax;
}


/* LDirectory::abbreviate

   returns an abbreviated form of the pathname that is absolute, but
   fits within the requested path length cchMax.  It uses two
   techniques to shorten the pathname.  It will replace a path prefix
   with the tilde (~) notation for the current user or another user.
   It will replace path elements starting at the beginning of the
   path, exempting a home directory reference, until all but the last
   path element is replaced by ellipsis.  It some circumstances the
   abbreviation may be longer than cchMax, but what can we do, eh?

*/

const char* LDirectory::abbreviate (int cchMax)
{
  static char sz[PATH_MAX];
  struct passwd* pEnt;
  size_t cchMatch = 0;

  setpwent ();
  while ((pEnt = getpwent ())) {
    if (   pEnt->pw_dir[0] == '/' // Ignore root cause it's an
	&& pEnt->pw_dir[1] == 0) // unhelpful abbreviation 
      continue;
    if (!pEnt->pw_uid		// Ignore daemons and other systems stuff
	|| !pEnt->pw_gid
	|| !pEnt->pw_shell)
      continue;
    if (strncmp (m_szPath, pEnt->pw_dir, strlen (pEnt->pw_dir)))
      continue;
    if (pEnt->pw_uid == getuid ()) {
      strcpy (sz, "~");	
      strcat (sz, m_szPath + (cchMatch = strlen (pEnt->pw_dir)));
      break;			// This is always the end of the line
    }
    if (cchMatch >= strlen (pEnt->pw_dir))
      continue;
    sprintf (sz, "~%s", pEnt->pw_name);
    strcat (sz, m_szPath + (cchMatch = strlen (pEnt->pw_dir)));
  }
  if (!cchMatch)
    strcpy (sz, m_szPath);

  char* pch = sz;
  if (*pch == '~') {
    int cch = strcspn (pch, "/");
    cchMax -= cch;
    pch += cch;
  }
  if (*pch != '/')
    return sz;

  int cchEllipsis = 0;
  int cchPath = strlen (pch);
  while (cchPath - cchEllipsis + (cchEllipsis ? 3 : 0) > cchMax) {
    int cch = strcspn (pch + cchEllipsis + 1, "/");
    if (pch[cchEllipsis + 1 + cch] != '/') {
      if (pch[cchEllipsis] == '/')
	++cchEllipsis;
      break;
    }
    cchEllipsis += 1 + cch;
  }
  if (cchEllipsis > 3) {
    memmove (pch + 3, pch + cchEllipsis, cchPath - cchEllipsis + 1);
    memcpy (pch, "...", 3);
  }

  return sz;
}


/* LDirectory::init

   initializes the directory structures.  If the path pointer is NULL,
   we do not release the current path.  Instead, we only discard the
   directory's contents.

*/

void LDirectory::init (const char* szPath)
{
  if (!szPath)
    release_files ();
  else {
    release_this ();
    m_szPath = (char*) malloc (strlen (szPath) + 2); // Always room for "/\0"
    strcpy (m_szPath, szPath);
    normalize ();
  }
}


/* LDirectory::exists

   quick function to check that a directory is readable on the
   system.  If not, we can short circuit some of the higher level
   code. 

*/

bool LDirectory::exists (void)
{
  struct stat stat;
  return ::stat (m_szPath, &stat) == 0;
}


/* LDirectory::find

   returns the file number with the given name.  We could do a binary
   search, but for now we will do a linear one.  The return value is
   -1 if the file is not found.

*/

int LDirectory::find (const char* szFile, int cch)
{
  if (!cch)
    cch = strlen (szFile);
  int iFile;
  for (iFile = m_cFiles; iFile--; )
    if (strncmp (szFile, m_rgInfo[iFile].szPath, cch) == 0
	&& m_rgInfo[iFile].szPath[cch] == 0)
      break;
  return iFile;
}

/* LDirectory::normalize

   coerces the path into cannonical form.  This means eliminating '.'
   and '..' references as well as making the path absolute.

*/

void LDirectory::normalize (void)
{
				// Replace ~ prefix with absolute path
  if (*m_szPath == '~') {
    int cchUser = strcspn (m_szPath + 1, "/");
    setpwent ();
    struct passwd* pEnt;
    while ((pEnt = getpwent ())) {
      if (cchUser) {
	if (strncmp (pEnt->pw_name, m_szPath + 1, cchUser) == 0
	    && !pEnt->pw_name[cchUser])
	  break;
      }
      else if (pEnt->pw_uid == getuid ())
	break;
    }
    if (!pEnt)
      return;			// FIXME: this is a rather serious error

    char sz[256];
    strcpy (sz, pEnt->pw_dir);
    strcat (sz, m_szPath + 1 + cchUser);
    free (m_szPath);
    m_szPath = (char*) malloc (strlen (sz) + 2); // Always room for "/\0"
    strcpy (m_szPath, sz);
  }

  if (*m_szPath != '/') {
    char sz[256];
    getcwd (sz, sizeof (sz));
    strcat (sz, "/");
    strcat (sz, m_szPath);
    free (m_szPath);
    m_szPath = (char*) malloc (strlen (sz) + 2); // Always room for "/\0"
    strcpy (m_szPath, sz);
  }
  int ichLastPath = strlen (m_szPath);
  bool fStrip = false;
  int cStrip = 0;
  for (int ich = ichLastPath; ich--; ) {
    if (m_szPath[ich] != '/')
      continue;
    if (m_szPath[ich + 1] == '.' && m_szPath[ich + 2] == 0)
      m_szPath[ich] = 0;
    else if (m_szPath[ich + 1] == '.' && m_szPath[ich + 2] == '/') {
      if (!fStrip)
	memmove (&m_szPath[ich], &m_szPath[ichLastPath], 
		 strlen (&m_szPath[ichLastPath]) + 1);
      else
	continue;
    }
    else if (m_szPath[ich + 1] == '.' 
	     && m_szPath[ich + 2] == '.'
	     && (m_szPath[ich + 3] == '/' || !m_szPath[ich + 3])) {
      ++cStrip;
      fStrip = true;
      continue;
    }
    if (fStrip) {
      if (--cStrip == 0) {
	memmove (&m_szPath[ich], &m_szPath[ichLastPath], 
		 strlen (&m_szPath[ichLastPath]) + 1);
	fStrip = false;
      }
      else
	continue;		// Must not reset ichLastPath
    }
    ichLastPath = ich;
  }
  if (fStrip || !*m_szPath)
    strcpy (m_szPath, "/");
}

void LDirectory::release_this (void)
{
  release_files ();
  if (m_szPath) {
    free (m_szPath);
    m_szPath = NULL;
  }
}


void LDirectory::release_files (void)
{
  if (m_rgInfo) {
    while (m_cFiles--)
      free (m_rgInfo[m_cFiles].szPath);
    free (m_rgInfo);
    m_rgInfo = NULL;
  }
  m_cFiles = 0;
  m_cFilesMax = 0;
}


int LDirectory::enumerate (PFN_DIRECTORY_SELECT pfn, void* pv)
{
  release_files ();

  DIR* pdir = opendir (m_szPath);
		// -- This is kinda special.  If we cannot read the
		// directory, we want a minimum of a '..' entry so
		// that the user can navigate out of it.
  if (!pdir) {
    if (!strcmp (m_szPath, "/")) // Nothing we can do at the root
      return 0;

    char* szPath = "..";
    if (pfn && !pfn (szPath, pv))
      return 0;

    alloc_to (1);
    char* szNew = (char*) malloc (strlen (szPath) + 1);
    if (!szNew) {
      release_files ();
      return 0;
    }
    m_rgInfo[m_cFiles].szPath = strcpy (szNew, szPath);

    _stat (m_cFiles);
    m_rgInfo[m_cFiles].mode |= S_IFDIR; // Coerce it into directory-ness
    return ++m_cFiles;
  }

  struct dirent* pdirent;
  while ((pdirent = readdir (pdir))) {
    if (pfn && !pfn (pdirent->d_name, pv))
      continue;

    alloc_to (m_cFiles + 1);
    char* szNew = (char*) malloc (strlen (pdirent->d_name) + 1);
    if (!szNew) {
      release_files ();
      return 0;
    }
    m_rgInfo[m_cFiles].szPath = strcpy (szNew, pdirent->d_name);

    _stat (m_cFiles++);
  }
  closedir (pdir);
  return m_cFiles;
}

bool LDirectory::is_directory (FILE_INFO* pInfo)
{
  return S_ISDIR (pInfo->mode);
}

bool LDirectory::is_directory (int iFile)
{
  if (!_is_ok (iFile))
    return false;

  return S_ISDIR (m_rgInfo[iFile].mode);
}

bool LDirectory::is_file (FILE_INFO* pInfo)
{
  return S_ISREG (pInfo->mode);
}

bool LDirectory::is_file (int iFile)
{
  if (!_is_ok (iFile))
    return false;

  return S_ISREG (m_rgInfo[iFile].mode);
}


bool LDirectory::is_softlink (int iFile)
{
  if (!_is_ok (iFile))
    return false;

  return S_ISLNK (m_rgInfo[iFile].mode);
}

int LDirectory::rank (FILE_INFO* pInfo)
{
  if (is_directory (pInfo))
    return fileRankDirectory;
  if (is_file (pInfo))
    return fileRankFile;
  return fileRankOther;
}


int LDirectory::sort_alpha_compare (const void* pv0, const void* pv1)
{
  //  fprintf (stderr, "sort_alpha_compare: %s %s\n", 
  //	   *(const char**) pv0, *(const char**) pv1);
  return strcmp (*(const char**) pv0, *(const char**) pv1);
}


void LDirectory::sort (PFN_DIRECTORY_COMPARE pfn)
{
  if (m_cFiles && m_rgInfo)
    qsort (m_rgInfo, m_cFiles, sizeof (FILE_INFO), 
	   pfn ? pfn : sort_alpha_compare);
}


/* LDirectory::stat 

   fetches filesystem `stat' information for the file at index iFile.
   This call will always request stat information from the operating
   system so it may be used to update the stat data.

*/

void LDirectory::_stat (int iFile)
{
  if (!m_rgInfo || iFile < 0 || iFile >= m_cFiles)
    return;
  struct stat stat;
  memset (&stat, 0, sizeof (stat));
  char sz[256];
  strcpy (sz, m_szPath);
  strcat (sz, "/");
  strcat (sz, m_rgInfo[iFile].szPath);
  ::lstat (sz, &stat);
  m_rgInfo[iFile].cb   = stat.st_size;
  m_rgInfo[iFile].mode = stat.st_mode;
  m_rgInfo[iFile].time = stat.st_mtime;
}
