/*
 *
 *  Copyright (C) 2022, OFFIS e.V.
 *  All rights reserved.  See COPYRIGHT file for details.
 *
 *  This software and supporting documentation were developed by
 *
 *    OFFIS e.V.
 *    R&D Division Health
 *    Escherweg 2
 *    D-26121 Oldenburg, Germany
 *
 *
 *  Module:  oficonv
 *
 *  Author:  Marco Eichelberg
 *
 *  Purpose: GB18030 related tables for oficonv unit tests
 *
 */

#ifndef TABLES_GB18030_H
#define TABLES_GB18030_H

// GB18030 (Chinese) is a true superset of GBK (and thus of GB2312).
// Every character is either
// - a single byte < 128, or
// - a sequence of two bytes where the first byte has a value >= 0x81,
//   and the second byte has a value >= 0x40,
// - a sequence of four bytes where the first byte has a value in
//   the range 0x81..0x8F or 0x90..0xFE,
//   the second byte has a value in the range 0x30..0x39 (which is
//   never used in two-byte sequences),
//   the third byte is in the range 0x81..0xFE and the fourth byte
//   in the range 0x30..0x39.
// This provides about 1.59 million four-byte sequences, sufficient
// to map the entire currently used Unicode set.
// However, oficonv only supports the Unicode Basic Multilingual Plane (BMP),
// i.e. Unicode code points 0000..FFFF where the four-byte GB18030 sequences
// always start with 0x81..0x84.
//
// In this file we only define a few examples of four-byte sequences
// and do not systematically exercise the entire BMP.

// Latin characters with accent
unsigned char gb18030_latin_with_accents[] = {
    0x81, 0x30, 0x86, 0x38, 0x81, 0x30, 0x86, 0x39, 0x81, 0x30, 0x87, 0x30, 0x81, 0x30, 0x87, 0x31,
    0x81, 0x30, 0x87, 0x32, 0x81, 0x30, 0x87, 0x33, 0x81, 0x30, 0x87, 0x34, 0x81, 0x30, 0x87, 0x35,
    0x81, 0x30, 0x87, 0x36, 0x81, 0x30, 0x87, 0x37, 0x81, 0x30, 0x87, 0x38, 0x81, 0x30, 0x87, 0x39,
    0x81, 0x30, 0x88, 0x30, 0x81, 0x30, 0x88, 0x31, 0x81, 0x30, 0x88, 0x32, 0x81, 0x30, 0x88, 0x33,
    0x81, 0x30, 0x88, 0x34, 0x81, 0x30, 0x88, 0x35, 0x81, 0x30, 0x88, 0x36, 0x81, 0x30, 0x88, 0x37,
    0x81, 0x30, 0x88, 0x38, 0x81, 0x30, 0x88, 0x39, 0x81, 0x30, 0x89, 0x30, 0x81, 0x30, 0x89, 0x31,
    0x81, 0x30, 0x89, 0x32, 0x81, 0x30, 0x89, 0x33, 0x81, 0x30, 0x89, 0x34, 0x81, 0x30, 0x89, 0x35,
    0x81, 0x30, 0x89, 0x36, 0x81, 0x30, 0x89, 0x37, 0x81, 0x30, 0x89, 0x38, 0x81, 0x30, 0x89, 0x39,
    0x81, 0x30, 0x8A, 0x30, 0x81, 0x30, 0x8A, 0x31, 0x81, 0x30, 0x8A, 0x32, 0x81, 0x30, 0x8A, 0x33,
    0x81, 0x30, 0x8A, 0x34, 0x81, 0x30, 0x8A, 0x35, 0x81, 0x30, 0x8A, 0x36, 0x81, 0x30, 0x8A, 0x37,
    0x81, 0x30, 0x8A, 0x38, 0x81, 0x30, 0x8A, 0x39, 0x81, 0x30, 0x8B, 0x30, 0x81, 0x30, 0x8B, 0x31,
    0x81, 0x30, 0x8B, 0x32, 0x81, 0x30, 0x8B, 0x33, 0x81, 0x30, 0x8B, 0x34, 0x81, 0x30, 0x8B, 0x35,
    0x81, 0x30, 0x8B, 0x36, 0x81, 0x30, 0x8B, 0x37, 0x81, 0x30, 0x8B, 0x38, 0x81, 0x30, 0x8B, 0x39,
    0x81, 0x30, 0x8C, 0x30, 0x81, 0x30, 0x8C, 0x31, 0x81, 0x30, 0x8C, 0x32, 0x81, 0x30, 0x8C, 0x33,
    0x81, 0x30, 0x8C, 0x34, 0x81, 0x30, 0x8C, 0x35, 0x81, 0x30, 0x8C, 0x36, 0x81, 0x30, 0x8C, 0x37,
    0x81, 0x30, 0x8C, 0x38, 0x81, 0x30, 0x8C, 0x39, 0x81, 0x30, 0x8D, 0x30, 0x81, 0x30, 0x8D, 0x31
};

// Cyrillic characters with accents
unsigned char gb18030_cyrillic_with_accents[] = {
    0x81, 0x30, 0xD8, 0x36, 0x81, 0x30, 0xD8, 0x37, 0x81, 0x30, 0xD8, 0x38, 0x81, 0x30, 0xD8, 0x39,
    0x81, 0x30, 0xD9, 0x30, 0x81, 0x30, 0xD9, 0x31, 0x81, 0x30, 0xD9, 0x32, 0x81, 0x30, 0xD9, 0x33,
    0x81, 0x30, 0xD9, 0x34, 0x81, 0x30, 0xD9, 0x35, 0x81, 0x30, 0xD9, 0x36, 0x81, 0x30, 0xD9, 0x37,
    0x81, 0x30, 0xD9, 0x38, 0x81, 0x30, 0xD9, 0x39, 0x81, 0x30, 0xDA, 0x30, 0x81, 0x30, 0xDA, 0x31,
    0x81, 0x30, 0xDA, 0x32, 0x81, 0x30, 0xDA, 0x33, 0x81, 0x30, 0xDA, 0x34, 0x81, 0x30, 0xDA, 0x35,
    0x81, 0x30, 0xDA, 0x36, 0x81, 0x30, 0xDA, 0x37, 0x81, 0x30, 0xDA, 0x38, 0x81, 0x30, 0xDA, 0x39,
    0x81, 0x30, 0xDB, 0x30, 0x81, 0x30, 0xDB, 0x31, 0x81, 0x30, 0xDB, 0x32, 0x81, 0x30, 0xDB, 0x33,
    0x81, 0x30, 0xDB, 0x34, 0x81, 0x30, 0xDB, 0x35, 0x81, 0x30, 0xDB, 0x36, 0x81, 0x30, 0xDB, 0x37,
    0x81, 0x30, 0xDB, 0x38, 0x81, 0x30, 0xDB, 0x39, 0x81, 0x30, 0xDC, 0x30, 0x81, 0x30, 0xDC, 0x31,
    0x81, 0x30, 0xDC, 0x32, 0x81, 0x30, 0xDC, 0x33, 0x81, 0x30, 0xDC, 0x34, 0x81, 0x30, 0xDC, 0x35,
    0x81, 0x30, 0xDC, 0x36, 0x81, 0x30, 0xDC, 0x37, 0x81, 0x30, 0xDC, 0x38, 0x81, 0x30, 0xDC, 0x39,
    0x81, 0x30, 0xDD, 0x30, 0x81, 0x30, 0xDD, 0x31, 0x81, 0x30, 0xDD, 0x32, 0x81, 0x30, 0xDD, 0x33,
    0x81, 0x30, 0xDD, 0x34, 0x81, 0x30, 0xDD, 0x35, 0x81, 0x30, 0xDD, 0x36, 0x81, 0x30, 0xDD, 0x37,
    0x81, 0x30, 0xDD, 0x38, 0x81, 0x30, 0xDD, 0x39, 0x81, 0x30, 0xDE, 0x30, 0x81, 0x30, 0xDE, 0x31,
    0x81, 0x30, 0xDE, 0x32, 0x81, 0x30, 0xDE, 0x33, 0x81, 0x30, 0xDE, 0x34, 0x81, 0x30, 0xDE, 0x35,
    0x81, 0x30, 0xDE, 0x36, 0x81, 0x30, 0xDE, 0x37, 0x81, 0x30, 0xDE, 0x38, 0x81, 0x30, 0xDE, 0x39
};

// Armenian characters
unsigned char gb18030_armenian[] = {
    0x81, 0x30, 0xEE, 0x31, 0x81, 0x30, 0xEE, 0x32, 0x81, 0x30, 0xEE, 0x33, 0x81, 0x30, 0xEE, 0x34,
    0x81, 0x30, 0xEE, 0x35, 0x81, 0x30, 0xEE, 0x36, 0x81, 0x30, 0xEE, 0x37, 0x81, 0x30, 0xEE, 0x38,
    0x81, 0x30, 0xEE, 0x39, 0x81, 0x30, 0xEF, 0x30, 0x81, 0x30, 0xEF, 0x31, 0x81, 0x30, 0xEF, 0x32,
    0x81, 0x30, 0xEF, 0x33, 0x81, 0x30, 0xEF, 0x34, 0x81, 0x30, 0xEF, 0x35, 0x81, 0x30, 0xEF, 0x36,
    0x81, 0x30, 0xEF, 0x37, 0x81, 0x30, 0xEF, 0x38, 0x81, 0x30, 0xEF, 0x39, 0x81, 0x30, 0xF0, 0x30,
    0x81, 0x30, 0xF0, 0x31, 0x81, 0x30, 0xF0, 0x32, 0x81, 0x30, 0xF0, 0x33, 0x81, 0x30, 0xF0, 0x34,
    0x81, 0x30, 0xF0, 0x35, 0x81, 0x30, 0xF0, 0x36, 0x81, 0x30, 0xF0, 0x37, 0x81, 0x30, 0xF0, 0x38,
    0x81, 0x30, 0xF0, 0x39, 0x81, 0x30, 0xF1, 0x30, 0x81, 0x30, 0xF1, 0x31, 0x81, 0x30, 0xF1, 0x32
};

// Devanagari characters
unsigned char gb18030_devanagari[] = {
    0x81, 0x31, 0xCD, 0x32, 0x81, 0x31, 0xCD, 0x33, 0x81, 0x31, 0xCD, 0x34, 0x81, 0x31, 0xCD, 0x35,
    0x81, 0x31, 0xCD, 0x36, 0x81, 0x31, 0xCD, 0x37, 0x81, 0x31, 0xCD, 0x38, 0x81, 0x31, 0xCD, 0x39,
    0x81, 0x31, 0xCE, 0x30, 0x81, 0x31, 0xCE, 0x31, 0x81, 0x31, 0xCE, 0x32, 0x81, 0x31, 0xCE, 0x33,
    0x81, 0x31, 0xCE, 0x34, 0x81, 0x31, 0xCE, 0x35, 0x81, 0x31, 0xCE, 0x36, 0x81, 0x31, 0xCE, 0x37,
    0x81, 0x31, 0xCE, 0x38, 0x81, 0x31, 0xCE, 0x39, 0x81, 0x31, 0xCF, 0x30, 0x81, 0x31, 0xCF, 0x31,
    0x81, 0x31, 0xCF, 0x32, 0x81, 0x31, 0xCF, 0x33, 0x81, 0x31, 0xCF, 0x34, 0x81, 0x31, 0xCF, 0x35,
    0x81, 0x31, 0xCF, 0x36, 0x81, 0x31, 0xCF, 0x37, 0x81, 0x31, 0xCF, 0x38, 0x81, 0x31, 0xCF, 0x39,
    0x81, 0x31, 0xD0, 0x30, 0x81, 0x31, 0xD0, 0x31, 0x81, 0x31, 0xD0, 0x32, 0x81, 0x31, 0xD0, 0x33,
    0x81, 0x31, 0xD0, 0x34, 0x81, 0x31, 0xD0, 0x35, 0x81, 0x31, 0xD0, 0x36, 0x81, 0x31, 0xD0, 0x37,
    0x81, 0x31, 0xD0, 0x38, 0x81, 0x31, 0xD0, 0x39, 0x81, 0x31, 0xD1, 0x30, 0x81, 0x31, 0xD1, 0x31,
    0x81, 0x31, 0xD1, 0x32, 0x81, 0x31, 0xD1, 0x33, 0x81, 0x31, 0xD1, 0x34, 0x81, 0x31, 0xD1, 0x35,
    0x81, 0x31, 0xD1, 0x36, 0x81, 0x31, 0xD1, 0x37, 0x81, 0x31, 0xD1, 0x38, 0x81, 0x31, 0xD1, 0x39,
    0x81, 0x31, 0xD2, 0x30, 0x81, 0x31, 0xD2, 0x31, 0x81, 0x31, 0xD2, 0x32, 0x81, 0x31, 0xD2, 0x33,
    0x81, 0x31, 0xD2, 0x34, 0x81, 0x31, 0xD2, 0x35, 0x81, 0x31, 0xD2, 0x36, 0x81, 0x31, 0xD2, 0x37,
    0x81, 0x31, 0xD2, 0x38, 0x81, 0x31, 0xD2, 0x39
};

// Amaric/Ethiopic characters
unsigned char gb18030_ethiopic[] = {
    0x81, 0x33, 0xB7, 0x32, 0x81, 0x33, 0xB7, 0x33, 0x81, 0x33, 0xB7, 0x34, 0x81, 0x33, 0xB7, 0x35,
    0x81, 0x33, 0xB7, 0x36, 0x81, 0x33, 0xB7, 0x37, 0x81, 0x33, 0xB7, 0x38, 0x81, 0x33, 0xB7, 0x39,
    0x81, 0x33, 0xB8, 0x30, 0x81, 0x33, 0xB8, 0x31, 0x81, 0x33, 0xB8, 0x32, 0x81, 0x33, 0xB8, 0x33,
    0x81, 0x33, 0xB8, 0x34, 0x81, 0x33, 0xB8, 0x35, 0x81, 0x33, 0xB8, 0x36, 0x81, 0x33, 0xB8, 0x37,
    0x81, 0x33, 0xB8, 0x38, 0x81, 0x33, 0xB8, 0x39, 0x81, 0x33, 0xB9, 0x30, 0x81, 0x33, 0xB9, 0x31,
    0x81, 0x33, 0xB9, 0x32, 0x81, 0x33, 0xB9, 0x33, 0x81, 0x33, 0xB9, 0x34, 0x81, 0x33, 0xB9, 0x35,
    0x81, 0x33, 0xB9, 0x36, 0x81, 0x33, 0xB9, 0x37, 0x81, 0x33, 0xB9, 0x38, 0x81, 0x33, 0xB9, 0x39,
    0x81, 0x33, 0xBA, 0x30, 0x81, 0x33, 0xBA, 0x31, 0x81, 0x33, 0xBA, 0x32, 0x81, 0x33, 0xBA, 0x33,
    0x81, 0x33, 0xBA, 0x34, 0x81, 0x33, 0xBA, 0x35, 0x81, 0x33, 0xBA, 0x36, 0x81, 0x33, 0xBA, 0x37,
    0x81, 0x33, 0xBA, 0x38, 0x81, 0x33, 0xBA, 0x39, 0x81, 0x33, 0xBB, 0x30, 0x81, 0x33, 0xBB, 0x31,
    0x81, 0x33, 0xBB, 0x32, 0x81, 0x33, 0xBB, 0x33, 0x81, 0x33, 0xBB, 0x34, 0x81, 0x33, 0xBB, 0x35,
    0x81, 0x33, 0xBB, 0x36, 0x81, 0x33, 0xBB, 0x37, 0x81, 0x33, 0xBB, 0x38, 0x81, 0x33, 0xBB, 0x39,
    0x81, 0x33, 0xBC, 0x30, 0x81, 0x33, 0xBC, 0x31, 0x81, 0x33, 0xBC, 0x32, 0x81, 0x33, 0xBC, 0x33,
    0x81, 0x33, 0xBC, 0x34, 0x81, 0x33, 0xBC, 0x35, 0x81, 0x33, 0xBC, 0x36, 0x81, 0x33, 0xBC, 0x37,
    0x81, 0x33, 0xBC, 0x38, 0x81, 0x33, 0xBC, 0x39, 0x81, 0x33, 0xBD, 0x30, 0x81, 0x33, 0xBD, 0x31,
    0x81, 0x33, 0xBD, 0x32, 0x81, 0x33, 0xBD, 0x33, 0x81, 0x33, 0xBD, 0x34, 0x81, 0x33, 0xBD, 0x35,
};

// Braille characters
unsigned char gb18030_braille[] = {
    0x81, 0x37, 0xD5, 0x35, 0x81, 0x37, 0xD5, 0x36, 0x81, 0x37, 0xD5, 0x37, 0x81, 0x37, 0xD5, 0x38,
    0x81, 0x37, 0xD5, 0x39, 0x81, 0x37, 0xD6, 0x30, 0x81, 0x37, 0xD6, 0x31, 0x81, 0x37, 0xD6, 0x32,
    0x81, 0x37, 0xD6, 0x33, 0x81, 0x37, 0xD6, 0x34, 0x81, 0x37, 0xD6, 0x35, 0x81, 0x37, 0xD6, 0x36,
    0x81, 0x37, 0xD6, 0x37, 0x81, 0x37, 0xD6, 0x38, 0x81, 0x37, 0xD6, 0x39, 0x81, 0x37, 0xD7, 0x30,
    0x81, 0x37, 0xD7, 0x31, 0x81, 0x37, 0xD7, 0x32, 0x81, 0x37, 0xD7, 0x33, 0x81, 0x37, 0xD7, 0x34,
    0x81, 0x37, 0xD7, 0x35, 0x81, 0x37, 0xD7, 0x36, 0x81, 0x37, 0xD7, 0x37, 0x81, 0x37, 0xD7, 0x38,
    0x81, 0x37, 0xD7, 0x39, 0x81, 0x37, 0xD8, 0x30, 0x81, 0x37, 0xD8, 0x31, 0x81, 0x37, 0xD8, 0x32,
    0x81, 0x37, 0xD8, 0x33, 0x81, 0x37, 0xD8, 0x34, 0x81, 0x37, 0xD8, 0x35, 0x81, 0x37, 0xD8, 0x36,
    0x81, 0x37, 0xD8, 0x37, 0x81, 0x37, 0xD8, 0x38, 0x81, 0x37, 0xD8, 0x39, 0x81, 0x37, 0xD9, 0x30,
    0x81, 0x37, 0xD9, 0x31, 0x81, 0x37, 0xD9, 0x32, 0x81, 0x37, 0xD9, 0x33, 0x81, 0x37, 0xD9, 0x34,
    0x81, 0x37, 0xD9, 0x35, 0x81, 0x37, 0xD9, 0x36, 0x81, 0x37, 0xD9, 0x37, 0x81, 0x37, 0xD9, 0x38,
    0x81, 0x37, 0xD9, 0x39, 0x81, 0x37, 0xDA, 0x30, 0x81, 0x37, 0xDA, 0x31, 0x81, 0x37, 0xDA, 0x32,
    0x81, 0x37, 0xDA, 0x33, 0x81, 0x37, 0xDA, 0x34, 0x81, 0x37, 0xDA, 0x35, 0x81, 0x37, 0xDA, 0x36,
    0x81, 0x37, 0xDA, 0x37, 0x81, 0x37, 0xDA, 0x38, 0x81, 0x37, 0xDA, 0x39, 0x81, 0x37, 0xDB, 0x30,
    0x81, 0x37, 0xDB, 0x31, 0x81, 0x37, 0xDB, 0x32, 0x81, 0x37, 0xDB, 0x33, 0x81, 0x37, 0xDB, 0x34,
    0x81, 0x37, 0xDB, 0x35, 0x81, 0x37, 0xDB, 0x36, 0x81, 0x37, 0xDB, 0x37, 0x81, 0x37, 0xDB, 0x38,
};

// Chinese
unsigned char gb18030_chinese[] = {
    0x81, 0x39, 0xF8, 0x32, 0x81, 0x39, 0xF8, 0x33, 0x81, 0x39, 0xF8, 0x34, 0x81, 0x39, 0xF8, 0x35,
    0x81, 0x39, 0xF8, 0x36, 0x81, 0x39, 0xF8, 0x37, 0x81, 0x39, 0xF8, 0x38, 0x81, 0x39, 0xF8, 0x39,
    0x81, 0x39, 0xF9, 0x30, 0x81, 0x39, 0xF9, 0x31, 0x81, 0x39, 0xF9, 0x32, 0x81, 0x39, 0xF9, 0x33,
    0x81, 0x39, 0xF9, 0x34, 0x81, 0x39, 0xF9, 0x35, 0x81, 0x39, 0xF9, 0x36, 0x81, 0x39, 0xF9, 0x37,
    0x81, 0x39, 0xF9, 0x38, 0x81, 0x39, 0xF9, 0x39, 0x81, 0x39, 0xFA, 0x30, 0x81, 0x39, 0xFA, 0x31,
    0x81, 0x39, 0xFA, 0x32, 0x81, 0x39, 0xFA, 0x33, 0x81, 0x39, 0xFA, 0x34, 0x81, 0x39, 0xFA, 0x35,
    0x81, 0x39, 0xFA, 0x36, 0x81, 0x39, 0xFA, 0x37, 0x81, 0x39, 0xFA, 0x38, 0x81, 0x39, 0xFA, 0x39,
    0x81, 0x39, 0xFB, 0x30, 0x81, 0x39, 0xFB, 0x31, 0x81, 0x39, 0xFB, 0x32, 0x81, 0x39, 0xFB, 0x33,
    0x81, 0x39, 0xFB, 0x34, 0x81, 0x39, 0xFB, 0x35, 0x81, 0x39, 0xFB, 0x36, 0x81, 0x39, 0xFB, 0x37,
    0x81, 0x39, 0xFB, 0x38, 0x81, 0x39, 0xFB, 0x39, 0x81, 0x39, 0xFC, 0x30, 0x81, 0x39, 0xFC, 0x31,
    0x81, 0x39, 0xFC, 0x32, 0x81, 0x39, 0xFC, 0x33, 0x81, 0x39, 0xFC, 0x34, 0x81, 0x39, 0xFC, 0x35,
    0x81, 0x39, 0xFC, 0x36, 0x81, 0x39, 0xFC, 0x37, 0x81, 0x39, 0xFC, 0x38, 0x81, 0x39, 0xFC, 0x39,
    0x81, 0x39, 0xFD, 0x30, 0x81, 0x39, 0xFD, 0x31, 0x81, 0x39, 0xFD, 0x32, 0x81, 0x39, 0xFD, 0x33,
    0x81, 0x39, 0xFD, 0x34, 0x81, 0x39, 0xFD, 0x35, 0x81, 0x39, 0xFD, 0x36, 0x81, 0x39, 0xFD, 0x37,
    0x81, 0x39, 0xFD, 0x38, 0x81, 0x39, 0xFD, 0x39, 0x81, 0x39, 0xFE, 0x30, 0x81, 0x39, 0xFE, 0x31,
    0x81, 0x39, 0xFE, 0x32, 0x81, 0x39, 0xFE, 0x33, 0x81, 0x39, 0xFE, 0x34, 0x81, 0x39, 0xFE, 0x35,
    0x82, 0x30, 0x84, 0x37, 0x82, 0x30, 0x84, 0x38, 0x82, 0x30, 0x84, 0x39, 0x82, 0x30, 0x85, 0x30,
    0x82, 0x30, 0x85, 0x31, 0x82, 0x30, 0x85, 0x32, 0x82, 0x30, 0x85, 0x33, 0x82, 0x30, 0x85, 0x34,
    0x82, 0x30, 0x85, 0x35, 0x82, 0x30, 0x85, 0x36, 0x82, 0x30, 0x85, 0x37, 0x82, 0x30, 0x85, 0x38,
    0x82, 0x30, 0x85, 0x39, 0x82, 0x30, 0x86, 0x30, 0x82, 0x30, 0x86, 0x31, 0x82, 0x30, 0x86, 0x32,
    0x82, 0x30, 0x86, 0x33, 0x82, 0x30, 0x86, 0x34, 0x82, 0x30, 0x86, 0x35, 0x82, 0x30, 0x86, 0x36,
    0x82, 0x30, 0x86, 0x37, 0x82, 0x30, 0x86, 0x38, 0x82, 0x30, 0x86, 0x39, 0x82, 0x30, 0x87, 0x30,
    0x82, 0x30, 0x87, 0x31, 0x82, 0x30, 0x87, 0x32, 0x82, 0x30, 0x87, 0x33, 0x82, 0x30, 0x87, 0x34,
    0x82, 0x30, 0x87, 0x35, 0x82, 0x30, 0x87, 0x36, 0x82, 0x30, 0x87, 0x37, 0x82, 0x30, 0x87, 0x38,
    0x82, 0x30, 0x87, 0x39, 0x82, 0x30, 0x88, 0x30, 0x82, 0x30, 0x88, 0x31, 0x82, 0x30, 0x88, 0x32,
    0x82, 0x30, 0x88, 0x33, 0x82, 0x30, 0x88, 0x34, 0x82, 0x30, 0x88, 0x35, 0x82, 0x30, 0x88, 0x36,
    0x82, 0x30, 0x88, 0x37, 0x82, 0x30, 0x88, 0x38, 0x82, 0x30, 0x88, 0x39, 0x82, 0x30, 0x89, 0x30,
    0x82, 0x30, 0x89, 0x31, 0x82, 0x30, 0x89, 0x32, 0x82, 0x30, 0x89, 0x33, 0x82, 0x30, 0x89, 0x34,
    0x82, 0x30, 0x89, 0x35, 0x82, 0x30, 0x89, 0x36, 0x82, 0x30, 0x89, 0x37, 0x82, 0x30, 0x89, 0x38,
    0x82, 0x30, 0x89, 0x39, 0x82, 0x30, 0x8A, 0x30, 0x82, 0x30, 0x8A, 0x31, 0x82, 0x30, 0x8A, 0x32,
    0x82, 0x30, 0x8A, 0x33, 0x82, 0x30, 0x8A, 0x34, 0x82, 0x30, 0x8A, 0x35, 0x82, 0x30, 0x8A, 0x36,
    0x82, 0x30, 0x8A, 0x37, 0x82, 0x30, 0x8A, 0x38, 0x82, 0x30, 0x8A, 0x39, 0x82, 0x30, 0x8B, 0x30,
    0x82, 0x30, 0x8F, 0x31, 0x82, 0x30, 0x8F, 0x32, 0x82, 0x30, 0x8F, 0x33, 0x82, 0x30, 0x8F, 0x34,
    0x82, 0x30, 0x8F, 0x35, 0x82, 0x30, 0x8F, 0x36, 0x82, 0x30, 0x8F, 0x37, 0x82, 0x30, 0x8F, 0x38,
    0x82, 0x30, 0x8F, 0x39, 0x82, 0x30, 0x90, 0x30, 0x82, 0x30, 0x90, 0x31, 0x82, 0x30, 0x90, 0x32,
    0x82, 0x30, 0x90, 0x33, 0x82, 0x30, 0x90, 0x34, 0x82, 0x30, 0x90, 0x35, 0x82, 0x30, 0x90, 0x36,
    0x82, 0x30, 0x90, 0x37, 0x82, 0x30, 0x90, 0x38, 0x82, 0x30, 0x90, 0x39, 0x82, 0x30, 0x91, 0x30,
    0x82, 0x30, 0x91, 0x31, 0x82, 0x30, 0x91, 0x32, 0x82, 0x30, 0x91, 0x33, 0x82, 0x30, 0x91, 0x34,
    0x82, 0x30, 0x91, 0x35, 0x82, 0x30, 0x91, 0x36, 0x82, 0x30, 0x91, 0x37, 0x82, 0x30, 0x91, 0x38,
    0x82, 0x30, 0x91, 0x39, 0x82, 0x30, 0x92, 0x30, 0x82, 0x30, 0x92, 0x31, 0x82, 0x30, 0x92, 0x32,
    0x82, 0x30, 0x92, 0x33, 0x82, 0x30, 0x92, 0x34, 0x82, 0x30, 0x92, 0x35, 0x82, 0x30, 0x92, 0x36,
    0x82, 0x30, 0x92, 0x37, 0x82, 0x30, 0x92, 0x38, 0x82, 0x30, 0x92, 0x39, 0x82, 0x30, 0x93, 0x30,
    0x82, 0x30, 0x93, 0x31, 0x82, 0x30, 0x93, 0x32, 0x82, 0x30, 0x93, 0x33, 0x82, 0x30, 0x93, 0x34,
    0x82, 0x30, 0x93, 0x35, 0x82, 0x30, 0x93, 0x36, 0x82, 0x30, 0x93, 0x37, 0x82, 0x30, 0x93, 0x38,
    0x82, 0x30, 0x93, 0x39, 0x82, 0x30, 0x94, 0x30, 0x82, 0x30, 0x94, 0x31, 0x82, 0x30, 0x94, 0x32,
    0x82, 0x30, 0x94, 0x33, 0x82, 0x30, 0x94, 0x34, 0x82, 0x30, 0x94, 0x35, 0x82, 0x30, 0x94, 0x36,
    0x82, 0x30, 0x94, 0x37, 0x82, 0x30, 0x94, 0x38, 0x82, 0x30, 0x94, 0x39, 0x82, 0x30, 0x95, 0x30,
    0x82, 0x30, 0x95, 0x31, 0x82, 0x30, 0x95, 0x32, 0x82, 0x30, 0x95, 0x33, 0x82, 0x30, 0x95, 0x34,
    0x82, 0x30, 0x99, 0x36, 0x82, 0x30, 0x99, 0x37, 0x82, 0x30, 0x99, 0x38, 0x82, 0x30, 0x99, 0x39,
    0x82, 0x30, 0x9A, 0x30, 0x82, 0x30, 0x9A, 0x31, 0x82, 0x30, 0x9A, 0x32, 0x82, 0x30, 0x9A, 0x33,
    0x82, 0x30, 0x9A, 0x34, 0x82, 0x30, 0x9A, 0x35, 0x82, 0x30, 0x9A, 0x36, 0x82, 0x30, 0x9A, 0x37,
    0x82, 0x30, 0x9A, 0x38, 0x82, 0x30, 0x9A, 0x39, 0x82, 0x30, 0x9B, 0x30, 0x82, 0x30, 0x9B, 0x31,
    0x82, 0x30, 0x9B, 0x32, 0x82, 0x30, 0x9B, 0x33, 0x82, 0x30, 0x9B, 0x34, 0x82, 0x30, 0x9B, 0x35,
    0x82, 0x30, 0x9B, 0x36, 0x82, 0x30, 0x9B, 0x37, 0x82, 0x30, 0x9B, 0x38, 0x82, 0x30, 0x9B, 0x39,
    0x82, 0x30, 0x9C, 0x30, 0x82, 0x30, 0x9C, 0x31, 0x82, 0x30, 0x9C, 0x32, 0x82, 0x30, 0x9C, 0x33,
    0x82, 0x30, 0x9C, 0x34, 0x82, 0x30, 0x9C, 0x35, 0x82, 0x30, 0x9C, 0x36, 0x82, 0x30, 0x9C, 0x37,
    0x82, 0x30, 0x9C, 0x38, 0x82, 0x30, 0x9C, 0x39, 0x82, 0x30, 0x9D, 0x30, 0x82, 0x30, 0x9D, 0x31,
    0x82, 0x30, 0x9D, 0x32, 0x82, 0x30, 0x9D, 0x33, 0x82, 0x30, 0x9D, 0x34, 0x82, 0x30, 0x9D, 0x35,
    0x82, 0x30, 0x9D, 0x36, 0x82, 0x30, 0x9D, 0x37, 0x82, 0x30, 0x9D, 0x38, 0x82, 0x30, 0x9D, 0x39,
    0x82, 0x30, 0x9E, 0x30, 0x82, 0x30, 0x9E, 0x31, 0x82, 0x30, 0x9E, 0x32, 0x82, 0x30, 0x9E, 0x33,
    0x82, 0x30, 0x9E, 0x34, 0x82, 0x30, 0x9E, 0x35, 0x82, 0x30, 0x9E, 0x36, 0x82, 0x30, 0x9E, 0x37,
    0x82, 0x30, 0x9E, 0x38, 0x82, 0x30, 0x9E, 0x39, 0x82, 0x30, 0x9F, 0x30, 0x82, 0x30, 0x9F, 0x31,
    0x82, 0x30, 0x9F, 0x32, 0x82, 0x30, 0x9F, 0x33, 0x82, 0x30, 0x9F, 0x34, 0x82, 0x30, 0x9F, 0x35,
    0x82, 0x30, 0x9F, 0x36, 0x82, 0x30, 0x9F, 0x37, 0x82, 0x30, 0x9F, 0x38, 0x82, 0x30, 0x9F, 0x39,
    0x82, 0x34, 0x83, 0x36, 0x82, 0x34, 0x83, 0x37, 0x82, 0x34, 0x83, 0x38, 0x82, 0x34, 0x83, 0x39,
    0x82, 0x34, 0x84, 0x30, 0x82, 0x34, 0x84, 0x31, 0x82, 0x34, 0x84, 0x32, 0x82, 0x34, 0x84, 0x33,
    0x82, 0x34, 0x84, 0x34, 0x82, 0x34, 0x84, 0x35, 0x82, 0x34, 0x84, 0x36, 0x82, 0x34, 0x84, 0x37,
    0x82, 0x34, 0x84, 0x38, 0x82, 0x34, 0x84, 0x39, 0x82, 0x34, 0x85, 0x30, 0x82, 0x34, 0x85, 0x31,
    0x82, 0x34, 0x85, 0x32, 0x82, 0x34, 0x85, 0x33, 0x82, 0x34, 0x85, 0x34, 0x82, 0x34, 0x85, 0x35,
    0x82, 0x34, 0x85, 0x36, 0x82, 0x34, 0x85, 0x37, 0x82, 0x34, 0x85, 0x38, 0x82, 0x34, 0x85, 0x39,
    0x82, 0x34, 0x86, 0x30, 0x82, 0x34, 0x86, 0x31, 0x82, 0x34, 0x86, 0x32, 0x82, 0x34, 0x86, 0x33,
    0x82, 0x34, 0x86, 0x34, 0x82, 0x34, 0x86, 0x35, 0x82, 0x34, 0x86, 0x36, 0x82, 0x34, 0x86, 0x37,
    0x82, 0x34, 0x86, 0x38, 0x82, 0x34, 0x86, 0x39, 0x82, 0x34, 0x87, 0x30, 0x82, 0x34, 0x87, 0x31,
    0x82, 0x34, 0x87, 0x32, 0x82, 0x34, 0x87, 0x33, 0x82, 0x34, 0x87, 0x34, 0x82, 0x34, 0x87, 0x35,
    0x82, 0x34, 0x87, 0x36, 0x82, 0x34, 0x87, 0x37, 0x82, 0x34, 0x87, 0x38, 0x82, 0x34, 0x87, 0x39,
    0x82, 0x34, 0x88, 0x30, 0x82, 0x34, 0x88, 0x31, 0x82, 0x34, 0x88, 0x32, 0x82, 0x34, 0x88, 0x33,
    0x82, 0x34, 0x88, 0x34, 0x82, 0x34, 0x88, 0x35, 0x82, 0x34, 0x88, 0x36, 0x82, 0x34, 0x88, 0x37,
    0x82, 0x34, 0x88, 0x38, 0x82, 0x34, 0x88, 0x39, 0x82, 0x34, 0x89, 0x30, 0x82, 0x34, 0x89, 0x31,
    0x82, 0x34, 0x89, 0x32, 0x82, 0x34, 0x89, 0x33, 0x82, 0x34, 0x89, 0x34, 0x82, 0x34, 0x89, 0x35,
    0x82, 0x34, 0x89, 0x36, 0x82, 0x34, 0x89, 0x37, 0x82, 0x34, 0x89, 0x38, 0x82, 0x34, 0x89, 0x39,
    0x82, 0x34, 0x8E, 0x31, 0x82, 0x34, 0x8E, 0x32, 0x82, 0x34, 0x8E, 0x33, 0x82, 0x34, 0x8E, 0x34,
    0x82, 0x34, 0x8E, 0x35, 0x82, 0x34, 0x8E, 0x36, 0x82, 0x34, 0x8E, 0x37, 0x82, 0x34, 0x8E, 0x38,
    0x82, 0x34, 0x8E, 0x39, 0x82, 0x34, 0x8F, 0x30, 0x82, 0x34, 0x8F, 0x31, 0x82, 0x34, 0x8F, 0x32,
    0x82, 0x34, 0x8F, 0x33, 0x82, 0x34, 0x8F, 0x34, 0x82, 0x34, 0x8F, 0x35, 0x82, 0x34, 0x8F, 0x36,
    0x82, 0x34, 0x8F, 0x37, 0x82, 0x34, 0x8F, 0x38, 0x82, 0x34, 0x8F, 0x39, 0x82, 0x34, 0x90, 0x30,
    0x82, 0x34, 0x90, 0x31, 0x82, 0x34, 0x90, 0x32, 0x82, 0x34, 0x90, 0x33, 0x82, 0x34, 0x90, 0x34,
    0x82, 0x34, 0x90, 0x35, 0x82, 0x34, 0x90, 0x36, 0x82, 0x34, 0x90, 0x37, 0x82, 0x34, 0x90, 0x38,
    0x82, 0x34, 0x90, 0x39, 0x82, 0x34, 0x91, 0x30, 0x82, 0x34, 0x91, 0x31, 0x82, 0x34, 0x91, 0x32,
    0x82, 0x34, 0x91, 0x33, 0x82, 0x34, 0x91, 0x34, 0x82, 0x34, 0x91, 0x35, 0x82, 0x34, 0x91, 0x36,
    0x82, 0x34, 0x91, 0x37, 0x82, 0x34, 0x91, 0x38, 0x82, 0x34, 0x91, 0x39, 0x82, 0x34, 0x92, 0x30,
    0x82, 0x34, 0x92, 0x31, 0x82, 0x34, 0x92, 0x32, 0x82, 0x34, 0x92, 0x33, 0x82, 0x34, 0x92, 0x34,
    0x82, 0x34, 0x92, 0x35, 0x82, 0x34, 0x92, 0x36, 0x82, 0x34, 0x92, 0x37, 0x82, 0x34, 0x92, 0x38,
    0x82, 0x34, 0x92, 0x39, 0x82, 0x34, 0x93, 0x30, 0x82, 0x34, 0x93, 0x31, 0x82, 0x34, 0x93, 0x32,
    0x82, 0x34, 0x93, 0x33, 0x82, 0x34, 0x93, 0x34, 0x82, 0x34, 0x93, 0x35, 0x82, 0x34, 0x93, 0x36,
    0x82, 0x34, 0x93, 0x37, 0x82, 0x34, 0x93, 0x38, 0x82, 0x34, 0x93, 0x39, 0x82, 0x34, 0x94, 0x30,
    0x82, 0x34, 0x94, 0x31, 0x82, 0x34, 0x94, 0x32, 0x82, 0x34, 0x94, 0x33, 0x82, 0x34, 0x94, 0x34,
    0x82, 0x34, 0x98, 0x36, 0x82, 0x34, 0x98, 0x37, 0x82, 0x34, 0x98, 0x38, 0x82, 0x34, 0x98, 0x39,
    0x82, 0x34, 0x99, 0x30, 0x82, 0x34, 0x99, 0x31, 0x82, 0x34, 0x99, 0x32, 0x82, 0x34, 0x99, 0x33,
    0x82, 0x34, 0x99, 0x34, 0x82, 0x34, 0x99, 0x35, 0x82, 0x34, 0x99, 0x36, 0x82, 0x34, 0x99, 0x37,
    0x82, 0x34, 0x99, 0x38, 0x82, 0x34, 0x99, 0x39, 0x82, 0x34, 0x9A, 0x30, 0x82, 0x34, 0x9A, 0x31,
    0x82, 0x34, 0x9A, 0x32, 0x82, 0x34, 0x9A, 0x33, 0x82, 0x34, 0x9A, 0x34, 0x82, 0x34, 0x9A, 0x35,
    0x82, 0x34, 0x9A, 0x36, 0x82, 0x34, 0x9A, 0x37, 0x82, 0x34, 0x9A, 0x38, 0x82, 0x34, 0x9A, 0x39,
    0x82, 0x34, 0x9B, 0x30, 0x82, 0x34, 0x9B, 0x31, 0x82, 0x34, 0x9B, 0x32, 0x82, 0x34, 0x9B, 0x33,
    0x82, 0x34, 0x9B, 0x34, 0x82, 0x34, 0x9B, 0x35, 0x82, 0x34, 0x9B, 0x36, 0x82, 0x34, 0x9B, 0x37,
    0x82, 0x34, 0x9B, 0x38, 0x82, 0x34, 0x9B, 0x39, 0x82, 0x34, 0x9C, 0x30, 0x82, 0x34, 0x9C, 0x31,
    0x82, 0x34, 0x9C, 0x32, 0x82, 0x34, 0x9C, 0x33, 0x82, 0x34, 0x9C, 0x34, 0x82, 0x34, 0x9C, 0x35,
    0x82, 0x34, 0x9C, 0x36, 0x82, 0x34, 0x9C, 0x37, 0x82, 0x34, 0x9C, 0x38, 0x82, 0x34, 0x9C, 0x39,
    0x82, 0x34, 0x9D, 0x30, 0x82, 0x34, 0x9D, 0x31, 0x82, 0x34, 0x9D, 0x32, 0x82, 0x34, 0x9D, 0x33,
    0x82, 0x34, 0x9D, 0x34, 0x82, 0x34, 0x9D, 0x35, 0x82, 0x34, 0x9D, 0x36, 0x82, 0x34, 0x9D, 0x37,
    0x82, 0x34, 0x9D, 0x38, 0x82, 0x34, 0x9D, 0x39, 0x82, 0x34, 0x9E, 0x30, 0x82, 0x34, 0x9E, 0x31,
    0x82, 0x34, 0x9E, 0x32, 0x82, 0x34, 0x9E, 0x33, 0x82, 0x34, 0x9E, 0x34, 0x82, 0x34, 0x9E, 0x35,
    0x82, 0x34, 0x9E, 0x36, 0x82, 0x34, 0x9E, 0x37, 0x82, 0x34, 0x9E, 0x38, 0x82, 0x34, 0x9E, 0x39,
    0x82, 0x34, 0xA3, 0x31, 0x82, 0x34, 0xA3, 0x32, 0x82, 0x34, 0xA3, 0x33, 0x82, 0x34, 0xA3, 0x34,
    0x82, 0x34, 0xA3, 0x35, 0x82, 0x34, 0xA3, 0x36, 0x82, 0x34, 0xA3, 0x37, 0x82, 0x34, 0xA3, 0x38,
    0x82, 0x34, 0xA3, 0x39, 0x82, 0x34, 0xA4, 0x30, 0x82, 0x34, 0xA4, 0x31, 0x82, 0x34, 0xA4, 0x32,
    0x82, 0x34, 0xA4, 0x33, 0x82, 0x34, 0xA4, 0x34, 0x82, 0x34, 0xA4, 0x35, 0x82, 0x34, 0xA4, 0x36,
    0x82, 0x34, 0xA4, 0x37, 0x82, 0x34, 0xA4, 0x38, 0x82, 0x34, 0xA4, 0x39, 0x82, 0x34, 0xA5, 0x30,
    0x82, 0x34, 0xA5, 0x31, 0x82, 0x34, 0xA5, 0x32, 0x82, 0x34, 0xA5, 0x33, 0x82, 0x34, 0xA5, 0x34,
    0x82, 0x34, 0xA5, 0x35, 0x82, 0x34, 0xA5, 0x36, 0x82, 0x34, 0xA5, 0x37, 0x82, 0x34, 0xA5, 0x38,
    0x82, 0x34, 0xA5, 0x39, 0x82, 0x34, 0xA6, 0x30, 0x82, 0x34, 0xA6, 0x31, 0x82, 0x34, 0xA6, 0x32,
    0x82, 0x34, 0xA6, 0x33, 0x82, 0x34, 0xA6, 0x34, 0x82, 0x34, 0xA6, 0x35, 0x82, 0x34, 0xA6, 0x36,
    0x82, 0x34, 0xA6, 0x37, 0x82, 0x34, 0xA6, 0x38, 0x82, 0x34, 0xA6, 0x39, 0x82, 0x34, 0xA7, 0x30,
    0x82, 0x34, 0xA7, 0x31, 0x82, 0x34, 0xA7, 0x32, 0x82, 0x34, 0xA7, 0x33, 0x82, 0x34, 0xA7, 0x34,
    0x82, 0x34, 0xA7, 0x35, 0x82, 0x34, 0xA7, 0x36, 0x82, 0x34, 0xA7, 0x37, 0x82, 0x34, 0xA7, 0x38,
    0x82, 0x34, 0xA7, 0x39, 0x82, 0x34, 0xA8, 0x30, 0x82, 0x34, 0xA8, 0x31, 0x82, 0x34, 0xA8, 0x32,
    0x82, 0x34, 0xA8, 0x33, 0x82, 0x34, 0xA8, 0x34, 0x82, 0x34, 0xA8, 0x35, 0x82, 0x34, 0xA8, 0x36,
    0x82, 0x34, 0xA8, 0x37, 0x82, 0x34, 0xA8, 0x38, 0x82, 0x34, 0xA8, 0x39, 0x82, 0x34, 0xA9, 0x30,
    0x82, 0x34, 0xA9, 0x31, 0x82, 0x34, 0xA9, 0x32, 0x82, 0x34, 0xA9, 0x33, 0x82, 0x34, 0xA9, 0x34
};

// when used as part of the GB18030 encoding, character A4 (interpunct)
// is mapped to a four-byte sequence (0x8139a739) when converting back from UTF-8
unsigned char gb2312_row_01_converted_from_utf8_gb18030_alternate[] = {
  0xa1, 0xa1, 0xa1, 0xa2, 0xa1, 0xa3, 0x81, 0x39, 0xa7, 0x39, 0xa1, 0xa5,
  0xa1, 0xa6, 0xa1, 0xa7, 0xa1, 0xa8, 0xa1, 0xa9, 0xa8, 0x44, 0xa1, 0xab,
  0xa1, 0xac, 0xa1, 0xad, 0xa1, 0xae, 0xa1, 0xaf, 0xa1, 0xb0, 0xa1, 0xb1,
  0xa1, 0xb2, 0xa1, 0xb3, 0xa1, 0xb4, 0xa1, 0xb5, 0xa1, 0xb6, 0xa1, 0xb7,
  0xa1, 0xb8, 0xa1, 0xb9, 0xa1, 0xba, 0xa1, 0xbb, 0xa1, 0xbc, 0xa1, 0xbd,
  0xa1, 0xbe, 0xa1, 0xbf, 0xa1, 0xc0, 0xa1, 0xc1, 0xa1, 0xc2, 0xa1, 0xc3,
  0xa1, 0xc4, 0xa1, 0xc5, 0xa1, 0xc6, 0xa1, 0xc7, 0xa1, 0xc8, 0xa1, 0xc9,
  0xa1, 0xca, 0xa1, 0xcb, 0xa1, 0xcc, 0xa1, 0xcd, 0xa1, 0xce, 0xa1, 0xcf,
  0xa1, 0xd0, 0xa1, 0xd1, 0xa1, 0xd2, 0xa1, 0xd3, 0xa1, 0xd4, 0xa1, 0xd5,
  0xa1, 0xd6, 0xa1, 0xd7, 0xa1, 0xd8, 0xa1, 0xd9, 0xa1, 0xda, 0xa1, 0xdb,
  0xa1, 0xdc, 0xa1, 0xdd, 0xa1, 0xde, 0xa1, 0xdf, 0xa1, 0xe0, 0xa1, 0xe1,
  0xa1, 0xe2, 0xa1, 0xe3, 0xa1, 0xe4, 0xa1, 0xe5, 0xa1, 0xe6, 0xa1, 0xe7,
  0xa1, 0xe8, 0xa1, 0xe9, 0xa1, 0xea, 0xa1, 0xeb, 0xa1, 0xec, 0xa1, 0xed,
  0xa1, 0xee, 0xa1, 0xef, 0xa1, 0xf0, 0xa1, 0xf1, 0xa1, 0xf2, 0xa1, 0xf3,
  0xa1, 0xf4, 0xa1, 0xf5, 0xa1, 0xf6, 0xa1, 0xf7, 0xa1, 0xf8, 0xa1, 0xf9,
  0xa1, 0xfa, 0xa1, 0xfb, 0xa1, 0xfc, 0xa1, 0xfd, 0xa1, 0xfe
};

unsigned char gb18030_latin_converted_to_utf8[] = {
  0xc3, 0x80, 0xc3, 0x81, 0xc3, 0x82, 0xc3, 0x83, 0xc3, 0x84, 0xc3, 0x85,
  0xc3, 0x86, 0xc3, 0x87, 0xc3, 0x88, 0xc3, 0x89, 0xc3, 0x8a, 0xc3, 0x8b,
  0xc3, 0x8c, 0xc3, 0x8d, 0xc3, 0x8e, 0xc3, 0x8f, 0xc3, 0x90, 0xc3, 0x91,
  0xc3, 0x92, 0xc3, 0x93, 0xc3, 0x94, 0xc3, 0x95, 0xc3, 0x96, 0xc3, 0x98,
  0xc3, 0x99, 0xc3, 0x9a, 0xc3, 0x9b, 0xc3, 0x9c, 0xc3, 0x9d, 0xc3, 0x9e,
  0xc3, 0x9f, 0xc3, 0xa2, 0xc3, 0xa3, 0xc3, 0xa4, 0xc3, 0xa5, 0xc3, 0xa6,
  0xc3, 0xa7, 0xc3, 0xab, 0xc3, 0xae, 0xc3, 0xaf, 0xc3, 0xb0, 0xc3, 0xb1,
  0xc3, 0xb4, 0xc3, 0xb5, 0xc3, 0xb6, 0xc3, 0xb8, 0xc3, 0xbb, 0xc3, 0xbd,
  0xc3, 0xbe, 0xc3, 0xbf, 0xc4, 0x80, 0xc4, 0x82, 0xc4, 0x83, 0xc4, 0x84,
  0xc4, 0x85, 0xc4, 0x86, 0xc4, 0x87, 0xc4, 0x88, 0xc4, 0x89, 0xc4, 0x8a,
  0xc4, 0x8b, 0xc4, 0x8c, 0xc4, 0x8d, 0xc4, 0x8e
};

unsigned char gb18030_cyrillic_converted_to_utf8[] = {
  0xd2, 0x8a, 0xd2, 0x8b, 0xd2, 0x8c, 0xd2, 0x8d, 0xd2, 0x8e, 0xd2, 0x8f,
  0xd2, 0x90, 0xd2, 0x91, 0xd2, 0x92, 0xd2, 0x93, 0xd2, 0x94, 0xd2, 0x95,
  0xd2, 0x96, 0xd2, 0x97, 0xd2, 0x98, 0xd2, 0x99, 0xd2, 0x9a, 0xd2, 0x9b,
  0xd2, 0x9c, 0xd2, 0x9d, 0xd2, 0x9e, 0xd2, 0x9f, 0xd2, 0xa0, 0xd2, 0xa1,
  0xd2, 0xa2, 0xd2, 0xa3, 0xd2, 0xa4, 0xd2, 0xa5, 0xd2, 0xa6, 0xd2, 0xa7,
  0xd2, 0xa8, 0xd2, 0xa9, 0xd2, 0xaa, 0xd2, 0xab, 0xd2, 0xac, 0xd2, 0xad,
  0xd2, 0xae, 0xd2, 0xaf, 0xd2, 0xb0, 0xd2, 0xb1, 0xd2, 0xb2, 0xd2, 0xb3,
  0xd2, 0xb4, 0xd2, 0xb5, 0xd2, 0xb6, 0xd2, 0xb7, 0xd2, 0xb8, 0xd2, 0xb9,
  0xd2, 0xba, 0xd2, 0xbb, 0xd2, 0xbc, 0xd2, 0xbd, 0xd2, 0xbe, 0xd2, 0xbf,
  0xd3, 0x80, 0xd3, 0x81, 0xd3, 0x82, 0xd3, 0x83, 0xd3, 0x84, 0xd3, 0x85,
  0xd3, 0x86, 0xd3, 0x87, 0xd3, 0x88, 0xd3, 0x89
};

unsigned char gb18030_armenian_converted_to_utf8[] = {
  0xd5, 0xa1, 0xd5, 0xa2, 0xd5, 0xa3, 0xd5, 0xa4, 0xd5, 0xa5, 0xd5, 0xa6,
  0xd5, 0xa7, 0xd5, 0xa8, 0xd5, 0xa9, 0xd5, 0xaa, 0xd5, 0xab, 0xd5, 0xac,
  0xd5, 0xad, 0xd5, 0xae, 0xd5, 0xaf, 0xd5, 0xb0, 0xd5, 0xb1, 0xd5, 0xb2,
  0xd5, 0xb3, 0xd5, 0xb4, 0xd5, 0xb5, 0xd5, 0xb6, 0xd5, 0xb7, 0xd5, 0xb8,
  0xd5, 0xb9, 0xd5, 0xba, 0xd5, 0xbb, 0xd5, 0xbc, 0xd5, 0xbd, 0xd5, 0xbe,
  0xd5, 0xbf, 0xd6, 0x80
};

unsigned char gb18030_devanagari_converted_to_utf8[] = {
  0xe0, 0xa4, 0x84, 0xe0, 0xa4, 0x85, 0xe0, 0xa4, 0x86, 0xe0, 0xa4, 0x87,
  0xe0, 0xa4, 0x88, 0xe0, 0xa4, 0x89, 0xe0, 0xa4, 0x8a, 0xe0, 0xa4, 0x8b,
  0xe0, 0xa4, 0x8c, 0xe0, 0xa4, 0x8d, 0xe0, 0xa4, 0x8e, 0xe0, 0xa4, 0x8f,
  0xe0, 0xa4, 0x90, 0xe0, 0xa4, 0x91, 0xe0, 0xa4, 0x92, 0xe0, 0xa4, 0x93,
  0xe0, 0xa4, 0x94, 0xe0, 0xa4, 0x95, 0xe0, 0xa4, 0x96, 0xe0, 0xa4, 0x97,
  0xe0, 0xa4, 0x98, 0xe0, 0xa4, 0x99, 0xe0, 0xa4, 0x9a, 0xe0, 0xa4, 0x9b,
  0xe0, 0xa4, 0x9c, 0xe0, 0xa4, 0x9d, 0xe0, 0xa4, 0x9e, 0xe0, 0xa4, 0x9f,
  0xe0, 0xa4, 0xa0, 0xe0, 0xa4, 0xa1, 0xe0, 0xa4, 0xa2, 0xe0, 0xa4, 0xa3,
  0xe0, 0xa4, 0xa4, 0xe0, 0xa4, 0xa5, 0xe0, 0xa4, 0xa6, 0xe0, 0xa4, 0xa7,
  0xe0, 0xa4, 0xa8, 0xe0, 0xa4, 0xa9, 0xe0, 0xa4, 0xaa, 0xe0, 0xa4, 0xab,
  0xe0, 0xa4, 0xac, 0xe0, 0xa4, 0xad, 0xe0, 0xa4, 0xae, 0xe0, 0xa4, 0xaf,
  0xe0, 0xa4, 0xb0, 0xe0, 0xa4, 0xb1, 0xe0, 0xa4, 0xb2, 0xe0, 0xa4, 0xb3,
  0xe0, 0xa4, 0xb4, 0xe0, 0xa4, 0xb5, 0xe0, 0xa4, 0xb6, 0xe0, 0xa4, 0xb7,
  0xe0, 0xa4, 0xb8, 0xe0, 0xa4, 0xb9, 0xe0, 0xa4, 0xba, 0xe0, 0xa4, 0xbb,
  0xe0, 0xa4, 0xbc, 0xe0, 0xa4, 0xbd
};

unsigned char gb18030_ethiopic_converted_to_utf8[] = {
  0xe1, 0x88, 0x80, 0xe1, 0x88, 0x81, 0xe1, 0x88, 0x82, 0xe1, 0x88, 0x83,
  0xe1, 0x88, 0x84, 0xe1, 0x88, 0x85, 0xe1, 0x88, 0x86, 0xe1, 0x88, 0x87,
  0xe1, 0x88, 0x88, 0xe1, 0x88, 0x89, 0xe1, 0x88, 0x8a, 0xe1, 0x88, 0x8b,
  0xe1, 0x88, 0x8c, 0xe1, 0x88, 0x8d, 0xe1, 0x88, 0x8e, 0xe1, 0x88, 0x8f,
  0xe1, 0x88, 0x90, 0xe1, 0x88, 0x91, 0xe1, 0x88, 0x92, 0xe1, 0x88, 0x93,
  0xe1, 0x88, 0x94, 0xe1, 0x88, 0x95, 0xe1, 0x88, 0x96, 0xe1, 0x88, 0x97,
  0xe1, 0x88, 0x98, 0xe1, 0x88, 0x99, 0xe1, 0x88, 0x9a, 0xe1, 0x88, 0x9b,
  0xe1, 0x88, 0x9c, 0xe1, 0x88, 0x9d, 0xe1, 0x88, 0x9e, 0xe1, 0x88, 0x9f,
  0xe1, 0x88, 0xa0, 0xe1, 0x88, 0xa1, 0xe1, 0x88, 0xa2, 0xe1, 0x88, 0xa3,
  0xe1, 0x88, 0xa4, 0xe1, 0x88, 0xa5, 0xe1, 0x88, 0xa6, 0xe1, 0x88, 0xa7,
  0xe1, 0x88, 0xa8, 0xe1, 0x88, 0xa9, 0xe1, 0x88, 0xaa, 0xe1, 0x88, 0xab,
  0xe1, 0x88, 0xac, 0xe1, 0x88, 0xad, 0xe1, 0x88, 0xae, 0xe1, 0x88, 0xaf,
  0xe1, 0x88, 0xb0, 0xe1, 0x88, 0xb1, 0xe1, 0x88, 0xb2, 0xe1, 0x88, 0xb3,
  0xe1, 0x88, 0xb4, 0xe1, 0x88, 0xb5, 0xe1, 0x88, 0xb6, 0xe1, 0x88, 0xb7,
  0xe1, 0x88, 0xb8, 0xe1, 0x88, 0xb9, 0xe1, 0x88, 0xba, 0xe1, 0x88, 0xbb,
  0xe1, 0x88, 0xbc, 0xe1, 0x88, 0xbd, 0xe1, 0x88, 0xbe, 0xe1, 0x88, 0xbf
};

unsigned char gb18030_braille_converted_to_utf8[] = {
  0xe2, 0xa0, 0x81, 0xe2, 0xa0, 0x82, 0xe2, 0xa0, 0x83, 0xe2, 0xa0, 0x84,
  0xe2, 0xa0, 0x85, 0xe2, 0xa0, 0x86, 0xe2, 0xa0, 0x87, 0xe2, 0xa0, 0x88,
  0xe2, 0xa0, 0x89, 0xe2, 0xa0, 0x8a, 0xe2, 0xa0, 0x8b, 0xe2, 0xa0, 0x8c,
  0xe2, 0xa0, 0x8d, 0xe2, 0xa0, 0x8e, 0xe2, 0xa0, 0x8f, 0xe2, 0xa0, 0x90,
  0xe2, 0xa0, 0x91, 0xe2, 0xa0, 0x92, 0xe2, 0xa0, 0x93, 0xe2, 0xa0, 0x94,
  0xe2, 0xa0, 0x95, 0xe2, 0xa0, 0x96, 0xe2, 0xa0, 0x97, 0xe2, 0xa0, 0x98,
  0xe2, 0xa0, 0x99, 0xe2, 0xa0, 0x9a, 0xe2, 0xa0, 0x9b, 0xe2, 0xa0, 0x9c,
  0xe2, 0xa0, 0x9d, 0xe2, 0xa0, 0x9e, 0xe2, 0xa0, 0x9f, 0xe2, 0xa0, 0xa0,
  0xe2, 0xa0, 0xa1, 0xe2, 0xa0, 0xa2, 0xe2, 0xa0, 0xa3, 0xe2, 0xa0, 0xa4,
  0xe2, 0xa0, 0xa5, 0xe2, 0xa0, 0xa6, 0xe2, 0xa0, 0xa7, 0xe2, 0xa0, 0xa8,
  0xe2, 0xa0, 0xa9, 0xe2, 0xa0, 0xaa, 0xe2, 0xa0, 0xab, 0xe2, 0xa0, 0xac,
  0xe2, 0xa0, 0xad, 0xe2, 0xa0, 0xae, 0xe2, 0xa0, 0xaf, 0xe2, 0xa0, 0xb0,
  0xe2, 0xa0, 0xb1, 0xe2, 0xa0, 0xb2, 0xe2, 0xa0, 0xb3, 0xe2, 0xa0, 0xb4,
  0xe2, 0xa0, 0xb5, 0xe2, 0xa0, 0xb6, 0xe2, 0xa0, 0xb7, 0xe2, 0xa0, 0xb8,
  0xe2, 0xa0, 0xb9, 0xe2, 0xa0, 0xba, 0xe2, 0xa0, 0xbb, 0xe2, 0xa0, 0xbc,
  0xe2, 0xa0, 0xbd, 0xe2, 0xa0, 0xbe, 0xe2, 0xa0, 0xbf, 0xe2, 0xa1, 0x80
};

unsigned char gb18030_chinese_converted_to_utf8[] = {
  0xe3, 0x91, 0x9e, 0xe3, 0x91, 0x9f, 0xe3, 0x91, 0xa0, 0xe3, 0x91, 0xa1,
  0xe3, 0x91, 0xa2, 0xe3, 0x91, 0xa3, 0xe3, 0x91, 0xa4, 0xe3, 0x91, 0xa5,
  0xe3, 0x91, 0xa6, 0xe3, 0x91, 0xa7, 0xe3, 0x91, 0xa8, 0xe3, 0x91, 0xa9,
  0xe3, 0x91, 0xaa, 0xe3, 0x91, 0xab, 0xe3, 0x91, 0xac, 0xe3, 0x91, 0xad,
  0xe3, 0x91, 0xae, 0xe3, 0x91, 0xaf, 0xe3, 0x91, 0xb0, 0xe3, 0x91, 0xb1,
  0xe3, 0x91, 0xb2, 0xe3, 0x91, 0xb4, 0xe3, 0x91, 0xb5, 0xe3, 0x91, 0xb6,
  0xe3, 0x91, 0xb7, 0xe3, 0x91, 0xb8, 0xe3, 0x91, 0xb9, 0xe3, 0x91, 0xba,
  0xe3, 0x91, 0xbb, 0xe3, 0x91, 0xbc, 0xe3, 0x91, 0xbd, 0xe3, 0x91, 0xbe,
  0xe3, 0x91, 0xbf, 0xe3, 0x92, 0x80, 0xe3, 0x92, 0x81, 0xe3, 0x92, 0x82,
  0xe3, 0x92, 0x83, 0xe3, 0x92, 0x84, 0xe3, 0x92, 0x85, 0xe3, 0x92, 0x86,
  0xe3, 0x92, 0x87, 0xe3, 0x92, 0x88, 0xe3, 0x92, 0x89, 0xe3, 0x92, 0x8a,
  0xe3, 0x92, 0x8b, 0xe3, 0x92, 0x8c, 0xe3, 0x92, 0x8d, 0xe3, 0x92, 0x8e,
  0xe3, 0x92, 0x8f, 0xe3, 0x92, 0x90, 0xe3, 0x92, 0x91, 0xe3, 0x92, 0x92,
  0xe3, 0x92, 0x93, 0xe3, 0x92, 0x94, 0xe3, 0x92, 0x95, 0xe3, 0x92, 0x96,
  0xe3, 0x92, 0x97, 0xe3, 0x92, 0x98, 0xe3, 0x92, 0x99, 0xe3, 0x92, 0x9a,
  0xe3, 0x92, 0x9b, 0xe3, 0x92, 0x9c, 0xe3, 0x92, 0x9d, 0xe3, 0x92, 0x9e,
  0xe3, 0x93, 0x88, 0xe3, 0x93, 0x89, 0xe3, 0x93, 0x8a, 0xe3, 0x93, 0x8b,
  0xe3, 0x93, 0x8c, 0xe3, 0x93, 0x8d, 0xe3, 0x93, 0x8e, 0xe3, 0x93, 0x8f,
  0xe3, 0x93, 0x90, 0xe3, 0x93, 0x91, 0xe3, 0x93, 0x92, 0xe3, 0x93, 0x93,
  0xe3, 0x93, 0x94, 0xe3, 0x93, 0x95, 0xe3, 0x93, 0x96, 0xe3, 0x93, 0x97,
  0xe3, 0x93, 0x98, 0xe3, 0x93, 0x99, 0xe3, 0x93, 0x9a, 0xe3, 0x93, 0x9b,
  0xe3, 0x93, 0x9c, 0xe3, 0x93, 0x9d, 0xe3, 0x93, 0x9e, 0xe3, 0x93, 0x9f,
  0xe3, 0x93, 0xa0, 0xe3, 0x93, 0xa1, 0xe3, 0x93, 0xa2, 0xe3, 0x93, 0xa3,
  0xe3, 0x93, 0xa4, 0xe3, 0x93, 0xa5, 0xe3, 0x93, 0xa6, 0xe3, 0x93, 0xa7,
  0xe3, 0x93, 0xa8, 0xe3, 0x93, 0xa9, 0xe3, 0x93, 0xaa, 0xe3, 0x93, 0xab,
  0xe3, 0x93, 0xac, 0xe3, 0x93, 0xad, 0xe3, 0x93, 0xae, 0xe3, 0x93, 0xaf,
  0xe3, 0x93, 0xb0, 0xe3, 0x93, 0xb1, 0xe3, 0x93, 0xb2, 0xe3, 0x93, 0xb3,
  0xe3, 0x93, 0xb4, 0xe3, 0x93, 0xb5, 0xe3, 0x93, 0xb6, 0xe3, 0x93, 0xb7,
  0xe3, 0x93, 0xb8, 0xe3, 0x93, 0xb9, 0xe3, 0x93, 0xba, 0xe3, 0x93, 0xbb,
  0xe3, 0x93, 0xbc, 0xe3, 0x93, 0xbd, 0xe3, 0x93, 0xbe, 0xe3, 0x93, 0xbf,
  0xe3, 0x94, 0x80, 0xe3, 0x94, 0x81, 0xe3, 0x94, 0x82, 0xe3, 0x94, 0x83,
  0xe3, 0x94, 0x84, 0xe3, 0x94, 0x85, 0xe3, 0x94, 0x86, 0xe3, 0x94, 0x87,
  0xe3, 0x94, 0xb0, 0xe3, 0x94, 0xb1, 0xe3, 0x94, 0xb2, 0xe3, 0x94, 0xb3,
  0xe3, 0x94, 0xb4, 0xe3, 0x94, 0xb5, 0xe3, 0x94, 0xb6, 0xe3, 0x94, 0xb7,
  0xe3, 0x94, 0xb8, 0xe3, 0x94, 0xb9, 0xe3, 0x94, 0xba, 0xe3, 0x94, 0xbb,
  0xe3, 0x94, 0xbc, 0xe3, 0x94, 0xbd, 0xe3, 0x94, 0xbe, 0xe3, 0x94, 0xbf,
  0xe3, 0x95, 0x80, 0xe3, 0x95, 0x81, 0xe3, 0x95, 0x82, 0xe3, 0x95, 0x83,
  0xe3, 0x95, 0x84, 0xe3, 0x95, 0x85, 0xe3, 0x95, 0x86, 0xe3, 0x95, 0x87,
  0xe3, 0x95, 0x88, 0xe3, 0x95, 0x89, 0xe3, 0x95, 0x8a, 0xe3, 0x95, 0x8b,
  0xe3, 0x95, 0x8c, 0xe3, 0x95, 0x8d, 0xe3, 0x95, 0x8e, 0xe3, 0x95, 0x8f,
  0xe3, 0x95, 0x90, 0xe3, 0x95, 0x91, 0xe3, 0x95, 0x92, 0xe3, 0x95, 0x93,
  0xe3, 0x95, 0x94, 0xe3, 0x95, 0x95, 0xe3, 0x95, 0x96, 0xe3, 0x95, 0x97,
  0xe3, 0x95, 0x98, 0xe3, 0x95, 0x99, 0xe3, 0x95, 0x9a, 0xe3, 0x95, 0x9b,
  0xe3, 0x95, 0x9c, 0xe3, 0x95, 0x9d, 0xe3, 0x95, 0x9e, 0xe3, 0x95, 0x9f,
  0xe3, 0x95, 0xa0, 0xe3, 0x95, 0xa1, 0xe3, 0x95, 0xa2, 0xe3, 0x95, 0xa3,
  0xe3, 0x95, 0xa4, 0xe3, 0x95, 0xa5, 0xe3, 0x95, 0xa6, 0xe3, 0x95, 0xa7,
  0xe3, 0x95, 0xa8, 0xe3, 0x95, 0xa9, 0xe3, 0x95, 0xaa, 0xe3, 0x95, 0xab,
  0xe3, 0x95, 0xac, 0xe3, 0x95, 0xad, 0xe3, 0x95, 0xae, 0xe3, 0x95, 0xaf,
  0xe3, 0x96, 0x99, 0xe3, 0x96, 0x9a, 0xe3, 0x96, 0x9b, 0xe3, 0x96, 0x9c,
  0xe3, 0x96, 0x9d, 0xe3, 0x96, 0x9f, 0xe3, 0x96, 0xa0, 0xe3, 0x96, 0xa1,
  0xe3, 0x96, 0xa2, 0xe3, 0x96, 0xa3, 0xe3, 0x96, 0xa4, 0xe3, 0x96, 0xa5,
  0xe3, 0x96, 0xa6, 0xe3, 0x96, 0xa7, 0xe3, 0x96, 0xa8, 0xe3, 0x96, 0xa9,
  0xe3, 0x96, 0xaa, 0xe3, 0x96, 0xab, 0xe3, 0x96, 0xac, 0xe3, 0x96, 0xad,
  0xe3, 0x96, 0xae, 0xe3, 0x96, 0xaf, 0xe3, 0x96, 0xb0, 0xe3, 0x96, 0xb1,
  0xe3, 0x96, 0xb2, 0xe3, 0x96, 0xb3, 0xe3, 0x96, 0xb4, 0xe3, 0x96, 0xb5,
  0xe3, 0x96, 0xb6, 0xe3, 0x96, 0xb7, 0xe3, 0x96, 0xb8, 0xe3, 0x96, 0xb9,
  0xe3, 0x96, 0xba, 0xe3, 0x96, 0xbb, 0xe3, 0x96, 0xbc, 0xe3, 0x96, 0xbd,
  0xe3, 0x96, 0xbe, 0xe3, 0x96, 0xbf, 0xe3, 0x97, 0x80, 0xe3, 0x97, 0x81,
  0xe3, 0x97, 0x82, 0xe3, 0x97, 0x83, 0xe3, 0x97, 0x84, 0xe3, 0x97, 0x85,
  0xe3, 0x97, 0x86, 0xe3, 0x97, 0x87, 0xe3, 0x97, 0x88, 0xe3, 0x97, 0x89,
  0xe3, 0x97, 0x8a, 0xe3, 0x97, 0x8b, 0xe3, 0x97, 0x8c, 0xe3, 0x97, 0x8d,
  0xe3, 0x97, 0x8e, 0xe3, 0x97, 0x8f, 0xe3, 0x97, 0x90, 0xe3, 0x97, 0x91,
  0xe3, 0x97, 0x92, 0xe3, 0x97, 0x93, 0xe3, 0x97, 0x94, 0xe3, 0x97, 0x95,
  0xe3, 0x97, 0x96, 0xe3, 0x97, 0x97, 0xe3, 0x97, 0x98, 0xe3, 0x97, 0x99,
  0xe4, 0xa2, 0x86, 0xe4, 0xa2, 0x87, 0xe4, 0xa2, 0x88, 0xe4, 0xa2, 0x89,
  0xe4, 0xa2, 0x8a, 0xe4, 0xa2, 0x8b, 0xe4, 0xa2, 0x8c, 0xe4, 0xa2, 0x8d,
  0xe4, 0xa2, 0x8e, 0xe4, 0xa2, 0x8f, 0xe4, 0xa2, 0x90, 0xe4, 0xa2, 0x91,
  0xe4, 0xa2, 0x92, 0xe4, 0xa2, 0x93, 0xe4, 0xa2, 0x94, 0xe4, 0xa2, 0x95,
  0xe4, 0xa2, 0x96, 0xe4, 0xa2, 0x97, 0xe4, 0xa2, 0x98, 0xe4, 0xa2, 0x99,
  0xe4, 0xa2, 0x9a, 0xe4, 0xa2, 0x9b, 0xe4, 0xa2, 0x9c, 0xe4, 0xa2, 0x9d,
  0xe4, 0xa2, 0x9e, 0xe4, 0xa2, 0x9f, 0xe4, 0xa2, 0xa0, 0xe4, 0xa2, 0xa1,
  0xe4, 0xa2, 0xa2, 0xe4, 0xa2, 0xa3, 0xe4, 0xa2, 0xa4, 0xe4, 0xa2, 0xa5,
  0xe4, 0xa2, 0xa6, 0xe4, 0xa2, 0xa7, 0xe4, 0xa2, 0xa8, 0xe4, 0xa2, 0xa9,
  0xe4, 0xa2, 0xaa, 0xe4, 0xa2, 0xab, 0xe4, 0xa2, 0xac, 0xe4, 0xa2, 0xad,
  0xe4, 0xa2, 0xae, 0xe4, 0xa2, 0xaf, 0xe4, 0xa2, 0xb0, 0xe4, 0xa2, 0xb1,
  0xe4, 0xa2, 0xb2, 0xe4, 0xa2, 0xb3, 0xe4, 0xa2, 0xb4, 0xe4, 0xa2, 0xb5,
  0xe4, 0xa2, 0xb6, 0xe4, 0xa2, 0xb7, 0xe4, 0xa2, 0xb8, 0xe4, 0xa2, 0xb9,
  0xe4, 0xa2, 0xba, 0xe4, 0xa2, 0xbb, 0xe4, 0xa2, 0xbc, 0xe4, 0xa2, 0xbd,
  0xe4, 0xa2, 0xbe, 0xe4, 0xa2, 0xbf, 0xe4, 0xa3, 0x80, 0xe4, 0xa3, 0x81,
  0xe4, 0xa3, 0x82, 0xe4, 0xa3, 0x83, 0xe4, 0xa3, 0x84, 0xe4, 0xa3, 0x85,
  0xe4, 0xa3, 0xaf, 0xe4, 0xa3, 0xb0, 0xe4, 0xa3, 0xb1, 0xe4, 0xa3, 0xb2,
  0xe4, 0xa3, 0xb3, 0xe4, 0xa3, 0xb4, 0xe4, 0xa3, 0xb5, 0xe4, 0xa3, 0xb6,
  0xe4, 0xa3, 0xb7, 0xe4, 0xa3, 0xb8, 0xe4, 0xa3, 0xb9, 0xe4, 0xa3, 0xba,
  0xe4, 0xa3, 0xbb, 0xe4, 0xa3, 0xbc, 0xe4, 0xa3, 0xbd, 0xe4, 0xa3, 0xbe,
  0xe4, 0xa3, 0xbf, 0xe4, 0xa4, 0x80, 0xe4, 0xa4, 0x81, 0xe4, 0xa4, 0x82,
  0xe4, 0xa4, 0x83, 0xe4, 0xa4, 0x84, 0xe4, 0xa4, 0x85, 0xe4, 0xa4, 0x86,
  0xe4, 0xa4, 0x87, 0xe4, 0xa4, 0x88, 0xe4, 0xa4, 0x89, 0xe4, 0xa4, 0x8a,
  0xe4, 0xa4, 0x8b, 0xe4, 0xa4, 0x8c, 0xe4, 0xa4, 0x8d, 0xe4, 0xa4, 0x8e,
  0xe4, 0xa4, 0x8f, 0xe4, 0xa4, 0x90, 0xe4, 0xa4, 0x91, 0xe4, 0xa4, 0x92,
  0xe4, 0xa4, 0x93, 0xe4, 0xa4, 0x94, 0xe4, 0xa4, 0x95, 0xe4, 0xa4, 0x96,
  0xe4, 0xa4, 0x97, 0xe4, 0xa4, 0x98, 0xe4, 0xa4, 0x99, 0xe4, 0xa4, 0x9a,
  0xe4, 0xa4, 0x9b, 0xe4, 0xa4, 0x9c, 0xe4, 0xa4, 0x9d, 0xe4, 0xa4, 0x9e,
  0xe4, 0xa4, 0x9f, 0xe4, 0xa4, 0xa0, 0xe4, 0xa4, 0xa1, 0xe4, 0xa4, 0xa2,
  0xe4, 0xa4, 0xa3, 0xe4, 0xa4, 0xa4, 0xe4, 0xa4, 0xa5, 0xe4, 0xa4, 0xa6,
  0xe4, 0xa4, 0xa7, 0xe4, 0xa4, 0xa8, 0xe4, 0xa4, 0xa9, 0xe4, 0xa4, 0xaa,
  0xe4, 0xa4, 0xab, 0xe4, 0xa4, 0xac, 0xe4, 0xa4, 0xad, 0xe4, 0xa4, 0xae,
  0xe4, 0xa5, 0x99, 0xe4, 0xa5, 0x9a, 0xe4, 0xa5, 0x9b, 0xe4, 0xa5, 0x9c,
  0xe4, 0xa5, 0x9d, 0xe4, 0xa5, 0x9e, 0xe4, 0xa5, 0x9f, 0xe4, 0xa5, 0xa0,
  0xe4, 0xa5, 0xa1, 0xe4, 0xa5, 0xa2, 0xe4, 0xa5, 0xa3, 0xe4, 0xa5, 0xa4,
  0xe4, 0xa5, 0xa5, 0xe4, 0xa5, 0xa6, 0xe4, 0xa5, 0xa7, 0xe4, 0xa5, 0xa8,
  0xe4, 0xa5, 0xa9, 0xe4, 0xa5, 0xaa, 0xe4, 0xa5, 0xab, 0xe4, 0xa5, 0xac,
  0xe4, 0xa5, 0xad, 0xe4, 0xa5, 0xae, 0xe4, 0xa5, 0xaf, 0xe4, 0xa5, 0xb0,
  0xe4, 0xa5, 0xb1, 0xe4, 0xa5, 0xb2, 0xe4, 0xa5, 0xb3, 0xe4, 0xa5, 0xb4,
  0xe4, 0xa5, 0xb5, 0xe4, 0xa5, 0xb6, 0xe4, 0xa5, 0xb7, 0xe4, 0xa5, 0xb8,
  0xe4, 0xa5, 0xb9, 0xe4, 0xa5, 0xbb, 0xe4, 0xa5, 0xbc, 0xe4, 0xa5, 0xbe,
  0xe4, 0xa5, 0xbf, 0xe4, 0xa6, 0x80, 0xe4, 0xa6, 0x81, 0xe4, 0xa6, 0x84,
  0xe4, 0xa6, 0x87, 0xe4, 0xa6, 0x88, 0xe4, 0xa6, 0x89, 0xe4, 0xa6, 0x8a,
  0xe4, 0xa6, 0x8b, 0xe4, 0xa6, 0x8c, 0xe4, 0xa6, 0x8d, 0xe4, 0xa6, 0x8e,
  0xe4, 0xa6, 0x8f, 0xe4, 0xa6, 0x90, 0xe4, 0xa6, 0x91, 0xe4, 0xa6, 0x92,
  0xe4, 0xa6, 0x93, 0xe4, 0xa6, 0x94, 0xe4, 0xa6, 0x95, 0xe4, 0xa6, 0x96,
  0xe4, 0xa6, 0x97, 0xe4, 0xa6, 0x98, 0xe4, 0xa6, 0x99, 0xe4, 0xa6, 0x9a,
  0xe4, 0xa6, 0x9c, 0xe4, 0xa6, 0x9d, 0xe4, 0xa6, 0x9e, 0xe4, 0xa6, 0xa0,
  0xe4, 0xa7, 0x8c, 0xe4, 0xa7, 0x8d, 0xe4, 0xa7, 0x8e, 0xe4, 0xa7, 0x8f,
  0xe4, 0xa7, 0x90, 0xe4, 0xa7, 0x91, 0xe4, 0xa7, 0x92, 0xe4, 0xa7, 0x93,
  0xe4, 0xa7, 0x94, 0xe4, 0xa7, 0x95, 0xe4, 0xa7, 0x96, 0xe4, 0xa7, 0x97,
  0xe4, 0xa7, 0x98, 0xe4, 0xa7, 0x99, 0xe4, 0xa7, 0x9a, 0xe4, 0xa7, 0x9b,
  0xe4, 0xa7, 0x9c, 0xe4, 0xa7, 0x9d, 0xe4, 0xa7, 0x9e, 0xe4, 0xa7, 0x9f,
  0xe4, 0xa7, 0xa0, 0xe4, 0xa7, 0xa1, 0xe4, 0xa7, 0xa2, 0xe4, 0xa7, 0xa3,
  0xe4, 0xa7, 0xa4, 0xe4, 0xa7, 0xa5, 0xe4, 0xa7, 0xa6, 0xe4, 0xa7, 0xa7,
  0xe4, 0xa7, 0xa8, 0xe4, 0xa7, 0xa9, 0xe4, 0xa7, 0xaa, 0xe4, 0xa7, 0xab,
  0xe4, 0xa7, 0xac, 0xe4, 0xa7, 0xad, 0xe4, 0xa7, 0xae, 0xe4, 0xa7, 0xaf,
  0xe4, 0xa7, 0xb0, 0xe4, 0xa7, 0xb1, 0xe4, 0xa7, 0xb2, 0xe4, 0xa7, 0xb3,
  0xe4, 0xa7, 0xb4, 0xe4, 0xa7, 0xb5, 0xe4, 0xa7, 0xb6, 0xe4, 0xa7, 0xb7,
  0xe4, 0xa7, 0xb8, 0xe4, 0xa7, 0xb9, 0xe4, 0xa7, 0xba, 0xe4, 0xa7, 0xbb,
  0xe4, 0xa7, 0xbc, 0xe4, 0xa7, 0xbd, 0xe4, 0xa7, 0xbe, 0xe4, 0xa7, 0xbf,
  0xe4, 0xa8, 0x80, 0xe4, 0xa8, 0x81, 0xe4, 0xa8, 0x82, 0xe4, 0xa8, 0x83,
  0xe4, 0xa8, 0x84, 0xe4, 0xa8, 0x85, 0xe4, 0xa8, 0x86, 0xe4, 0xa8, 0x87,
  0xe4, 0xa8, 0x88, 0xe4, 0xa8, 0x89, 0xe4, 0xa8, 0x8a, 0xe4, 0xa8, 0x8b
};

#endif

