/*
  CoreLinux++ 
  Copyright (C) 2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   

/** \example examp19.cpp
   This example is to show use of the Mediator and Memento patterns.
   We create three (3) Colleagues:
   
      ListColleague     - Maintains a list of sentences that the user can edit
      SelectColleague   - Sets up the select of the sentence to edit
      EditColleague     - Provides editing for a sentence
      
   When something occurs in a Colleague, it captures the state in a Memento
   which is made part of the Event. The Colleague then calls the Mediator
   action interface with the event.
   
   The flow of traffic is:
   
   A) ListColleague updates the list and calls the mediator with a memento
      that contains the current list --> B
      
   B) SelectColleague is interested in list changes, so it receives an
      event with the current list.
      
   C) The ListMediator invokes SelectColleague to do it's thing, which
      is to display the list and allow for a selection. When a valid
      selection is received, it calls the mediator with a memento containing
      the text of the selection ---> D
      
   D) The EditColleague receives the event and displays the selected string
      with a edit prompt. It calls the mediator with a memento that contains
      the original string, and the replacement string ---> E
      
   E) The ListColleague receives the event and updates it list and A then C.      
            
*/                   


#include <corelinux/Common.hpp>

#if   !defined(__LISTMEDIATOR__HPP)
#include <ListMediator.hpp>
#endif

using namespace corelinux;

#include <iostream>
#include <exception>

//
// In module function prototypes
//

int   main( void );

//
// General Functions 
//

void  handleAssertion( AssertionCref );
void  handleException( ExceptionCref );

//
// Global data
//

//
// Main entry point
//


int main( void )
{

   cout << endl;

   //
   // Practice graceful exception management
   //


   try
   {
      ListMediator   aMediator;

      aMediator.run();
   }

   catch( NullPointerException aException )
   {
      cerr  << "Received NullPointerException!" << endl;
      handleException(aException);
   }
   catch( AssertionRef aAssert )
   {
      handleAssertion(aAssert);
   }
   catch( ExceptionRef aException )
   {
      handleException(aException);
   }
   catch( std::exception & e )
   {
      cerr  << e.what() << endl;
   }
   catch( ... )
   {
      cerr  << "Unknown exception." << endl;
   }

   return 0;               
}

//
// Some utility functions
//

//
// Error handlers
//

void  handleAssertion( AssertionCref aAssert )
{
   cerr << aAssert.getFile() << ":" << aAssert.getLine() << ":" << 
      "Assertion: ";

   if( aAssert.getType() == Assertion::NEVERGETHERE )
   {
      cerr << "NEVER_GET_HERE";
   }
   else
   {
      if( aAssert.getType() == Assertion::REQUIRE )
      {
         cerr  << "REQUIRE";
      }
      else if( aAssert.getType() == Assertion::ENSURE )
      {
         cerr  << "ENSURE";
      }
      else if( aAssert.getType() == Assertion::CHECK )
      {
         cerr  << "CHECK";
      }
      else 
      {
         cerr  << "ASSERT";
      }
      cerr << "( " << aAssert.getWhy() << " )";
   }

   cerr << endl;
}

void  handleException( ExceptionCref aExcp )
{
   cerr << aExcp.getFile() << ":" << aExcp.getLine() << ":" <<
      "Exception: " << aExcp.getWhy() << endl;
}

/*
   Common rcs information do not modify
   $Author: prudhomm $
   $Revision: 1.2 $
   $Date: 2000/08/31 22:49:01 $
   $Locker:  $
*/


