/* ----- encode/653x1541round, derived from supercop/crypto_encode/try.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <ntruprime.h>
#include "ntruprime_test.h"

#define fail ((ok = 0),printf)
static const char *encode_653x1541round_checksums[] = {
  "d2bf70e2ae1f712bcc35b342c7a4b9ca35dd41f3bc48badd81cb089c820a3e8b",
  "a81054ade0c8798b3b305e7ba21c764c63f4ce82a5f318828adc6a99c42fd820",
} ;

static void (*crypto_encode)(unsigned char *,const void *);
#define crypto_encode_STRBYTES ntruprime_encode_653x1541round_STRBYTES
#define crypto_encode_ITEMS ntruprime_encode_653x1541round_ITEMS
#define crypto_encode_ITEMBYTES ntruprime_encode_653x1541round_ITEMBYTES

static void *storage_encode_653x1541round_s;
static unsigned char *test_encode_653x1541round_s;
static void *storage_encode_653x1541round_x;
static unsigned char *test_encode_653x1541round_x;
static void *storage_encode_653x1541round_s2;
static unsigned char *test_encode_653x1541round_s2;
static void *storage_encode_653x1541round_x2;
static unsigned char *test_encode_653x1541round_x2;

#define precomputed_encode_653x1541round_NUM 4

static const unsigned char precomputed_encode_653x1541round_s[precomputed_encode_653x1541round_NUM][crypto_encode_STRBYTES] = {
  {32,99,173,238,170,58,22,225,172,196,119,166,57,45,163,22,161,87,53,6,131,62,90,52,210,178,36,176,12,246,102,97,70,156,106,153,23,186,130,58,169,156,233,127,26,225,124,224,45,181,218,67,171,45,113,191,135,30,21,251,244,236,29,57,43,9,232,145,175,94,89,16,54,4,181,134,181,175,57,159,137,127,51,35,10,50,89,121,25,108,39,201,107,158,41,89,58,16,62,109,196,128,173,59,116,136,182,176,77,84,34,16,223,110,212,107,60,185,220,117,146,123,156,143,103,184,12,55,88,66,196,103,13,157,164,119,166,29,126,41,130,36,144,209,44,171,2,87,103,34,108,133,9,94,211,176,193,30,119,211,41,197,237,53,201,229,191,128,35,47,11,159,88,78,22,32,122,78,85,118,87,225,46,75,168,28,124,117,127,209,208,149,7,21,100,231,153,154,241,31,171,19,21,133,184,218,89,73,7,7,194,163,106,190,53,186,71,128,229,18,216,108,171,221,168,25,132,153,52,183,183,33,185,166,158,82,131,157,227,133,16,142,131,218,232,112,247,141,1,35,130,230,195,169,144,180,54,91,195,121,210,89,31,104,235,174,96,180,200,167,23,83,86,173,81,94,66,102,101,104,83,86,47,32,172,15,96,200,47,18,213,172,223,150,153,200,95,115,220,79,70,79,203,132,51,185,163,107,3,234,190,59,236,51,232,98,178,133,146,68,253,231,167,211,139,203,110,236,119,80,3,172,132,28,78,45,90,128,109,120,30,14,37,231,201,145,66,23,164,95,60,204,27,73,198,5,149,208,149,135,48,184,159,139,29,211,0,129,215,214,125,243,70,50,252,162,96,227,155,226,40,4,50,202,179,217,70,151,126,194,113,8,21,116,100,71,238,142,174,162,244,80,71,171,200,145,98,119,40,59,71,72,83,200,112,109,128,223,122,21,107,68,13,54,171,65,251,71,191,145,98,196,236,91,101,210,196,28,194,138,155,159,133,47,244,66,4,104,232,124,114,168,87,233,202,150,253,49,10,175,210,41,19,227,81,245,193,116,63,150,212,176,85,0,79,13,199,205,211,108,154,41,190,62,176,188,62,161,223,87,152,49,8,216,8,146,63,137,136,235,10,224,78,183,226,186,112,213,152,99,74,184,151,21,173,254,217,215,30,85,159,0,35,190,94,117,67,1,173,222,134,80,233,122,147,170,108,34,82,11,159,177,13,106,162,228,84,248,254,198,77,88,114,89,180,40,248,164,237,79,237,102,190,23,62,175,116,215,63,201,108,60,245,92,40,119,112,220,1,250,104,162,79,183,101,219,56,128,251,239,120,74,230,103,167,48,55,240,20,130,103,52,103,189,251,239,128,25,15,238,30,116,108,61,1,122,184,84,233,224,216,87,33,232,120,17,34,198,61,184,216,252,150,57,52,173,41,83,238,53,77,212,44,207,136,162,46,147,233,41,179,20,0,127,223,49,18,2,171,163,8,105,155,128,223,203,163,20,187,92,243,220,20,229,94,85,247,237,156,23,228,254,128,253,114,130,141,149,246,235,170,148,105,3,91,2,224,185,29,92,126,16,58,201,90,237,63,197,240,105,187,63,228,95,85,76,182,62,25,52,255,222,163,207,85,210,174,202,99,39,171,77,157,143,102,10,10,120,138,66,114,250,135,59,140,181,179,232,91,219,247,73,242,69,40,245,234,85,95,56,86,144,65,75,172,219,28,28,101,211,127,243,49,176,144,161,11,215,13,123,144,20,163,107,228,101,87,99,232,201,0,175,28,198,86,15,106,53,144,52,184,146,56,28,196,13,16,86,51,118,233,201,213,98,235,129,190,56,72,32,192,189,70,110,103,114,48,88,233,224,111,2,154,155,138,141,179,66,253,37,128,112,88,42,80,220,82,15,252,102,109,139,48,171,144,147,93,127,13,165,189,187,137,34,22},
  {211,156,35,196,20,204,225,201,101,97,11,223,156,148,88,117,193,48,222,16,24,205,0,202,231,143,65,74,179,5,217,13,27,219,244,138,142,148,198,142,177,143,231,147,43,200,166,201,202,238,245,148,110,84,115,118,69,11,54,172,195,121,179,175,32,243,135,241,13,38,89,157,88,28,164,176,9,249,27,72,4,21,118,148,134,82,102,110,249,165,214,86,89,139,139,139,251,146,146,246,47,234,222,136,251,25,96,203,40,81,243,35,32,188,100,20,122,20,107,189,46,3,236,160,119,44,175,20,107,19,66,177,64,16,62,185,205,39,161,43,2,0,125,191,236,214,73,126,118,136,102,221,236,10,179,94,51,133,170,77,191,30,54,20,201,136,201,134,195,162,192,242,228,110,242,207,51,184,178,150,223,131,215,71,67,59,240,119,48,65,157,77,49,140,115,86,18,224,192,21,22,12,244,247,32,156,26,45,122,112,190,118,163,226,246,211,108,159,246,131,4,160,79,127,134,19,202,145,32,34,88,171,246,145,16,36,62,16,79,193,247,85,79,23,91,250,91,178,120,170,104,73,133,133,51,10,70,202,146,225,150,157,125,218,157,128,84,180,185,63,119,126,153,160,46,111,208,97,27,50,2,149,210,250,18,242,242,89,226,124,210,67,21,234,98,89,212,83,215,85,63,59,171,201,146,64,173,69,143,76,41,70,198,68,76,46,88,120,217,253,58,140,102,54,90,137,137,105,234,2,242,133,194,180,167,176,255,94,9,121,247,231,30,124,113,62,154,41,162,202,166,70,129,74,110,89,168,36,25,179,206,212,238,251,188,254,188,13,154,110,44,164,147,57,92,167,143,183,231,169,84,198,52,219,89,185,179,48,184,44,113,46,235,83,111,232,242,203,15,117,231,109,162,70,190,234,19,27,35,167,170,163,94,100,16,4,152,193,202,243,226,209,138,223,189,134,242,44,55,184,70,95,232,129,153,147,28,255,44,15,250,251,142,214,241,253,26,123,71,12,38,226,63,53,104,49,235,46,142,185,199,8,45,25,221,25,2,13,76,142,175,165,235,126,10,0,128,211,10,4,0,22,137,181,103,229,169,30,197,145,14,121,212,132,44,101,102,13,53,164,230,111,126,93,235,84,117,245,219,121,84,222,102,216,195,4,30,37,132,144,94,80,185,13,59,37,167,136,129,217,110,210,6,203,183,89,171,53,86,8,44,101,27,169,225,71,61,44,69,121,218,85,125,115,160,135,169,190,201,115,15,142,126,229,58,15,67,82,27,101,116,224,162,108,159,115,153,16,3,107,177,114,95,76,64,160,187,244,61,228,139,211,33,89,14,224,52,71,13,135,168,245,53,129,151,94,193,2,21,156,117,112,12,236,38,124,159,108,92,19,150,80,137,152,119,50,92,172,27,153,22,17,189,226,242,159,179,135,210,99,117,115,21,172,42,149,224,133,210,2,229,218,164,1,79,78,0,9,52,249,171,160,106,240,211,148,47,130,10,151,55,145,141,146,255,89,5,253,231,97,224,2,218,255,107,231,134,30,181,87,67,104,4,72,229,189,221,149,88,7,168,83,200,191,223,180,21,144,147,22,64,90,45,232,88,45,90,26,56,246,66,197,44,108,157,86,0,215,147,65,89,4,115,243,38,227,11,39,206,43,208,53,6,76,150,91,29,57,9,175,247,138,4,177,193,169,125,241,100,69,249,3,25,233,220,235,59,202,130,196,198,85,168,204,5,64,214,190,106,193,21,22,229,250,243,100,200,7,81,11,40,225,242,74,222,68,129,101,241,131,167,224,111,20,11,76,249,16,206,255,97,73,36,53,65,147,126,90,133,48,146,12,254,104,109,89,145,208,141,167,136,105,93,137,162,2,220,186,103,72,253,34,92,141,104,188,8,127,219,185,254,147,23,64,174,207,231,102,146,87,223,190,90,64,27},
  {235,218,92,61,128,86,209,13,124,219,179,113,32,116,249,183,75,245,196,112,222,100,79,123,98,10,228,57,117,229,136,190,246,42,93,234,94,136,31,25,29,175,28,93,245,79,216,231,153,62,177,199,101,29,7,160,189,46,72,248,134,128,45,219,26,187,138,233,100,40,58,236,65,35,160,245,100,171,125,55,223,61,222,86,13,35,41,16,127,50,93,172,69,115,93,154,198,191,86,197,199,177,161,182,39,213,52,130,52,105,158,211,239,20,178,204,228,40,176,3,84,13,179,178,138,16,240,216,253,73,99,223,246,47,165,159,135,79,246,0,84,66,215,57,102,237,38,188,87,178,244,250,241,105,18,70,58,44,240,43,251,142,147,90,178,236,222,122,95,67,100,246,58,233,58,112,123,22,117,39,166,144,189,101,156,47,112,248,66,82,139,104,197,13,89,13,208,215,51,130,248,178,77,182,236,210,78,67,110,133,109,35,186,183,81,174,193,134,24,87,25,3,178,103,239,210,234,169,139,219,193,5,141,208,149,41,117,246,186,225,8,237,137,157,158,17,47,81,229,204,194,134,106,15,92,253,170,140,50,254,132,234,190,140,126,204,255,32,124,83,1,55,101,72,84,253,231,218,13,104,147,150,210,26,84,76,82,183,204,80,29,164,13,55,0,172,70,156,8,218,169,56,122,197,149,124,211,149,221,62,58,25,202,159,197,34,136,22,87,126,145,176,150,164,47,250,175,235,61,101,55,242,234,183,14,140,129,34,133,151,165,176,162,131,147,142,16,74,233,213,49,235,254,65,198,106,190,146,20,100,157,214,96,47,64,155,179,59,182,115,255,26,32,236,102,72,81,145,176,192,124,81,56,34,170,146,162,243,221,7,207,1,165,26,236,67,75,118,37,138,0,171,80,24,176,47,252,121,168,223,143,177,189,68,25,158,96,188,160,72,148,182,164,28,221,72,254,30,37,39,167,21,232,95,29,219,215,7,79,99,172,252,9,253,210,245,41,237,188,178,180,152,107,119,171,161,155,44,45,172,45,177,151,182,204,65,137,103,235,180,44,234,234,179,66,113,115,1,213,7,73,107,188,255,27,248,47,114,227,83,116,72,92,21,14,227,234,111,202,238,127,77,38,61,155,176,77,168,238,93,152,6,119,173,13,109,137,226,31,20,52,90,59,248,37,202,241,65,142,221,79,43,25,223,169,115,9,98,230,85,33,98,85,63,106,180,248,96,128,134,205,122,223,142,181,174,213,153,193,223,86,238,76,120,130,89,39,19,51,80,134,164,126,84,39,198,232,99,205,114,166,171,5,225,127,91,232,151,68,157,196,14,56,77,182,103,22,196,15,150,42,182,252,65,36,19,12,93,238,231,231,249,160,133,141,189,166,10,191,13,237,169,51,82,183,238,227,153,105,129,252,29,124,172,57,181,255,163,59,177,200,202,66,136,161,194,253,119,143,35,214,149,83,215,249,248,68,226,49,90,125,2,138,49,248,196,30,26,216,148,190,219,85,201,225,132,118,230,2,186,70,54,185,171,245,31,7,30,160,48,235,96,223,117,212,52,136,140,86,99,58,149,121,177,217,134,73,233,225,28,103,230,13,95,136,80,110,238,249,167,88,243,24,78,95,150,11,236,244,180,152,149,200,143,21,192,171,228,92,46,65,161,135,75,130,22,70,12,231,109,57,114,170,180,161,23,159,78,99,1,54,201,81,60,39,135,225,2,79,250,47,161,121,80,122,236,149,211,70,50,33,247,33,181,185,3,124,55,45,49,151,241,178,248,128,195,166,29,73,170,129,25,113,200,138,44,48,222,196,145,199,206,75,2,151,152,47,199,65,38,158,14,78,60,42,93,198,118,47,70,53,62,145,212,246,211,183,247,109,43,13,65,243,219,109,163,135,158,210,15,63,71,165,181,157,38,121,30,63,152,101,129,90,199,32},
  {104,221,47,27,49,99,222,26,242,239,241,249,53,141,239,201,177,129,30,196,147,180,67,227,181,208,84,153,90,15,63,119,54,250,203,236,200,220,224,56,38,40,189,242,92,87,88,137,80,185,119,56,157,45,38,249,81,141,154,198,200,197,230,161,244,88,42,59,13,246,47,163,202,67,6,61,36,8,214,36,145,12,38,72,178,14,191,44,29,216,35,187,239,178,110,77,215,156,240,129,144,213,160,165,141,111,87,31,187,157,154,116,38,76,94,148,12,114,248,82,234,104,228,60,11,142,54,20,82,222,183,55,106,73,231,116,242,18,90,221,249,253,252,122,241,228,218,190,101,255,30,247,41,52,113,226,25,13,54,28,33,71,158,154,45,104,229,117,7,137,120,234,83,212,130,169,212,189,66,209,3,119,208,94,69,235,231,250,131,199,235,95,35,251,188,173,152,11,96,112,129,252,72,101,49,132,249,79,248,243,185,161,158,144,77,193,86,58,0,56,31,234,55,104,24,9,216,73,79,189,15,120,21,133,106,119,115,4,62,41,241,11,27,175,141,125,139,177,64,206,31,47,144,240,26,105,136,31,192,189,45,130,157,94,249,17,107,54,223,136,214,125,108,232,164,9,218,41,220,15,235,69,50,166,148,44,60,1,12,25,142,239,167,209,123,194,168,187,33,204,192,137,161,64,232,226,171,48,151,149,67,179,71,162,8,47,54,222,78,135,243,153,85,60,26,20,133,164,162,19,224,14,56,140,240,172,124,34,89,98,160,0,130,24,100,232,105,213,67,180,58,180,187,136,101,24,109,228,127,101,51,230,31,178,5,100,63,98,34,78,12,254,77,42,141,24,13,135,44,194,14,84,172,43,113,231,240,83,143,207,19,49,157,144,187,53,67,40,103,188,136,255,54,116,204,103,211,29,82,168,96,151,17,212,75,60,185,199,86,64,178,138,128,159,29,132,245,89,184,113,212,176,249,15,171,90,161,74,236,141,103,35,77,237,84,95,99,184,28,115,44,245,246,180,51,146,97,231,41,230,194,177,145,17,228,192,42,241,171,247,46,16,78,146,139,108,222,101,18,81,104,4,243,234,34,49,179,151,80,59,209,133,127,16,69,176,155,64,104,89,243,164,97,108,18,40,48,236,129,80,190,227,174,69,134,194,15,190,170,9,65,130,129,230,136,169,255,209,138,103,179,13,171,57,142,0,201,9,171,26,9,219,160,3,87,114,195,36,156,131,251,241,189,219,238,158,52,118,255,199,29,96,15,100,145,48,100,61,236,99,96,65,23,212,184,51,5,4,38,211,136,94,198,67,179,18,190,159,181,227,89,13,136,240,250,135,36,80,174,146,140,152,154,49,120,202,250,157,207,120,210,29,50,222,219,88,180,185,207,249,193,133,72,124,243,137,173,107,4,50,176,36,140,236,3,93,67,52,131,155,51,85,46,147,32,141,176,116,174,215,128,205,33,140,245,30,95,204,8,62,82,144,161,238,71,104,149,249,227,214,37,158,17,85,16,78,15,31,125,38,135,130,55,233,112,231,169,10,69,43,42,84,233,48,176,66,55,73,253,118,151,145,252,152,109,214,37,54,64,157,88,47,189,22,38,220,99,211,74,106,136,94,118,128,196,232,201,3,67,155,150,95,217,235,32,179,99,239,148,155,227,212,163,21,96,120,216,108,191,164,193,236,11,213,141,67,126,76,83,19,144,111,11,82,255,237,123,68,86,90,213,95,160,134,224,254,94,85,41,253,250,50,20,73,102,47,8,167,77,101,66,20,29,166,67,14,16,10,234,189,145,114,107,33,89,158,220,182,150,43,206,183,19,174,173,112,48,75,253,228,153,254,195,10,144,39,233,74,223,81,179,154,79,63,176,177,138,247,243,20,89,191,169,69,226,162,126,92,130,1,3,163,28,165,170,120,105,205,204,101,135,189,222,85,33},
} ;

static const int16_t precomputed_encode_653x1541round_x[precomputed_encode_653x1541round_NUM][crypto_encode_ITEMS] = {
  {5165,24118,2917,1773,-28166,4860,21109,-4291,18122,-1636,-26642,-1196,2188,24249,-26278,-12330,-2845,5529,25697,552,-8245,4733,1194,20693,-20173,-28806,5966,-9724,-2843,-7172,-9714,11422,16459,-3868,4469,-15697,20209,-24443,-28089,-30631,18787,-6502,-13534,10980,-7486,7175,28510,-12334,25824,-32538,-595,15223,-24748,-30054,30866,27156,-10054,-15346,-27514,-31984,28041,4635,17230,-14659,9428,10436,11503,91,16389,-2964,89,13737,7851,-14722,-20787,-28297,-8405,-21848,-21937,5235,-11531,-14954,-21586,-23087,-26043,-11452,-5231,18371,2467,7656,12375,-10744,-27370,10662,-23246,-3414,1840,-6748,-25647,13883,10957,17544,24418,-23125,29497,-21928,15408,-2957,-20106,6590,22905,-31044,-32407,3229,29908,-9093,25207,18774,-24355,-5018,18397,6496,-26661,25436,-8483,10872,-18210,-9693,-29325,4092,15838,31339,-2461,-29889,32521,23427,-19371,-28895,-6570,30446,-12917,5189,-28064,-30939,27252,-29537,-24386,21003,-6302,-3553,-31685,-449,-25886,7787,-13319,3787,14898,10922,-664,16147,-16117,20862,30054,-5070,-10029,5510,25835,6112,7333,3294,-5899,-31461,19805,20047,16297,-11538,2758,26746,14107,-1176,-22591,7044,-11571,-14927,3365,17613,19419,-25603,-9866,2247,-22005,2503,-15493,-16381,16501,19436,5317,17196,21138,-11593,-12052,-9460,-24511,9939,9318,5146,-4627,-16861,9992,-18011,31917,-6615,8019,-15888,-1880,-16062,-26757,-24252,-19253,-15664,18574,-12506,-8875,-9589,-25815,-32338,4016,-10829,-2934,1732,-7948,-27175,-924,-18627,5382,-25904,-8817,-9476,-4367,-24353,-32668,-25332,-19069,-25606,-7604,19632,27207,12659,-29825,-16301,26239,-28797,-4426,18755,27281,22525,-11107,-31778,-1651,31748,14966,-23463,23583,-3483,-2903,8364,16391,1930,26052,26831,29488,16286,2646,-31972,25995,2849,9016,-5676,7730,-12076,24973,6802,6982,20839,-19672,-30390,-18980,3967,-16010,-6914,-16473,-20978,-16971,-29734,7051,30381,-31545,25564,-5615,-29990,-26886,1800,28188,19935,-29733,-22775,-14712,28388,-31342,603,-18247,25918,-21500,-18935,9862,-1346,-26425,16104,-23270,-21335,23998,-16504,-30523,6881,-7232,29282,14188,7530,-18181,6003,27373,-1145,-12322,22591,26774,27741,27932,-31835,-13394,2649,-21069,-22048,-9702,9504,12167,14766,-8050,8664,28544,13396,-17197,8317,-23878,-13178,-18783,316,2827,-21920,17864,17996,26407,-1039,-29774,-5487,4764,-32316,-10403,13433,7125,-32715,-5639,1701,-16417,-22554,-12387,-3755,53,-15097,1862,15978,1853,-19654,-13483,-28655,23510,-28049,13152,-3649,-23130,-6892,-16567,-32750,-556,-12502,-21618,-3322,5791,4509,-23379,-18247,21774,24159,-15443,-24347,-21875,4126,-12698,10395,17301,14463,-5113,-5304,-3632,1797,-1824,-29139,14737,-24757,19358,23620,15220,9406,28287,18858,24196,24863,-13471,32752,23375,-23539,11459,22798,12877,21987,-19880,-1892,-12384,-12290,3039,5078,-18311,-28725,-30615,5926,9877,-7049,13990,19704,21003,-2928,-30879,-19925,-22437,30467,20607,6306,-10360,10800,-28828,-30992,-29658,16635,21477,-27473,11754,-25028,20765,19596,-26783,-3976,15438,-2481,-25067,-6793,6019,-29889,32127,31644,15623,-13853,10626,-22908,30750,2324,-3665,-10407,-14931,29736,-26076,23626,2444,31571,-15364,6082,9136,32707,-10452,14118,-250,2981,-862,-24073,5200,12147,28934,-25182,-10543,11214,6137,22763,-10684,9252,-4992,8499,-18836,12467,-21303,-19894,3691,-24068,-14912,18053,-28615,-5494,-1493,21764,6399,22225,-20546,-27394,367,-1511,-3517,-2778,1415,-7502,-22417,-15882,-24614,7219,22012,1186,-7246,-8951,11496,29183,15059,5278,-5271,-31106,2775,-17195,19916,-8917,-4837,9111,14531,9333,9933,28123,20736,12844,-11485,1982,28818,-16696,31830,25340,-21141,10058,22133,30291,2248,9586,31018,16695,-10704,-10124,1288,-23341,-18391,-13711,-21616,-23226,-6702,-887,9143,-19920,-13088,24565,10457,-29675,-10027,8915,-9427,19817,-7128,-25735,-24248,-18505,-29424,-6880,18542,7597,-15332,-30709,3466,14997,-17781,-4561,-30957,-21502,21977,-27189,-20189,15003,-30164,11917,12351,10719,-392,25939,725,2955,-8325,-402,-5573,6924,18057,7786,-12347,29299,-1573,-26974,7766,-26105,-20999,8941,23925,-18316,6933,4901,12003,-18345,30640,7585,24834,29715,27394,-18047,9265,22633,-8640},
  {9689,-5761,-27129,-5135,-1800,681,-6713,-27829,-29265,-11089,-3121,-20095,-1134,32202,-21991,13625,30409,12762,24703,-31614,-4415,-8025,774,7191,5844,-19408,25364,23426,17838,-7720,27119,20907,26787,-26134,-18523,18370,20525,18756,-2550,-2713,-29404,9780,-5623,16500,-10762,558,-5647,-21128,32591,28380,-9004,17906,-13643,24318,-13558,26608,-12479,22462,-25069,-5332,17623,-30893,2643,4847,6864,13689,25613,-9043,-4256,-29674,-11209,-26199,11393,-14283,6580,12896,-3434,16980,-1750,-15649,9962,14646,31536,-12524,4017,9312,-11510,-27667,13494,10684,-19563,28142,-23404,19368,-23925,-427,-7305,22516,-21097,10550,-23534,-14402,-30037,-18944,-12332,2629,3367,-20340,16248,14067,19986,-21287,-23098,8853,28416,17973,29136,3313,30503,31014,-25983,-4114,7111,-24751,-1237,-15881,29070,-7303,-26980,-28842,2601,-996,-25315,-21124,-12547,30603,1868,-32062,15119,-21438,27531,5959,-6467,-8296,-10301,28425,-23361,-24038,-1581,8241,16663,-32108,-8325,-3559,16572,19563,1186,29982,-3256,16253,4825,-16430,-19585,-14558,-18153,8712,690,7469,-26138,11344,9356,21314,31258,-29797,16627,-21541,-9894,26902,22986,2150,19563,22321,-2333,5528,-25138,12346,-2744,-28659,6975,26279,-18635,25668,226,-23297,-17113,-7400,20262,-31464,-5190,-8959,-22395,-1746,14511,16586,18604,-26939,2237,-3392,29435,-11785,7929,-10281,-26741,32227,-10335,-23107,-32722,-25691,-6338,21945,14418,28897,-8491,-16719,-13945,18673,1494,-11662,-5334,24211,6019,-7906,29450,8205,18441,31134,22181,-3354,30093,-25109,-26916,16925,-21047,-18986,-21284,18523,-18118,-31867,-22260,-5364,-24394,-14504,23552,31586,-4224,-19831,-13150,19329,-25554,1173,-8116,-8482,-23979,-4372,-29481,-25251,-4058,6600,-29101,28891,30044,-23873,22417,1675,-182,7271,3054,14176,30773,-10744,-12618,-10230,4222,-6636,27554,-19571,31057,4442,-3159,15259,-23467,9322,-11796,19037,-11238,-40,26282,24914,-14366,13189,-26803,-5695,13938,27085,-20270,4700,-29159,1428,23909,20511,23703,12718,8748,-9355,-27046,5074,-15790,-2583,6957,-26591,-5473,6065,20213,-15908,29083,710,-17910,-8700,-6229,7452,-16781,-26006,14146,13464,29172,-3508,8746,3276,30699,-32261,26872,12756,6558,19205,-671,4105,15872,14398,-2078,-31905,-27236,31472,-28395,-25475,-14447,176,748,-15972,18778,-28553,22788,-22707,14358,476,30504,26482,2722,-17967,-6499,25193,8884,29074,30404,-10601,-12811,3803,-31668,-4293,-26521,-24304,4071,13404,501,-20231,19679,-10721,20100,12390,-23475,25188,11502,31280,-3208,-19884,30271,-28964,2616,9877,3117,-21094,20080,31575,-8179,-31334,-25542,7295,-18100,-7144,-25141,32178,-32246,-6569,1591,-15004,28823,8746,-10156,-1025,24358,17616,-26540,-28989,-2750,-21143,-16210,22893,-8932,19631,1526,31067,-25383,-29207,-9818,-24496,-30885,-3615,30224,1828,22572,9280,30039,-6985,-23893,9680,-17112,187,30726,18244,21800,6364,-20807,25365,-32534,21369,2877,23723,-32243,-26099,-286,5906,-9015,7788,8882,29602,29819,1908,-16160,-31807,23469,17582,-22838,8781,3591,-9083,10880,-3504,17539,5645,32306,-24883,-6830,-12930,-5190,4409,24545,-32360,170,21887,5989,-7241,-28155,16987,-9058,5187,30339,22143,-9704,17878,14080,-18436,-19671,-12342,-4141,9876,-15211,-8865,3727,7161,14453,-3379,30539,-3805,31772,13289,22929,14576,-19162,19122,-3046,-12318,-14797,29360,-24896,-17409,-22184,-27288,-5450,-11074,7961,-29816,-8040,-13373,2266,21044,26336,15110,22748,-2245,10077,22542,17503,-28162,20522,17930,31928,21860,-22715,-32076,-29332,-14745,20525,-31843,-17165,10718,-19817,-9552,-20664,-2820,28113,15754,-14020,14833,-19009,31329,-13570,-18363,-28077,16360,23151,-23516,13649,21472,27452,24907,-28199,7582,26664,5292,-21399,9620,9549,-27645,30104,21048,-7761,12873,-28778,15261,-15363,-6706,-24019,29736,-15158,23632,-29951,-6612,27466,15212,-17989,-12791,-32439,-24659,12803,-6473,-8354,24398,17825,19436,10257,-1971,18132,-30010,-14187,-15037,-5203,7014,14023,-2264,-21293,-8655,-14216,-9425,1946,-19142,22670,28143,24860,-32275,-24364,-29645,-13491,-18475,12889,-7395,8236,12549,-15848,3550,8625,25402,-12347,-6564,26213,-3987,-7157,8192,-30014,9459,27885,-9421,19981,5345,10057},
  {18418,-2833,3030,21894,-31930,29016,-8855,16445,9264,21822,16346,25346,-28256,-4724,-28644,22165,25815,14722,-29789,-20181,14191,-29769,4233,-11792,-30551,20256,-5488,-16502,960,28315,18604,-496,1123,-5664,-7511,-7420,-16404,17882,-5268,-17010,620,30773,-11116,-4634,-2966,18351,12550,-3927,16696,-5948,-22726,3060,5363,30342,-23683,-25799,-28541,19903,-10372,-17146,-822,12067,-21851,27365,-23097,-24727,-24485,32665,15844,20844,-7864,-6740,-19920,-984,-6230,22108,-11910,7873,-23323,-19438,-10638,3162,-21162,-18859,-963,-17255,-22142,-3868,915,4873,12999,5273,29618,10840,-25650,-28989,-9053,29136,-12604,3881,12625,-4524,7512,9670,29110,-30187,-18261,14775,-20377,6583,-28545,31858,1049,-18329,-17920,-21991,26713,-16414,-29008,26681,-30500,16467,26223,-18841,-9987,124,22771,8371,-4093,-26305,-1502,253,9613,-2155,-31054,13460,-28872,-14036,-3962,18833,-31996,-29819,19275,-6228,-11524,15754,-12932,-15984,-32423,-24911,-15828,-30126,30589,6018,-24740,-22602,3353,-31832,29180,-4786,-12299,-1457,19628,-27702,-4835,21727,-16355,11558,-359,-18000,-1803,8977,7199,18955,-3949,-27417,2584,1242,-20292,18522,19339,-7095,29503,28911,13724,-7114,29627,9088,27268,-23067,25427,-15494,-21924,27078,-25565,19813,-5504,19938,-9861,-6211,29810,-6974,6056,8719,-22075,-3224,-31495,-25545,29647,-30947,1561,6567,-10063,30045,29767,-22623,30733,-10269,-22846,20294,6085,17100,14833,15742,-3918,-23140,27444,5935,23630,4642,19810,30406,4422,32474,-8942,-8330,8099,9385,1079,-9079,-20217,29431,-4081,5221,13414,-20910,4534,-4903,-26213,-14084,27164,-6403,9074,-10842,-8537,29685,-26411,-3378,-28877,-14820,-3454,28904,-2511,-7819,-8963,-4057,3239,28046,-20003,14080,13427,14402,22267,-28030,31722,10848,10633,1085,-20113,28287,5819,11627,12386,-9613,6103,22353,26442,-32138,24382,10127,-11237,5249,-22300,-19944,-24882,4213,-6495,6668,-14270,17292,-6271,28173,22808,-14759,-30558,3076,26869,3642,4787,-17337,16433,21480,23053,13594,-21875,16223,18663,-14245,-19826,16428,16412,24202,13204,-21350,-22645,-12490,30958,-13055,27042,-18261,-17141,21740,-1667,-24779,18861,-12971,-14104,31133,-9209,7199,-23386,-7762,21190,11161,-403,18028,14813,31680,-31150,14862,19071,-23259,-939,-12903,-10521,6360,-11953,-12291,-24546,-23179,1986,-13924,9941,21696,1196,9391,-22620,-12523,4900,-24596,-3153,26816,-19880,-2773,-9144,11289,-9821,-8755,5019,-502,90,21511,14445,-10411,7460,-305,-4163,25002,-12123,26791,-19688,2455,23423,18936,27591,14575,-461,-16727,16266,-9838,2332,5973,3276,-32398,10439,14336,15927,12789,-5166,-5435,-23989,10960,21149,3234,17142,-8515,-29326,17753,12626,-5319,23160,5001,-2474,-5635,3677,18364,5055,-3620,16000,-17492,31422,-23336,21238,21205,-16274,-24420,15085,-17805,-9966,6289,-5288,-21920,-18045,-442,-16027,11240,5737,16982,-12682,20697,28672,-1879,943,7165,-381,15943,-9233,11089,-11223,-17597,5463,-11991,31062,25631,-25538,3240,23195,-11637,-23934,29600,27809,-4980,-6445,2375,-7201,13971,18867,-5562,3782,2904,10377,-19021,27154,23581,-14918,25077,16896,20209,-28983,12387,-3297,22525,23533,11340,-19500,24987,9230,17740,9228,-25432,-7960,9329,-14205,14334,8127,13792,-8210,-17861,21721,-30099,12845,24097,12835,-25253,11945,10590,-26117,7601,-15305,-15379,-28378,15119,29310,27912,27791,-21268,-26890,7349,24781,11445,-9312,-2066,-31040,21102,-9421,8756,-12583,13132,-20791,31951,-15755,19326,-12091,-27761,-30766,-29733,-2566,-21515,22268,10234,-9075,24285,3353,13898,-18817,25699,29887,21364,-17889,-5889,14669,886,24184,-13712,6646,-12677,29195,-10336,6057,-11867,-3082,-15932,9885,9584,27033,17892,-11404,31579,27587,-509,11194,12210,11760,-27653,8800,-8620,-6405,-17890,-137,-31823,22310,28352,18748,-16054,26507,-10641,-25922,15014,-25817,25104,-17539,-20613,28645,-26283,10001,28122,14912,29693,-8884,-8892,-5186,-6266,-8394,9887,-2681,-1121,-28889,31042,20461,32597,-25236,6047,-3506,-6048,21219,21831,5390,11298,-4246,25741,-31143,-13007,6204,7897,5596,25255,23006,12084,22114,-26778,-27751,-12237,7157,26110,-12004,27418,-21719,-30276,-12182,-4146,13724,9225,-20195},
  {4632,-8500,9754,16403,27627,18304,-5426,-12882,-9516,20716,6266,32641,12906,-26464,-27075,-18588,17933,23278,15104,17699,-3969,-26817,4010,11627,-23337,20976,-31376,16033,30346,9938,-29449,-21640,28926,15513,-18972,-23779,1710,16363,-4844,-12128,-5759,-12130,-4783,9147,-24264,19939,21452,15348,-15469,16456,-8404,32672,-887,8976,28935,-18499,18981,5212,13074,-12853,-8211,-2933,-15683,-23472,-3258,-29527,29688,-30520,-31210,-22671,-5144,-18187,-10945,-26059,14345,27208,-21274,-29868,-26867,26290,-27913,20192,8446,-13141,5881,22687,-29921,-2273,-31990,11874,-24032,15607,-55,-32556,1667,664,5825,29700,23550,-21039,-22090,26755,22911,-12131,26547,15517,-8143,-13558,25549,3406,20984,-32620,-9094,-27115,7850,-265,-20729,-5649,27964,17866,13877,-25750,82,-30060,2068,22399,15371,19443,30843,-8794,-14223,-25624,-16883,-7355,-27104,-10575,7469,23186,-28509,-13255,-16980,-27864,30060,13024,25118,17979,29537,-7614,18855,28022,-19811,18277,9832,30255,11154,379,-22222,-28300,31090,8338,-5139,10888,-20329,31818,-9943,-9055,-19011,11108,-6545,-7805,11104,-31297,-28014,-25927,31523,-3366,5307,3096,29274,21917,-11570,-24343,-30632,-3496,4490,9989,-6540,-20810,20023,4708,16886,-16905,21487,16716,-9606,-32498,-27243,-17657,-20215,-2285,-23449,-24104,-8177,1398,-24494,4426,-7473,3992,-23822,30595,-11733,-716,-24907,-19084,-17967,7701,16367,27093,-32559,-15723,13712,30726,13388,1838,3236,3669,31627,-22360,25228,-28187,16383,-25311,-27198,17728,-9940,-21551,24474,13117,-21231,28711,-16751,-18480,12309,-2507,-27740,5883,5297,-2360,10072,-20496,9136,15404,58,9537,-27029,-2063,15511,22440,20779,-11760,-16641,-2711,28955,9338,17160,13572,-12765,-25771,24540,-19826,-21922,8646,31122,-8081,-21288,-12531,22951,-31573,-24454,-25700,-2029,-13224,5051,-19345,3603,-27044,24521,25883,-29923,-31968,15181,12965,14721,-19406,18303,-25975,-996,17253,1157,12662,-13504,20735,-1472,-10725,-25268,-24263,16004,29233,-16474,-30688,-25579,-5249,6048,-14051,1813,-6301,-29518,24976,-22390,-2782,-18604,31382,5500,31550,22603,10671,-18645,-17534,18808,304,24037,5284,22337,-3312,-20382,-15263,-26626,-15155,-17875,22441,-29926,26328,18729,-6121,14011,-10848,1911,-13266,-4844,-9712,-13289,-22690,-17692,3147,12899,10276,-10025,32296,20180,-10420,-3637,-27178,-16574,14798,-26700,9043,-351,-32492,-651,-28608,-12082,22724,-22120,24219,-26952,11267,13930,-16947,-25314,-27693,29625,16177,18826,3556,26369,2624,6371,1942,-9537,-15956,-31274,-3376,9298,-25477,27896,15312,-29690,21349,32710,-21020,-2601,32694,-24350,9213,4738,-297,-4687,-2287,-19933,12816,9087,2708,28740,5504,13961,5236,-27879,-15222,-26368,-8154,-26437,28433,27433,30494,-23281,-1809,31490,9090,1814,2749,8046,22718,4954,6273,-25786,-25686,402,-16317,-15951,-28218,9031,-9577,23960,-24470,-1408,19457,5213,9151,31766,17976,21113,-333,18860,-17370,29531,-9605,10733,13408,-29610,6388,-14891,-22464,-12839,-22195,-19252,-392,509,-32181,-19900,10387,27607,5347,28361,-19893,430,2367,-10102,-613,19514,29448,-15846,-3297,-9278,-39,23104,14284,-2308,-6148,-26038,30171,14181,-4814,-27224,-26116,32041,-15678,21460,-8151,-25553,-2348,-28389,-4958,12112,28883,-21697,-1060,26615,23232,17619,5446,-25526,2447,-5305,28682,14342,8100,-27466,3043,27310,-29468,-23160,6393,-9893,-21704,-22825,-31134,-20041,-22357,8301,26386,-18889,7981,-13625,18040,16229,4051,24535,20410,30688,1720,-19031,-10408,21209,29312,29876,2121,-29309,-11259,8993,-7186,-29273,3448,6133,-26496,-23754,13574,-2369,23385,4688,-28325,-15210,-5272,9490,-8876,31896,16735,20789,-13484,-21344,18176,-25105,26547,22467,30405,29886,27656,21435,7898,-601,11954,-2951,3075,-6293,-28348,1616,-16000,16303,-15480,8251,28472,8792,-1035,-15058,-4113,22678,-31921,18331,-14763,29654,-13392,29218,-29387,17841,-17366,23753,-25395,29408,-32419,-20429,-19433,10873,22020,18273,-15128,-26804,10503,6898,-24077,-20498,1009,-5335,-26033,-2940,18252,15411,19073,-2591,9160,-5429,878,17653,-17691,-5145,-7238,14497,-15878,11287,-26380,29823,25445,-10347,23150,190,1122,-32438,15749,31229,-18504,9181,-8841,-30308,22196,12821},
} ;

static void test_encode_653x1541round_impl(long long impl)
{
  unsigned char *s = test_encode_653x1541round_s;
  unsigned char *x = test_encode_653x1541round_x;
  unsigned char *s2 = test_encode_653x1541round_s2;
  unsigned char *x2 = test_encode_653x1541round_x2;
  long long slen = crypto_encode_STRBYTES;
  long long xwords = crypto_encode_ITEMS;
  long long xlen;

  if (targeti && strcmp(targeti,".") && strcmp(targeti,ntruprime_dispatch_encode_653x1541round_implementation(impl))) return;
  if (targetn && atol(targetn) != impl) return;
  if (impl >= 0) {
    crypto_encode = ntruprime_dispatch_encode_653x1541round(impl);
    printf("encode_653x1541round %lld implementation %s compiler %s\n",impl,ntruprime_dispatch_encode_653x1541round_implementation(impl),ntruprime_dispatch_encode_653x1541round_compiler(impl));
  } else {
    crypto_encode = ntruprime_encode_653x1541round;
    printf("encode_653x1541round selected implementation %s compiler %s\n",ntruprime_encode_653x1541round_implementation(),ntruprime_encode_653x1541round_compiler());
  }
  for (long long checksumbig = 0;checksumbig < 2;++checksumbig) {
    long long loops = checksumbig ? 4096 : 1024;

    checksum_clear();

    for (long long loop = 0;loop < loops;++loop) {
      xlen = xwords*crypto_encode_ITEMBYTES;

      output_prepare(s2,s,slen);
      input_prepare(x2,x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      secret(x,xlen);
      crypto_encode(s,x);
      public(x,xlen);
      endianness(x,xwords,crypto_encode_ITEMBYTES);
      public(s,slen);
      checksum(s,slen);
      output_compare(s2,s,slen,"crypto_encode");
      input_compare(x2,x,xlen,"crypto_encode");

      double_canary(s2,s,slen);
      double_canary(x2,x,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      secret(x2,xlen);
      crypto_encode(s2,x2);
      public(x2,xlen);
      endianness(x2,xwords,crypto_encode_ITEMBYTES);
      public(s2,slen);
      if (memcmp(s2,s,slen) != 0) fail("failure: crypto_encode is nondeterministic\n");
    }
    checksum_expected(encode_653x1541round_checksums[checksumbig]);
  }
  for (long long precomp = 0;precomp < precomputed_encode_653x1541round_NUM;++precomp) {
    output_prepare(s2,s,crypto_encode_STRBYTES);
    input_prepare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x,precomputed_encode_653x1541round_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    memcpy(x2,precomputed_encode_653x1541round_x[precomp],crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
    crypto_encode(s,x);
    if (memcmp(s,precomputed_encode_653x1541round_s[precomp],crypto_encode_STRBYTES)) {
      fail("failure: crypto_encode fails precomputed test vectors\n");
      printf("expected s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",((unsigned char *) precomputed_encode_653x1541round_s[precomp])[pos]);
      printf("\n");
      printf("received s: ");
      for (long long pos = 0;pos < crypto_encode_STRBYTES;++pos) printf("%02x",s[pos]);
      printf("\n");
    }
    output_compare(s2,s,crypto_encode_STRBYTES,"crypto_encode");
    input_compare(x2,x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES,"crypto_encode");
  }
}

void test_encode_653x1541round(void)
{
  long long maxalloc = 0;
  if (targeto && strcmp(targeto,"encode")) return;
  if (targetp && strcmp(targetp,"653x1541round")) return;
  storage_encode_653x1541round_s = callocplus(crypto_encode_STRBYTES);
  test_encode_653x1541round_s = aligned(storage_encode_653x1541round_s,crypto_encode_STRBYTES);
  if (crypto_encode_STRBYTES > maxalloc) maxalloc = crypto_encode_STRBYTES;
  storage_encode_653x1541round_x = callocplus(crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  test_encode_653x1541round_x = aligned(storage_encode_653x1541round_x,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);
  if (crypto_encode_ITEMS*crypto_encode_ITEMBYTES > maxalloc) maxalloc = crypto_encode_ITEMS*crypto_encode_ITEMBYTES;
  storage_encode_653x1541round_s2 = callocplus(maxalloc);
  test_encode_653x1541round_s2 = aligned(storage_encode_653x1541round_s2,crypto_encode_STRBYTES);
  storage_encode_653x1541round_x2 = callocplus(maxalloc);
  test_encode_653x1541round_x2 = aligned(storage_encode_653x1541round_x2,crypto_encode_ITEMS*crypto_encode_ITEMBYTES);

  for (long long offset = 0;offset < 1;++offset) {
    if (targetoffset && atol(targetoffset) != offset) continue;
    if (offset && valgrind) break;
    printf("encode_653x1541round offset %lld\n",offset);
    for (long long impl = -1;impl < ntruprime_numimpl_encode_653x1541round();++impl)
      forked(test_encode_653x1541round_impl,impl);
    ++test_encode_653x1541round_s;
    ++test_encode_653x1541round_x;
    ++test_encode_653x1541round_s2;
    ++test_encode_653x1541round_x2;
  }
  free(storage_encode_653x1541round_x2);
  free(storage_encode_653x1541round_s2);
  free(storage_encode_653x1541round_x);
  free(storage_encode_653x1541round_s);
}
#undef crypto_encode_STRBYTES
#undef crypto_encode_ITEMS
#undef crypto_encode_ITEMBYTES

