with Standard_Floating_Numbers;          use Standard_Floating_Numbers;
with Octo_Double_Numbers;                use Octo_Double_Numbers;

package Octo_Double_Constants is

-- DESCRIPTION :
--   This package collects common definitions of octo double constants
--   for use in octo double mathematical functions.

  od_eps : constant double_float := 4.616489308892868e-128; -- 2^(-52*8-7)

-- PI and multiples and factors :

  pi : constant octo_double
     := create( 3.14159265358979312E+00, 1.22464679914735321E-16,
               -2.99476980971833967E-33, 1.11245422086336528E-49,
                5.67223197964031574E-66, 1.74498621613524860E-83,
                6.02937273224953984E-100, 1.91012354687998999E-116);

  twopi : constant octo_double
        := create( 6.28318530717958623E+00, 2.44929359829470641E-16,
                  -5.98953961943667933E-33, 2.22490844172673056E-49,
                   1.13444639592806315E-65, 3.48997243227049720E-83,
                   1.20587454644990797E-99, 3.82024709375997998E-116);

  pi2 : constant octo_double
      := create( 1.57079632679489656E+00, 6.12323399573676604E-17,
                -1.49738490485916983E-33, 5.56227110431682641E-50,
                 2.83611598982015787E-66, 8.72493108067624301E-84,
                 3.01468636612476992E-100, 9.55061773439994996E-117);

  pi4 : constant octo_double
      := create( 7.85398163397448279E-01, 3.06161699786838302E-17,
                -7.48692452429584916E-34, 2.78113555215841320E-50,
                 1.41805799491007894E-66, 4.36246554033812150E-84,
                 1.50734318306238496E-100, 4.77530886719997498E-117);

  threepi4 : constant octo_double
           := create( 2.35619449019234484E+00, 9.18485099360514844E-17,
                      3.91689846475040032E-33, -2.58679816327048642E-49,
                     -4.93609888149662566E-67, 1.30873966210143654E-83,
                     -4.62186958107104523E-100, 3.97053449747564186E-116);

  pi1024 : constant octo_double
         := create( 3.06796157577128234E-03, 1.19594413979233712E-19,
                   -2.92457989230306608E-36, 1.08638107506188016E-52,
                    5.53928904261749584E-69, 1.70408810169457871E-86,
                    5.88805930883744125E-103, 1.86535502624999023E-119);

-- INVERSE FACTORIALS FOR TAYLOR EXPANSION :

  i_fac0 : constant octo_double
         := Create( 1.66666666666666657E-01, 9.25185853854297066E-18,
                    5.13581318503262866E-34, 2.85094902409834186E-50,
                    1.58259462429329970E-66, 8.78516495269210220E-83,
                    4.87674620280437299E-99, 2.70713795980335902E-115);
  i_fac1 : constant octo_double
         := Create( 4.16666666666666644E-02, 2.31296463463574266E-18,
                    1.28395329625815716E-34, 7.12737256024585466E-51,
                    3.95648656073324926E-67, 2.19629123817302555E-83,
                    1.21918655070109325E-99, 6.76784489950839755E-116);
  i_fac2 : constant octo_double
         := Create( 8.33333333333333322E-03, 1.15648231731787138E-19,
                    1.60494162032269652E-36, 2.22730392507682967E-53,
                    3.09100512557285111E-70, 4.28963132455669071E-87,
                    5.95305932959518212E-104, 8.26152941834521220E-121);
  i_fac3 : constant octo_double
         := Create( 1.38888888888888894E-03, -5.30054395437357706E-20,
                   -1.73868675534958776E-36, -1.63335621172300840E-52,
                   -5.35774221765960817E-69, -5.03316742081318367E-85,
                   -1.65098178740773037E-101, -1.55096445613734114E-117);
  i_fac4 : constant octo_double
         := Create( 1.98412698412698413E-04, 1.72095582934207053E-22,
                    1.49269123913941271E-40, 1.29470326746002471E-58,
                    1.12297607624339191E-76, 9.74026481209866379E-95,
                    8.44833301588918793E-113, 7.32776080776645696E-131);
  i_fac5 : constant octo_double
         := Create( 2.48015873015873016E-05, 2.15119478667758816E-23,
                    1.86586404892426588E-41, 1.61837908432503088E-59,
                    1.40372009530423989E-77, 1.21753310151233297E-95,
                    1.05604162698614849E-113, 9.15970100970807120E-132);
  i_fac6 : constant octo_double
         := Create( 2.75573192239858925E-06, -1.85839327404647208E-22,
                    8.49175460488199287E-39, -5.72661640789429621E-55,
                    2.61672391582886888E-71, -1.76464992319726308E-87,
                    8.06340311310246955E-104, -5.43774740551399185E-120);
  i_fac7 : constant octo_double
         := Create( 2.75573192239858883E-07, 2.37677146222502973E-23,
                   -3.26318890334088294E-40, 1.61435111860404415E-56,
                   -9.99798078191450073E-74, -5.23082523455780306E-91,
                   -2.46767621898134459E-107, 8.08497474286222354E-124);
  i_fac8 : constant octo_double
         := Create( 2.50521083854417202E-08, -1.44881407093591197E-24,
                    2.04267351467144546E-41, -8.49632672007163175E-58,
                   -9.08907343810409139E-75, -2.26065446324988261E-91,
                   -7.19805892199252488E-108, -6.40212493137993289E-125);
  i_fac9 : constant octo_double
         := Create( 2.08767569878681002E-09, -1.20734505911325997E-25,
                    1.70222792889287100E-42, 1.41609532150396700E-58,
                    5.13820161376913309E-75, 1.44797661649508516E-91,
                    1.30256332445267049E-107, 3.72847700075020451E-124);
  i_fac10 : constant octo_double
          := Create( 1.60590438368216133E-10, 1.25852945887520981E-26,
                    -5.31334602762985031E-43, 3.54021472597605528E-59,
                    -1.98567896059148884E-75, -1.02148490610754577E-91,
                     5.19442455358700245E-108, 2.03226501937080225E-124);
  i_fac11 : constant octo_double
          := Create( 1.14707455977297245E-11, 2.06555127528307454E-28,
                     6.88907923246664603E-45, 5.72920002655109095E-61,
                    -3.65551458930952487E-78, -1.73752114063892946E-94,
                    -1.29464103159491163E-110, -4.67626589929406776E-127);
  i_fac12 : constant octo_double
          := Create( 7.64716373181981641E-13, 7.03872877733453001E-30,
                    -7.82753927716258345E-48, 1.92138649443790242E-64,
                    -1.43027453759388186E-80, 7.76151305461206293E-97,
                     3.19012607380048399E-114, 1.78550560936942146E-130);
  i_fac13 : constant octo_double
          := Create( 4.77947733238738525E-14, 4.39920548583408126E-31,
                    -4.89221204822661465E-49, 1.20086655902368901E-65,
                    -8.93921585996176165E-82, 4.85094565913253933E-98,
                     1.99382879612530249E-115, 1.11594100585588841E-131);
  i_fac14 : constant octo_double
          := Create( 2.81145725434552060E-15, 1.65088427308614326E-31,
                    -2.87777179307447918E-50, 4.27110689256293549E-67,
                    -2.93287743014724397E-83, -1.23436334109628973E-99,
                    -2.14851021924804564E-118, 2.19396004831602516E-134);
  i_fac15 : constant octo_double
          := Create( 1.56192069685862253E-16, 1.19106796602737540E-32,
                    -4.57750605962998323E-49, 2.87494142340899603E-67,
                     2.03110904687324058E-84, -6.85757411720160904E-101,
                    -5.65180691747504322E-117, 2.48255175510937718E-133);
  i_fac16 : constant octo_double
          := Create( 8.22063524662432950E-18, 2.21418941196042654E-34,
                    -1.50891402377419897E-50, 1.40072951514781548E-67,
                    -5.96179899774792636E-84, -3.40489743808039776E-100,
                    -3.82514166264397287E-149, -6.59553593472873417E-166);
  i_fac17 : constant octo_double
          := Create( 4.11031762331216484E-19, 1.44129733786595272E-36,
                    -5.28562754878981208E-53, -4.14764725635765685E-70,
                    -2.10538849298467203E-87, -5.82681010557201984E-104,
                     1.30202886782577633E-120, -3.27971291402577718E-137);
  i_fac18 : constant octo_double
          := Create( 1.95729410633912626E-20, -1.36435038300879085E-36,
                     1.33923482511250642E-53, -6.82108942414933122E-70,
                    -1.54203684683207840E-86, -7.25646161501115955E-103,
                     1.18641862580086732E-119, 4.32477008266988955E-136);
  i_fac19 : constant octo_double
          := Create( 8.89679139245057408E-22, -7.91140261487237622E-38,
                    -3.18779767905709333E-54, 1.27057810175205662E-70,
                    -6.55042310022825056E-87, 2.10550328869752187E-103,
                    -3.96700758009699443E-120, 1.44732684633021301E-136);
  i_fac20 : constant octo_double
          := Create( 3.86817017063068413E-23, -8.84317765548234385E-40,
                     3.87181571061732467E-56, -1.95652575315225570E-72,
                     5.39691325170565736E-89, 1.87480587934294261E-105,
                     1.81105888742894516E-121, 1.00313695134730449E-137);
  i_fac21 : constant octo_double
          := Create( 1.61173757109611839E-24, -3.68465735645097660E-41,
                     1.61325654609055195E-57, -8.15219063813439928E-74,
                     7.48508021786159889E-90, 7.81169116392892833E-107,
                    -5.21821124982141188E-124, 2.60976234532966935E-140);
  i_fac22 : constant octo_double
          := Create( 6.44695028438447359E-26, -1.93304042337034648E-42,
                    -1.52130238070391442E-58, 6.64377273721295753E-75,
                    -2.89688007121263253E-91, 2.49254308464133028E-107,
                     2.21164149678796755E-124, -9.03290922321876539E-141);
  i_fac23 : constant octo_double
          := Create( 2.47959626322479759E-27, -1.29537309647652288E-43,
                     6.40339015984996240E-60, -8.46024562770674585E-77,
                    -5.24149370502727977E-93, 4.15713747064484237E-110,
                    -1.94825926264674307E-126, 1.57228559591277112E-142);
  i_fac24 : constant octo_double
          := Create( 9.18368986379554601E-29, 1.43031503967873220E-45,
                    -8.55122677465050480E-62, 8.38146710023453832E-78,
                     3.03029953301710766E-94, 2.32237352854655252E-110,
                    -1.27586459249083458E-126, 3.01211536378102089E-143);
  i_fac25 : constant octo_double
          := Create( 3.27988923706983776E-30, 1.51175427440298787E-46,
                     8.05851771951971593E-63, -9.09648053071092885E-81,
                     1.21523807193217733E-97, 1.49898243133566639E-114,
                     5.73056590844191453E-131, -2.50926562681977975E-147);
  i_fac26 : constant octo_double
          := Create( 1.13099628864477159E-31, 1.04980154129595060E-47,
                    -4.34615092939779518E-64, -4.96677980014005582E-81,
                    -2.54108908976355161E-97, 2.74015709219560046E-114,
                    -9.12989153567239791E-131, -8.37100817409627246E-147);
  i_fac27 : constant octo_double
          := Create( 3.76998762881590539E-33, 2.58703478327503238E-49,
                     3.23789002742563999E-66, 2.56128591057885727E-82,
                    -6.67503041391508032E-100, -3.04826384512978187E-116,
                     1.46499316215822502E-132, 3.37918778191504535E-149);
  i_fac28 : constant octo_double
          := Create( 1.21612504155351789E-34, 5.58629056788880577E-51,
                     6.61594857808279193E-68, -3.16204422895208591E-84,
                     7.96409263336490057E-102, 1.93557272906923875E-118,
                     7.49217813286381361E-135, -3.68425172923987781E-151);

  n_inv_fact : constant natural := 29;
  i_fac : array(0..n_inv_fact-1) of octo_double
        := (i_fac0,i_fac1,i_fac2,i_fac3,i_fac4,i_fac5,i_fac6,i_fac7,
            i_fac8,i_fac9,i_fac10,i_fac11,i_fac12,i_fac13,i_fac14,
            i_fac15,i_fac16,i_fac17,i_fac18,i_fac19,i_fac20,i_fac21,
            i_fac22,i_fac23,i_fac24,i_fac25,i_fac26,i_fac27,i_fac28);

-- TABLES of sin(k * pi/1024) and cos(k * pi/1024).

  sin_t0   : constant octo_double
           := create( 3.06795676296597614E-03, 1.26902790854559250E-19,
                      5.28794642453283888E-36,-1.78203340819552983E-52,
                      1.24867176529600562E-68, 9.55672442768777125E-85,
                     -4.78635016554141373E-101,-8.83022048198150353E-118);
  sin_t1   : constant octo_double
           := create( 6.13588464915447527E-03, 9.05452574824749325E-20,
                      1.62601131337453203E-37,-9.74920012087674099E-55,
                      4.17002567067724779E-71,-1.96032877561331419E-87,
                      9.74228205614807724E-104,-1.98833998927193889E-120);
  sin_t2   : constant octo_double
           := create( 9.20375478205981944E-03,-1.21365916935359338E-19,
                      5.56969039494255674E-36, 1.25056357919369509E-52,
                     -6.20114835670991562E-69,-3.60171168599636468E-85,
                     -3.46259917683846748E-102,-1.28977874687952156E-118);
  sin_t3   : constant octo_double
           := create( 1.22715382857199254E-02, 6.91979076402831699E-19,
                     -4.02037267134355554E-36,-2.06887036069528163E-52,
                     -1.64091649492352269E-68, 9.99330273992003136E-85,
                      2.79487261553066006E-102,-9.12920554076969506E-119);
  sin_t4   : constant octo_double
           := create( 1.53392062849881019E-02,-8.44625788654016957E-19,
                      4.65358975050586285E-35,-1.39236829785704672E-51,
                      1.34115203025621548E-67, 1.61333777554372834E-86,
                      5.08208232933158994E-103, 1.69914332528431097E-119);
  sin_t5   : constant octo_double
           := create( 1.84067299058048202E-02, 7.41955338128331604E-19,
                      3.90684764867876066E-35, 3.63933212928986139E-52,
                     -2.19353176581903619E-68,-4.29241259606783756E-85,
                     -2.59677938349607068E-101, 2.51792510872904365E-118);
  sin_t6   : constant octo_double
           := create( 2.14740802754695079E-02,-4.54079602076885659E-19,
                     -2.20317701197230047E-35, 1.27098146548337412E-51,
                     -2.50649453460352302E-68, 3.88605110431610064E-85,
                     -2.16505604540818196E-101, 9.05360629654707290E-118);
  sin_t7   : constant octo_double
           := create( 2.45412285229122881E-02,-9.18684901257787818E-20,
                      4.87061487044670605E-36,-2.81539478554692237E-52,
                      7.95004118779174058E-69,-1.22229777976689277E-85,
                      1.40195062694131108E-102, 2.51107195779642889E-119);
  sin_t8   : constant octo_double
           := create( 2.76081457789657432E-02,-1.59323588313892692E-18,
                     -7.04754162427760300E-35,-2.75184941766027441E-51,
                      1.18327750332464105E-67,-1.05832781649026863E-84,
                      5.50679374753733312E-101,-8.08900495379845942E-119);
  sin_t9   : constant octo_double
           := create( 3.06748031766366260E-02,-1.69360548441079213E-20,
                      1.00531123166080144E-36, 4.37274329418026507E-54,
                      1.99763190099018758E-70, 1.20943664362514015E-86,
                     -8.66316411970888068E-103, 1.48262526031177689E-119);
  sin_t10  : constant octo_double
           := create( 3.37411718513775868E-02,-2.00960742923683398E-18,
                     -1.35482370165371341E-34, 6.55548818758999731E-51,
                     -3.72227736510150863E-67, 2.70492399703400928E-83,
                      2.35695981258085198E-100,-1.54103417314555294E-116);
  sin_t11  : constant octo_double
           := create( 3.68072229413588317E-02, 6.10600888035298419E-19,
                     -4.04487212598527268E-35,-2.11110567656714955E-51,
                     -9.23904371219639896E-69, 2.14598176126270627E-85,
                      1.26338281770142051E-101, 2.22676073313206986E-118);
  sin_t12  : constant octo_double
           := create( 3.98729275877398107E-02, 4.66574534811832892E-19,
                      3.41193335622886843E-35, 2.40075347261875112E-51,
                      5.25864729846962600E-68, 2.52845885078272082E-85,
                      9.36713417811774289E-102, 3.90798751798522276E-118);
  sin_t13  : constant octo_double
           := create( 4.29382569349408202E-02, 2.83519405886609072E-18,
                      1.69913096011864755E-34, 6.80265360986726293E-51,
                      4.33267797183762470E-67,-2.87883945215165775E-83,
                     -5.93728361883032950E-100, 4.04652741874106067E-116);
  sin_t14  : constant octo_double
           := create( 4.60031821309146299E-02,-1.11828139401577879E-18,
                      7.52350202703789460E-35, 4.11873049554937220E-52,
                      2.71658912037149379E-68,-1.36786339319123030E-84,
                      1.12307260321677812E-100, 1.42561559916829998E-117);
  sin_t15  : constant octo_double
           := create( 4.90676743274180149E-02,-6.79610372051828011E-19,
                     -4.43188681247183253E-35,-9.93766281325253159E-52,
                     -3.24651050179866359E-68, 1.61864844909595580E-84,
                      5.12323924879172258E-101,-2.62019795916202101E-117);
  sin_t16  : constant octo_double
           := create( 5.21317046802833237E-02,-2.42436952919537795E-18,
                     -1.36754053200922982E-34,-8.39381376211450699E-51,
                     -1.74550009910030472E-67, 9.69062131685494505E-84,
                      1.74418166782269283E-100,-4.48318060509326491E-117);
  sin_t17  : constant octo_double
           := create( 5.51952443496899411E-02,-1.33402998608911026E-18,
                     -3.43595741256656076E-35, 1.19114627554093692E-51,
                      2.55262260673375794E-68,-1.86772762327757241E-84,
                      4.21123652198814818E-101,-1.04993035483230225E-117);
  sin_t18  : constant octo_double
           := create( 5.82582645004357594E-02, 2.32999054960774918E-19,
                      1.93761089906286601E-36,-5.12737757100953014E-53,
                      3.51220352568903212E-69,-2.12103407570678699E-85,
                     -9.34585002006586359E-102, 3.50205723963112311E-118);
  sin_t19  : constant octo_double
           := create( 6.13207363022085783E-02,-5.11811340646381077E-19,
                     -4.27263358667063129E-35, 2.63684955574406907E-51,
                      6.60361134854085147E-68, 3.80427283057729097E-84,
                      1.30658598433432592E-100, 2.78618091452163955E-118);
  sin_t20  : constant octo_double
           := create( 6.43826309298574651E-02,-4.23259970000527053E-18,
                      3.32601177118559366E-35, 1.47362677067183522E-51,
                      1.09501347268971187E-67,-7.29750189938547575E-84,
                     -3.97035459020836904E-100,-2.42662381311286416E-116);
  sin_t21  : constant octo_double
           := create( 6.74439195636640648E-02,-6.92217965569836359E-18,
                      1.59092863589110404E-34,-7.88289468918352183E-51,
                     -3.40224950677578928E-67, 3.35486710132031578E-84,
                      6.12345214854455644E-101, 3.11252771036061230E-117);
  sin_t22  : constant octo_double
           := create( 7.05045733896138699E-02,-6.85527911073428832E-18,
                     -1.99611776308415798E-34, 2.01271295804853003E-50,
                      5.78139634342966359E-67, 4.30525599157503775E-84,
                     -3.54383487423268438E-100, 6.79651882460436489E-117);
  sin_t23  : constant octo_double
           := create( 7.35645635996674263E-02,-2.77849415062735933E-18,
                     -9.12403754898528210E-35,-1.95897520235467951E-51,
                      1.72892595832864613E-68, 3.30047289861029170E-85,
                     -1.10397616676790883E-101,-7.25541818801301675E-118);
  sin_t24  : constant octo_double
           := create( 7.66238613920314920E-02, 2.32537002879588012E-19,
                     -1.31860839212134400E-36,-4.99278726080996730E-53,
                     -6.75622016413686270E-70, 1.90640045815686919E-86,
                      1.14704672601347686E-102, 6.08235791186362312E-119);
  sin_t25  : constant octo_double
           := create( 7.96824379714301256E-02,-4.48676643113730408E-18,
                      2.85407891436502636E-34, 2.84913485832627411E-51,
                      2.34228764339898482E-67,-3.73261010818490682E-84,
                     -2.62166830049628519E-101,-1.25938489278639043E-118);
  sin_t26  : constant octo_double
           := create( 8.27402645493756916E-02, 1.47359835308777599E-18,
                      3.72840934522337134E-35, 2.90244300367240881E-52,
                     -1.57048255247903112E-68,-6.39124711770101570E-85,
                     -1.80116514887867025E-101, 1.00868908818279292E-117);
  sin_t27  : constant octo_double
           := create( 8.57973123444398939E-02,-3.38818938306840289E-18,
                     -1.61355295315082582E-34, 7.72946516205880486E-51,
                      3.89394073854801227E-67, 2.03594242671766312E-83,
                     -3.52108548965970156E-102,-1.64968845981580663E-118);
  sin_t28  : constant octo_double
           := create( 8.88535525825246003E-02,-3.75017758302906906E-18,
                      3.75436063739115777E-34,-2.05305335070232712E-50,
                     -4.65456325283412581E-67,-3.84824924104213256E-84,
                      4.66089732693259822E-101, 3.09756551661741010E-117);
  sin_t29  : constant octo_double
           := create( 9.19089564971327239E-02, 4.76315948542745641E-18,
                      1.57228746429393444E-34,-4.84641454478314562E-51,
                     -3.52793110726296069E-68,-6.95220710003810539E-85,
                     -1.50348895443414149E-101,-5.25177904048400548E-118);
  sin_t30  : constant octo_double
           := create( 9.49634953296390055E-02,-6.58858864004175640E-18,
                     -2.13711169916419652E-34, 1.38193705592492998E-50,
                      4.56104393082810257E-67, 2.58758898446223002E-83,
                     -1.37753556057404643E-99, 5.00185132538643362E-116);
  sin_t31  : constant octo_double
           := create( 9.80171403295606036E-02,-1.63458236224425599E-18,
                     -1.32092388100064538E-35,-3.56910600491179416E-52,
                     -2.80870560395247626E-70,-8.56590019023935848E-87,
                      6.48842344630234380E-103, 5.71394239789677917E-120);
  sin_t32  : constant octo_double
           := create( 1.01069862754827822E-01, 3.31643257193086556E-18,
                     -1.20042248851322819E-34, 7.20288284954186312E-51,
                      5.40076219370887877E-67,-2.11093157441623204E-83,
                     -1.70701640802714587E-99,-3.65931298372649385E-116);
  sin_t33  : constant octo_double
           := create( 1.04121633872054573E-01, 6.57602540853851002E-18,
                      1.70662461712192141E-34,-4.94993409968935140E-51,
                      2.07083651001785294E-67, 5.34794700622502467E-84,
                      2.82034205793675201E-100,-9.26284597153554991E-117);
  sin_t34  : constant octo_double
           := create( 1.07172424956808843E-01, 6.44240442790261976E-18,
                     -8.39569764996981391E-35,-4.06677302133183208E-51,
                     -1.60605665974102045E-67, 1.54397645981811330E-83,
                      7.69490527926525277E-100, 4.30449084427257403E-116);
  sin_t35  : constant octo_double
           := create( 1.10222207293883059E-01,-5.67895035378232325E-19,
                      1.03802747923832327E-35, 1.52139979184566952E-52,
                     -8.51557122494676361E-69, 4.89688150976223126E-85,
                     -5.09638612553203020E-102, 2.22462970218491877E-118);
  sin_t36  : constant octo_double
           := create( 1.13270952177564346E-01, 2.71004810121328995E-18,
                      1.53232929994916192E-35, 4.95644328103608790E-52,
                      2.20365393113530988E-69,-8.10820043479993803E-86,
                      4.29522884165877683E-102,-2.38699269859091463E-118);
  sin_t37  : constant octo_double
           := create( 1.16318630911904766E-01, 1.02949148775097048E-18,
                     -9.39757349489930378E-35, 1.35348273237197078E-52,
                      6.72558403404510899E-69,-2.30016685085211769E-85,
                      3.81872231599344731E-102, 2.64469382670329387E-118);
  sin_t38  : constant octo_double
           := create( 1.19365214810991369E-01,-3.95000893918985055E-18,
                      3.53173499782273111E-34, 1.88560468070122748E-51,
                      1.10020978623025641E-67, 6.78805784330018206E-84,
                      3.08307827722845500E-100, 2.15390763199215144E-116);
  sin_t39  : constant octo_double
           := create( 1.22410675199216196E-01, 2.83545014899653353E-18,
                      1.81516557514933045E-34,-2.87165921779151915E-51,
                     -3.42479358759733376E-68, 1.92569167918437660E-84,
                     -1.00490567858699142E-100,-1.85045471075088884E-117);
  sin_t40  : constant octo_double
           := create( 1.25454983411546234E-01, 4.86867517631482349E-18,
                      5.98781052580979364E-35,-3.35346290987221065E-51,
                      2.95636376128706732E-67, 2.14159509153177471E-84,
                     -1.61454442335754400E-101,-1.35017597542095033E-117);
  sin_t41  : constant octo_double
           := create( 1.28498110793793169E-01, 3.81986039549888022E-18,
                     -1.86275014559477977E-34,-2.43081611335277907E-51,
                     -9.37108850912359351E-68,-3.03520412072352890E-84,
                     -1.81686426547226526E-100, 1.01017345049459640E-116);
  sin_t42  : constant octo_double
           := create( 1.31540028702883116E-01,-5.00397082622138130E-18,
                     -1.29830041592455520E-34,-4.68720349157941220E-51,
                      1.43606684924838004E-67, 8.09870234180351493E-84,
                      3.86732674792833942E-100, 1.22603596075179043E-117);
  sin_t43  : constant octo_double
           := create( 1.34580708507126195E-01,-9.16703591714806995E-18,
                      1.59164930070739730E-34, 4.02370024843668325E-51,
                     -1.02717877978233728E-67, 6.23598935311893397E-84,
                     -4.49215719918551625E-100,-9.95078960347874627E-117);
  sin_t44  : constant octo_double
           := create( 1.37620121586486038E-01, 6.62532558667744818E-18,
                     -2.37465830314014593E-34,-9.37038761730932495E-52,
                      2.42411173021019925E-68,-2.84602100800227938E-85,
                      2.59763439629371080E-101, 1.55086840385294101E-117);
  sin_t45  : constant octo_double
           := create( 1.40658239332849239E-01,-7.91939329655247413E-18,
                      6.09724642021083971E-34, 2.45666232410357969E-50,
                     -2.10711102615167760E-66,-1.28052439913987791E-82,
                      1.60259496952782748E-100, 3.09004581205557797E-117);
  sin_t46  : constant octo_double
           := create( 1.43695033150294443E-01, 1.14727230166186663E-17,
                     -5.18849545575764348E-35,-4.22206848321866068E-51,
                     -1.10194648716812339E-67,-1.16705291578018202E-84,
                      4.50875588782240834E-101,-1.97165234463791243E-117);
  sin_t47  : constant octo_double
           := create( 1.46730474455361748E-01, 3.72694714704656775E-18,
                      3.73523981512508271E-34,-4.08818222895086342E-51,
                      1.55176390752754226E-68, 9.78372264456757519E-85,
                     -1.81754325445953092E-101,-4.47428701685497407E-118);
  sin_t48  : constant octo_double
           := create( 1.49764534677321509E-01, 8.08121141312851509E-18,
                      1.29791425549173248E-34, 9.93806674877362537E-51,
                      5.73740287319276834E-68, 3.38366048667785719E-84,
                      6.61985823237986893E-101, 5.96157298606967856E-117);
  sin_t49  : constant octo_double
           := create( 1.52797185258443435E-01,-7.63135739384168378E-18,
                      5.77146904502841248E-34,-3.77311325829866872E-50,
                     -1.12335157722231889E-66, 1.37371132596995389E-83,
                     -1.22858617667660062E-100, 7.32642194958219683E-117);
  sin_t50  : constant octo_double
           := create( 1.55828397654265233E-01, 3.03513071876782208E-18,
                     -1.09769423151761844E-34, 7.87346476852578675E-51,
                      4.24392301767977857E-67,-2.21959429914176478E-83,
                     -3.18443002931819062E-100,-1.13688942801505243E-116);
  sin_t51  : constant octo_double
           := create( 1.58858143333861446E-01,-4.01632005738590786E-18,
                     -9.28405802576288120E-35,-2.85674200292748750E-51,
                      1.78746767309220706E-67,-1.62515863533510290E-83,
                      4.31051834545506324E-100, 1.82823036365252451E-116);
  sin_t52  : constant octo_double
           := create( 1.61886393780111826E-01, 1.18505196435735283E-17,
                     -5.04409905191629570E-34, 3.05100287079280095E-50,
                     -4.30947315708130251E-67,-3.19794395349932226E-83,
                     -1.03680203634885848E-99, 1.97873415107642996E-116);
  sin_t53  : constant octo_double
           := create( 1.64913120489969922E-01,-7.04052883191667383E-19,
                      3.32111074912455269E-35, 8.66632992546860315E-52,
                      6.70716075759242849E-68, 3.77905336370558319E-84,
                     -1.60556977933041428E-101,-4.53336802474728143E-118);
  sin_t54  : constant octo_double
           := create( 1.67938294974731173E-01, 5.42845337215581389E-18,
                     -3.32633393361813689E-34,-1.85363673351238475E-50,
                      6.37212819004787885E-67,-3.02607576348603856E-83,
                     -7.54950355176388181E-101, 1.68878087995375344E-117);
  sin_t55  : constant octo_double
           := create( 1.70961888760301217E-01, 9.19199801817590944E-18,
                     -6.76887439409826057E-34,-1.03777113843183886E-50,
                      1.86569247309858739E-67,-1.08062018121900044E-83,
                      5.83208295835655638E-100, 1.92369394449120577E-116);
  sin_t56  : constant octo_double
           := create( 1.73983873387463822E-01, 5.81519946181079284E-18,
                     -1.67510142983016057E-34,-6.69822597971649632E-51,
                      2.68981537458849633E-67,-1.45125866211886185E-83,
                     -6.18686844530808796E-100,-2.56175648208501112E-116);
  sin_t57  : constant octo_double
           := create( 1.77004220412148749E-01, 6.73293006354081671E-18,
                      2.80427366442466233E-34, 3.67868882327935990E-51,
                      1.38727517272938610E-67,-2.11240491184557759E-84,
                      9.43779315921888920E-101, 4.15288787564091342E-117);
  sin_t58  : constant octo_double
           := create( 1.80022901405699515E-01, 7.97018260473921427E-18,
                     -7.07659201105249775E-34, 1.96225126084617837E-50,
                      1.01456273354976480E-66,-2.16414525056017038E-83,
                     -1.02935181286685096E-99, 9.20921620032132056E-116);
  sin_t59  : constant octo_double
           := create( 1.83039887955140951E-01, 7.73499186886373835E-18,
                     -4.48037699681450831E-34, 1.12011487933288905E-50,
                     -5.70484980282081912E-68,-2.58441598521855717E-84,
                      2.14990804274506987E-100, 8.45857082297176154E-117);
  sin_t60  : constant octo_double
           := create( 1.86055151663446661E-01,-1.25648930076795521E-17,
                      7.59538442485308101E-34,-3.84716951324150389E-51,
                      2.93494161751752629E-67, 1.51466405418760871E-83,
                      5.87161205470116746E-100,-2.59706696508825614E-116);
  sin_t61  : constant octo_double
           := create( 1.89068664149806220E-01,-7.62089558035277776E-18,
                     -4.47922986566629895E-34, 4.13916466263052541E-50,
                     -7.02958654675518609E-67, 5.78703022169765588E-83,
                     -3.71978807477315389E-100, 9.73865940228132258E-117);
  sin_t62  : constant octo_double
           := create( 1.92080397049892437E-01, 4.33483439411749035E-18,
                     -2.34041218481399366E-34, 1.57899709626118563E-50,
                     -3.57254602216994629E-67, 2.87240307276431677E-83,
                      3.74606176073540396E-100, 1.19551285078245729E-116);
  sin_t63  : constant octo_double
           := create( 1.95090322016128276E-01,-7.99107906846173126E-18,
                      6.18462700242207127E-34,-3.58402709180329370E-50,
                     -9.07403900926455768E-67,-3.41055954373888195E-83,
                      9.54337952630623091E-100, 2.36834608203971332E-117);
  sin_t64  : constant octo_double
           := create( 1.98098410717953588E-01,-1.84344118006894450E-18,
                      1.41390313182372847E-34, 1.05428111253438094E-50,
                      5.04786838497206121E-67,-7.58831413866410294E-84,
                     -3.20942765989283302E-100, 1.62926053987493820E-116);
  sin_t65  : constant octo_double
           := create( 2.01104634842091901E-01, 1.10100326693007386E-17,
                     -3.91235767574137906E-34, 2.40848525000635315E-51,
                      5.10723190909262353E-68, 2.57466667969522560E-84,
                     -2.20644792926840395E-100,-9.12131216069626137E-117);
  sin_t66  : constant octo_double
           := create( 2.04108966092816868E-01, 6.09412977739577521E-18,
                     -2.82754099704496409E-34, 4.61010085635329893E-51,
                     -2.34161035446829554E-67, 1.22639916348963898E-83,
                      1.78248112199774571E-100,-1.76637863202369565E-117);
  sin_t67  : constant octo_double
           := create( 2.07111376192218560E-01,-1.06133625289713558E-17,
                      2.24568051126908840E-34, 1.34837361252809044E-50,
                      9.08964713237916564E-67,-3.18691210021983370E-84,
                     -1.85758364055659700E-100,-1.01768078020383412E-117);
  sin_t68  : constant octo_double
           := create( 2.10111836880469610E-01, 1.15615484765128436E-17,
                      6.03559056104012542E-34, 3.33299096184056749E-50,
                     -2.21966170571301981E-66, 2.93685728359414516E-83,
                      5.70850349398451920E-100, 4.12171713298719901E-116);
  sin_t69  : constant octo_double
           := create( 2.13110319916091362E-01, 1.20318738210638598E-17,
                     -3.41426997196956354E-34,-1.24362627802417777E-50,
                     -2.65527034405922961E-67, 9.23715882768353469E-84,
                     -3.88430382525409780E-100,-3.39610516242628183E-118);
  sin_t70  : constant octo_double
           := create( 2.16106797076219520E-01,-1.01111960826091175E-17,
                      7.27895453351896431E-34,-2.93475403652586095E-50,
                      1.18882891152961383E-66,-3.88293994734515188E-83,
                      2.10742693032734185E-100, 3.89313339745640687E-117);
  sin_t71  : constant octo_double
           := create( 2.19101240156869798E-01,-3.65138122991507758E-19,
                     -2.33594994186064421E-35, 3.17852981984586531E-52,
                      1.63757063858449075E-68, 4.96563698141715738E-85,
                      2.59699358857451374E-101, 1.36969124298657282E-117);
  sin_t72  : constant octo_double
           := create( 2.22093620973203537E-01,-3.03372109958121623E-18,
                      6.66546680336329981E-35, 2.01108623226569421E-51,
                      6.15985763115796938E-68, 1.49991836323130075E-84,
                     -4.67327451259251333E-101, 2.91885891613829426E-117);
  sin_t73  : constant octo_double
           := create( 2.25083911359792832E-01, 3.95070408225565101E-18,
                      2.42879939583053746E-35, 5.66627975130203223E-52,
                      1.50886598195277854E-68, 5.91715463771356825E-85,
                     -3.40819580874471617E-101,-9.84762128874968364E-118);
  sin_t74  : constant octo_double
           := create( 2.28072083170885731E-01, 8.23618373392580116E-18,
                      6.97867813163979374E-34,-6.41229624826395036E-51,
                     -4.03894879011491570E-67, 6.04543349327994713E-84,
                      2.06739106807475888E-101, 9.44078955647149053E-118);
  sin_t75  : constant octo_double
           := create( 2.31058108280671110E-01, 1.01297871497618686E-17,
                     -6.93592346158160440E-34,-2.88773556048837821E-50,
                      1.59688388065607953E-66,-8.33259843562148299E-83,
                     -3.78334872586390280E-99,-2.48981034606182887E-115);
  sin_t76  : constant octo_double
           := create( 2.34041958583543430E-01,-6.99224026961011726E-18,
                     -5.73230319227502798E-34, 5.30925799668727269E-51,
                     -1.33071021660334324E-67, 6.87979134931095779E-84,
                     -3.50704739589457128E-100,-1.62990922734006731E-116);
  sin_t77  : constant octo_double
           := create( 2.37023605994367198E-01, 8.85448522850399180E-18,
                      1.35884808263541335E-34, 1.03810225202138672E-50,
                     -4.78361238841613321E-67, 2.13573033443488305E-83,
                     -5.81317983874262780E-100, 1.11859921298045118E-116);
  sin_t78  : constant octo_double
           := create( 2.40003022448741499E-01,-1.21377589756321639E-17,
                     -2.64488077317038911E-34,-1.99297338006704726E-51,
                     -7.70893711383538847E-68,-2.25220579180167779E-84,
                      1.82601089979310064E-100, 4.69253293840164990E-117);
  sin_t79  : constant octo_double
           := create( 2.42980179903263899E-01,-8.75143152971966316E-18,
                     -6.57232603730794308E-34,-1.03331580831721772E-50,
                     -7.67303665548996940E-68,-1.64207904731718957E-85,
                     -7.72587767773904461E-102,-4.19026554882611894E-118);
  sin_t80  : constant octo_double
           := create( 2.45955050335794623E-01,-1.11290440527418318E-17,
                      4.38059982028833967E-34, 1.22193995546862912E-50,
                      3.02355370311391133E-67, 2.48465122464541366E-83,
                     -3.44756092825038672E-100, 1.16426914159946003E-116);
  sin_t81  : constant octo_double
           := create( 2.48927605745720176E-01,-8.17834361000209898E-18,
                      5.56668752611118396E-34, 3.80804730587481670E-50,
                     -1.10140116930784744E-66, 4.42088080681673841E-83,
                      2.99921010737057309E-99,-1.34096087439631684E-115);
  sin_t82  : constant octo_double
           := create( 2.51897818154216968E-01,-1.75914360325170387E-17,
                     -1.09596812325252852E-33, 5.62094260202324558E-50,
                     -4.28112473828177472E-66,-3.58163542964403378E-83,
                     -2.40556113146921592E-99,-2.98396626783772226E-116);
  sin_t83  : constant octo_double
           := create( 2.54865659604514572E-01,-1.36022998069014613E-19,
                     -6.00738446427625355E-36,-3.00727513118938785E-52,
                     -1.22704495507404083E-68,-8.03034598714414042E-85,
                     -5.46922327998671672E-101, 1.39912585450205386E-117);
  sin_t84  : constant octo_double
           := create( 2.57831102162158987E-01, 1.84800386308799567E-17,
                      3.32016647140475995E-34,-5.55478192905767638E-51,
                      3.69201202971236860E-67, 1.69300719286625139E-83,
                     -3.74999602985559879E-100, 5.60028936238932763E-117);
  sin_t85  : constant octo_double
           := create( 2.60794117915275514E-01, 4.27214209835500749E-18,
                      5.67821269347779200E-35, 3.14283380843653967E-51,
                      1.83862654848826093E-67, 1.02585540224865971E-83,
                     -3.08805031036283959E-100, 1.85281267072560652E-116);
  sin_t86  : constant octo_double
           := create( 2.63754678974831402E-01,-1.88379476800386996E-17,
                      1.37201290457547940E-33,-8.27634066659660331E-50,
                      4.03232747874737897E-66,-1.97524885946060865E-82,
                      1.06634388098399124E-99, 2.08519022343101556E-116);
  sin_t87  : constant octo_double
           := create( 2.66712757474898365E-01, 2.09412225788266884E-17,
                     -1.13034665247279894E-33, 1.99542240505089630E-50,
                     -1.15018997376335405E-66,-3.53906497779610396E-83,
                      2.01080326428584159E-99,-5.51062706665966420E-116);
  sin_t88  : constant octo_double
           := create( 2.69668325572915091E-01, 1.57656576181332587E-17,
                     -6.96961421733700858E-34,-4.04553468791467760E-50,
                     -1.42275530120596467E-66, 5.06661291343617801E-83,
                      3.29710618519863633E-99, 1.32985242526478086E-115);
  sin_t89  : constant octo_double
           := create( 2.72621355449948977E-01, 7.86971660763878500E-18,
                      6.61793886029333718E-35,-2.76429036963862670E-51,
                      1.66189023046745488E-67, 1.06574295253096451E-83,
                     -4.65554643535425639E-100, 2.73346293776479004E-116);
  sin_t90  : constant octo_double
           := create( 2.75571819310958144E-01, 1.93203289625565816E-17,
                      1.39320941801002799E-33, 1.36172539200181157E-50,
                     -8.59903883753700632E-67,-2.45914777761687186E-83,
                      2.76443812552678445E-100, 7.29703813514032750E-117);
  sin_t91  : constant octo_double
           := create( 2.78519689385053115E-01,-1.00302737195435440E-17,
                      7.25921153256892535E-34,-1.00685162966558511E-50,
                     -4.53535017230020529E-67, 2.62187814298213566E-83,
                      6.27276949158705022E-100, 4.21437996812238222E-116);
  sin_t92  : constant octo_double
           := create( 2.81464937925757996E-01,-1.23222996412740093E-17,
                     -1.05647887063864348E-34, 7.51374242512658853E-51,
                      3.58665823582072999E-67,-2.47392221022071423E-83,
                     -1.70635428822967862E-99, 6.92276798385997583E-116);
  sin_t93  : constant octo_double
           := create( 2.84407537211271821E-01, 2.22092685106614753E-17,
                     -9.18230956295237080E-34,-5.21928753088922177E-50,
                     -1.03445935928443390E-66, 5.13647689477255948E-83,
                     -1.63059935127107675E-99,-5.04002555843145888E-116);
  sin_t94  : constant octo_double
           := create( 2.87347459544729511E-01, 1.54611173676457170E-17,
                     -6.32639736634440759E-34,-2.29825384164762143E-50,
                     -9.62276988638651218E-68,-1.00730303857294149E-84,
                      4.98254688228871583E-101, 8.01701050452985206E-118);
  sin_t95  : constant octo_double
           := create( 2.90284677254462387E-01,-1.89279787077742515E-17,
                      1.15229531571423147E-33, 7.47386556547165956E-50,
                     -3.41637702108171592E-66, 8.71048343217906042E-83,
                     -2.23401472329868530E-99,-4.69668073672083961E-116);
  sin_t96  : constant octo_double
           := create( 2.93219162694258628E-01, 2.23854308119018331E-17,
                      1.36624846465396802E-33,-4.24513252539969377E-50,
                     -1.38873788365405033E-66, 1.00620826781653700E-82,
                      2.07492508636586772E-100,-2.83425474648516188E-117);
  sin_t97  : constant octo_double
           := create( 2.96150888243623844E-01,-2.02207363608769380E-17,
                     -7.92522125339204134E-35,-2.89905777295724702E-51,
                      8.36274648052424006E-70,-1.49841704580304461E-86,
                     -5.30876075145934179E-103,-2.41162210997145315E-119);
  sin_t98  : constant octo_double
           := create( 2.99079826308040475E-01, 1.67011816092194466E-18,
                      8.60911511173162921E-35, 3.99312862300121018E-52,
                      3.42705154233895770E-68, 7.58741280448239564E-85,
                      7.11192384277790360E-102, 2.29703490939575048E-118);
  sin_t99  : constant octo_double
           := create( 3.02005949319228084E-01,-1.71676662352624742E-17,
                      2.33361821490080688E-34, 8.30253345552200042E-51,
                     -8.49514740336632099E-68, 1.50435736158517223E-84,
                      5.80260295138377131E-101,-4.02615119145615817E-117);
  sin_t100 : constant octo_double
           := create( 3.04929229735402429E-01,-2.29890338981912620E-17,
                     -1.45989010996611334E-34, 3.77604876931218273E-51,
                     -4.36859442473001706E-68, 1.81704008552938600E-84,
                     -1.05219904323698258E-100,-3.21896656097575478E-117);
  sin_t101 : constant octo_double
           := create( 3.07849640041534867E-01, 2.70740885272451850E-17,
                      1.25688582068992839E-33, 7.29318151059016452E-50,
                     -4.34403975620837728E-66, 2.59392226067851353E-82,
                     -1.19045910069809790E-98,-3.71047086983658618E-115);
  sin_t102 : constant octo_double
           := create( 3.10767152749611475E-01, 2.08870763640485131E-17,
                     -3.01305907910659423E-34, 1.38767390099351791E-51,
                      7.05850317048596351E-68, 1.64691692440556230E-84,
                     -3.23065832264728263E-101, 3.53582022202242265E-119);
  sin_t103 : constant octo_double
           := create( 3.13681740398891462E-01, 1.45604472999689122E-17,
                      3.65641868980115946E-34, 1.16542647349993749E-50,
                      2.20627987078892904E-67, 1.44241743542421191E-83,
                      3.93419225290242587E-100, 1.92409876585653877E-116);
  sin_t104 : constant octo_double
           := create( 3.16593375556165846E-01, 2.14352925127262832E-17,
                      1.23381692313773158E-33, 3.39635421009892933E-50,
                     -1.93728229714585417E-66, 1.23572972769286304E-82,
                     -3.99620694623083571E-99,-1.08259753528319005E-115);
  sin_t105 : constant octo_double
           := create( 3.19502030816015692E-01,-1.39815624910966264E-17,
                      8.17300006974113497E-34,-7.76710962702109518E-50,
                      2.01858875781622656E-66,-9.80058308180401479E-84,
                      5.54096900974084408E-100, 2.00824774002142862E-117);
  sin_t106 : constant octo_double
           := create( 3.22407678801069852E-01,-4.05190399379593982E-18,
                      3.74383027802967961E-34, 8.79367310466391951E-51,
                     -8.55228338329538054E-68, 3.55838653159921593E-84,
                     -4.69655825682594782E-101, 1.92516635676656938E-117);
  sin_t107 : constant octo_double
           := create( 3.25310292162262926E-01, 7.91712494637658925E-18,
                     -6.75766220681463915E-35, 2.30216550669295381E-51,
                     -1.19255802605666878E-67, 5.65529219697017358E-84,
                      9.69701113768915973E-101, 3.14387562288893286E-117);
  sin_t108 : constant octo_double
           := create( 3.28209843579092553E-01,-2.66931407196418959E-17,
                      7.89288514475347883E-34, 2.55251638219878091E-51,
                     -1.19204092531719092E-67, 2.68221919301586565E-85,
                      3.80270310625008163E-102, 2.98968140168781494E-118);
  sin_t109 : constant octo_double
           := create( 3.31106305759876429E-01,-2.74694654747786940E-17,
                     -1.34012459166102057E-33, 6.55317624899761627E-50,
                      4.16101557520311717E-66, 1.39395796768404633E-82,
                      2.38622353481304296E-99,-4.86989223390849381E-116);
  sin_t110 : constant octo_double
           := create( 3.33999651442009382E-01, 2.25989868062881418E-17,
                      7.80630571925861152E-34, 2.04276008954866827E-50,
                     -5.00592200578385284E-67,-9.79997812035375432E-84,
                      7.87951887594339420E-100,-5.04488260683143345E-116);
  sin_t111 : constant octo_double
           := create( 3.36889853392220051E-01,-4.20009400334750924E-19,
                     -2.91786529699854378E-36,-1.15973764370367485E-52,
                      8.13714082453078616E-69, 4.95777216853127969E-85,
                     -8.93396613272086084E-103, 2.44496916950042010E-119);
  sin_t112 : constant octo_double
           := create( 3.39776884406826851E-01, 6.60286794994182819E-18,
                      1.25750099886696825E-34, 2.55690676990083039E-51,
                     -1.06829152215916100E-67, 8.53518010985346333E-85,
                     -1.76434729744117232E-101, 1.43187327275854025E-117);
  sin_t113 : constant octo_double
           := create( 3.42660717311994378E-01, 1.92615184493063189E-17,
                     -9.27541891359908673E-34, 8.54399966873901659E-50,
                      9.32963304574866163E-68,-8.03396329164393270E-84,
                     -3.28401482209420453E-100,-1.48798544009986274E-116);
  sin_t114 : constant octo_double
           := create( 3.45541324963989038E-01, 2.72511436729161234E-17,
                      7.01381636019417366E-34,-1.41762921974540153E-50,
                      9.42576230036345916E-67,-1.68806522916280685E-83,
                     -3.82669209245553588E-100, 2.47123766397266121E-116);
  sin_t115 : constant octo_double
           := create( 3.48418680249434565E-01, 3.69744205142049177E-18,
                      3.55321468784999958E-34, 1.95654625445013218E-50,
                      2.02352641219016910E-67, 3.74958059978421188E-84,
                      1.36249301317932123E-100,-5.53771067941087221E-117);
  sin_t116 : constant octo_double
           := create( 3.51292756085567148E-01,-2.26707120987958437E-17,
                     -1.69942166731396328E-34, 9.11056160345304866E-51,
                     -5.17862842813958020E-67, 2.73370049819498641E-83,
                     -1.70217726184498056E-99,-1.46843964222269468E-118);
  sin_t117 : constant octo_double
           := create( 3.54163525420490399E-01,-1.69517633057648601E-17,
                      1.27723317778146171E-33,-3.37037854358433103E-50,
                      1.79793457972255570E-66,-8.95427957871373118E-83,
                     -3.59760475053243210E-99, 1.40941204508121450E-115);
  sin_t118 : constant octo_double
           := create( 3.57030961233430033E-01,-4.82181911379191658E-19,
                     -4.16724369944923610E-35,-7.15311671493643523E-52,
                      6.20967593983977441E-68,-7.31675547696572083E-85,
                     -2.99949921625586749E-101,-9.99398420040930671E-118);
  sin_t119 : constant octo_double
           := create( 3.59895036534988166E-01,-1.76016871238392825E-17,
                      1.33751254730467910E-33, 7.94678155935843403E-50,
                     -3.56009437536984743E-66, 1.64674395332976333E-83,
                     -4.63807576067395367E-100,-4.52965212454481292E-116);
  sin_t120 : constant octo_double
           := create( 3.62755724367397225E-01,-9.16683526637498492E-18,
                     -7.43178439569367352E-34,-2.01995825118045641E-50,
                     -1.17856641422082888E-66, 6.72281042886257665E-84,
                      1.45648315934707610E-101,-9.77920682049448279E-118);
  sin_t121 : constant octo_double
           := create( 3.65612997804773854E-01, 1.62178987704575461E-17,
                      1.12869701519610553E-33,-7.18252873181390098E-50,
                      2.49980797789504764E-66,-2.14890693511455580E-82,
                     -9.96977516832914542E-99,-3.07864567708923780E-116);
  sin_t122 : constant octo_double
           := create( 3.68466829953372321E-01, 1.04636407961592684E-17,
                      2.05549847385173044E-35, 1.04418613056187690E-51,
                      6.63908600018281761E-68, 3.30903847157543902E-84,
                      7.72218329838374556E-101, 5.25922086312600548E-119);
  sin_t123 : constant octo_double
           := create( 3.71317193951837543E-01, 3.47492396482382656E-19,
                     -7.51510530428666715E-37,-2.81534684386508515E-53,
                      1.54670382177623091E-69, 4.76151224631190364E-86,
                     -1.55237787921883323E-102, 2.98041432263897182E-119);
  sin_t124 : constant octo_double
           := create( 3.74164062971457989E-01, 8.01141037619621177E-18,
                      5.34295998134060520E-34, 1.03513787965392099E-50,
                      1.26070376979965845E-67, 5.71975152402306685E-84,
                     -1.08743408690080497E-100, 5.39776830662065205E-117);
  sin_t125 : constant octo_double
           := create( 3.77007410216418259E-01,-2.72553020419569299E-18,
                      6.36465864450181373E-35, 8.30486571765035594E-52,
                     -1.47241614510515818E-68,-1.28446846708744689E-85,
                      8.79014244401513679E-103,-9.73195178218488810E-120);
  sin_t126 : constant octo_double
           := create( 3.79847208924051161E-01, 9.91513058551723705E-18,
                      4.87614096972248857E-34, 1.40250840007767047E-50,
                      1.50124853831174294E-67, 2.07221245953549079E-84,
                      1.56274968439721697E-100,-1.13477932768015179E-116);
  sin_t127 : constant octo_double
           := create( 3.82683432365089782E-01,-1.00507726964615876E-17,
                     -2.06053163028066947E-34,-1.27177246980852050E-50,
                      6.07736561771507054E-67,-2.18213633675164148E-83,
                     -8.42501228457432599E-100,-4.55038691861614569E-116);
  sin_t128 : constant octo_double
           := create( 3.85516053843918849E-01, 1.51776653964723134E-17,
                      1.41982305180165351E-33, 5.89551671599042348E-50,
                     -6.60618952452351693E-67,-8.38410465649804950E-84,
                      4.47036513330724044E-100,-2.26738741002153613E-116);
  sin_t129 : constant octo_double
           := create( 3.88345046698826302E-01,-1.00537705983987169E-17,
                      7.59429992550571310E-34,-3.19679740466542192E-50,
                     -5.54944516189153787E-68, 1.81788043273150058E-84,
                     -3.12226350160964040E-101, 1.78299606718499133E-117);
  sin_t130 : constant octo_double
           := create( 3.91170384302253871E-01, 1.79977878582439952E-17,
                     -1.06134824026098561E-33,-5.45821488177910320E-50,
                     -9.71868199622811197E-67,-6.50416395633437059E-83,
                      2.48199913080417198E-99, 3.40963125384574983E-116);
  sin_t131 : constant octo_double
           := create( 3.93992040061048099E-01, 9.76492416412393360E-18,
                     -2.12335994412846166E-34,-5.55298367953408191E-51,
                      3.90719210847308485E-67, 3.06283560819357113E-83,
                     -1.71821561323903406E-100,-8.25917849179282102E-117);
  sin_t132 : constant octo_double
           := create( 3.96809987416710308E-01, 2.05450636708401260E-17,
                      6.13470588019228416E-34, 1.07337881506364299E-50,
                      4.16361014621386142E-67,-1.90389115936262464E-83,
                     -1.15158381762066368E-99,-8.62145960853751498E-116);
  sin_t133 : constant octo_double
           := create( 3.99624199845646844E-01,-1.50654974761893716E-17,
                     -9.96532588818672981E-34,-5.75243237127253548E-50,
                      6.21979448232916408E-67,-6.03832619886155209E-83,
                     -4.22356612309472786E-101,-2.94108201068720585E-117);
  sin_t134 : constant octo_double
           := create( 4.02434650859418430E-01, 1.09026193393282696E-17,
                      7.39985281259897649E-34, 2.27457848068234989E-50,
                      1.60225633135131042E-66, 4.43217656927782080E-83,
                     -1.02315094071943250E-100,-4.27996593315948612E-117);
  sin_t135 : constant octo_double
           := create( 4.05241314004989861E-01, 9.91114019428998842E-18,
                     -2.51690708954346477E-34, 9.27729848184365734E-53,
                      4.19931751188889834E-69,-7.68595548245619712E-86,
                     -3.88866589693576813E-102,-1.32848673446719518E-118);
  sin_t136 : constant octo_double
           := create( 4.08044162864978688E-01,-7.00060151373513107E-18,
                     -1.41082073342682282E-34, 1.51755469975771358E-52,
                     -8.58011210399058743E-69,-4.35983787910480077E-85,
                      2.18950602338367537E-101, 7.39785014279769569E-119);
  sin_t137 : constant octo_double
           := create( 4.10843171057903966E-01,-2.42198351903554993E-17,
                     -1.14189029253133137E-33,-2.09968431650934681E-50,
                     -3.58396626808437780E-67, 4.68857604112512300E-84,
                     -1.58524753709479526E-100,-1.09295480339526143E-116);
  sin_t138 : constant octo_double
           := create( 4.13638312238434558E-01,-1.03939849405978709E-17,
                     -1.14816811745038799E-34,-2.02810528510286799E-51,
                     -1.32198238091455296E-67,-3.18722046570921303E-84,
                     -1.10457573774508088E-100,-3.27769307628216940E-117);
  sin_t139 : constant octo_double
           := create( 4.16429560097637208E-01,-2.54755804131317323E-17,
                     -3.44826785061128235E-34, 7.17886193518654803E-51,
                      4.67879696786894412E-67,-2.95932264463873355E-83,
                     -1.60900321794911505E-99, 1.07371726320995739E-116);
  sin_t140 : constant octo_double
           := create( 4.19216888363223961E-01,-4.22324637501105896E-18,
                     -3.60530230452557941E-34, 2.05545621514494970E-50,
                      1.16846638534326586E-67, 4.33787084636179890E-84,
                     -3.55027388994698667E-100,-3.86111864440964539E-117);
  sin_t141 : constant octo_double
           := create( 4.22000270799799682E-01, 4.35432669941285275E-18,
                      3.17343102722511904E-34,-1.35732479807386682E-50,
                      1.04323924361772305E-66, 1.65084724226252926E-83,
                     -2.79274295985653372E-100,-2.34858230027357275E-116);
  sin_t142 : constant octo_double
           := create( 4.24779681209108806E-01, 2.74623122042772811E-17,
                     -4.65528478021119482E-34, 6.59617810991931222E-51,
                     -1.34133607229630039E-67,-6.38348523402678775E-84,
                     -6.35930435215519189E-101, 6.28545860427507004E-117);
  sin_t143 : constant octo_double
           := create( 4.27555093430282085E-01, 9.41118981629547262E-18,
                     -1.74466824265988006E-34,-2.20544926264801693E-51,
                     -5.15215851093596162E-68, 1.03208115815492520E-84,
                      6.53956613897519999E-101, 1.99227926691509150E-117);
  sin_t144 : constant octo_double
           := create( 4.30326481340082612E-01, 2.22596869740926898E-17,
                      8.59725913140850753E-34,-2.94208978890030203E-50,
                      1.75926705124073484E-66,-2.11017639613453163E-83,
                     -1.70028712866050335E-100,-4.18327177821188638E-117);
  sin_t145 : constant octo_double
           := create( 4.33093818853151957E-01, 1.12242833298475168E-17,
                      5.32237480410756612E-35,-5.29843957766736142E-51,
                      6.76146648124111829E-68,-3.01616882584315782E-84,
                     -2.70870956048747896E-101,-1.46405093803331340E-117);
  sin_t146 : constant octo_double
           := create( 4.35857079922255475E-01, 1.62305154506445271E-17,
                     -6.43714490635794420E-35, 3.78081519223010690E-51,
                     -1.05443929214259962E-67, 6.55631269454937484E-84,
                      1.65208077276557663E-100, 1.11624333718965509E-117);
  sin_t147 : constant octo_double
           := create( 4.38616238538527659E-01,-2.08833158310750902E-17,
                     -1.42595835408918766E-34, 6.38647635906570774E-52,
                      1.76788005853839990E-68,-9.42253457285591945E-85,
                      5.61541325456574757E-101,-3.18038472694768712E-118);
  sin_t148 : constant octo_double
           := create( 4.41371268731716671E-01, 2.23607838869649689E-17,
                      1.18647696035157695E-34,-3.80870032661892321E-51,
                     -9.31638331325958014E-68,-5.88740846958704811E-84,
                      2.41598936545314791E-101,-1.00530219499800980E-118);
  sin_t149 : constant octo_double
           := create( 4.44122144570429256E-01,-2.42188744221783147E-17,
                      2.22052308387039070E-34, 9.21330359113562581E-51,
                      1.75117701549931936E-67, 4.34385628654253481E-84,
                     -1.59650430253157971E-100, 1.05368100122087423E-116);
  sin_t150 : constant octo_double
           := create( 4.46868840162374215E-01,-1.92221361423093823E-17,
                     -4.44256785897320493E-35,-1.36736092921495352E-51,
                     -7.40632758976008092E-68,-2.87677356289550370E-84,
                      3.38297030420845157E-101,-2.17930928276559544E-117);
  sin_t151 : constant octo_double
           := create( 4.49611329654606595E-01, 4.88319242320352435E-18,
                      2.71510844981913810E-34,-1.56539931716131535E-50,
                     -9.84328637079578728E-67, 5.42013325779863042E-84,
                      3.06626570250924240E-100, 2.49557333332829462E-116);
  sin_t152 : constant octo_double
           := create( 4.52349587233770889E-01,-1.48279774721961223E-17,
                     -7.69475010889723245E-34, 1.76568568820313190E-50,
                      4.78856030489328032E-67, 1.05757287492197087E-83,
                     -8.96139401905120995E-100, 3.56844859016001897E-116);
  sin_t153 : constant octo_double
           := create( 4.55083587126343836E-01,-1.23799067581164721E-17,
                      5.52896889555426431E-34,-8.53823128402093862E-51,
                      4.35761034232550185E-67, 1.06275321495045154E-83,
                     -3.83289275430714933E-100, 2.48658300545672635E-116);
  sin_t154 : constant octo_double
           := create( 4.57813303598877230E-01,-8.45542549222959487E-18,
                     -6.37703942467642634E-34, 3.17782535755642491E-50,
                      1.63954755347278651E-66, 9.94921380496366312E-83,
                     -2.34369765957016175E-99, 9.27923204206177782E-116);
  sin_t155 : constant octo_double
           := create( 4.60538710958240005E-01, 1.84887774921778717E-17,
                     -1.05277321542097255E-33, 3.32355934909471024E-50,
                      1.48893748691692587E-66,-8.05039814004150823E-83,
                      5.16182103159212165E-100,-3.68003478412553394E-116);
  sin_t156 : constant octo_double
           := create( 4.63259783551860205E-01,-7.35149245332317067E-18,
                      6.71753968817070354E-34, 3.95941276121233792E-50,
                      6.23152884028039170E-67, 4.84227359210742245E-83,
                      5.50191026148806027E-100, 2.64394033125971875E-116);
  sin_t157 : constant octo_double
           := create( 4.65976495767966181E-01,-3.30235477782351354E-18,
                      3.49046770504768909E-35,-1.89714389379696671E-51,
                     -2.57573587999938835E-68,-7.20917984486729402E-87,
                     -2.33081932508047651E-103, 1.27248390569055507E-119);
  sin_t158 : constant octo_double
           := create( 4.68688822035827957E-01,-2.29492516818450541E-17,
                     -1.13647576418236581E-33, 6.88405225019186122E-50,
                      2.37995478956060175E-66,-1.76643806017233826E-82,
                     -1.33557531813477976E-99,-2.70042577896921252E-116);
  sin_t159 : constant octo_double
           := create( 4.71396736825997642E-01, 6.51667813606901296E-18,
                      2.94575469662359839E-34,-6.21597177388366305E-51,
                      4.90363168400334462E-67,-1.54734589077876804E-83,
                      5.01089000683439180E-100, 8.62267126631534475E-117);
  sin_t160 : constant octo_double
           := create( 4.74100214650550023E-01,-8.14516015489780752E-18,
                     -3.47894485556144223E-34,-1.16819439746585078E-50,
                     -1.00679196118875977E-66,-8.14318999812223865E-84,
                     -3.35311680847673620E-100,-2.55012544402769616E-117);
  sin_t161 : constant octo_double
           := create( 4.76799230063322144E-01,-1.02935153383057945E-17,
                     -3.65820450083699518E-34, 1.74241314791764755E-50,
                      7.64290068643720225E-68,-7.94821442098581988E-84,
                      4.02089145833352637E-100, 4.53026336969603360E-119);
  sin_t162 : constant octo_double
           := create( 4.79493757660153008E-01, 1.84199996626847708E-17,
                     -1.30408386212733120E-33, 1.09771318222464710E-50,
                     -1.01452353259774662E-66,-4.92872791661231722E-83,
                     -2.87470906671517064E-99, 1.66793539379023301E-115);
  sin_t163 : constant octo_double
           := create( 4.82183772079122774E-01,-2.58615009255204420E-17,
                     -6.29131976065000067E-36, 4.08023598086847262E-52,
                     -3.27122422823965083E-68,-9.81131579124486336E-85,
                      5.13316853105007495E-101,-8.00846692593906573E-118);
  sin_t164 : constant octo_double
           := create( 4.84869248000791120E-01,-1.80340042032622450E-17,
                     -3.52442769069580443E-34,-1.71383186547492462E-50,
                      2.73490284006916009E-67, 1.54269479875197795E-84,
                      1.03368944895574248E-100,-5.76898104380222011E-117);
  sin_t165 : constant octo_double
           := create( 4.87550160148435940E-01, 1.42310909312736532E-17,
                     -1.82777330732626972E-34,-1.52082917904295566E-51,
                      1.90835864893336768E-68,-1.67966347634175453E-84,
                     -2.59839893899196797E-101,-1.47212545304801719E-117);
  sin_t166 : constant octo_double
           := create( 4.90226483288291159E-01,-5.14961456434404041E-18,
                     -3.69030274052841041E-34, 1.51729400951513044E-50,
                      9.51177428868984434E-67, 5.21488749326745008E-83,
                      3.09097894796168815E-101, 9.92307920152893204E-118);
  sin_t167 : constant octo_double
           := create( 4.92898192229784038E-01,-1.02578316765621855E-18,
                      6.95208177608850686E-35,-2.42609612140903889E-51,
                      8.42984754135809199E-68, 6.29390299185880572E-84,
                      3.22328918712547932E-100,-1.06177307471458175E-116);
  sin_t168 : constant octo_double
           := create( 4.95565261825772541E-01,-9.43232419423653616E-18,
                      3.12129186576991429E-35, 4.20090723752427356E-52,
                      3.33881771940916825E-68,-4.85326709650289342E-85,
                      1.50436676336302327E-101,-1.01429678586771007E-117);
  sin_t169 : constant octo_double
           := create( 4.98227666972781869E-01,-1.61263838305407977E-17,
                     -1.50928973192988707E-33, 1.10492988908959171E-50,
                     -8.19156114106015927E-67, 2.86055839075918389E-83,
                      8.20013328268562607E-100,-6.49778303276352489E-117);
  sin_t170 : constant octo_double
           := create( 5.00885382611240826E-01,-3.96040151470746387E-17,
                     -2.22083952018980066E-33, 1.36482027358394166E-49,
                      1.84246249086235095E-66,-8.64504173724520648E-83,
                      2.44523582308599862E-99, 9.12743793511014537E-116);
  sin_t171 : constant octo_double
           := create( 5.03538383725717575E-01,-1.67313082049674969E-17,
                     -1.01402336440747857E-33, 4.09530719376714769E-50,
                      1.70589927959393278E-66, 7.04473113528281631E-83,
                      4.03917247767082378E-99,-2.54218377887325899E-115);
  sin_t172 : constant octo_double
           := create( 5.06186645345155339E-01,-4.83215929864937112E-17,
                      9.28581072266422520E-34, 4.26998024010370046E-50,
                     -1.26563820798698634E-66, 1.25786903562437943E-82,
                      4.30621553335155989E-99, 5.24668620748806158E-116);
  sin_t173 : constant octo_double
           := create( 5.08830142543106989E-01, 4.78369682680141297E-17,
                     -1.07270229288060348E-33, 2.73093745136727574E-50,
                      1.55903014238376149E-66, 8.60429738645002481E-83,
                      4.59186550036684138E-99, 3.41794553305563843E-115);
  sin_t174 : constant octo_double
           := create( 5.11468850437970413E-01,-1.30880012210075789E-17,
                      4.09290333633668988E-34,-3.79521901534779262E-50,
                      7.49210687958463522E-67, 1.93556276607286329E-83,
                      1.73325249346675038E-99,-2.17196172188454166E-116);
  sin_t175 : constant octo_double
           := create( 5.14102744193221772E-01,-4.57127075236156240E-17,
                      1.54882794422382834E-33,-2.58539593055211305E-50,
                     -5.61934740949359583E-67, 2.78238575662766112E-83,
                      1.66343671905130830E-99, 1.02611613943763643E-116);
  sin_t176 : constant octo_double
           := create( 5.16731799017649873E-01, 8.30186172338365153E-18,
                      5.82510274676952021E-34,-2.28123971905350764E-50,
                     -1.55795495996593512E-66, 5.07807951710912863E-83,
                      2.21573327195689219E-99,-1.89401446286088564E-115);
  sin_t177 : constant octo_double
           := create( 5.19355990165589643E-01,-5.53312481441711451E-17,
                     -3.16283756097690261E-35,-2.40919720511885711E-51,
                     -8.78120270230154531E-68,-7.39730320419324519E-84,
                      3.88695960892179327E-100,-2.12502728831123566E-116);
  sin_t178 : constant octo_double
           := create( 5.21975292937154389E-01,-4.65557956920888829E-17,
                      4.63789809368504302E-34,-3.34705429346895319E-51,
                     -7.95475580886532071E-68,-2.82832721823990219E-84,
                     -7.84623444115327049E-102, 5.51090942525187956E-118);
  sin_t179 : constant octo_double
           := create( 5.24589682678468949E-01,-4.30688690400823448E-17,
                     -4.20131552919320554E-34,-1.50960699267002743E-50,
                     -1.78160444348773757E-67, 7.91114491416668663E-84,
                     -2.00081317689602499E-100, 9.16435719981167617E-118);
  sin_t180 : constant octo_double
           := create( 5.27199134781901391E-01,-4.22029834805606191E-17,
                      8.55859161848672945E-34, 7.99743393367323068E-50,
                      2.22268804804906685E-66, 5.29121454912949153E-84,
                     -2.16739715725003772E-102,-1.17525981418490342E-118);
  sin_t181 : constant octo_double
           := create( 5.29803624686294716E-01,-4.80678417064823419E-17,
                      5.83097210466302958E-34,-8.97407615217566599E-51,
                      2.97235810125729858E-67, 1.22666055200279936E-83,
                     -6.14416122053005295E-102, 3.01011294842719771E-118);
  sin_t182 : constant octo_double
           := create( 5.32403127877198012E-01,-4.10203061358008955E-17,
                     -1.92399963742308207E-33,-1.53269879138121844E-49,
                      8.20626131901962789E-66,-2.46700363515096821E-82,
                      1.05024145684911709E-98,-6.14470941700834373E-115);
  sin_t183 : constant octo_double
           := create( 5.34997619887097264E-01,-5.36831327083581340E-17,
                     -1.39005699188381124E-33, 2.71540847264740923E-50,
                      2.88906455000423559E-68, 1.90484320728190398E-84,
                     -2.77564890279787574E-101,-1.24434420730948051E-117);
  sin_t184 : constant octo_double
           := create( 5.37587076295645505E-01,-2.26173653884030542E-17,
                     -5.97872790334470747E-34, 3.12044197290436254E-51,
                      1.89704306032484395E-67,-1.44937350656576784E-83,
                      5.25087918828526031E-100,-3.88326271881161204E-116);
  sin_t185 : constant octo_double
           := create( 5.40171472729892854E-01, 2.70724479659358388E-17,
                      1.16987997092138293E-33,-5.90946685158815002E-50,
                     -2.39979693168273975E-66,-6.32478836374786016E-83,
                     -2.98607182580382153E-99, 1.01015328857454325E-115);
  sin_t186 : constant octo_double
           := create( 5.42750784864515889E-01, 1.71482610047571013E-17,
                     -1.35259059252008698E-33, 4.97244112907273233E-50,
                      3.92021159501083647E-66, 1.86734876881603372E-82,
                     -6.18559281949176881E-99,-1.73979338208946707E-115);
  sin_t187 : constant octo_double
           := create( 5.45324988422046464E-01,-4.15178175383842578E-17,
                     -1.53189302193859406E-33, 6.36299211014139742E-50,
                      1.31608964229229922E-67,-8.08831724257032142E-84,
                      1.68091787417465648E-100,-1.78894677150068080E-117);
  sin_t188 : constant octo_double
           := create( 5.47894059173100190E-01,-2.40658782971133627E-17,
                     -3.56392136693626063E-36,-2.60132708542716446E-52,
                     -3.91404198232171823E-69, 1.80489682903023005E-85,
                     -8.29286151273548521E-103, 1.55261799951404738E-119);
  sin_t189 : constant octo_double
           := create( 5.50457972936604811E-01,-8.33199030158076632E-18,
                     -2.30584540357676377E-34, 2.11529449291061230E-50,
                     -6.24983431736289734E-67,-3.52468697446179383E-83,
                     -1.41148589389242068E-99, 6.79709637298784201E-116);
  sin_t190 : constant octo_double
           := create( 5.53016705580027579E-01,-4.70615366237982039E-17,
                     -1.06171115459180559E-33,-1.61963161444073792E-50,
                      4.87793776246993030E-67, 1.17681865745260970E-83,
                      3.43733278217739062E-100, 4.17617202321411486E-117);
  sin_t191 : constant octo_double
           := create( 5.55570233019602178E-01, 4.70941094056167682E-17,
                     -2.06405203836829206E-33, 1.22901631885671377E-49,
                      3.88576868196360505E-66, 1.74396944780609574E-82,
                     -7.72108496973436171E-100, 2.33433622424404137E-116);
  sin_t192 : constant octo_double
           := create( 5.58118531220556102E-01, 1.34811763247652260E-17,
                     -5.50167438730114378E-34,-2.34848227393354160E-50,
                      1.85461240105610639E-66,-1.14220541744252336E-82,
                      4.30846725253398281E-99, 2.65153846674908728E-115);
  sin_t193 : constant octo_double
           := create( 5.60661576197336031E-01,-7.39564181534761517E-18,
                      3.96806206117311926E-34, 3.19959522008362225E-50,
                     -1.71630231497906093E-66, 1.15907607797441955E-82,
                      5.45606522286511890E-99,-1.92200124014174877E-115);
  sin_t194 : constant octo_double
           := create( 5.63199344013834091E-01, 2.38357751468548289E-17,
                      1.35117931737698137E-34, 9.32013115812481430E-51,
                      5.61530717534183517E-67, 1.14353859651037923E-83,
                      4.77637103994015678E-100,-3.99240028754291830E-116);
  sin_t195 : constant octo_double
           := create( 5.65731810783613232E-01,-3.40960795965904665E-17,
                     -1.70732897443035458E-33, 8.91470899754045074E-50,
                      5.22879378448664091E-66,-1.44308158716926903E-82,
                     -8.30710739397395802E-99, 1.41797146649581310E-115);
  sin_t196 : constant octo_double
           := create( 5.68258952670131601E-01,-5.09356736427692475E-17,
                     -1.62743563510282493E-33, 9.81831515617029655E-51,
                      5.67886317959306896E-67, 2.62172797814423790E-83,
                      3.38967068442399839E-100,-1.28671608601508849E-116);
  sin_t197 : constant octo_double
           := create( 5.70780745886967256E-01, 2.45681514555662075E-17,
                     -1.28444812475603499E-33,-1.80376343769362605E-50,
                     -4.95734042353730418E-67,-2.65873506350543923E-83,
                     -1.37511717085720690E-99,-3.21421169642697006E-116);
  sin_t198 : constant octo_double
           := create( 5.73297166698042204E-01, 8.51766116693063998E-18,
                     -6.44432087880267658E-34, 2.25461055432730032E-50,
                      4.88161839186071226E-67, 5.27204544525972028E-84,
                      2.07064177132721181E-100,-3.86801831267257127E-118);
  sin_t199 : constant octo_double
           := create( 5.75808191417845339E-01,-3.79094954589427341E-17,
                     -2.74337380468543095E-33, 1.11308415242167973E-49,
                     -7.14503871863004114E-66,-2.73674562669213643E-82,
                      1.52977853937329281E-98,-1.94207287043890982E-115);
  sin_t200 : constant octo_double
           := create( 5.78313796411655590E-01,-2.62376915123728308E-17,
                      1.36790516807381666E-33,-3.14098089353359002E-50,
                     -2.83084859311821715E-67,-1.57083163570623726E-83,
                     -3.96076254798867961E-100, 6.06601324357856541E-117);
  sin_t201 : constant octo_double
           := create( 5.80813958095764527E-01, 1.85853385866134080E-17,
                      2.76738431145491813E-34, 1.96053496198369368E-50,
                      3.11628270874890089E-68, 1.59824195468454168E-84,
                      1.08971005354611251E-100, 2.72047115264662162E-117);
  sin_t202 : constant octo_double
           := create( 5.83308652937698291E-01, 3.45166010790448584E-18,
                      1.80659774789463059E-34,-6.39539580385446461E-51,
                     -1.45041239803651036E-68,-6.08578264858702688E-85,
                     -5.01763861911483901E-101, 4.04090144980844087E-118);
  sin_t203 : constant octo_double
           := create( 5.85797857456438864E-01,-3.74855019643112936E-18,
                      2.79654037755366136E-34,-7.18169360241572021E-51,
                      4.90046109353813952E-68,-3.21996048009553181E-84,
                     -1.46263724693933932E-100, 8.41524248467131943E-118);
  sin_t204 : constant octo_double
           := create( 5.88281548222645334E-01,-2.92921667250068459E-17,
                     -2.37449546036939336E-33,-1.15716311915124803E-50,
                      9.67744974151397136E-67, 1.35219864237296078E-83,
                     -4.29342968033117638E-100,-1.19040160603063274E-116);
  sin_t205 : constant octo_double
           := create( 5.90759701858874275E-01,-4.70135841706595421E-17,
                      2.48084176117683556E-33, 1.25989076736431978E-50,
                     -5.00053739901223323E-67, 8.23634571976930574E-84,
                     -1.02523310024957001E-100,-2.12752832591997812E-117);
  sin_t206 : constant octo_double
           := create( 5.93232295039799795E-01, 1.28923209441890533E-17,
                      5.30583647763595832E-34, 4.11416746993900523E-50,
                      1.01735017952806524E-66,-2.19320706723724907E-84,
                     -2.02361402030226122E-100, 2.15314176083255220E-118);
  sin_t207 : constant octo_double
           := create( 5.95699304492433357E-01,-1.34386419365794672E-17,
                     -6.78776879077210602E-35, 5.08636508720029357E-51,
                      9.10243433159323455E-68,-3.45925835501244779E-84,
                     -1.62179721114190179E-100,-1.51245216723603694E-117);
  sin_t208 : constant octo_double
           := create( 5.98160706996342273E-01, 3.88018857830006574E-17,
                     -1.20841658580946627E-33,-4.04566108434300615E-50,
                      6.16553756722891208E-68,-3.65306365180145374E-84,
                      1.06574964070496396E-100, 2.41407055143999798E-117);
  sin_t209 : constant octo_double
           := create( 6.00616479383868973E-01,-4.63981982294619322E-17,
                     -1.66734930037108008E-33, 5.19828243784914448E-50,
                      8.73012275500980064E-67,-4.78514390797267157E-83,
                     -1.94572541126167107E-99,-1.39123157516787629E-115);
  sin_t210 : constant octo_double
           := create( 6.03066598540348164E-01, 3.73233576805596498E-17,
                      2.77719208669743046E-33,-1.61942296497424577E-49,
                     -7.37508411905744173E-66, 3.35682700091095669E-82,
                     -1.45830099724495385E-98, 1.26824086440384910E-115);
  sin_t211 : constant octo_double
           := create( 6.05511041404325545E-01,-3.12026724933056770E-17,
                      1.27612673386809158E-33,-4.08593685983796467E-50,
                      1.75900787110181626E-66,-4.99620587203248457E-83,
                      3.63225256318426350E-99,-1.40119484298225805E-115);
  sin_t212 : constant octo_double
           := create( 6.07949784967773632E-01, 3.51608323620966603E-17,
                     -2.55462427767783942E-34,-1.40853135512206942E-50,
                     -4.97540725567108556E-67,-2.51649414854921912E-83,
                      3.42086219399250868E-100,-2.49340451067699471E-116);
  sin_t213 : constant octo_double
           := create( 6.10382806276309475E-01,-2.25632656482291687E-17,
                      1.31855750112267303E-33, 8.23166914200634603E-50,
                      8.02255577567658064E-67,-1.07114392644803108E-83,
                     -7.98904799405371899E-100,-4.22687515933408549E-116);
  sin_t214 : constant octo_double
           := create( 6.12810082429409708E-01,-4.26934765684096849E-18,
                      2.58399658866503197E-34, 1.68844120056225371E-50,
                     -7.71518677044124058E-67,-1.25354218337244017E-84,
                     -1.01234080309617067E-100,-6.02526760735004833E-117);
  sin_t215 : constant octo_double
           := create( 6.15231590580626819E-01, 2.62314177672669503E-17,
                     -1.40953666211067163E-33, 7.20586904913045583E-50,
                     -3.42839232565893414E-66,-1.03821645383225418E-82,
                      3.28136913536308384E-99, 1.14531125173619919E-115);
  sin_t216 : constant octo_double
           := create( 6.17647307937803980E-01,-4.74785945109024519E-17,
                     -7.29865582631239963E-34,-3.01523275174391538E-50,
                      1.91686923688726259E-66,-1.16501753240120418E-82,
                      1.85558804656382216E-99, 1.13554717159740694E-115);
  sin_t217 : constant octo_double
           := create( 6.20057211763289207E-01,-2.79834108376811183E-17,
                      1.16499510561389229E-33,-5.45390891171352071E-50,
                     -1.71369509801439094E-66,-5.34482672784073242E-83,
                     -3.58130779365839579E-99, 1.18474976780496365E-115);
  sin_t218 : constant octo_double
           := create( 6.22461279374149967E-01, 5.29407286065730019E-18,
                     -4.84864112159458267E-35, 1.26965276419801086E-52,
                      2.88343763031018788E-69, 1.25936300469886697E-85,
                      4.91359376727124150E-102, 1.12266202694961301E-118);
  sin_t219 : constant octo_double
           := create( 6.24859488142386343E-01, 3.36718460372439003E-17,
                     -2.78460533910120962E-33, 5.61027181200121039E-50,
                      3.54966437573498937E-66,-1.70890972826580455E-82,
                      7.16907172991214882E-99,-2.66293016338513025E-116);
  sin_t220 : constant octo_double
           := create( 6.27251815495144083E-01, 3.07635851812532250E-17,
                      2.70689302734981384E-34,-1.11722403092864839E-50,
                      3.03575976560143181E-67, 1.80124091593841570E-84,
                     -9.81430462918888961E-101,-1.88555761473032233E-117);
  sin_t221 : constant octo_double
           := create( 6.29638238914926984E-01, 4.11153340496268057E-17,
                     -1.91674735802307466E-33, 1.11184240281617301E-49,
                      7.98870570644857055E-66,-4.94790815584510336E-82,
                      4.56218011450812815E-100,-7.75576462608736961E-117);
  sin_t222 : constant octo_double
           := create( 6.32018735939809062E-01,-4.01649422964636124E-17,
                     -7.22086436417367226E-34, 3.78289204705443437E-50,
                     -7.78594540583411322E-67, 5.91519959245143264E-83,
                     -6.15471885300076844E-100, 2.35473358174957182E-116);
  sin_t223 : constant octo_double
           := create( 6.34393284163645488E-01, 1.04209019292800346E-17,
                      4.11745589292804917E-34,-1.44641529866307047E-51,
                      2.58028489626117526E-68,-9.85971293301378270E-85,
                      5.03159228129794532E-101, 1.01470068282795913E-118);
  sin_t224 : constant octo_double
           := create( 6.36761861236284199E-01, 3.14190487119016107E-17,
                     -2.26937384151264490E-33,-1.60235842042973904E-49,
                      4.66517715059801333E-66,-1.91564222697059998E-82,
                      1.78998793247835687E-99,-7.87849978693768483E-116);
  sin_t225 : constant octo_double
           := create( 6.39124444863775731E-01, 1.24167963122710428E-17,
                     -6.20954196263566053E-34, 2.77620659995066033E-50,
                     -1.28017953588201167E-66, 8.49646667332401847E-83,
                      1.03436403840429052E-99,-9.00467922822119601E-116);
  sin_t226 : constant octo_double
           := create( 6.41481012808583162E-01,-9.98834301159433096E-18,
                      4.19692303767301283E-34, 5.69805437992575975E-51,
                      1.44517967098287091E-67, 6.09337922469104617E-84,
                      4.14683958488517766E-100, 2.36498118600062820E-116);
  sin_t227 : constant octo_double
           := create( 6.43831542889791497E-01,-3.20847987950468858E-17,
                     -1.25953119070533050E-33,-4.02058852308415357E-50,
                     -1.38114204746378912E-66,-6.54648672541083828E-83,
                      2.31668885549647098E-99,-5.16836465067820877E-116);
  sin_t228 : constant octo_double
           := create( 6.46176012983316395E-01,-2.97561373822808148E-17,
                     -1.02753700775182591E-33, 8.08524786658930136E-51,
                      1.91397713644455189E-67, 1.53959405645638030E-83,
                      6.66005425961994199E-100, 4.24357047037356515E-116);
  sin_t229 : constant octo_double
           := create( 6.48514401022112441E-01, 3.98702703133868310E-18,
                      1.94083885095407877E-34,-5.17984206361931902E-51,
                      3.34308197241945212E-68, 2.66728122249075353E-85,
                      1.43224015135665033E-102,-3.57452016333645263E-119);
  sin_t230 : constant octo_double
           := create( 6.50846684996380875E-01, 3.97146707105002572E-17,
                      2.91785467870029633E-34, 3.81406355082932784E-51,
                     -2.16931148649947718E-67,-4.40024493760145796E-84,
                     -3.59742102616582646E-100,-1.88148968553399107E-118);
  sin_t231 : constant octo_double
           := create( 6.53172842953776756E-01, 8.56956420600262380E-18,
                     -6.91653223050706326E-34, 2.38737512241853949E-50,
                     -2.09029827983373456E-66,-8.92356003171879901E-84,
                     -6.13388293101643140E-100,-2.94372987468226103E-117);
  sin_t232 : constant octo_double
           := create( 6.55492852999615350E-01, 3.56387344263850047E-17,
                      1.26953657908898113E-33, 4.39849528654120501E-50,
                     -1.51248625610693883E-66,-1.03435673428736812E-82,
                     -4.91149054184080853E-99,-1.61142807318108414E-115);
  sin_t233 : constant octo_double
           := create( 6.57806693297078637E-01, 1.95809430584685454E-17,
                     -1.19442722566271922E-33, 2.85564026164368583E-50,
                     -1.12528639300865706E-66,-2.78426371277035596E-83,
                      1.14322103229636655E-99,-4.26329524978391191E-117);
  sin_t234 : constant octo_double
           := create( 6.60114342067420479E-01,-1.39600543868236375E-19,
                      6.15157779314940466E-36, 5.35104988756226604E-52,
                     -3.63482042656562813E-68,-1.70195469543568536E-86,
                     -7.78554030198436450E-103, 1.97629154088938690E-119);
  sin_t235 : constant octo_double
           := create( 6.62415777590171784E-01,-2.26155088857645914E-17,
                      5.01770503181268624E-34, 2.91625323995307621E-50,
                      2.19113648801362274E-66, 7.93016547020295871E-84,
                     -5.22054014891269000E-102, 3.64196782119481451E-118);
  sin_t236 : constant octo_double
           := create( 6.64710978203344904E-01,-3.62277935980343671E-17,
                     -9.06079347655404268E-34, 3.09170363423802134E-50,
                      1.39074854554920572E-66,-7.72423934836349767E-83,
                     -3.03564997990684066E-99, 4.05243227784261206E-116);
  sin_t237 : constant octo_double
           := create( 6.66999922303637471E-01, 3.52843649974281657E-17,
                     -1.03820572324582381E-33, 7.38127565501676261E-50,
                      4.20051962214655963E-66,-2.02776115737064969E-82,
                     -6.34674974309307474E-99,-2.63554828904906076E-115);
  sin_t238 : constant octo_double
           := create( 6.69282588346636120E-01,-5.45926524174479126E-17,
                     -2.51810147096951515E-33,-1.68678759994371737E-49,
                      4.50952366574911060E-66,-2.07788424762868051E-82,
                     -7.50899394130098507E-99, 5.31506492308346382E-115);
  sin_t239 : constant octo_double
           := create( 6.71558954847018441E-01,-4.04890377492966925E-17,
                      3.19958356253556814E-34,-1.40444146556709596E-50,
                     -1.03266886487120775E-66,-4.74892468697725435E-83,
                      9.08704136447958725E-100,-9.13491206186852835E-117);
  sin_t240 : constant octo_double
           := create( 6.73829000378756038E-01, 2.30919012361610865E-17,
                      5.74280371928813190E-34, 1.12406683546259775E-50,
                     -1.62809842465025644E-67,-4.46988689971823802E-85,
                     -1.59180582284883064E-101,-2.32631271283867128E-118);
  sin_t241 : constant octo_double
           := create( 6.76092703575315923E-01, 3.72569022480494661E-17,
                      1.70594178957643746E-33, 9.73263477953006713E-50,
                     -9.15621915906168915E-66, 4.42028658126193742E-82,
                      2.73525011751112449E-98,-1.41103290490708240E-115);
  sin_t242 : constant octo_double
           := create( 6.78350043129861469E-01, 1.83020930418631220E-17,
                      9.52416757468130715E-34, 5.03281011161335034E-50,
                      1.77584393257119488E-66, 4.83734937530211747E-83,
                      2.74414806212966615E-99, 9.52505689910944107E-116);
  sin_t243 : constant octo_double
           := create( 6.80600997795453022E-01, 2.84732933545220469E-17,
                      4.13318059772709032E-34, 4.25790305107485759E-50,
                      1.98049760789605078E-66, 5.68072312123240008E-83,
                      3.55211326950322693E-99, 8.65305755659055024E-117);
  sin_t244 : constant octo_double
           := create( 6.82845546385248081E-01,-1.29580580615245309E-17,
                      1.82923869593306981E-34, 3.45362091160444873E-51,
                      2.76743539748949421E-67, 8.58649007405974912E-84,
                      8.84975004002922769E-100, 2.72778103605443430E-116);
  sin_t245 : constant octo_double
           := create( 6.85083667772700355E-01, 2.59481351946451372E-17,
                     -8.50307431295007020E-34,-6.95720861410099303E-50,
                      1.90404618335118095E-66, 6.47582439850383951E-83,
                     -1.41006843714793515E-100, 4.84222183266288088E-117);
  sin_t246 : constant octo_double
           := create( 6.87315340891759163E-01,-5.51561587149171682E-17,
                      1.18964898542668288E-33,-7.85058962182206623E-51,
                      8.49101994996188345E-68, 5.16466772729114140E-84,
                      4.10144252328766083E-100, 1.66832767261761327E-117);
  sin_t247 : constant octo_double
           := create( 6.89540544737066941E-01,-1.58893232948067899E-17,
                      9.12423562402057118E-34, 3.83154541522676376E-50,
                      3.64527748294119434E-68,-8.75561504452244138E-86,
                     -2.45530731811725988E-102, 1.15935254851880374E-119);
  sin_t248 : constant octo_double
           := create( 6.91759258364157748E-01, 2.74060784724106682E-17,
                      1.32865089432020923E-33, 1.06518691295800791E-51,
                      2.44528726725697874E-68,-7.66214977588925940E-85,
                      1.62231696330129848E-101,-1.06915549012911892E-117);
  sin_t249 : constant octo_double
           := create( 6.93971460889654002E-01, 7.43450769562801370E-18,
                      7.50615283881974596E-34,-1.59280002406865830E-50,
                     -7.97272367397494873E-67, 6.18303563207484492E-83,
                      1.72425841139322194E-99, 8.32126128004373864E-116);
  sin_t250 : constant octo_double
           := create( 6.96177131491462986E-01,-4.12240812135828892E-17,
                     -3.18387167620832964E-35, 1.38297067897247816E-51,
                     -9.84905733325786196E-68, 7.64604544769398702E-84,
                      1.59166828507059935E-100, 6.74612815742894458E-117);
  sin_t251 : constant octo_double
           := create( 6.98376249408972805E-01, 4.89882824356677676E-17,
                      1.91340104132441523E-33, 2.61611532437939887E-50,
                      1.03327200245448079E-66, 5.66297561556869681E-83,
                      1.89789287942147789E-99,-1.54016547167223941E-116);
  sin_t252 : constant octo_double
           := create( 7.00568793943248336E-01, 3.10279601929929218E-17,
                      9.56382505091799968E-34, 4.58969161381070479E-51,
                     -6.76731624709943623E-68, 2.24668316240760074E-84,
                     -2.51155109372052608E-101, 1.51192186123573206E-117);
  sin_t253 : constant octo_double
           := create( 7.02754744457225300E-01, 2.52782943836298217E-18,
                     -8.69855612106749531E-35, 5.00107260958748422E-51,
                     -2.62731902530561481E-67,-4.33503773631568896E-84,
                      2.48068789806382173E-100,-6.25512428574325721E-117);
  sin_t254 : constant octo_double
           := create( 7.04934080375904881E-01, 2.76087255857485017E-17,
                      2.98165994716291371E-34, 1.15330441851112058E-50,
                      7.33114367656022885E-67, 5.71629261673559666E-83,
                      4.65380830452457425E-100,-3.57402117616356843E-116);
  sin_t255 : constant octo_double
           := create( 7.07106781186547573E-01,-4.83364665672645673E-17,
                      2.06933765434970678E-33, 2.46777349573417546E-50,
                      2.04470214305994790E-67, 1.55277156441246667E-84,
                     -1.00055013355304061E-100, 1.53225205737542922E-117);
  cos_t0   : constant octo_double
           := create( 9.99995293809576191E-01,-1.96680642853221887E-17,
                     -6.30539550958834812E-34, 5.32661108557267312E-52,
                      1.91390915786795595E-68,-1.77893176531166562E-84,
                     -5.42005211648319282E-101, 1.34126812728719205E-117);
  cos_t1   : constant octo_double
           := create( 9.99981175282601109E-01, 3.35681035228955846E-17,
                     -1.47401325593680629E-35, 9.86030975947555961E-52,
                      5.12832731239829224E-68, 2.11240472258526906E-84,
                      1.62258206144986809E-100, 1.10147920527880245E-117);
  cos_t2   : constant octo_double
           := create( 9.99957644551963898E-01,-3.15278368666472871E-17,
                      2.63632511866384370E-33, 1.00075048154884004E-49,
                     -2.45224545327877928E-66, 1.71651249137784331E-82,
                     -2.46027292142140532E-99, 1.74126858063330681E-115);
  cos_t3   : constant octo_double
           := create( 9.99924701839144503E-01, 3.79310825126680122E-17,
                     -8.50999186605014844E-35,-4.99569732232951532E-51,
                      2.74762680122087446E-67,-3.64771596864154011E-84,
                     -1.61142890761787735E-100,-3.11125047596179137E-117);
  cos_t4   : constant octo_double
           := create( 9.99882347454212561E-01,-3.54778148724085383E-17,
                      1.71020010353039741E-33,-1.07253885190265420E-49,
                     -1.77152475524247922E-66,-1.21776746997724741E-82,
                      6.74468501957268618E-99,-3.37677940468724558E-115);
  cos_t5   : constant octo_double
           := create( 9.99830581795823403E-01, 1.88251405175511193E-17,
                     -5.13835134576169366E-34,-3.83788279954037865E-50,
                     -1.86398261465470787E-66,-1.16941268279622312E-82,
                     -2.14322784657043002E-99,-5.99285423894471043E-117);
  cos_t6   : constant octo_double
           := create( 9.99769405351215279E-01, 4.26811770322890116E-17,
                      1.90623023597370993E-33,-6.02211532628811598E-50,
                     -2.42650316157993754E-66, 2.24728463834524652E-82,
                     -4.50986553604074784E-99, 3.08587221421861494E-115);
  cos_t7   : constant octo_double
           := create( 9.99698818696204250E-01,-2.98514864037997529E-17,
                     -1.90847873707337374E-33, 5.59802603440292015E-51,
                     -2.01704480376572711E-67, 8.44721223054990384E-85,
                      5.67606186799352514E-101, 2.19026107928370261E-117);
  cos_t8   : constant octo_double
           := create( 9.99618822495178638E-01,-4.11819655214247342E-17,
                      2.09153655936999163E-33, 8.14033909209037336E-50,
                     -2.70869324880270646E-66, 1.39240583354108177E-82,
                     -3.06055521334881155E-100,-1.04322463102580305E-116);
  cos_t9   : constant octo_double
           := create( 9.99529417501093143E-01, 2.05179178237555913E-17,
                     -4.76738025857065202E-34,-2.94436041986567723E-50,
                     -1.59147008083688614E-66,-2.04615123609567971E-83,
                      1.10653355585645945E-99,-6.02444725763214112E-116);
  cos_t10  : constant octo_double
           := create( 9.99430604555461732E-01, 3.96444977522577981E-17,
                     -2.37572237167224280E-34,-1.28567590113617260E-51,
                     -1.03609800671890584E-68, 1.02101442113116168E-84,
                      6.69500473317417997E-102, 1.89392181562864510E-118);
  cos_t11  : constant octo_double
           := create( 9.99322384588349544E-01,-4.28585384408456820E-17,
                      3.32351016051465654E-34,-8.35542723770575433E-51,
                     -4.07583171927762145E-67,-2.51945007792082703E-83,
                      4.20362925049073802E-100, 1.17846080823709937E-116);
  cos_t12  : constant octo_double
           := create( 9.99204758618363886E-01, 9.17963171103856931E-18,
                      5.54162081858685698E-34, 8.02670467176153108E-52,
                     -3.29221777194787692E-68,-1.75681319385651342E-84,
                      9.57922678619483470E-101,-6.00855034919348925E-118);
  cos_t13  : constant octo_double
           := create( 9.99077727752645362E-01, 2.14190076535870320E-17,
                     -7.90482033185296178E-34,-5.31662961811127117E-50,
                      2.94307850181673490E-66, 6.16832063740849062E-83,
                      3.27169346199369429E-99,-6.77311908881136941E-116);
  cos_t14  : constant octo_double
           := create( 9.98941293186856871E-01,-2.06106419100586385E-17,
                     -1.25465254859134845E-33,-7.51758888061570636E-50,
                     -2.31695880524365778E-66, 3.95477556392195026E-83,
                      4.03337145690082665E-100,-1.62322395449940097E-117);
  cos_t15  : constant octo_double
           := create( 9.98795456205172405E-01,-1.22916933370754648E-17,
                      2.44684467864912710E-34, 1.07238910852102681E-50,
                      3.86850910136902605E-67, 7.93561503494129318E-84,
                     -3.43091070055007960E-100,-1.15497612893909168E-116);
  cos_t16  : constant octo_double
           := create( 9.98640218180265271E-01,-4.86902543129233022E-17,
                     -2.94708819679091466E-34,-1.30006507613469068E-50,
                      2.75300694627370753E-67, 4.42074826996169982E-84,
                      1.66204534475812494E-101,-8.86365685626486004E-118);
  cos_t17  : constant octo_double
           := create( 9.98475580573294774E-01,-2.20029311827787949E-17,
                     -1.23715094549449923E-33,-2.49112251312320649E-50,
                     -9.53932577873467762E-67,-3.59464073777121486E-83,
                      1.96601640849066130E-99, 9.57793171322828412E-116);
  cos_t18  : constant octo_double
           := create( 9.98301544933892893E-01,-5.18694027027922775E-17,
                      1.04801954936334517E-33,-2.89956491431555115E-50,
                     -3.45152433771346240E-67, 2.54702953902471068E-83,
                      1.71976295722971083E-99,-2.99498300351969390E-116);
  cos_t19  : constant octo_double
           := create( 9.98118112900149179E-01, 2.79354875581138326E-17,
                      2.44233412558303451E-33,-6.76466991753344167E-50,
                     -1.17295727773754094E-66, 1.24698153437164865E-83,
                      1.74668518670530080E-100, 8.29136148131427661E-117);
  cos_t20  : constant octo_double
           := create( 9.97925286198595996E-01, 1.71436597788863622E-17,
                      5.78858409028874604E-34,-9.26014326038945965E-51,
                     -2.38564730192505536E-67,-1.23823775422421644E-83,
                     -6.39107972759589716E-100, 2.15310180883531581E-116);
  cos_t21  : constant octo_double
           := create( 9.97723066644191636E-01,-2.63944752748987206E-17,
                     -1.61762230876617832E-34,-9.99249428893622806E-51,
                      2.28197066750363497E-67,-3.33737051252090983E-84,
                     -6.55945492379988131E-101,-2.64496449991983451E-117);
  cos_t22  : constant octo_double
           := create( 9.97511456140303454E-01, 5.60072059198069370E-18,
                     -5.94776735146856903E-35,-1.41668071627436270E-54,
                      6.20989982174911772E-71, 3.39472756627353839E-87,
                      3.52914218753507108E-104, 2.32243478412648976E-120);
  cos_t23  : constant octo_double
           := create( 9.97290456678690207E-01, 9.16476953711017346E-18,
                      6.78241343097392959E-34,-8.61913927955433572E-52,
                     -4.26913497573766520E-68, 3.98204022985038455E-84,
                      1.14911844320716265E-101,-6.53274052186337734E-119);
  cos_t24  : constant octo_double
           := create( 9.97060070339482962E-01, 1.67340935462419634E-17,
                     -1.31699514407800281E-33, 1.03110487679524771E-50,
                     -9.67800121860686581E-68,-4.12614329061330995E-84,
                      2.51709590242781601E-100, 7.78058965308424546E-117);
  cos_t25  : constant octo_double
           := create( 9.96820299291165668E-01, 4.70628207086156550E-17,
                      2.84120410764749371E-33,-8.00061556702636224E-50,
                     -2.39978097004205408E-67,-7.91228636384700789E-85,
                     -2.55336981288535038E-101,-1.07849752967779775E-117);
  cos_t26  : constant octo_double
           := create( 9.96571145790554835E-01, 1.17071790883909861E-17,
                     -7.59344132630246627E-34, 2.84748484369260084E-50,
                     -8.22113003166783761E-67,-2.14933562267061605E-83,
                     -1.02979629006746281E-99,-8.36515691106041903E-116);
  cos_t27  : constant octo_double
           := create( 9.96312612182778001E-01, 1.13364978916247349E-17,
                      3.40024586744143600E-34, 7.74190759215449010E-52,
                      5.78821081875662121E-68, 8.98103577480125886E-85,
                      5.57648189278119921E-101,-2.11813781613971451E-117);
  cos_t28  : constant octo_double
           := create( 9.96044700901251967E-01, 2.28700317076706948E-17,
                     -3.91848394050131483E-34,-3.70812604162463749E-50,
                      2.59243183926170746E-67, 1.50642008469898392E-83,
                     -3.19903903320151441E-101,-2.70902797785187790E-117);
  cos_t29  : constant octo_double
           := create( 9.95767414467659817E-01,-2.31519083230943588E-17,
                     -1.63065129319445913E-34,-1.59254207838631923E-51,
                     -8.56758743545876085E-69,-4.26861198082222194E-86,
                     -3.40759466853772268E-102, 1.77187118118850337E-118);
  cos_t30  : constant octo_double
           := create( 9.95480755491926939E-01, 3.20846214122265538E-18,
                     -4.95012921460130232E-36,-2.78114288508785162E-52,
                      7.80587653851640341E-69,-9.82898958779051486E-86,
                      5.50719840551550884E-102,-2.44090326891330398E-118);
  cos_t31  : constant octo_double
           := create( 9.95184726672196929E-01,-4.24869136783044096E-17,
                      1.33155107725046145E-33, 6.79279874170518884E-50,
                      2.61716123052701635E-66, 2.54535704018462273E-82,
                     -1.40434345143490885E-98, 6.61348469925497652E-115);
  cos_t32  : constant octo_double
           := create( 9.94879330794805616E-01, 4.21308132849436624E-18,
                     -4.20625974882884518E-35, 2.51570645560876200E-51,
                     -4.59520978357192429E-68,-3.97944467335093376E-84,
                      3.71473526111383334E-101,-2.58749500208764656E-117);
  cos_t33  : constant octo_double
           := create( 9.94564570734255415E-01, 3.67450696415280579E-17,
                     -3.06033041054710104E-33, 1.03978722804875260E-49,
                      2.43829312533316654E-66, 7.11965928997294609E-83,
                      4.52238414444860216E-99,-2.67430162239639854E-115);
  cos_t34  : constant octo_double
           := create( 9.94240449453187902E-01, 4.41294234724626735E-17,
                     -3.01072317082380655E-33, 7.42015829068618922E-50,
                     -1.38519263790160233E-66,-5.58257119565188988E-83,
                      8.75531365791685464E-100, 2.39559030120501934E-116);
  cos_t35  : constant octo_double
           := create( 9.93906970002356061E-01,-1.89648494711237457E-17,
                     -1.59808537779377516E-35,-8.53748071505970816E-52,
                      6.83422434140127349E-68,-1.99103719912616617E-84,
                      3.51780225012217231E-101,-1.23324071200568117E-117);
  cos_t36  : constant octo_double
           := create( 9.93564135520595304E-01, 2.97523099277974285E-17,
                     -4.50667073311342330E-34,-3.35481916338050359E-50,
                      1.10333814987137020E-66, 9.53644935793553151E-84,
                     -1.12199329342380592E-101,-6.99255441037706881E-118);
  cos_t37  : constant octo_double
           := create( 9.93211949234794500E-01, 3.30969062612722618E-17,
                      1.55928119732495674E-33, 1.43739777332535918E-50,
                     -2.40789043417042093E-67,-6.98524848899963475E-84,
                     -1.17207320840425796E-100, 7.08785285194052238E-117);
  cos_t38  : constant octo_double
           := create( 9.92850414459865105E-01,-1.40945177336933023E-17,
                     -1.19545581316169161E-33, 1.87618737428679832E-50,
                     -6.30757528536727055E-67,-1.00831351979554370E-83,
                     -4.79575999677828619E-100, 5.18232243863142938E-117);
  cos_t39  : constant octo_double
           := create( 9.92479534598709967E-01, 3.10930550954289061E-17,
                     -1.83795947578180191E-33,-3.98857585593813140E-51,
                     -9.15534345276831994E-68,-7.98152145507745253E-84,
                     -2.47215520637108643E-100, 3.98015918362302325E-119);
  cos_t40  : constant octo_double
           := create( 9.92099313142191797E-01,-3.94319261495887785E-17,
                     -6.27580629110472295E-34,-1.29609295592123895E-50,
                     -7.55907361742899420E-67,-1.07907269674512670E-83,
                     -4.09468956873849556E-100, 1.70260990022612595E-116);
  cos_t41  : constant octo_double
           := create( 9.91709753669099525E-01,-2.33728913118836612E-18,
                      2.70732988249685913E-35,-1.25694594418028720E-51,
                     -1.55863808007098786E-68, 3.20025026010067097E-85,
                     -1.96903486450261532E-101, 7.60434400565379034E-118);
  cos_t42  : constant octo_double
           := create( 9.91310859846115444E-01,-2.51921115833721052E-17,
                     -1.28524715673808870E-33, 5.23852125843104833E-50,
                      1.39395955888094931E-66, 3.10603359677739087E-83,
                     -2.26307982992181427E-100,-8.74838925732097158E-117);
  cos_t43  : constant octo_double
           := create( 9.90902635427780010E-01, 1.53945650945667038E-17,
                     -1.07999841331845674E-33, 2.74511159601335946E-51,
                      2.74295307140821957E-67,-1.37202956639149234E-83,
                      5.34351238546212769E-100, 2.19900704682449814E-116);
  cos_t44  : constant octo_double
           := create( 9.90485084256457093E-01,-5.54114375537808667E-17,
                     -1.46140172107535854E-33,-3.83393743973876203E-50,
                     -1.54304701790074939E-66, 1.01701108038048691E-83,
                      7.74444204472651311E-100, 5.22300753537535776E-117);
  cos_t45  : constant octo_double
           := create( 9.90058210262297123E-01,-1.70554859062339627E-17,
                      1.34549396857587774E-33, 7.31175891373000357E-50,
                      9.18936572556167788E-67,-6.27355679215791661E-83,
                      3.55294653952654093E-99, 1.25743152900242351E-115);
  cos_t46  : constant octo_double
           := create( 9.89622017463200887E-01,-5.23982179681325303E-17,
                      1.34631892114562195E-33, 5.80216405548948719E-50,
                     -1.99415305390991408E-67, 1.02917596544388029E-83,
                     -9.05314181948351785E-100,-1.75056666479271795E-116);
  cos_t47  : constant octo_double
           := create( 9.89176509964781014E-01,-4.09873099370471114E-17,
                     -4.48575605520484373E-34,-3.94145045028711250E-50,
                     -1.10813384628218254E-66, 5.86311831108411458E-83,
                     -7.73345487155608671E-100, 3.45953928855912604E-116);
  cos_t48  : constant octo_double
           := create( 9.88721691960323779E-01,-1.09762272066561250E-17,
                      3.23113425776537643E-34, 9.63679465835750406E-51,
                     -4.13408597463624609E-67, 2.95103410580208815E-83,
                      1.37479860004498271E-99,-8.30843157948635366E-116);
  cos_t49  : constant octo_double
           := create( 9.88257567730749464E-01, 2.70306077843726315E-17,
                      7.75148664886013771E-35, 2.10196449568649380E-51,
                     -4.86891372106515279E-68, 1.32264721680838175E-84,
                     -8.04049227617538703E-101, 9.56166660227564236E-118);
  cos_t50  : constant octo_double
           := create( 9.87784141644572178E-01,-2.36006933971590207E-17,
                     -1.23232837697078607E-33, 3.01309007168033386E-50,
                     -9.94701579676429690E-67, 2.68010215897179680E-83,
                     -3.10949141672238648E-100,-2.21818902684167404E-116);
  cos_t51  : constant octo_double
           := create( 9.87301418157858435E-01,-5.23322612557156525E-17,
                     -2.79376443331524725E-33, 1.20741605679584084E-49,
                      1.13657463527745169E-66,-2.22817119579308375E-83,
                     -1.08358952798673913E-99, 8.31823007992088613E-117);
  cos_t52  : constant octo_double
           := create( 9.86809401814185527E-01,-5.02872143510610749E-17,
                     -2.26815262381444611E-33, 4.40036943201691332E-50,
                      4.54603713298411555E-66, 4.00096603785398856E-83,
                      3.37713551127682296E-99,-2.38844061870790620E-117);
  cos_t53  : constant octo_double
           := create( 9.86308097244598669E-01,-2.15208771030133409E-17,
                      1.18665280541877157E-33,-7.85321991998138360E-50,
                     -4.06443126012895357E-66,-1.93515701218937409E-82,
                     -2.65618250804936956E-99, 1.69968596349126790E-115);
  cos_t54  : constant octo_double
           := create( 9.85797509167567476E-01,-5.14394529799530116E-17,
                      2.62764393099967253E-33, 7.54235527832863473E-50,
                     -1.47430810478738511E-67,-4.01579087923588949E-84,
                     -1.26088839343716867E-100,-8.07383059760856692E-117);
  cos_t55  : constant octo_double
           := create( 9.85277642388941222E-01, 2.31556370279002067E-17,
                     -7.52759715457648330E-34, 1.05822316604560938E-50,
                     -2.20931362370312839E-67, 1.51527219652343606E-83,
                      1.82657895910291741E-100, 3.29974259983817193E-117);
  cos_t56  : constant octo_double
           := create( 9.84748501801904208E-01, 1.05481440618299566E-17,
                      2.87861452662673056E-34,-3.67822100814661117E-51,
                     -9.59929240016175224E-68, 4.86899611148854190E-84,
                      1.60498396044187144E-100, 1.12118079979530586E-116);
  cos_t57  : constant octo_double
           := create( 9.84210092386929025E-01, 4.79839226270506910E-17,
                      2.25974196450705877E-34, 1.75738758148634002E-50,
                     -8.34269785789292007E-67,-5.82693326643445060E-83,
                      1.23922943076403646E-99,-1.36854293683982094E-116);
  cos_t58  : constant octo_double
           := create( 9.83662419211730255E-01, 1.98649482016352550E-17,
                     -1.07430462812110326E-35, 1.79756627965581001E-52,
                      2.61289340783778145E-69, 1.84203296318514549E-86,
                      9.54919422326801129E-103,-4.17582728358588429E-119);
  cos_t59  : constant octo_double
           := create( 9.83105487431216285E-01, 4.21700075228886281E-17,
                      8.23962656564409036E-34,-8.08037001390965615E-50,
                      3.22407590704885472E-66, 8.03895491713134467E-83,
                      5.05327202364715016E-100, 3.05151279797412319E-116);
  cos_t60  : constant octo_double
           := create( 9.82539302287441241E-01, 1.51495808137772237E-17,
                     -4.18027714221862370E-34,-2.21501743262261598E-50,
                      3.44393741838749221E-67, 1.29417545356935641E-83,
                      8.90427309390118165E-100,-2.52906278192474884E-116);
  cos_t61  : constant octo_double
           := create( 9.81963869109555243E-01, 2.11084437115130841E-17,
                     -1.52530134428960536E-33,-6.83880820793379687E-50,
                     -4.60505604057983488E-66, 4.71926919976347839E-83,
                      2.23615821902927661E-99,-7.78272760767018931E-116);
  cos_t62  : constant octo_double
           := create( 9.81379193313754561E-01, 1.34281632603556334E-17,
                     -6.52942904699629863E-34, 2.79654122874562679E-51,
                      1.53443005157334637E-67, 7.20444050501193441E-84,
                      2.41781571636343697E-100, 1.56611910549533379E-119);
  cos_t63  : constant octo_double
           := create( 9.80785280403230431E-01, 1.85469399978250057E-17,
                     -1.06965644455307566E-33, 6.66681744752649606E-50,
                      1.25019906217330185E-66, 4.29563840715566795E-83,
                      2.47946548647872342E-99,-3.38217924974884715E-116);
  cos_t64  : constant octo_double
           := create( 9.80182135968117430E-01,-3.68017869638561585E-17,
                      6.32451713879928420E-34, 1.86001761371759714E-50,
                      1.14347049424142521E-66,-4.29757498053305257E-83,
                      3.52925073215367280E-99, 1.87216027409591895E-115);
  cos_t65  : constant octo_double
           := create( 9.79569765685440519E-01, 1.55739915849904202E-17,
                     -1.34010660297829900E-33,-1.72637021998621493E-50,
                     -7.98673886311878563E-67, 6.25949845075466295E-83,
                     -2.02118324860428437E-99, 2.00291649541741970E-116);
  cos_t66  : constant octo_double
           := create( 9.78948175319062197E-01,-2.38177279611480529E-18,
                     -1.06947503703816611E-34,-8.22930471960874622E-51,
                     -3.84909112809551807E-67,-1.27036580697574222E-83,
                      7.42123263314149340E-100, 3.08074248920414000E-116);
  cos_t67  : constant octo_double
           := create( 9.78317370719627655E-01,-2.16230822333448952E-17,
                      1.09704030120280318E-33, 7.70919230993693393E-50,
                     -3.57933922537670313E-66,-2.22788512267157500E-82,
                     -3.38218053395532520E-99,-1.06499935361126710E-115);
  cos_t68  : constant octo_double
           := create( 9.77677357824509929E-01, 5.05141361670596278E-17,
                     -1.32547517014287881E-33, 7.01612543121245383E-50,
                     -4.22808578258530649E-66, 2.55511503489806707E-82,
                     -2.73917739122753280E-99,-1.04196996278482272E-115);
  cos_t69  : constant octo_double
           := create( 9.77028142657754395E-01,-4.33538757515559965E-17,
                      5.49488398315354785E-34,-9.27552631053773058E-51,
                     -4.25058150239580349E-67, 1.10847815393431722E-83,
                     -4.64420883033577086E-100, 1.21103933628312211E-116);
  cos_t70  : constant octo_double
           := create( 9.76369731330021140E-01, 9.30939315262137798E-18,
                     -4.11849491556856653E-34,-3.19139260313936898E-50,
                      1.40853962602612807E-66, 4.64413981554671916E-83,
                      7.15207724189136246E-100, 7.32480312701871755E-117);
  cos_t71  : constant octo_double
           := create( 9.75702130038528570E-01,-2.55725560812596857E-17,
                     -9.31742445089422234E-34,-8.36758632116468632E-51,
                      2.12334620080920379E-68, 1.63450441303316426E-84,
                     -4.95349273130162712E-101, 2.69361997203231145E-117);
  cos_t72  : constant octo_double
           := create( 9.75025345066994120E-01, 2.66426606518991355E-17,
                      1.78193927393538529E-34,-3.31596253856489471E-51,
                      2.95007891993014475E-68,-9.34914614065628015E-85,
                     -2.35973433953258568E-101,-1.34156308884222877E-117);
  cos_t73  : constant octo_double
           := create( 9.74339382785575858E-01, 2.30412214761515121E-18,
                      1.07586860050314303E-34, 5.10741164328094776E-51,
                      7.72778458824261812E-68,-1.62633643973408358E-84,
                      3.19655323125310531E-101,-2.20615608093413326E-117);
  cos_t74  : constant octo_double
           := create( 9.73644249650811977E-01,-5.17298086910058710E-17,
                     -1.55084730059898873E-33,-1.65051259176754027E-49,
                      7.96915446049543911E-66,-1.66344698689485860E-83,
                     -1.31220862667275281E-99,-1.99689379354672836E-116);
  cos_t75  : constant octo_double
           := create( 9.72939952205560177E-01,-3.13112111222817999E-17,
                     -2.68740877890061412E-33,-2.16524348188221452E-51,
                      4.79414741594388014E-68, 1.20190725928193214E-84,
                     -3.95462135206638842E-101, 8.42016898959033959E-118);
  cos_t76  : constant octo_double
           := create( 9.72226497078936269E-01, 3.64611697859382213E-17,
                      3.03096368838831329E-33,-1.27027169079673056E-51,
                      3.66763585449818093E-68,-1.69863399345301020E-84,
                      7.84561740807817259E-101, 3.84283121023098868E-117);
  cos_t77  : constant octo_double
           := create( 9.71503890986251784E-01,-7.98654211222890458E-18,
                     -4.36284172112633798E-34, 3.43075177987593518E-51,
                     -1.71729077144990356E-67, 1.02816740876531348E-83,
                     -1.38560997377986189E-100,-6.60797140033146819E-117);
  cos_t78  : constant octo_double
           := create( 9.70772140728950350E-01,-4.79921633251149218E-17,
                      3.03475289109757830E-33, 8.59891995064797013E-50,
                     -7.49481543042292715E-66, 2.94997856929937779E-82,
                     -2.36546149345046766E-98,-2.95245850600016617E-115);
  cos_t79  : constant octo_double
           := create( 9.70031253194543974E-01, 1.83653003484288444E-17,
                     -1.43110975719449184E-33, 8.58467819987406972E-51,
                      1.62164617019844889E-67, 9.93971074826211097E-84,
                      6.35412877280818009E-100, 3.28427780309489795E-116);
  cos_t80  : constant octo_double
           := create( 9.69281235356548532E-01,-4.56636602619278955E-17,
                      9.61475269172393871E-34, 8.12676052078713300E-51,
                     -1.20711052357953293E-67, 6.09453764434430899E-84,
                      3.26073337427688091E-100, 2.12671505876037430E-116);
  cos_t81  : constant octo_double
           := create( 9.68522094274417267E-01, 4.94750749182447709E-17,
                      2.85587383519112405E-33, 6.29484223165074613E-50,
                     -1.85092668189800111E-67,-1.60239927272611526E-83,
                     -5.00466528836349518E-100,-4.59670977119475995E-116);
  cos_t82  : constant octo_double
           := create( 9.67753837093475511E-01,-4.55121328255158199E-17,
                     -1.41276179887190934E-33,-8.46206090897045783E-50,
                      1.24395936473099010E-66, 1.14293321503006500E-82,
                      7.04399065125957426E-99,-3.23066297849334417E-115);
  cos_t83  : constant octo_double
           := create( 9.66976471044852071E-01, 3.84962288373378642E-17,
                     -5.38816315427456469E-34,-3.52218631714589587E-50,
                      1.44938221650369558E-66, 3.79895418846207444E-83,
                     -2.28718367225149526E-99,-1.08006133273322478E-115);
  cos_t84  : constant octo_double
           := create( 9.66190003445412504E-01, 5.12988404016654925E-17,
                      1.45640759047698075E-34, 1.00959739713774320E-50,
                     -1.51706285249723938E-67, 8.66894665478795992E-84,
                     -7.17117120878777403E-100, 7.26842554113666122E-117);
  cos_t85  : constant octo_double
           := create( 9.65394441697689398E-01,-2.37453899183921563E-17,
                      5.92215155900538620E-34,-3.88111925562310936E-50,
                     -9.42019874222734875E-67, 4.59000233797429829E-83,
                      3.34225989389553376E-99,-1.22327783966708738E-115);
  cos_t86  : constant octo_double
           := create( 9.64589793289812758E-01,-3.41894707359597856E-17,
                      2.29820741554635218E-33,-4.51287910456076340E-50,
                     -2.90435472553590522E-66, 2.27851433769356981E-82,
                     -7.57748661918793489E-99, 3.53663721834692119E-115);
  cos_t87  : constant octo_double
           := create( 9.63776065795439840E-01, 2.64639505612200288E-17,
                     -2.90732345901993230E-36,-1.29383286293956014E-52,
                      5.42377504217757778E-69, 2.31568658554524316E-85,
                     -7.78256473857223002E-102, 7.39933411115412867E-119);
  cos_t88  : constant octo_double
           := create( 9.62953266873683877E-01, 8.93419604043136339E-18,
                     -3.90712446610201256E-34, 1.62120911168473936E-50,
                      5.45609708106822516E-67, 7.45013740618382878E-84,
                     -1.79824171468069716E-101, 1.82075512815275648E-118);
  cos_t89  : constant octo_double
           := create( 9.62121404269041580E-01, 1.52367704538463048E-17,
                     -1.30501735255971420E-33, 7.90161223940926659E-50,
                      3.09258698989453650E-66, 2.42945747300312265E-82,
                      8.46011306621555756E-99, 7.54340212118548061E-115);
  cos_t90  : constant octo_double
           := create( 9.61280485811320640E-01, 2.09339552166740391E-18,
                      1.07686074690156925E-34,-5.94536393043617736E-51,
                     -5.39774379960807517E-67,-1.02390685673233149E-83,
                     -6.92621794150414054E-100, 4.37032948036597055E-116);
  cos_t91  : constant octo_double
           := create( 9.60430519415565787E-01, 2.46539048153171851E-17,
                     -1.37921694109063224E-33,-4.77265983785069034E-51,
                      2.43312491357664852E-67,-1.18220254871496246E-83,
                      4.78736167565198069E-100, 3.96379054800794783E-117);
  cos_t92  : constant octo_double
           := create( 9.59571513081984517E-01, 1.10006400850009570E-17,
                     -4.20360308282239746E-34, 4.00237048426065730E-51,
                      2.18497216748766813E-67,-9.20882126898136034E-84,
                     -6.80679007032517504E-100,-1.66513242856990270E-116);
  cos_t93  : constant octo_double
           := create( 9.58703474895871599E-01,-4.36850143923720533E-17,
                      2.20018006627291311E-33,-1.05537213243580767E-49,
                      7.59233777821841529E-66, 8.73471225330366257E-83,
                     -6.77293432348759834E-99,-2.47456894842052508E-115);
  cos_t94  : constant octo_double
           := create( 9.57826413027532908E-01,-1.76967100753712627E-17,
                      1.91640341103821897E-34, 8.14892350717548131E-51,
                     -2.70794569294906723E-67, 9.07441597478512394E-84,
                     -4.68016123362953683E-100, 5.65851735365578967E-117);
  cos_t95  : constant octo_double
           := create( 9.56940335732208824E-01, 4.05538698618757006E-17,
                     -1.71470133643021491E-33, 2.57367452953294547E-50,
                     -1.06073717989001806E-66,-6.06834981124942937E-83,
                     -3.06437169362617683E-99, 1.95997253676932792E-115);
  cos_t96  : constant octo_double
           := create( 9.56045251349996406E-01, 3.77050452795890670E-17,
                      1.96786999973475710E-33, 8.50931777312301799E-50,
                      1.78072153859820630E-66, 4.81230769400565826E-83,
                     -3.18116888009302172E-99,-8.72512963029218373E-116);
  cos_t97  : constant octo_double
           := create( 9.55141168305770671E-01, 5.00886529550146681E-17,
                     -2.69831818380592114E-33, 1.01023235755964930E-49,
                      8.70736756883692803E-66,-2.09861613758072834E-82,
                     -4.49423470421902387E-99,-3.79216290693442012E-115);
  cos_t98  : constant octo_double
           := create( 9.54228095109105667E-01,-3.75459016906268741E-17,
                      1.49516192412577644E-33,-8.27173331513949730E-50,
                      4.64025572425820192E-66, 1.35102594875214550E-82,
                     -1.12109676531992514E-98, 7.77172974375096880E-115);
  cos_t99  : constant octo_double
           := create( 9.53306040354193862E-01,-2.51907387799199336E-17,
                     -1.42722398211343787E-33,-4.67172868092831554E-50,
                      1.07355803110695776E-66, 8.98787279082884347E-84,
                      3.91384245651778361E-100, 1.47666486707701230E-116);
  cos_t100 : constant octo_double
           := create( 9.52375012719765879E-01,-2.02693004622992721E-17,
                     -1.06359568872462463E-33,-3.55145376664876193E-50,
                     -1.16913925385888480E-66, 3.05382459417423301E-83,
                     -1.02660805067577601E-99, 3.88681271571159073E-116);
  cos_t101 : constant octo_double
           := create( 9.51435020969008338E-01, 3.13505841232666951E-17,
                     -2.48248334527378125E-33, 9.54503355253806126E-51,
                      5.53918767059611572E-67,-2.84449994016305898E-83,
                      1.45756841212546658E-99, 7.95274417611133811E-116);
  cos_t102 : constant octo_double
           := create( 9.50486073949481702E-01, 1.94100975626304361E-17,
                     -8.15593939498167894E-34,-1.05012097201645618E-50,
                     -2.46659050179783976E-67,-5.13938908505698639E-84,
                     -1.25938517274140014E-100,-1.32275148405962913E-117);
  cos_t103 : constant octo_double
           := create( 9.49528180593036675E-01,-7.55441519280432984E-18,
                     -5.12602450240466856E-34, 1.80936433890404058E-50,
                      1.06925435595184381E-66, 1.85580824775036884E-83,
                      9.52521545126716726E-100,-1.36672850084663406E-116);
  cos_t104 : constant octo_double
           := create( 9.48561349915730268E-01, 2.06682622623332316E-17,
                     -5.94407302436673059E-34, 1.42688531115543005E-50,
                     -7.32845891867839664E-67,-1.61488109785671798E-83,
                      5.00580589956716106E-100,-2.05576822144118270E-116);
  cos_t105 : constant octo_double
           := create( 9.47585591017741091E-01, 4.34179938521259905E-17,
                     -2.77286678898403727E-34, 5.57091601965199683E-51,
                     -4.65524084437226262E-67, 1.05674991192480383E-83,
                     -1.24989038979712829E-100, 1.36648459676357626E-117);
  cos_t106 : constant octo_double
           := create( 9.46600913083283535E-01, 3.50568002106807304E-17,
                      9.85785369403181171E-34, 6.60359110645851972E-50,
                     -8.63226846881280089E-67,-4.96118217013441903E-83,
                      2.10018870791639280E-99, 1.52417880924195845E-115);
  cos_t107 : constant octo_double
           := create( 9.45607325380521280E-01, 4.60191024785237379E-17,
                     -6.25343847694520592E-34, 1.57589412157799615E-50,
                      3.50354275814951774E-67,-9.86913826329273034E-84,
                      4.76591532343491806E-100,-2.98448753664147515E-116);
  cos_t108 : constant octo_double
           := create( 9.44604837261480257E-01, 8.81005454766411651E-18,
                      5.22916956027578420E-34,-3.34872560184071233E-50,
                      2.39194940723815589E-67, 6.45407160912412937E-84,
                     -3.11215319917321197E-100, 6.28466941542440826E-117);
  cos_t109 : constant octo_double
           := create( 9.43593458161960386E-01,-2.40931278444042141E-17,
                      1.02832798568039392E-34,-2.33982326145313548E-51,
                     -1.00344539939127154E-67,-7.78093352395119302E-84,
                     -2.94134011522936065E-100, 5.04802149998341297E-117);
  cos_t110 : constant octo_double
           := create( 9.42573197601446866E-01, 1.32355648064368862E-17,
                     -5.70482628853869107E-35, 3.99470504427537436E-51,
                     -1.53360535292408179E-67, 1.22879371215148752E-83,
                     -3.73272933341177773E-100, 2.13712777453120318E-116);
  cos_t111 : constant octo_double
           := create( 9.41544065183020806E-01,-2.78963795476983411E-17,
                      1.62732363567338985E-33,-5.30759447084712028E-51,
                     -1.97563364352909530E-67,-9.84904967317238783E-84,
                     -7.22940592221540571E-100, 8.31254830802350592E-117);
  cos_t112 : constant octo_double
           := create( 9.40506070593268295E-01, 2.86104215671162683E-17,
                      2.92615011475388268E-33,-2.68498676908969248E-50,
                      2.09213372520891565E-66,-2.80351901621276408E-83,
                     -1.01978742425401925E-99, 4.80238403194639172E-116);
  cos_t113 : constant octo_double
           := create( 9.39459223602189919E-01,-7.01528679430986548E-18,
                     -5.63956938180112096E-34, 3.55681426789876511E-50,
                     -2.08561420747572766E-66, 8.43574259773126629E-83,
                     -1.83268145527305285E-99, 1.79672756959986535E-115);
  cos_t114 : constant octo_double
           := create( 9.38403534063108058E-01, 5.42425450447954901E-17,
                     -1.90399666078597588E-33,-1.56277929883412154E-49,
                     -5.21562779934203741E-66,-4.39115487120723673E-82,
                     -7.90809181835800649E-99, 5.86537199903655596E-115);
  cos_t115 : constant octo_double
           := create( 9.37339011912574960E-01,-3.65709262843627764E-17,
                     -1.19029400712732472E-33,-1.12150823315832229E-50,
                      7.60707300254348212E-67,-5.26062862939899059E-83,
                      8.57127824561348457E-100,-3.48852376555868662E-116);
  cos_t116 : constant octo_double
           := create( 9.36265667170278260E-01,-1.30137661454976537E-17,
                      5.22298700619905950E-34,-3.39727770756341084E-51,
                     -1.90341799749199951E-67,-3.15356159885427870E-84,
                      9.20670470559841481E-101,-6.21090077072843663E-117);
  cos_t117 : constant octo_double
           := create( 9.35183509938947610E-01,-3.26093953024850650E-17,
                     -8.18130152188752454E-34, 5.56421400249281388E-50,
                      3.39120020833333907E-66, 1.73251824602431850E-82,
                     -1.27465111561684163E-98, 3.09092949517218252E-115);
  cos_t118 : constant octo_double
           := create( 9.34092550404258870E-01, 4.46628243607675113E-17,
                     -2.59032430473969160E-33, 8.15052090043430426E-50,
                     -3.96084807942105700E-67,-2.03534861617404378E-83,
                      1.17672537711663367E-100,-2.83661734960174942E-117);
  cos_t119 : constant octo_double
           := create( 9.32992798834738846E-01, 4.20414155553843554E-17,
                      9.02858964955212762E-34, 5.30199849776612590E-50,
                     -2.20337124483564173E-66, 8.36670783722345020E-83,
                      1.23427699568136070E-99, 1.68964307056419849E-116);
  cos_t120 : constant octo_double
           := create( 9.31884265581668148E-01,-4.07859443773180948E-17,
                      1.76314502987541687E-33, 2.57764033055074533E-50,
                      7.73467038309955434E-67,-6.41087611395991873E-83,
                     -3.17471443066570237E-99, 9.64975860284331691E-116);
  cos_t121 : constant octo_double
           := create( 9.30766961078983712E-01, 1.97037751028383294E-17,
                      6.56579087182782053E-34,-1.94803479662595241E-51,
                     -9.73087044913080682E-68, 2.08236441694450518E-84,
                      1.09576586527377549E-100,-2.60750375818945636E-117);
  cos_t122 : constant octo_double
           := create( 9.29640895843181214E-01, 5.12825300168641071E-17,
                      2.37197398919162612E-34,-1.72300654269171267E-50,
                     -6.68792677476599850E-67, 1.23049318112029354E-83,
                      8.37244418565023376E-100, 4.02434900972948025E-116);
  cos_t123 : constant octo_double
           := create( 9.28506080473215589E-01,-2.33066398484859428E-17,
                     -7.77990843332085032E-34,-5.85975580093003054E-50,
                      1.75833592434031391E-66,-1.19262802341714440E-82,
                     -7.37736511562237771E-101, 4.33493425197844834E-117);
  cos_t124 : constant octo_double
           := create( 9.27362525650401115E-01,-2.76771116921554372E-17,
                      2.21102934501995759E-34, 2.03491908196806129E-50,
                      8.39032375347225070E-67,-9.32778897381976315E-84,
                      4.79684902392223340E-101, 1.18478488401045131E-117);
  cos_t125 : constant octo_double
           := create( 9.26210242138311379E-01,-3.73037545860990542E-17,
                      2.04644578099934049E-33, 1.38317996312318165E-49,
                     -6.80112806695844948E-66,-3.81403409254158291E-82,
                     -6.68485648218491344E-99, 6.88291084626239389E-116);
  cos_t126 : constant octo_double
           := create( 9.25049240782677584E-01, 6.05294474125761588E-18,
                     -8.82565177602785406E-35, 1.82854621223883285E-51,
                      4.09130392565241415E-68, 2.56828789316996150E-84,
                      1.08010652515540063E-100, 2.44618780187023231E-117);
  cos_t127 : constant octo_double
           := create( 9.23879532511286739E-01, 1.76450470843366771E-17,
                     -5.04425373215868178E-34,-4.04786777168238900E-50,
                      1.31735239700201084E-66, 6.54375320157104185E-84,
                     -1.42764261912597804E-100,-3.88751015742232925E-117);
  cos_t128 : constant octo_double
           := create( 9.22701128333878518E-01, 5.29637989185398138E-17,
                     -5.71356996288766851E-34, 3.01636717972190866E-50,
                      1.07463934459841561E-66, 1.11166515229030846E-84,
                      8.10373844610333058E-101, 1.72023532144832194E-117);
  cos_t129 : constant octo_double
           := create( 9.21514039342041902E-01, 4.16398433906846436E-17,
                      1.18914856047023557E-33, 2.08624375943803242E-50,
                     -6.91693942914795348E-67,-3.50049466036502569E-83,
                     -1.27214433223848543E-99,-7.31815958744299068E-116);
  cos_t130 : constant octo_double
           := create( 9.20318276709110594E-01,-2.78068887790368375E-17,
                      2.70110136770712743E-33, 1.19985787924554990E-49,
                      7.67396671848431259E-66,-1.54987464495404513E-82,
                      2.23718133923744613E-99,-1.48969758828585084E-115);
  cos_t131 : constant octo_double
           := create( 9.19113851690057770E-01,-2.64964846223447180E-17,
                      6.54036047634619200E-34,-2.89971802011860779E-50,
                     -2.26723278231783554E-66,-8.67903385550794176E-84,
                      3.17018237483716193E-100, 1.50379978355683784E-116);
  cos_t132 : constant octo_double
           := create( 9.17900775621390497E-01,-3.90745796808495149E-17,
                      2.30046365414902636E-33, 3.98517627444431070E-50,
                     -3.54790303670753146E-67,-1.24676514023696537E-83,
                      8.30992887322625520E-101,-5.48847252842469518E-117);
  cos_t133 : constant octo_double
           := create( 9.16679059921042705E-01,-4.17339786982875676E-17,
                      1.20944448043811722E-33, 4.93569168260978155E-50,
                      2.26213938243848806E-66, 5.55299561362914361E-83,
                     -2.36487390714231140E-99, 1.06016933744567057E-115);
  cos_t134 : constant octo_double
           := create( 9.15448716088267833E-01,-1.35910566929008941E-17,
                      5.99230274755947351E-34, 2.14032959259628792E-50,
                     -5.07426480084617918E-67,-2.67698374962148665E-83,
                     -1.00282497079407074E-99,-7.27567635875714328E-116);
  cos_t135 : constant octo_double
           := create( 9.14209755703530691E-01,-3.63161825278144230E-17,
                     -1.94388197771225540E-33, 2.83406792877283164E-50,
                      1.36617856001305524E-66,-6.24708794138598536E-83,
                      1.76722288872753994E-99, 4.01135210160478946E-117);
  cos_t136 : constant octo_double
           := create( 9.12962190428398213E-01,-4.79325052280394689E-17,
                     -1.77535518894286383E-33, 4.06077829038681596E-51,
                     -1.19218090888976604E-67, 7.34235365808370103E-84,
                     -4.55753775880261135E-100,-1.24301716589214678E-116);
  cos_t137 : constant octo_double
           := create( 9.11706032005429878E-01,-2.69132731750341301E-17,
                     -5.19281019161625276E-35, 1.13381759360906298E-51,
                      6.57931209831338710E-68,-3.38062832397093344E-84,
                     -5.68095864663377634E-101, 3.06965993165979683E-118);
  cos_t138 : constant octo_double
           := create( 9.10441292258067247E-01,-5.04330416733138196E-17,
                      1.09387462574043045E-33, 9.53782720841707308E-51,
                     -4.79587246974103756E-67, 2.74335939397916893E-83,
                      3.43640814825976249E-100, 1.93635703801525373E-116);
  cos_t139 : constant octo_double
           := create( 9.09167983090522380E-01,-3.68785640913598945E-18,
                      2.99513303105076927E-34,-1.22256661369199255E-50,
                     -8.30747552639331380E-67, 4.35558969147325212E-84,
                     -4.18323179208477636E-100,-4.61697468250600271E-117);
  cos_t140 : constant octo_double
           := create( 9.07886116487666262E-01,-4.94599643012258396E-17,
                     -1.65996827070753131E-33,-5.19252027126347157E-50,
                      2.44502604582037370E-66,-2.49025671686227720E-82,
                      1.02605003572779146E-98,-2.19500622166559831E-115);
  cos_t141 : constant octo_double
           := create( 9.06595704514915335E-01, 3.05067189554420234E-17,
                     -1.44788365571412037E-33, 1.89063737844487253E-50,
                     -3.68113553769296829E-67, 1.69521389347304407E-83,
                      1.70862682708653291E-99,-9.67120115591583006E-116);
  cos_t142 : constant octo_double
           := create( 9.05296759318118816E-01,-4.11530998268899011E-17,
                      2.98593687051842233E-33, 5.11452939174392112E-50,
                      4.42169060075559365E-66, 1.54752415043817613E-82,
                      9.62040222624798796E-99,-6.19453688724480299E-115);
  cos_t143 : constant octo_double
           := create( 9.03989293123443338E-01,-6.60975446874843085E-18,
                      1.27280130346803566E-34,-4.30260972340148227E-51,
                     -3.13816322520577034E-69, 3.68437472738792958E-86,
                      3.02492240548272987E-102,-1.08966512339559140E-118);
  cos_t144 : constant octo_double
           := create( 9.02673318237258826E-01,-1.92507870339614830E-17,
                      1.32421289932445272E-33,-5.29710306887036647E-50,
                      1.54789540534235651E-66, 7.54657066585132303E-83,
                      5.33555996595478614E-99,-3.75357027596638990E-117);
  cos_t145 : constant octo_double
           := create( 9.01348847046022028E-01,-1.35247893676986820E-17,
                      6.36053531158800906E-34, 3.62274006545738278E-50,
                     -1.66997184383873454E-67, 1.44483354167472010E-83,
                     -2.94194529894533352E-100,-2.28531935255889977E-116);
  cos_t146 : constant octo_double
           := create( 9.00015892016160279E-01,-5.06396180508022727E-17,
                      1.07835253840315763E-33, 2.81300163265151114E-50,
                     -1.59414118363857990E-66, 3.00173555796078788E-84,
                     -5.27467677803455111E-101, 2.40557423410227474E-117);
  cos_t147 : constant octo_double
           := create( 8.98674465693953817E-01, 2.63169064610330135E-17,
                      3.70031370477968400E-35,-2.34477199004659381E-51,
                      7.43800282401806768E-68,-3.78898833916026033E-84,
                      2.23786471472600773E-100,-1.24876106813507631E-116);
  cos_t148 : constant octo_double
           := create( 8.97324580705418318E-01,-3.63962833148672899E-17,
                     -2.36116498954748151E-33, 1.18372470479000824E-49,
                      2.36718765690566237E-66, 4.90824468118734516E-83,
                     -2.91626266939918703E-99,-1.19639288998458225E-115);
  cos_t149 : constant octo_double
           := create( 8.95966249756185107E-01, 4.90250991148118131E-17,
                     -1.94404898147953256E-33,-1.70704866677670326E-49,
                     -6.43085393921242098E-66, 1.36219297588791205E-82,
                      1.04564860732068228E-98, 4.15930978879588229E-115);
  cos_t150 : constant octo_double
           := create( 8.94599485631382696E-01,-1.75162263968149192E-17,
                     -1.32006700472469226E-33,-1.59530098843246948E-50,
                      9.56135529357425679E-67,-2.51680348975634531E-84,
                     -1.49070740490683001E-100,-1.12356218117560415E-117);
  cos_t151 : constant octo_double
           := create( 8.93224301195515324E-01,-4.11612391519089127E-18,
                      2.53802538057159990E-34, 4.28494555105161925E-51,
                      1.25260781969256085E-67,-6.27106178643637808E-85,
                      4.20635626812736010E-101, 3.99267357753090098E-118);
  cos_t152 : constant octo_double
           := create( 8.91840709392342723E-01, 4.66902281371245465E-18,
                      1.61502542868419821E-34,-3.96174488207250116E-51,
                      2.92506421225298359E-67, 3.03939365711243743E-84,
                      6.70823039866983636E-101,-6.23245824304293348E-117);
  cos_t153 : constant octo_double
           := create( 8.90448723244757878E-01, 1.17819314590518032E-17,
                     -1.33461422095719299E-34,-9.49823735307334314E-51,
                      3.87268252626796267E-67, 2.99662128219363086E-83,
                     -1.09907227994864728E-99, 4.58120531473482302E-116);
  cos_t154 : constant octo_double
           := create( 8.89048355854664574E-01,-1.11645149667666750E-17,
                     -3.47976361077987357E-34,-1.56050799970406315E-50,
                     -1.07184758158533505E-67, 6.26045919082496246E-85,
                      1.32746736678679254E-101, 3.54512774037962057E-118);
  cos_t155 : constant octo_double
           := create( 8.87639620402853935E-01, 1.28050919185879605E-17,
                      3.99487420595844644E-35,-1.45145778765057735E-51,
                     -2.02973120139504795E-68,-2.40935325013997952E-85,
                      8.26360871779154612E-102, 1.80624444278659178E-118);
  cos_t156 : constant octo_double
           := create( 8.86222530148880638E-01,-6.73073696002743145E-18,
                      1.23855934329174132E-34, 2.03640147591333204E-51,
                     -1.28817816294531839E-68,-9.94530821378804471E-85,
                      2.68224157902936971E-101,-1.31593959234570326E-117);
  cos_t157 : constant octo_double
           := create( 8.84797098430937790E-01,-9.43314696289726900E-18,
                     -5.71065414787014390E-34, 1.82601341119073970E-50,
                      6.90310493823286941E-67, 3.10303140437974252E-83,
                      6.15411222260062330E-100,-4.46600199860445290E-116);
  cos_t158 : constant octo_double
           := create( 8.83363338665731579E-01, 1.58226433802551274E-17,
                     -7.89213200075882501E-34,-1.47823210161798355E-50,
                      1.06126860704844144E-66, 6.28063470985803973E-83,
                     -7.86329224780793988E-100, 1.59246594473099978E-116);
  cos_t159 : constant octo_double
           := create( 8.81921264348355050E-01,-1.98432484058905621E-17,
                     -7.04121140076738337E-34,-1.06367701693891040E-50,
                     -3.71942427921525743E-67,-1.43513301946188705E-83,
                      1.09129200800172888E-100, 4.51208157504180717E-117);
  cos_t160 : constant octo_double
           := create( 8.80470889052160755E-01, 1.63110966029963503E-17,
                     -5.75413605947241719E-34,-4.01286118621700210E-50,
                      5.18486151133952877E-67, 5.33949721574292825E-84,
                      1.09810944827955464E-100, 2.11634291184424819E-117);
  cos_t161 : constant octo_double
           := create( 8.79012226428633525E-01,-4.73568372911180108E-17,
                      1.43887712979751925E-33,-2.90855543044791338E-50,
                     -2.07923337025886643E-66,-1.06401405709765915E-82,
                      1.08225061245609177E-99, 1.25858212647241052E-116);
  cos_t162 : constant octo_double
           := create( 8.77545290207261242E-01, 5.01133118464995501E-17,
                      2.83827690087395427E-34, 1.55506403931641403E-50,
                     -2.38305207285345852E-67,-7.10208977806837156E-84,
                     -2.56820620008990001E-100,-1.46963774505636116E-116);
  cos_t163 : constant octo_double
           := create( 8.76070094195406601E-01, 5.87290242351476768E-18,
                      2.79411443917384576E-34,-1.85360738465098282E-50,
                     -2.13205724534811585E-67,-1.49173050008941418E-83,
                     -9.05880020501538453E-100, 9.53798882263581157E-117);
  cos_t164 : constant octo_double
           := create( 8.74586652278176113E-01,-5.72166177303970651E-19,
                     -2.97058115036895959E-35, 8.73895939697967523E-52,
                      4.13155974958278277E-68,-5.85628965707227393E-85,
                      4.44925347421914587E-101,-2.70603793742448176E-117);
  cos_t165 : constant octo_double
           := create( 8.73094978418290091E-01, 7.84246729901299026E-18,
                     -4.86850158397971649E-34,-2.28155705874775273E-50,
                     -1.30529312774659988E-66,-1.37836697330166865E-83,
                      6.94443008987875970E-100, 2.72704916954505765E-116);
  cos_t166 : constant octo_double
           := create( 8.71595086655951090E-01,-5.52729980385510496E-17,
                     -2.21040902049849069E-33,-9.77497631876431909E-50,
                      7.60450749280500932E-66, 2.17416830794539454E-82,
                      9.65224708408154429E-99,-2.15051141127342394E-115);
  cos_t167 : constant octo_double
           := create( 8.70086991108711461E-01,-4.18885108685499682E-17,
                      7.09001858618784148E-34, 3.76002511151572602E-50,
                      1.59845586416776705E-66,-6.55752179908687736E-83,
                      1.53088887074347386E-99,-6.26017198352983389E-116);
  cos_t168 : constant octo_double
           := create( 8.68570705971340895E-01, 2.71927816897829032E-19,
                     -1.67101403969324282E-35,-1.26255147346379688E-51,
                     -5.98654226661371267E-68,-1.50955507390121791E-84,
                     -2.59103317009717227E-101, 1.22697484212396339E-117);
  cos_t169 : constant octo_double
           := create( 8.67046245515692648E-01, 3.02678595509305667E-18,
                     -1.15594387821715718E-34,-5.35805563978080119E-52,
                     -4.28506058754367816E-69,-2.08467770100785323E-85,
                      8.89424881347040456E-102,-6.14206564148871148E-118);
  cos_t170 : constant octo_double
           := create( 8.65513624090569089E-01,-6.37231135496288988E-18,
                      2.37255203217468325E-34, 1.59118803483951749E-50,
                      9.07552778100558198E-67,-5.44533838768258701E-84,
                      1.29883075832665651E-100, 1.07717916442005427E-116);
  cos_t171 : constant octo_double
           := create( 8.63972856121586696E-01, 4.14863559573616071E-17,
                      2.27099769322102664E-33,-8.12283856594799842E-50,
                     -3.76029505994081227E-66,-8.26358050616788289E-83,
                     -7.14825421516855036E-99, 2.38295041726871053E-115);
  cos_t172 : constant octo_double
           := create( 8.62423956111040502E-01, 3.70089925273831299E-17,
                      5.21284115427015728E-34, 2.69456000810268615E-50,
                     -2.77241410008718926E-67,-8.87692199390093526E-84,
                     -8.86121097456247394E-100,-4.77766867578843616E-116);
  cos_t173 : constant octo_double
           := create( 8.60866938637767309E-01,-3.00500488985736562E-17,
                     -8.87061830908921111E-34, 1.50053205580973008E-50,
                     -9.27953646128873950E-67, 3.96849153291894039E-83,
                      1.60329666370346146E-99,-3.60560408737110614E-116);
  cos_t174 : constant octo_double
           := create( 8.59301818357008362E-01, 4.24356558168506872E-17,
                      7.61818140599120249E-34,-3.95921278506587079E-50,
                      9.72622088830955615E-67,-1.77062992351235503E-85,
                      1.11706313469972439E-101,-3.51675909670601207E-118);
  cos_t175 : constant octo_double
           := create( 8.57728610000272118E-01,-4.81834479363366201E-17,
                     -1.10441305176875334E-33, 8.36567080012549594E-50,
                      2.10780578646816070E-66, 4.52933101817810101E-83,
                     -6.72264346600263696E-100,-2.01493744763302878E-116);
  cos_t176 : constant octo_double
           := create( 8.56147328375194472E-01, 9.18069256166062606E-18,
                      5.63286497859514699E-34, 2.33266461132173782E-51,
                      1.20279635197638733E-67, 2.20702248018362072E-84,
                     -7.47277371892129419E-101,-5.17221625097663478E-117);
  cos_t177 : constant octo_double
           := create( 8.54557988365400534E-01,-1.29911242363960916E-17,
                      1.28934077229480803E-34,-3.65069257475830527E-52,
                      1.29156216501808980E-68,-7.40461585629889932E-85,
                     -5.46350759368241801E-101,-2.10005744207930166E-117);
  cos_t178 : constant octo_double
           := create( 8.52960604930363631E-01, 2.71529842519813696E-17,
                      7.43364832831207193E-34, 4.21624176223506684E-50,
                     -3.99971106716386046E-67, 1.14048495105206873E-83,
                      1.01446380644951989E-101, 7.21424017847209023E-118);
  cos_t179 : constant octo_double
           := create( 8.51355193105265196E-01,-5.32798744460162087E-17,
                      2.22811563809199417E-33,-4.02818864041384766E-50,
                     -2.32974514667335517E-66,-9.54969389325849038E-83,
                     -6.70760407438132036E-100, 4.90970048913518058E-116);
  cos_t180 : constant octo_double
           := create( 8.49741768000852438E-01, 5.18123476599740155E-17,
                      3.08106260873312754E-33,-2.59313082019949651E-50,
                     -1.48769511923567653E-66, 1.29481628770209766E-83,
                     -2.85836681246348824E-100, 8.22589342191472184E-117);
  cos_t181 : constant octo_double
           := create( 8.48120344803297233E-01, 1.87625634152399810E-17,
                      1.40487733079196167E-33,-2.49152215099586914E-50,
                     -3.10879601359248259E-67,-6.28058823920548878E-84,
                     -1.27952757695492605E-100,-3.96326673293678578E-117);
  cos_t182 : constant octo_double
           := create( 8.46490938774052126E-01,-4.79694199585693451E-17,
                     -2.75182670978867026E-33,-7.35189597273133496E-50,
                     -1.79992545378431456E-66,-8.07549509177087165E-83,
                     -6.75692565312063963E-99, 3.69407563872155989E-115);
  cos_t183 : constant octo_double
           := create( 8.44853565249707117E-01,-4.36313602968796371E-17,
                     -2.03077268533675470E-33, 4.30972298198517614E-50,
                     -2.94316743336060953E-66, 2.08364884809147169E-82,
                      3.89194394324208042E-99,-1.34383529796321358E-115);
  cos_t184 : constant octo_double
           := create( 8.43208239641845436E-01, 9.65367070059590768E-19,
                      2.89951424315563637E-36, 9.67150768114802837E-53,
                      1.40225208152153184E-69, 1.39009498829323027E-86,
                     -5.18183413117357503E-105, 2.60545478032205367E-121);
  cos_t185 : constant octo_double
           := create( 8.41554977436898444E-01,-3.40954653913215567E-17,
                     -8.41302086075795948E-34,-4.94472839605686856E-50,
                      4.19142603088527515E-66, 1.15117232095336389E-82,
                      7.01650063703618002E-99, 1.45545069953295861E-115);
  cos_t186 : constant octo_double
           := create( 8.39893794195999521E-01,-1.66736948815114113E-17,
                     -1.47591841417502893E-33,-7.57950981619140581E-50,
                      4.49995646997760470E-66, 2.50140986439293233E-83,
                     -7.35614162110444218E-100, 3.58971486738000141E-116);
  cos_t187 : constant octo_double
           := create( 8.38224705554838079E-01,-3.55600850528550261E-17,
                      1.16897915770226427E-33,-5.86273473597234114E-50,
                      4.06735227318414874E-66, 2.52619828812764647E-82,
                     -7.73362195345344032E-99,-2.50688542276844309E-115);
  cos_t188 : constant octo_double
           := create( 8.36547727223512005E-01,-2.08990590270665325E-17,
                     -9.81040978210025854E-35,-3.16091778682298533E-51,
                     -9.63876215166712618E-68, 1.68028987100428272E-84,
                     -6.16764417741841199E-101,-2.94514807857070266E-117);
  cos_t189 : constant octo_double
           := create( 8.34862874986380010E-01, 4.60484306091596574E-17,
                     -5.18274232652399122E-34,-7.05053434355041089E-51,
                      5.86894939706931855E-68, 1.16026618462615502E-84,
                     -8.34924136658410209E-101, 6.14208978790693918E-117);
  cos_t190 : constant octo_double
           := create( 8.33170164701913185E-01, 1.32751295072297644E-18,
                      4.85891641153708626E-35, 4.54222813005068589E-51,
                      1.24684694920008172E-67,-5.06101801832919203E-84,
                      4.21277862781142705E-100,-2.04508804433076737E-116);
  cos_t191 : constant octo_double
           := create( 8.31469612302545236E-01, 1.40738569847280239E-18,
                      4.69513153839808352E-35,-2.02338815193825685E-52,
                      1.80862190676516987E-68,-8.69192376109987748E-85,
                     -1.44946744968666399E-101, 5.37893425803780335E-118);
  cos_t192 : constant octo_double
           := create( 8.29761233794523045E-01,-2.93491093764855972E-18,
                      1.14969179341498182E-34, 3.51866655449802329E-51,
                      2.78556961490874403E-67,-1.44309352476996495E-83,
                     -3.16443712628995738E-100,-2.01192204485881835E-116);
  cos_t193 : constant octo_double
           := create( 8.28045045257755796E-01,-4.41965932258715322E-17,
                      2.79678648552112514E-33, 1.00307772873935024E-49,
                      8.76178885992477621E-66, 3.13938859489836763E-82,
                     -9.00567528534413452E-99, 5.17081704082530844E-115);
  cos_t194 : constant octo_double
           := create( 8.26321062845663534E-01,-5.39574854536129017E-17,
                      6.89768961301385503E-34, 3.81061642741991960E-50,
                      1.51587929364825242E-66, 2.35521198265514756E-84,
                      6.89191177879717867E-102,-3.12863928351092492E-118);
  cos_t195 : constant octo_double
           := create( 8.24589302785025291E-01,-2.65123604888682752E-17,
                      1.69169643509143861E-34, 6.76939748135626486E-51,
                      2.77282255592114680E-67,-6.33966117698259256E-84,
                      4.08492416157300688E-100, 5.85303709901462227E-117);
  cos_t196 : constant octo_double
           := create( 8.22849781375826317E-01, 1.51930190345054947E-17,
                      9.68905472465216850E-34, 5.69945629236532643E-50,
                     -2.55485291460119648E-66,-1.76042948552635333E-82,
                     -1.47705996151342594E-99,-2.00820866203629458E-116);
  cos_t197 : constant octo_double
           := create( 8.21102514991104648E-01, 3.07151316096976817E-17,
                     -1.70371683258558788E-33,-1.11498624432838531E-49,
                     -4.57918879704099086E-66, 1.63510950982401734E-82,
                     -6.65040898151770911E-99,-3.36406740024041633E-115);
  cos_t198 : constant octo_double
           := create( 8.19347520076797009E-01,-4.82007369951911332E-17,
                     -1.55744896466727808E-35,-9.56478536145222163E-53,
                     -8.47815998866343963E-69, 6.79115283386241611E-86,
                     -3.18474277469231575E-102, 2.02263447604729452E-119);
  cos_t199 : constant octo_double
           := create( 8.17584813151583711E-01,-1.48831498124267717E-17,
                     -7.82732627712989171E-34, 4.13321491610315945E-50,
                      2.26241333804064943E-66, 4.57210587403679514E-83,
                     -1.47787828750723035E-99, 8.34737895353133910E-116);
  cos_t200 : constant octo_double
           := create( 8.15814410806733781E-01, 8.26526937821308706E-18,
                     -2.30287781351794708E-34, 1.51020713872498429E-50,
                      1.11074359445313109E-66,-3.53516918783849070E-83,
                     -1.11422455273816773E-99,-3.08729922156241098E-117);
  cos_t201 : constant octo_double
           := create( 8.14036329705948414E-01,-5.21273518770426242E-17,
                     -1.90476706113163596E-33,-1.69372695859415089E-49,
                      7.33648318429306217E-66,-2.91201700450456406E-82,
                     -2.29623110643455198E-99, 9.38703371803024460E-116);
  cos_t202 : constant octo_double
           := create( 8.12250586585203882E-01, 3.10545456092148025E-17,
                      2.26495419227072515E-34,-7.42216841546494051E-51,
                     -1.89983046279279691E-67,-7.63967592611586337E-84,
                     -1.90583599871037281E-100,-1.09375821503590715E-116);
  cos_t203 : constant octo_double
           := create( 8.10457198252594768E-01, 2.35203673498404990E-17,
                     -7.75300709048463408E-34,-7.27926163571971404E-50,
                      2.39892753940360295E-66,-1.39234436198979365E-82,
                     -7.38085246595720096E-99, 2.71706101226970742E-115);
  cos_t204 : constant octo_double
           := create( 8.08656181588174983E-01, 9.32515978797216739E-18,
                     -7.18233019330683940E-34, 2.39254408461321061E-50,
                      9.75100280469822160E-67, 2.87072101066184638E-83,
                      8.50269913792596222E-100,-3.11040045719388525E-116);
  cos_t205 : constant octo_double
           := create( 8.06847553543799223E-01, 4.92206037660955458E-17,
                      2.97960168999034868E-33, 1.52207542236157884E-49,
                     -8.20537240314446833E-66,-5.08801529400560496E-82,
                      2.19519574810594033E-98,-7.14998273967986694E-115);
  cos_t206 : constant octo_double
           := create( 8.05031331142963547E-01, 5.13682895682121485E-17,
                      6.30828074022565241E-34, 7.32776460851298267E-51,
                     -1.57429892135200407E-67,-1.57140477187390270E-83,
                      5.62586099970600174E-100, 1.40219819106494183E-116);
  cos_t207 : constant octo_double
           := create( 8.03207531480644943E-01,-3.30606098048149096E-17,
                     -1.22427262524204334E-33, 2.84136732686301168E-50,
                     -1.77472778752101069E-66,-8.08418180719305248E-83,
                      3.04868199060977942E-99, 1.34097321446384906E-115);
  cos_t208 : constant octo_double
           := create( 8.01376171723140240E-01,-2.09580134134958343E-17,
                     -4.37981621980069310E-34, 2.02356904977525149E-50,
                      7.99206533569950952E-67, 6.10990634086046154E-83,
                      3.33879628818755324E-100,-2.44560556815854340E-116);
  cos_t209 : constant octo_double
           := create( 7.99537269107905013E-01, 2.03567238220054308E-17,
                     -9.74485136968963603E-34, 5.36081095996960076E-52,
                      3.11416781644117344E-68, 8.01347482037836151E-85,
                     -1.60075613925974116E-101, 5.34952993625944697E-118);
  cos_t210 : constant octo_double
           := create( 7.97690840943391155E-01,-4.67307598847889437E-17,
                      2.30758970771917571E-33, 3.16055677746402533E-51,
                      2.95606214041708767E-67,-1.07914915777331787E-83,
                      3.60103478627628118E-100,-1.15937918663442059E-116);
  cos_t211 : constant octo_double
           := create( 7.95836904608883566E-01,-3.00627248519107210E-17,
                     -2.24962108320422348E-33,-6.58817741171830403E-50,
                     -2.26750460140100101E-66,-7.83779841151105783E-83,
                     -3.58687602231499806E-99, 2.81146703510034567E-116);
  cos_t212 : constant octo_double
           := create( 7.93975477554337172E-01,-7.41946317599214165E-18,
                      2.41243413046310687E-34,-4.99568086162449721E-51,
                      1.59478184276499887E-67,-4.34516373214877536E-84,
                     -3.58342168702975843E-100, 1.21714044941247247E-116);
  cos_t213 : constant octo_double
           := create( 7.92106577300212389E-01,-3.70878502023264666E-17,
                     -1.48744572672282636E-33, 2.93230972891535051E-50,
                      5.44957337644476069E-67,-1.36782586341549824E-83,
                     -1.52140483965265471E-101,-9.96570920545965255E-118);
  cos_t214 : constant octo_double
           := create( 7.90230221437310032E-01, 2.30569059549544919E-17,
                      1.44810805332601935E-33,-7.67252370572034879E-50,
                      1.15364173026066538E-67, 9.04606685951134895E-85,
                     -2.80446127907435379E-101, 1.02321738401556100E-117);
  cos_t215 : constant octo_double
           := create( 7.88346427626606228E-01, 3.43969931540597076E-17,
                      1.77106237467371704E-33, 1.70841590984174018E-49,
                      1.29989097232837413E-66, 4.80336976345926870E-84,
                      4.45987160141755700E-101, 1.83346199383946818E-117);
  cos_t216 : constant octo_double
           := create( 7.86455213599085767E-01,-9.78414299393052654E-18,
                      3.39060632724454723E-34, 5.72695053203825766E-51,
                     -5.59911540376708308E-67,-1.54042605807597777E-83,
                      7.50229619808078922E-100, 2.23797071884846207E-116);
  cos_t217 : constant octo_double
           := create( 7.84556597155575242E-01,-8.56279654231734757E-18,
                     -2.11068344590018491E-34,-1.68903221824696034E-50,
                      3.42873394313861129E-67, 1.11257203233140109E-83,
                      3.94934970011989446E-100, 3.28485113660475476E-117);
  cos_t218 : constant octo_double
           := create( 7.82650596166575729E-01, 9.07458669758088248E-18,
                      6.76238474042786663E-34,-1.71732377319872710E-50,
                     -9.93684984725503518E-67, 3.17443463756980676E-83,
                     -1.50513001978939340E-99,-2.11077380435569501E-116);
  cos_t219 : constant octo_double
           := create( 7.80737228572094488E-01,-9.91987820666788060E-18,
                     -2.12657940121627152E-36, 3.07721655989576468E-54,
                      1.80158482793033295E-70,-1.34669007254800522E-86,
                      7.80694206355215236E-103, 4.95134439641453292E-119);
  cos_t220 : constant octo_double
           := create( 7.78816512381475978E-01,-2.48913855799738065E-17,
                      6.76654970248079802E-35,-6.52185942817013320E-52,
                      1.58969924388680187E-69, 6.51853256499713246E-86,
                      6.58973835507586416E-102, 2.92696066902391406E-118);
  cos_t221 : constant octo_double
           := create( 7.76888465673232442E-01, 7.74186025706728636E-18,
                     -5.99865178721578973E-34, 3.05665482329589716E-50,
                     -2.12151636991502970E-66,-2.28800022085562485E-83,
                      5.73272126517782747E-102, 1.79971793772880788E-118);
  cos_t222 : constant octo_double
           := create( 7.74953106594873931E-01,-5.22090831898264327E-17,
                     -9.66535933936866120E-34, 3.70277500765625693E-50,
                     -2.32003665023097361E-66,-9.43873230178152217E-83,
                     -6.24058604140948038E-100, 2.11505546980347515E-116);
  cos_t223 : constant octo_double
           := create( 7.73010453362736993E-01,-3.25659070336497723E-17,
                      1.38608072515239293E-33,-3.99713299175860216E-50,
                     -5.87184241638023931E-67, 2.90836779315596850E-84,
                     -1.18751113655406475E-100,-4.31542979610777401E-117);
  cos_t224 : constant octo_double
           := create( 7.71060524261813818E-01,-4.45584423477692650E-17,
                     -2.98635656140837829E-33,-6.87952620835962359E-50,
                     -2.79058077850289113E-66,-3.00471395073884376E-83,
                      7.64461639901323789E-101, 1.31350710903665172E-117);
  cos_t225 : constant octo_double
           := create( 7.69103337645579588E-01, 5.15464551845648167E-17,
                      2.61428295535242917E-33,-1.61990236327732983E-49,
                      8.23753465067406557E-66,-4.81932572971675509E-82,
                      1.85356119378349438E-98,-1.41099731146406473E-114);
  cos_t226 : constant octo_double
           := create( 7.67138911935820400E-01,-1.88859036837507816E-17,
                     -1.36593593314954328E-33,-2.25388349629219338E-50,
                     -1.57641293479742966E-66, 5.37008075326108372E-83,
                      3.22758140620785202E-99,-6.88531766844382796E-116);
  cos_t227 : constant octo_double
           := create( 7.65167265622458959E-01,-3.27072256125345983E-17,
                      1.11771177470795276E-33,-3.70051822801757151E-50,
                      1.52856893645965124E-66,-9.88565803951930985E-83,
                     -6.41339777468150683E-99,-3.67702350647877322E-115);
  cos_t228 : constant octo_double
           := create( 7.63188417263381269E-01, 2.63147484167507484E-18,
                      1.40480390630959103E-34, 8.96018866266303208E-52,
                      5.92016319737862517E-68,-2.72152413358343248E-85,
                     -1.35512522028139270E-101, 7.71590612095592004E-118);
  cos_t229 : constant octo_double
           := create( 7.61202385484261779E-01, 3.53155108816905509E-17,
                      1.28335663818643591E-33,-8.48355062650099085E-50,
                      2.59574018798442827E-66,-2.16126931530912636E-82,
                     -6.02393371067981473E-99, 2.41906811037512169E-115);
  cos_t230 : constant octo_double
           := create( 7.59209188978388072E-01,-3.85588421755231226E-17,
                      2.97202412083327591E-34,-1.25213889282201633E-50,
                     -9.50796849787696662E-69,-5.69624742907900866E-85,
                      1.21722832225560743E-101,-2.29174159138829256E-118);
  cos_t231 : constant octo_double
           := create( 7.57208846506484568E-01,-1.99090987773355019E-17,
                      3.94092832661584820E-34, 2.07442542078029764E-50,
                      1.84967125011123820E-67,-3.21731322687720385E-84,
                     -8.11338110480626528E-101,-7.85111920398289294E-118);
  cos_t232 : constant octo_double
           := create( 7.55201376896536547E-01,-1.94022380018230173E-17,
                     -3.77562064447275734E-34,-2.12122423081782872E-50,
                     -4.35116900426565139E-68, 3.81396602626495010E-84,
                     -1.93406370120564640E-100,-1.03167960171057699E-116);
  cos_t233 : constant octo_double
           := create( 7.53186799043612520E-01,-3.79377898387365397E-17,
                     -6.70095399202315589E-34,-6.71285621150502140E-51,
                     -3.73199195403069783E-67, 3.09209542827111704E-83,
                     -1.04555741975290515E-99, 5.07450618893185909E-116);
  cos_t234 : constant octo_double
           := create( 7.51165131909686368E-01, 4.34997611586458684E-17,
                      2.52277189711022124E-33,-6.59697092127571017E-50,
                      1.70122728327206657E-66, 3.44850141010511855E-83,
                      1.02836296253189260E-99,-6.02766840068461899E-117);
  cos_t235 : constant octo_double
           := create( 7.49136394523459370E-01,-4.47290784470118887E-17,
                     -2.42060252499837676E-33, 1.13366813511164239E-49,
                     -3.85460906711813256E-66, 7.99340477938902223E-83,
                      1.63607522210387854E-99, 3.87361859246198560E-116);
  cos_t236 : constant octo_double
           := create( 7.47100605980180132E-01, 1.18748248759654303E-17,
                      2.19925238498335178E-34, 1.10250185646444830E-50,
                     -1.44201284749006903E-67,-7.65530350947307930E-85,
                      4.29655270597590688E-101,-5.83709489247448776E-118);
  cos_t237 : constant octo_double
           := create( 7.45057785441465947E-01, 1.50786869118778628E-17,
                      8.08989872129424711E-34, 8.26779587653235318E-50,
                      4.57146336616658912E-66, 2.43022816459765415E-82,
                     -1.15870031193112784E-98, 5.57734619918296287E-115);
  cos_t238 : constant octo_double
           := create( 7.43007952135121719E-01,-2.51446296697192650E-17,
                      7.11289895125261566E-34, 3.01816290778212201E-50,
                     -1.45718227109412454E-66, 4.48897408258470590E-84,
                      4.06791923875464614E-100, 1.02574651231260981E-116);
  cos_t239 : constant octo_double
           := create( 7.40951125354959106E-01,-1.47086169522973452E-17,
                     -4.95504338271420317E-34, 3.14341325337356709E-50,
                     -8.74572003205103851E-67, 5.95046846145053695E-83,
                      3.63711515266175249E-99, 1.91624730339486914E-115);
  cos_t240 : constant octo_double
           := create( 7.38887324460615114E-01, 3.43248748082250910E-17,
                     -1.37066394447176100E-33,-3.35208275307189379E-51,
                      1.34306817695004519E-67, 6.93350429935821635E-84,
                     -3.25811294101672528E-101,-9.74631440640070672E-119);
  cos_t241 : constant octo_double
           := create( 7.36816568877369904E-01,-2.89324681016562949E-17,
                     -3.46498871262023781E-34,-1.84844744762914758E-50,
                      1.10322689087000852E-66, 3.87239303525113149E-83,
                     -3.30158594540024819E-99,-1.22756570808872914E-115);
  cos_t242 : constant octo_double
           := create( 7.34738878095963499E-01,-3.45075959762639412E-17,
                     -2.37180006766664089E-33,-3.96960903871654021E-50,
                      5.56139266819586662E-67,-2.72404125872524782E-83,
                     -3.71762778283113930E-100, 2.20406154289584634E-117);
  cos_t243 : constant octo_double
           := create( 7.32654271672412816E-01, 1.89186734815735196E-17,
                     -1.51237195441198860E-33,-9.79221520116257279E-51,
                      2.15939020123068802E-67,-1.59912297639812090E-83,
                      6.07786768426215207E-100, 1.68180575837333321E-116);
  cos_t244 : constant octo_double
           := create( 7.30562769227827591E-01,-2.96899599044769280E-17,
                     -1.12768712442397442E-33,-3.05315209615390073E-50,
                      1.98951471390596132E-66, 9.76701320696808611E-84,
                      4.01417218036712560E-100,-1.45083788238704024E-116);
  cos_t245 : constant octo_double
           := create( 7.28464390448225196E-01, 1.19246423233707176E-19,
                      5.90018923166110111E-36, 1.21780890695027036E-52,
                     -6.21747046015240160E-69, 2.03668407075855177E-85,
                      5.73312785998625447E-102, 5.47819166222143882E-119);
  cos_t246 : constant octo_double
           := create( 7.26359155084346009E-01,-3.19175024434605418E-17,
                      7.70479124120393961E-34, 4.14558801601821234E-50,
                     -1.39953003097243823E-67, 1.79118956773450665E-84,
                     -1.06876854786331409E-101, 8.73454262795612774E-119);
  cos_t247 : constant octo_double
           := create( 7.24247082951466892E-01, 2.91984713344030044E-17,
                      2.30273249687394643E-33,-1.29288205338921832E-51,
                      4.16385887990780634E-68, 1.72327966387108462E-84,
                     -9.87183634655684642E-101, 3.76094569754187731E-117);
  cos_t248 : constant octo_double
           := create( 7.22128193929215345E-01,-2.38712620534520474E-17,
                      1.06361254328622726E-33,-4.45986388378025172E-50,
                      1.72751679066061951E-66,-5.53694502237793733E-83,
                     -3.57588524264425827E-99,-2.38384158661875562E-116);
  cos_t249 : constant octo_double
           := create( 7.20002507961381655E-01,-2.56896588544623333E-17,
                     -9.14925669485679253E-34, 4.44037808012677858E-50,
                     -2.12198361485441019E-66, 1.27884771498526192E-82,
                     -6.97751900566949928E-99, 3.31189435317157235E-115);
  cos_t250 : constant octo_double
           := create( 7.17870045055731709E-01, 2.70064760625114533E-17,
                     -2.28549565802153482E-34, 9.17269038902878672E-51,
                      1.44061942715007654E-68,-1.88923373167755234E-85,
                     -5.15336787790840759E-102,-1.05062779708296134E-118);
  cos_t251 : constant octo_double
           := create( 7.15730825283818706E-01,-5.15810184764102619E-17,
                     -1.37362713493002614E-34, 8.64750633662626798E-51,
                      1.30418039883353805E-67,-5.44803885877779374E-85,
                     -2.93010400980414548E-101, 2.85226611533079471E-117);
  cos_t252 : constant octo_double
           := create( 7.13584868780793635E-01,-4.23425044031335842E-17,
                     -4.26903661016172682E-34,-2.63523708830665222E-50,
                     -4.87237798306160764E-67,-3.14625609103327139E-83,
                     -1.32259290080673598E-99,-1.61642037983535338E-116);
  cos_t253 : constant octo_double
           := create( 7.11432195745216434E-01, 7.96432986138568126E-18,
                      2.94882395107214687E-34, 1.69852364376663557E-50,
                     -8.30614212941238664E-67,-3.05786707202141167E-83,
                     -1.29248426058282786E-100,-7.51574582703618824E-117);
  cos_t254 : constant octo_double
           := create( 7.09272826438865689E-01,-3.75973591102457304E-17,
                      1.06131259546451192E-34, 8.94654801854860323E-51,
                     -3.93309712103078023E-67,-1.23882277375143911E-83,
                      1.34363675560727344E-101, 1.05825732919371533E-118);
  cos_t255 : constant octo_double
           := create( 7.07106781186547573E-01,-4.83364665672645673E-17,
                      2.06933765434970678E-33, 2.46777349573417546E-50,
                      2.04470214305994790E-67, 1.55277156441246667E-84,
                     -1.00055013355304061E-100, 1.53225205737542922E-117);

  sin_table : array(0..255) of octo_double 
            := (sin_t0  ,sin_t1  ,sin_t2  ,sin_t3  ,sin_t4  ,sin_t5  ,
                sin_t6  ,sin_t7  ,sin_t8  ,sin_t9  ,sin_t10 ,sin_t11 ,
                sin_t12 ,sin_t13 ,sin_t14 ,sin_t15 ,sin_t16 ,sin_t17 ,
                sin_t18 ,sin_t19 ,sin_t20 ,sin_t21 ,sin_t22 ,sin_t23 ,
                sin_t24 ,sin_t25 ,sin_t26 ,sin_t27 ,sin_t28 ,sin_t29 ,
                sin_t30 ,sin_t31 ,sin_t32 ,sin_t33 ,sin_t34 ,sin_t35 ,
                sin_t36 ,sin_t37 ,sin_t38 ,sin_t39 ,sin_t40 ,sin_t41 ,
                sin_t42 ,sin_t43 ,sin_t44 ,sin_t45 ,sin_t46 ,sin_t47 ,
                sin_t48 ,sin_t49 ,sin_t50 ,sin_t51 ,sin_t52 ,sin_t53 ,
                sin_t54 ,sin_t55 ,sin_t56 ,sin_t57 ,sin_t58 ,sin_t59 ,
                sin_t60 ,sin_t61 ,sin_t62 ,sin_t63 ,sin_t64 ,sin_t65 ,
                sin_t66 ,sin_t67 ,sin_t68 ,sin_t69 ,sin_t70 ,sin_t71 ,
                sin_t72 ,sin_t73 ,sin_t74 ,sin_t75 ,sin_t76 ,sin_t77 ,
                sin_t78 ,sin_t79 ,sin_t80 ,sin_t81 ,sin_t82 ,sin_t83 ,
                sin_t84 ,sin_t85 ,sin_t86 ,sin_t87 ,sin_t88 ,sin_t89 ,
                sin_t90 ,sin_t91 ,sin_t92 ,sin_t93 ,sin_t94 ,sin_t95 ,
                sin_t96 ,sin_t97 ,sin_t98 ,sin_t99 ,sin_t100,sin_t101,
                sin_t102,sin_t103,sin_t104,sin_t105,sin_t106,sin_t107,
                sin_t108,sin_t109,sin_t110,sin_t111,sin_t112,sin_t113,
                sin_t114,sin_t115,sin_t116,sin_t117,sin_t118,sin_t119,
                sin_t120,sin_t121,sin_t122,sin_t123,sin_t124,sin_t125,
                sin_t126,sin_t127,sin_t128,sin_t129,sin_t130,sin_t131,
                sin_t132,sin_t133,sin_t134,sin_t135,sin_t136,sin_t137,
                sin_t138,sin_t139,sin_t140,sin_t141,sin_t142,sin_t143,
                sin_t144,sin_t145,sin_t146,sin_t147,sin_t148,sin_t149,
                sin_t150,sin_t151,sin_t152,sin_t153,sin_t154,sin_t155,
                sin_t156,sin_t157,sin_t158,sin_t159,sin_t160,sin_t161,
                sin_t162,sin_t163,sin_t164,sin_t165,sin_t166,sin_t167,
                sin_t168,sin_t169,sin_t170,sin_t171,sin_t172,sin_t173,
                sin_t174,sin_t175,sin_t176,sin_t177,sin_t178,sin_t179,
                sin_t180,sin_t181,sin_t182,sin_t183,sin_t184,sin_t185,
                sin_t186,sin_t187,sin_t188,sin_t189,sin_t190,sin_t191,
                sin_t192,sin_t193,sin_t194,sin_t195,sin_t196,sin_t197,
                sin_t198,sin_t199,sin_t200,sin_t201,sin_t202,sin_t203,
                sin_t204,sin_t205,sin_t206,sin_t207,sin_t208,sin_t209,
                sin_t210,sin_t211,sin_t212,sin_t213,sin_t214,sin_t215,
                sin_t216,sin_t217,sin_t218,sin_t219,sin_t220,sin_t221,
                sin_t222,sin_t223,sin_t224,sin_t225,sin_t226,sin_t227,
                sin_t228,sin_t229,sin_t230,sin_t231,sin_t232,sin_t233,
                sin_t234,sin_t235,sin_t236,sin_t237,sin_t238,sin_t239,
                sin_t240,sin_t241,sin_t242,sin_t243,sin_t244,sin_t245,
                sin_t246,sin_t247,sin_t248,sin_t249,sin_t250,sin_t251,
                sin_t252,sin_t253,sin_t254,sin_t255);

  cos_table : array(0..255) of octo_double
            := (cos_t0  ,cos_t1  ,cos_t2  ,cos_t3  ,cos_t4  ,cos_t5  ,
                cos_t6  ,cos_t7  ,cos_t8  ,cos_t9  ,cos_t10 ,cos_t11 ,
                cos_t12 ,cos_t13 ,cos_t14 ,cos_t15 ,cos_t16 ,cos_t17 ,
                cos_t18 ,cos_t19 ,cos_t20 ,cos_t21 ,cos_t22 ,cos_t23 ,
                cos_t24 ,cos_t25 ,cos_t26 ,cos_t27 ,cos_t28 ,cos_t29 ,
                cos_t30 ,cos_t31 ,cos_t32 ,cos_t33 ,cos_t34 ,cos_t35 ,
                cos_t36 ,cos_t37 ,cos_t38 ,cos_t39 ,cos_t40 ,cos_t41 ,
                cos_t42 ,cos_t43 ,cos_t44 ,cos_t45 ,cos_t46 ,cos_t47 ,
                cos_t48 ,cos_t49 ,cos_t50 ,cos_t51 ,cos_t52 ,cos_t53 ,
                cos_t54 ,cos_t55 ,cos_t56 ,cos_t57 ,cos_t58 ,cos_t59 ,
                cos_t60 ,cos_t61 ,cos_t62 ,cos_t63 ,cos_t64 ,cos_t65 ,
                cos_t66 ,cos_t67 ,cos_t68 ,cos_t69 ,cos_t70 ,cos_t71 ,
                cos_t72 ,cos_t73 ,cos_t74 ,cos_t75 ,cos_t76 ,cos_t77 ,
                cos_t78 ,cos_t79 ,cos_t80 ,cos_t81 ,cos_t82 ,cos_t83 ,
                cos_t84 ,cos_t85 ,cos_t86 ,cos_t87 ,cos_t88 ,cos_t89 ,
                cos_t90 ,cos_t91 ,cos_t92 ,cos_t93 ,cos_t94 ,cos_t95 ,
                cos_t96 ,cos_t97 ,cos_t98 ,cos_t99 ,cos_t100,cos_t101,
                cos_t102,cos_t103,cos_t104,cos_t105,cos_t106,cos_t107,
                cos_t108,cos_t109,cos_t110,cos_t111,cos_t112,cos_t113,
                cos_t114,cos_t115,cos_t116,cos_t117,cos_t118,cos_t119,
                cos_t120,cos_t121,cos_t122,cos_t123,cos_t124,cos_t125,
                cos_t126,cos_t127,cos_t128,cos_t129,cos_t130,cos_t131,
                cos_t132,cos_t133,cos_t134,cos_t135,cos_t136,cos_t137,
                cos_t138,cos_t139,cos_t140,cos_t141,cos_t142,cos_t143,
                cos_t144,cos_t145,cos_t146,cos_t147,cos_t148,cos_t149,
                cos_t150,cos_t151,cos_t152,cos_t153,cos_t154,cos_t155,
                cos_t156,cos_t157,cos_t158,cos_t159,cos_t160,cos_t161,
                cos_t162,cos_t163,cos_t164,cos_t165,cos_t166,cos_t167,
                cos_t168,cos_t169,cos_t170,cos_t171,cos_t172,cos_t173,
                cos_t174,cos_t175,cos_t176,cos_t177,cos_t178,cos_t179,
                cos_t180,cos_t181,cos_t182,cos_t183,cos_t184,cos_t185,
                cos_t186,cos_t187,cos_t188,cos_t189,cos_t190,cos_t191,
                cos_t192,cos_t193,cos_t194,cos_t195,cos_t196,cos_t197,
                cos_t198,cos_t199,cos_t200,cos_t201,cos_t202,cos_t203,
                cos_t204,cos_t205,cos_t206,cos_t207,cos_t208,cos_t209,
                cos_t210,cos_t211,cos_t212,cos_t213,cos_t214,cos_t215,
                cos_t216,cos_t217,cos_t218,cos_t219,cos_t220,cos_t221,
                cos_t222,cos_t223,cos_t224,cos_t225,cos_t226,cos_t227,
                cos_t228,cos_t229,cos_t230,cos_t231,cos_t232,cos_t233,
                cos_t234,cos_t235,cos_t236,cos_t237,cos_t238,cos_t239,
                cos_t240,cos_t241,cos_t242,cos_t243,cos_t244,cos_t245,
                cos_t246,cos_t247,cos_t248,cos_t249,cos_t250,cos_t251,
                cos_t252,cos_t253,cos_t254,cos_t255);

end Octo_Double_Constants;
