/*	Real_Range

PIRL CVS ID: Real_Range.java,v 1.13 2012/04/16 06:18:24 castalia Exp

Copyright (C) 2007-2012  Arizona Board of Regents on behalf of the
Planetary Image Research Laboratory, Lunar and Planetary Laboratory at
the University of Arizona.

This file is part of the PIRL Java Packages.

The PIRL Java Packages are free software; you can redistribute them
and/or modify them under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation, either version 3 of
the License, or (at your option) any later version.

The PIRL Java Packages are distributed in the hope that they will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser
General Public License for more details.

You should have received a copy of the GNU Lesser General Public License
along with this program. If not, see <http://www.gnu.org/licenses/>.

*******************************************************************************/

package PIRL.Utilities;

/**	A <i>Real_Range</i> specifies a range of integer values.
<p>
	The range of values is characterized by minimum and maximum values
	that identify the inclusive limits of the range. Single-valued and
	open-ended ranges are defined. An appropriate String representation
	is provided.
<p>
	@author		Bradford Castalia, UA/PIRL
	@version	1.13
*/
public class Real_Range
	implements Cloneable
{
public static final String
	ID = "PIRL.Utilities.Real_Range (1.13 2012/04/16 06:18:24)";

/**	The range minimum value.
*/
protected double
	Minimum	= 0.0;

/**	The range maximum value.
*/
protected double
	Maximum	= 0.0;


/**	Constructs a default Real_Range.
<p>
	The range values are initialized to zero.
*/
public Real_Range ()
{}

/**	Constructs a Real_Range from limiting values.
<p>
	Note: If the minimum is greater than the maximum then it is
	used as the maximum and the maximum used as the minimum.
<p>
	@param	minimum	The minimum inclusive value of the range.
	@param	maximum	The maximum inclusive value of the range.
	@see	#Range(double, double)
*/
public Real_Range
	(
	double	minimum,
	double	maximum
	)
{Range (minimum, maximum);}

/**	Constructs a single-valued Real_Range.
<p>
	@param	value	The value of the range.
	@see	#Value(double)
*/
public Real_Range
	(
	double	value
	)
{Value (value);}

/**	Constructs a Real_Range as a copy of another Real_Range.
<p>
	@param	range	The Real_Range to be copied. If null a default Real_Range
		is constructed.
	@see	#Range(Real_Range)
*/
public Real_Range
	(
	Real_Range	range
	)
{Range (range);}

/**	Constructs a Real_Range from an Integer_Range.
<p>
	@param	range	The Integer_Range to be copied. If null a default
		Real_Range is constructed.
	@see	#Range(Integer_Range)
*/
public Real_Range
	(
	Integer_Range	range
	)
{Range (range);}

/**	Clones this Real_Range.
<p>
	@return	A Real_Range constructed as a copy of this Real_Range.
*/
public Object clone ()
{return new Real_Range (this);}

/**	Sets the Range minimum and maximum.
<p>
	Note: If the minimum is greater than the maximum then it is
	used as the maximum and the maximum used as the minimum.
<p>
	@param	minimum	The minimum inclusive value of the range.
	@param	maximum	The maximum inclusive value of the range.
	@return	This Real_Range.
	@see	#Minimum(double)
	@see	#Maximum(double)
*/
public Real_Range Range
	(
	double	minimum,
	double	maximum
	)
{
if (minimum <= maximum)
	{
	Minimum (minimum);
	Maximum (maximum);
	}
else
	{
	Minimum (maximum);
	Maximum (minimum);
	}
return this;
}

/**	Sets the range from another Real_Range.
<p>
	<b>N.B.</b>: The values of the source range are copied without checking.
<p>
	@param	range	The Real_Range to be copied. If null nothing is done.
	@return	This Real_Range.
*/
public Real_Range Range
	(
	Real_Range	range
	)
{
if (range != null)
	{
	Minimum = range.Minimum;
	Maximum = range.Maximum;
	}
return this;
}

/**	Sets the range from an Integer_Range.
<p>
	Long.MIN_VALUE and Long.MAX_VALUE from the Integer_Range are
	reset to Double.MIN_VALUE and Double.MAX_VALUE respectively.
<p>
	@param	range	The Integer_Range limits.
	@return	This Real_Range.
	@see	#Minimum(double)
	@see	#Maximum(double)
*/
public Real_Range Range
	(
	Integer_Range	range
	)
{
if (range != null)
	{
	if (range.Minimum () == Long.MIN_VALUE)
		Minimum (Double.MIN_VALUE);
	else
	if (range.Minimum () == Long.MAX_VALUE)
		Minimum (Double.MAX_VALUE);
	else
		Minimum ((double)range.Minimum ());

	if (range.Maximum () == Long.MIN_VALUE)
		Maximum (Double.MIN_VALUE);
	else
	if (range.Maximum () == Long.MAX_VALUE)
		Maximum (Double.MAX_VALUE);
	else
		Maximum ((double)range.Maximum ());
	}
return this;
}

/**	Gets the minimum range value.
<p>
	@return	The minimum range value.
*/
public double Minimum ()
{return Minimum;}

/**	Sets the minimum range value.
<p>
	If the new minimum value is greater than the current {@link
	#Maximum() maximum} value, the maximum is reset to the new minimum.
<p>
	@param	minimum The minimum range value.
	@return	This Real_Range.
*/
public Real_Range Minimum
	(
	double	minimum
	)
{
if ((Minimum = minimum) > Maximum)
	 Maximum = minimum;
return this;
}

/**	Gets the maximum limit value.
<p>
	@return	The maximum limit value.
*/
public double Maximum ()
{return Maximum;}

/**	Sets the maximum range value.
<p>
	If the new maximum rage value is less than the current {@link
	#Minimum() minimum} value, the minimum is reset to the new maximum.
<p>
	@param	maximum	The maximum range value.
	@return	This Real_Range.
*/
public Real_Range Maximum
	(
	double	maximum
	)
{
if ((Maximum = maximum) < Minimum)
	 Minimum = maximum;
return this;
}

/**	Tests if the Real_Range is open-ended.
<p>
	@return	true	If either the {@link #Minimum() minimum} is
		{@link Double#MIN_VALUE} or the {@link #Maximum() maximum} is
		{@link Double#MAX_VALUE}.
*/
public boolean Is_Open_Ended ()
{
return
	Maximum == Double.MAX_VALUE ||
	Minimum == Double.MIN_VALUE;
}

/**	Gets the distance between the range limits.
<p>
	@return	The difference between the maximum and minimum limit values,
		or Double.MAX_VALUE if the range is {@link #Is_Open_Ended()
		open-ended}.
*/
public double Distance ()
{
if (Is_Open_Ended ())
	return Double.MAX_VALUE;
return Maximum - Minimum;
}

/**	Tests if the range is for a single value.
<p>
	For a single valued range the {@link #Minimum() minimum} and
	{@link #Maximum() maximum} values are identical.
<p>
	@return true if the range only has a single value; false otherwise.
*/
public boolean Is_Single_Valued ()
{return Minimum == Maximum;}

/**	Sets the range to a single value.
<p>
	Both the minimum and maximum are set to the value.
<p>
	@param	value	The single value for the range.
	@return	This Real_Range.
	@see	#Minimum(double)
	@see	#Maximum(double)
*/
public Real_Range Value
	(
	double	value
	)
{
Minimum (value);
return Maximum (value);
}

/**	Compares two Real_Ranges for equality.
<p>
	The two ranges are equal if, and only if, the argument is not null
	and the {@link #Minimum() minimum} and {@link #Maximum() maximum}
	values of the ranges are equal.
*/
public boolean equals
	(
	Real_Range	range
	)
{
return
	range != null &&
	range.Minimum == Minimum &&
	range.Maximum == Maximum;
}

/**	Gets a hash code for this Real_Range.
<p>
	The result is the upper 16 bits of the hash code of the range
	minimum as a Double concatenated with the upper 16 bits of the
	hash code of the range maximum as a Double. That is, the
	hash code is the value of the expression:
<p><blockquote>
	(new Double (Minimum ()).hashCode () & 0xFFFF0000) |
	(new Double (Maximum ()).hashCode () >>> 16)
</blockquote><p>
*/
public int hashCode ()
{
return
	(new Double (Minimum).hashCode () & 0xFFFF0000) |
	(new Double (Maximum).hashCode () >>> 16);
}

/**	Provides a String representation of the Real_Range.
<p>
	The format of the Integer_Range representation is:
<p><blockquote>
	<i>minimum</i><b>-</b><i>maximum</i>
</blockquote><p>
	If the minimum and maximum are identical then only a single value is
	represented.
<p>
	If the minimum is not {@link Double#MIN_VALUE} or the maximum is {@link
	Double#MAX_VALUE} then the minimum value is represented. Then a dash
	('-') delimiter is provided regardless of whether the minimum is
	represented or not.  If the maximum is not {@link Double#MAX_VALUE}
	then it is included in the representation after the '-' delimiter.
	Note that a range may be open-ended at its minimum or maximum, but
	in all cases at least one limit value will be represented with the
	minimum value being used if both both limits are open-ended.
<p>
	@return	A String representation of the Real_Range.
*/
public String toString ()
{
StringBuffer
	range = new StringBuffer ();
if (Minimum == Maximum)
	range.append (String.valueOf (Minimum));
else
	{
	if (Minimum != Double.MIN_VALUE ||
		Maximum == Double.MAX_VALUE)
		range.append (String.valueOf (Minimum));
	range.append ('-');
	if (Maximum != Double.MAX_VALUE)
		range.append (String.valueOf (Maximum));
	}
return range.toString ();
}

}	//	End of Real_Range class.
