#ifndef BIRTHDEATHPROBS_HH
#define BIRTHDEATHPROBS_HH

#include <iostream>
#include <string>
#include <iomanip>

#include "Beep.hh"
#include "BeepVector.hh"

namespace beep
{
  // Forward declarations.
  class Node;
  class Probability;
  class Tree;

  //------------------------------------------------------------------------
  //! We have to put a limit on the size of birth-death parameters
  //! or our program will crash when (birth_rate - death_rate) * t is too big.
  //------------------------------------------------------------------------
  const Real MAX_INTENSITY = 10.0;	// not used by Yang-Rannala
  
  //! This class calculates birth-death probabilities of a "guest-tree" (e.g.,
  //! a gene tree)  being generated by a birth-death process formaly being
  //! constrained by a "host-tree" (e.g., a species tree) by a a 
  //! reconciliation "gamma". However, we can also calculate the probabilities 
  //! of an unreconciled tree by using a fake host-tree, comprising only
  //! a root node, with a associated branch time (defaults to 1.0)
  //! Author: Lars Arvestad, SBC, � the MCMC-club, SBC, all rights reserved
  //
  class BirthDeathProbs 
  {
  public:
    //----------------------------------------------------------------------
    //
    //Constructors and Destructors and Assignment
    //
    //----------------------------------------------------------------------
    BirthDeathProbs(Tree &S, const Real& birth_rate, 
		    const Real& death_rate, Real* topTime = 0);


    BirthDeathProbs(const BirthDeathProbs &M);
    virtual ~BirthDeathProbs();

    // Assignment
    //----------------------------------------------------------------------
    virtual BirthDeathProbs& operator=(const BirthDeathProbs& dsp);

    //----------------------------------------------------------------------
    //
    // Interface
    //
    //----------------------------------------------------------------------
    //! Access to Tree S and topTime
    //---------------------------------------------------------------------
    virtual Tree& getStree() const;
    Real getTopTime() const;

    //! Access to rates
    //---------------------------------------------------------------------
    void getRates(Real& birthRate, Real& deathRate) const;

    Real& getBirthRate();
    Real& getDeathRate();

#ifdef THOMPSON
    //! If THOMPSON is defined this function will return the factor 2* p_1(t)
    //! of Thompson 1975. If multiplied with the result of recursively applied
    //! EdgeTimeProbs(...) it will yield Thompson's probability 
    //! Pr[t,G,l|lambda, mu, S], where G is a labeled history, t is the 
    //! divergence times and l the labels of G, and S is a single-leaved host tree. 
    //! Notice that S must be single-leaved!
    Probability getThompsons_p1()
    {
      Probability P;
      Probability u;
      calcPt_Ut(S.getRootNode()->getTime(), P, u);
      return 2.0 * pow(P*(1.0-u), 2.0);  // 2 * p1(t) = 2 * (P(t) * (1-u_t))^2
    };
#endif

    //! Parameter update.
    //! Assign new rates. There is an automatic update of all probabilities
    //! unless you use the optional parameter update = false.
    //---------------------------------------------------------------------
    void setRates(const Real newBirthRate, const Real newDeathRate, 
		  const bool doUpdate = true);

    //!
    //! Force an update of precomputed values
    //!
    void update();

    //! The conditional probability of n gene copies over an arc (x,y)
    //---------------------------------------------------------------------
    Probability partialProbOfCopies(const Node &y, unsigned n_kids) const;

    //! We treat the top slice differently since we actually don't know the
    //! time scale of the top slice. Therefore, we are adapting an exponential
    //! prior on the time before the species root.
    //---------------------------------------------------------------------
    virtual Probability topPartialProbOfCopies(unsigned n_kids) const;

    //! Returns the probability of going extinct in time t, given that
    //! we start with one gene.
    Probability extinctionProbability(Real t) const;

    //! The probability of extinction below a node v. 
    Probability extinctionProbability(Node *v) const;

    // Given a probability, P, and a vertex sn, sample c, the number of childs
    // from the cumulative probability function for c
    //----------------------------------------------------------------------
    unsigned sampleNumberOfChildren(Node& y, const Real& P) const;

    //! The calculation of 
    //! \f$(c-1)\tilde{P}(t)(1-\tilde{u}_t)/\tilde{u}_T,\f$
    //! in a slice, i.e., partial probabilities for each edge time in a slice.
    //! The full edgetimeprob of the slice is the product of
    //! partialEdgeTimeProbs for all edges an the (c-1)th power of
    //! EdgeTimeProbFactor  (see above) used, e.g., in ReconciliationTimeModel
    //! NOTE!! Despite the name, what is really sent as 't' is really the
    //! 'remaining time' in the slice, i.e., if 'u' is the node such that 
    //! |L(G_u)| = 'leaves', then 't' = u.getNodeTime() -y.getNodeTime() 
    //----------------------------------------------------------------------
    Probability partialEdgeTimeProb(Node& y, const unsigned& leaves, 
				    const Real& t) const;
    
    //! Computes the probability that a birth occur time div_time before
    //! Node x and produces a single lineage at x that survives to the 
    //! leaves of S. Used in ReconciledTreeTimeModel
    Probability bornLineageProbability(Node& x, Real div_time);

    //! The calculation of edge time
    //! Given a number of leaves at vertex sn of time maxT, sample the 
    //! time for the first birth
    //----------------------------------------------------------------------
    Real generateEdgeTime(Node& y, const unsigned& leaves, const Real& P, 
			  Real maxT = -1.0) const;


    //---------------------------------------------------------------------
    // I/O
    //---------------------------------------------------------------------
    friend std::ostream& operator<<(std::ostream &o, 
				    const BirthDeathProbs& tsm);
    virtual std::string print() const;
    


    //---------------------------------------------------------------------
    //
    // Implementation
    //
    //---------------------------------------------------------------------
  protected:
    // Used to calculate the Pt-value and Ut-value of a child Node. 
    //---------------------------------------------------------------------
    void calcPt_Ut(const Real t, Probability & Pt, Probability & Ut) const;

    // calcBirthDeathProbs()
    //******TO BEOVERLOADED IN INTEGRALDUPSPECPROBS*****
    //----------------------------------------------------------------------
    virtual void calcBirthDeathProbs(Node &sn);

    // calcBirthDeathProbs_recursive()
    // This function is called for nodes below the root.
    //----------------------------------------------------------------------
    virtual void calcBirthDeathProbs_recursive(Node &sn);

  protected:
    //------------------------------------------------------------------------
    //
    // Attributes
    //
    //------------------------------------------------------------------------
    Tree& S;			//!< Species tree
    const Real* topTime;

    Real birth_rate;		//!< \f$ \lambda \f$ in the papers
    Real death_rate;		//!< \f$ \mu \f$ in the papers
    Real db_diff;               //!< \f$ \lambda - \mu \f$

    //! For various nodes the values
    //! \f$ (P(t) * (1 - u_t)) / (1 - D * u_t)^2  = \tilde{P}(t) * (1-\tilde{u}_t)\f$
    //! are saved.
    ProbVector BD_const;
    ProbVector BD_var;		//!< Stores \f$ u_t / (1 - D * u_t) = \tilde{u}_t/(1 - D)\f$.
    ProbVector BD_zero;		//!< Stores extinction probabilities, i.e. probability of extinction below parent, \f$ 1 - \tilde{P}(t)(1-D) \f$.

    RealVector generalBirthRate;  //!< Stores for various nodes, the general birth rate \f$ \tilde\lambda \f$.
    RealVector generalDeathRate;  //!< Stores for various nodes, the general death rate \f$ \tilde\mu \f$.

    //------------------------------------------------------------------------
    //
    // NOT USED?
    //
    //------------------------------------------------------------------------

  public:
    //! For computing expected number of genes given one gene copy starting 
    //! at v in the species tree.
    //---------------------------------------------------------------------
    Probability expectedNumGenes() const;

    // Expected number of genes of one lineage during time t.
    //---------------------------------------------------------------------
    Probability expectedNumGenesLineage(Real t) const;

    //! Draws t = \tau - waiting time from Distribution function 
    //! F(t) = 1-((1-\tilde{u}_{\tau}) / (1-\tilde{u}_t))^n,
    //! where \tau = startTime, n = nLineages and F(t) = p
    //! Note! returns startTime - waiting time (a divergence time!)
    Real sampleWaitingTime(Node& x,  Real startTime,
			   Probability p);

  protected:
    // For computing expected number of genes
    //----------------------------------------------------------------------
    Probability expectedNumGenes(Node *v) const;




  };

}//end namespace beep

#endif
