% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/operators.R
\name{jacobian}
\alias{jacobian}
\alias{\%jacobian\%}
\title{Numerical and Symbolic Jacobian}
\usage{
jacobian(
  f,
  var,
  params = list(),
  coordinates = "cartesian",
  accuracy = 4,
  stepsize = NULL
)

f \%jacobian\% var
}
\arguments{
\item{f}{array of \code{characters} or a \code{function} returning a \code{numeric} array.}

\item{var}{vector giving the variable names with respect to which the derivatives are to be computed and/or the point where the derivatives are to be evaluated. See \code{\link{derivative}}.}

\item{params}{\code{list} of additional parameters passed to \code{f}.}

\item{coordinates}{coordinate system to use. One of: \code{cartesian}, \code{polar}, \code{spherical}, \code{cylindrical}, \code{parabolic}, \code{parabolic-cylindrical} or a vector of scale factors for each varibale.}

\item{accuracy}{degree of accuracy for numerical derivatives.}

\item{stepsize}{finite differences stepsize for numerical derivatives. It is based on the precision of the machine by default.}
}
\value{
\code{array}.
}
\description{
Computes the numerical Jacobian of \code{functions} or the symbolic Jacobian of \code{characters}
in arbitrary \href{https://en.wikipedia.org/wiki/Orthogonal_coordinates#Table_of_orthogonal_coordinates}{orthogonal coordinate systems}.
}
\details{
The function is basically a wrapper for \code{\link{gradient}} with \code{drop=FALSE}.
}
\section{Functions}{
\itemize{
\item \code{f \%jacobian\% var}: binary operator with default parameters.

}}
\examples{
### symbolic Jacobian 
jacobian("x*y*z", var = c("x", "y", "z"))

### numerical Jacobian in (x=1, y=2, z=3)
f <- function(x, y, z) x*y*z
jacobian(f = f, var = c(x=1, y=2, z=3))

### vectorized interface
f <- function(x) x[1]*x[2]*x[3]
jacobian(f = f, var = c(1, 2, 3))

### symbolic vector-valued functions
f <- c("y*sin(x)", "x*cos(y)")
jacobian(f = f, var = c("x","y"))

### numerical vector-valued functions
f <- function(x) c(sum(x), prod(x))
jacobian(f = f, var = c(0,0,0))

### binary operator
"x*y^2" \%jacobian\% c(x=1, y=3)

}
\references{
Guidotti E (2022). "calculus: High-Dimensional Numerical and Symbolic Calculus in R." Journal of Statistical Software, 104(5), 1-37. \doi{10.18637/jss.v104.i05}
}
\seealso{
Other differential operators: 
\code{\link{curl}()},
\code{\link{derivative}()},
\code{\link{divergence}()},
\code{\link{gradient}()},
\code{\link{hessian}()},
\code{\link{laplacian}()}
}
\concept{differential operators}
