% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knitr.R
\name{knit_ex}
\alias{knit_ex}
\alias{hook_try}
\alias{hook_backspace}
\alias{hook_toggle}
\title{Knitr Extensions}
\usage{
knit_ex(x, ..., quiet = TRUE, open = FALSE)

hook_try(before, options, envir)

hook_backspace()

hook_toggle()
}
\arguments{
\item{x}{text to knit as a character vector}

\item{...}{arguments passed to \code{\link[knitr]{knit2html}} or \code{\link[knitr]{knit}}}

\item{quiet}{logical that indicates if knitting should be quiet (no progress bars etc..).}

\item{open}{logical, only used when \code{x} is in .Rmd format, that indicates
if the generated document result should be open in a browse, instead of
being printed on screen.
Not that a browser will not open in non-interactive sessions, and the result will
be returned invisibly.}

\item{before}{logical that indicates when the hook is being called:
before or after the chunk is processed.}

\item{options}{list of current knitr chunk options}

\item{envir}{environment where the chunk is evaluated}
}
\value{
\code{knit_ex} returns the generated code, although invisibly when \code{open=TRUE}.

\itemize{
\item \code{hook_try} returns a function.
}

\itemize{
\item \code{hook_backspace} returns a function.
}

\itemize{
\item \code{hook_toggle}: returns a hook function.
}
}
\description{
\code{knit_ex} is a utility function for running small knitr examples,
e.g., to illustrate functionalities or issues.

\code{hook_backspace} is a chunk hook that enables the use of backspace
characters in the output (e.g., as used in progress bars), and still
obtain a final output as in the console.
}
\section{Functions}{
\itemize{
\item \code{hook_try()}: is a knitr hook to enable showing error
messages thrown by \code{\link{try}}.
The function is not meant to be called directly, but only registered
using \link[knitr:knit_hooks]{knitr::knit_hooks} (see details on this dedicated man page).

This simply defines a function \code{try} in \code{envir} that prints
the error message if any, and is called instead of base \code{\link{try}}.

\item \code{hook_toggle()}: a chunk hook that adds clickable elements to toggle \emph{indvidual}
code chunks in HTML documents generated from .Rmd files.

}}
\examples{

library(knitr)
knit_ex("1 + 1")


library(knitr)

# standard error message is caught
knit_ex("stop('ah ah')")

# with try the error is output on stderr but not caughted by knitr
knit_ex("try( stop('ah ah') )")

\donttest{
# no message caught
knit_ex("
^^^{r, include = FALSE}
knit_hooks$set(try = pkgmaker::hook_try)
^^^

^^^{r, try=TRUE}
try( stop('ah ah') )
^^^")
}


\donttest{
# Correctly formatting backspaces in chunk outputs
tmp <- tempfile(fileext = '.Rmd')
cat(file = tmp, "
^^^{r, include = FALSE}
library(knitr)
knit_hooks$set(backspace = pkgmaker::hook_backspace())
^^^
Default knitr does not handle backspace and adds a special character:
^^^{r}
cat('abc\bd')
^^^

Using the hook backspace solves the issue:
^^^{r, backspace=TRUE}
cat('abc\bd')
^^^
")

# knit
out <- knitr::knit2html(tmp, fragment.only = TRUE)

}

# look at output
\dontrun{
  browseURL(out)
  edit( file = out)
}

\donttest{    
# cleanup
out_files <- list.files(dirname(out), full.names = TRUE,
                         pattern = paste0("^", tools::file_path_sans_ext(out))) 
unlink(c(tmp, out_files))
}



\donttest{

knit_ex("

Declare chunk hook:
^^^{r, setup}
library(knitr)
knit_hooks$set(toggle = hook_toggle())
^^^

The R code of this chunk can be toggled on/off, and starts visible:
^^^{r, toggle=TRUE}
print(1:10)
^^^
The R code of this chunk can be toggled on/off, and starts hidden:
^^^{r, toggle=FALSE}
print(1:10)
^^^

This is a plain chunk that cannot be toggled on/off:
^^^{r}
print(1:10)
^^^

Now all chunks can be toggled and start visible:
^^^{r, toggle_all}
opts_chunk$set(toggle = TRUE)
^^^

^^^{r}
sample(5)
^^^

To disable the toggle link, one can pass anything except TRUE/FALSE:
^^^{r, toggle = NA}
sample(5)
^^^

", open = FALSE)

}

}
