% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/block.R
\name{block}
\alias{block}
\title{Break a vector into blocks}
\usage{
block(x, i, period, every = 1L, origin = NULL)
}
\arguments{
\item{x}{\verb{[vector]}

The vector to block.}

\item{i}{\verb{[Date / POSIXct / POSIXlt]}

The datetime index to block by.

There are 3 restrictions on the index:
\itemize{
\item The size of the index must match the size of \code{x}, they will not be
recycled to their common size.
\item The index must be an \emph{increasing} vector, but duplicate values
are allowed.
\item The index cannot have missing values.
}}

\item{period}{\verb{[character(1)]}

A string defining the period to group by. Valid inputs can be roughly
broken into:
\itemize{
\item \code{"year"}, \code{"quarter"}, \code{"month"}, \code{"week"}, \code{"day"}
\item \code{"hour"}, \code{"minute"}, \code{"second"}, \code{"millisecond"}
\item \code{"yweek"}, \code{"mweek"}
\item \code{"yday"}, \code{"mday"}
}}

\item{every}{\verb{[positive integer(1)]}

The number of periods to group together.

For example, if the period was set to \code{"year"} with an every value of \code{2},
then the years 1970 and 1971 would be placed in the same group.}

\item{origin}{\verb{[Date(1) / POSIXct(1) / POSIXlt(1) / NULL]}

The reference date time value. The default when left as \code{NULL} is the
epoch time of \verb{1970-01-01 00:00:00}, \emph{in the time zone of the index}.

This is generally used to define the anchor time to count from, which is
relevant when the every value is \verb{> 1}.}
}
\value{
A vector fulfilling the following invariants:
\itemize{
\item \code{vec_size(block(x)) == vec_size(unique(warp::warp_boundary(i)))}
\item \code{vec_ptype(block(x)) == list()}
\item \code{vec_ptype(block(x)[[1]]) == vec_ptype(x)}
}
}
\description{
\code{block()} breaks up the \code{i}-ndex by \code{period}, and then uses that to define
the indices to chop \code{x} with.

For example, it can split \code{x} into monthly or yearly blocks. Combined with
\code{purrr::map()}, it is a way to iterate over a vector in "time blocks".
}
\details{
\code{block()} determines the indices to block by with \code{\link[warp:warp_boundary]{warp::warp_boundary()}},
and splits \code{x} by those indices using \code{\link[vctrs:vec_chop]{vctrs::vec_chop()}}.

Like \code{\link[=slide]{slide()}}, \code{block()} splits data frame \code{x} values row wise.
}
\examples{
x <- 1:6
i <- as.Date("2019-01-01") + c(-2:2, 31)

block(i, i, period = "year")

# Data frames are split row wise
df <- data.frame(x = x, i = i)
block(df, i, period = "month")

# Iterate over these blocks to apply a function over
# non-overlapping period blocks. For example, to compute a
# mean over yearly or monthly blocks.
vapply(block(x, i, "year"), mean, numeric(1))
vapply(block(x, i, "month"), mean, numeric(1))

# block by every 2 months, ensuring that we start counting
# the 1st of the 2 months from `2019-01-01`
block(i, i, period = "month", every = 2, origin = as.Date("2019-01-01"))

# Use the `origin` to instead start counting from `2018-12-01`, meaning
# that [2018-12, 2019-01] gets bucketed together.
block(i, i, period = "month", every = 2, origin = as.Date("2018-12-01"))

}
\seealso{
\code{\link[=slide_period]{slide_period()}}, \code{\link[=slide]{slide()}}, \code{\link[=slide_index]{slide_index()}}
}
