//! Diffie-Helman over ECs in Weierstraß form.

use nettle_sys::{
    ecc_point, nettle_ecc_point_init, nettle_ecc_point_mul,
    nettle_ecc_point_mul_g,
};
use std::mem::zeroed;

use crate::{
    Error,
    Result,
    ecc::{Point, Scalar},
};

/// Derive ECDH shared secret.
///
/// Computes the shared secret for our private key `scalar` and the other parties public key `point`.
/// Results an error if `scalr` and `point` are not on the same curve.
pub fn point_mul(scalar: &Scalar, point: &Point) -> Result<Point> {
    if scalar.scalar.ecc != point.point.ecc {
        return Err(Error::InconsistentCurves.into());
    }

    unsafe {
        let mut ret: ecc_point = zeroed();

        nettle_ecc_point_init(&mut ret as *mut _, scalar.scalar.ecc);
        nettle_ecc_point_mul(&mut ret, &scalar.scalar, &point.point);

        Ok(Point { point: ret })
    }
}

/// Derive ECDH public key.
///
/// Computes the public key for a given secret `scalar`.
pub fn point_mul_g(scalar: &Scalar) -> Point {
    unsafe {
        let mut ret: ecc_point = zeroed();

        nettle_ecc_point_init(&mut ret as *mut _, scalar.scalar.ecc);
        nettle_ecc_point_mul_g(&mut ret, &scalar.scalar);

        Point { point: ret }
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::ecc::{Secp192r1, Secp224r1, Secp256r1, Secp384r1, Secp521r1};

    #[test]
    fn nist_p_192() {
        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x42\xea\x6d\xd9\x96\x9d\xd2\xa6\x1f\xea\x1a\xac\x7f\x8e\x98\xed\xcc\x89\x6c\x6e\x55\x85\x7c\xc0"[..],
                &b"\xdf\xbe\x5d\x7c\x61\xfa\xc8\x8b\x11\x81\x1b\xde\x32\x8e\x8a\x0d\x12\xbf\x01\xa9\xd2\x04\xb5\x23"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xf1\x7d\x3f\xea\x36\x7b\x74\xd3\x40\x85\x1c\xa4\x27\x0d\xcb\x24\xc2\x71\xf4\x45\xbe\xd9\xd5\x27"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(
                &b"\xb1\x50\x53\x40\x1f\x57\x28\x56\x37\xec\x32\x4c\x1c\xd2\x13\x9e\x3a\x67\xde\x37\x39\x23\x4b\x37"[..],
                &b"\xf2\x69\xc1\x58\x63\x74\x82\xaa\xd6\x44\xcd\x69\x2d\xd1\xd3\xef\x2c\x8a\x7c\x49\xe3\x89\xf7\xf6"[..]).unwrap();
            let z = &b"\x80\x3d\x8a\xb2\xe5\xb6\xe6\xfc\xa7\x15\x73\x7c\x3a\x82\xf7\xce\x3c\x78\x31\x24\xf6\xd5\x1c\xd0"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\xde\xb5\x71\x2f\xa0\x27\xac\x8d\x2f\x22\xc4\x55\xcc\xb7\x3a\x91\xe1\x7b\x65\x12\xb5\xe0\x30\xe7"[..],
                &b"\x7e\x26\x90\xa0\x2c\xc9\xb2\x87\x08\x43\x1a\x29\xfb\x54\xb8\x7b\x1f\x0c\x14\xe0\x11\xac\x21\x25"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x56\xe8\x53\x34\x9d\x96\xfe\x4c\x44\x24\x48\xda\xcb\x7c\xf9\x2b\xb7\xa9\x5d\xcf\x57\x4a\x9b\xd5"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xc0\x0d\x43\x57\x16\xff\xea\x53\xfd\x8c\x16\x27\x92\x41\x4c\x37\x66\x51\x87\xe5\x82\x71\x65\x39"[..],
                                                  &b"\xab\x71\x1c\x62\xaa\x71\xa5\xa1\x8e\x8a\x3c\x48\xf8\x9d\xc6\xfa\x52\xfa\xc0\x10\x8e\x52\xa8\xa0"[..]).unwrap();
            let z = &b"\xc2\x08\x84\x75\x68\xb9\x88\x35\xd7\x31\x2c\xef\x1f\x97\xf7\xaa\x29\x82\x83\x15\x23\x13\xc2\x9d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x4e\xda\xa8\xef\xc5\xa0\xf4\x0f\x84\x36\x63\xec\x58\x15\xe7\x76\x2d\xdd\xc0\x08\xe6\x63\xc2\x0f"[..],
                &b"\x0a\x9f\x8d\xc6\x7a\x3e\x60\xef\x6d\x64\xb5\x22\x18\x5d\x03\xdf\x1f\xc0\xad\xfd\x42\x47\x82\x79"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xc6\xef\x61\xfe\x12\xe8\x0b\xf5\x6f\x2d\x3f\x7d\x0b\xb7\x57\x39\x45\x19\x90\x6d\x55\x50\x09\x49"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xe1\x84\xbc\x18\x24\x82\xf3\x40\x3c\x87\x87\xb8\x38\x42\x47\x74\x67\xfc\xd0\x11\xdb\x0f\x6c\x64"[..],
                                                  &b"\xf9\xd1\xc1\x41\x42\xf4\x0d\xe8\x63\x9d\xb9\x7d\x51\xa6\x3d\x2c\xce\x10\x07\xcc\xf7\x73\xcd\xcb"[..]).unwrap();
            let z = &b"\x87\x22\x91\x07\x04\x7a\x3b\x61\x19\x20\xd6\xe3\xb2\xc0\xc8\x9b\xea\x4f\x49\x41\x22\x60\xb8\xdd"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x88\x87\xc2\x76\xed\xee\xd3\xe9\xe8\x66\xb4\x6d\x58\xd8\x95\xc7\x3f\xbd\x80\xb6\x3e\x38\x2e\x88"[..],
                &b"\x04\xc5\x09\x7b\xa6\x64\x5e\x16\x20\x6c\xfb\x70\xf7\x05\x26\x55\x94\x7d\xd4\x4a\x17\xf1\xf9\xd5"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xe6\x74\x7b\x9c\x23\xba\x70\x44\xf3\x8f\xf7\xe6\x2c\x35\xe4\x03\x89\x20\xf5\xa0\x16\x3d\x3c\xda"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x2b\x83\x8d\xbe\x73\x73\x5f\x37\xa3\x9a\x78\xd3\x19\x57\x83\xd2\x69\x91\xe8\x6f\xf4\xd9\x2d\x1a"[..],
                                                  &b"\x60\xd3\x44\x94\x22\x74\x48\x9f\x98\x90\x3b\x2e\x7f\x93\xf8\xd1\x97\xfc\x9a\xe6\x0a\x0e\xd5\x3a"[..]).unwrap();
            let z = &b"\xee\xc0\xbe\xd8\xfc\x55\xe1\xfe\xdd\xc8\x21\x58\xfd\x6d\xc0\xd4\x8a\x4d\x79\x6a\xaf\x47\xd4\x6c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x0d\x04\x5f\x30\x25\x4a\xdc\x1f\xce\xfa\x8a\x5b\x1f\x31\xbf\x4e\x73\x9d\xd3\x27\xcd\x18\xd5\x94"[..],
                &b"\x54\x2c\x31\x4e\x41\x42\x7c\x08\x27\x8a\x08\xce\x8d\x73\x05\xf3\xb5\xb8\x49\xc7\x2d\x8a\xff\x73"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xbe\xab\xed\xd0\x15\x4a\x1a\xfc\xfc\x85\xd5\x21\x81\xc1\x0f\x5e\xb4\x7a\xdc\x51\xf6\x55\x04\x7d"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x1f\x65\xcf\x6e\x89\x78\xe1\xc1\xbc\x10\xbb\x61\xa7\xdb\x31\x1d\xe3\x10\x08\x8c\x8c\xf9\x76\x8b"[..],
                                                  &b"\xf7\xd4\x38\x16\x8e\x7f\x42\xab\x14\xb1\x6a\xf5\x3a\x7a\x2f\x64\x6f\xf4\x0b\x53\xd7\x4c\xbc\xc7"[..]).unwrap();
            let z = &b"\x71\x6e\x74\x3b\x1b\x37\xa2\xcd\x84\x79\xf0\xa3\xd5\xa7\x4c\x10\xba\x25\x99\xbe\x18\xd7\xe2\xf4"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\xfb\x35\xca\x20\xd2\xe9\x66\x65\xc5\x1b\x98\xe8\xf6\xeb\x3d\x79\x11\x35\x08\xd8\xbc\xcd\x45\x16"[..],
                &b"\x36\x8e\xec\x0d\x5b\xfb\x84\x77\x21\xdf\x6a\xaf\xf0\xe5\xd4\x8c\x44\x4f\x74\xbf\x9c\xd8\xa5\xa7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xcf\x70\x35\x42\x26\x66\x73\x21\xd6\xe2\xba\xf4\x09\x99\xe2\xfd\x74\xc7\xa0\xf7\x93\xfa\x86\x99"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x5f\x48\x44\xff\xcc\xe6\x10\x05\xd2\x4f\x73\x7d\xb9\x86\x75\xe9\x2f\x7b\x65\x43\xae\xb6\x10\x6c"[..],
                                                  &b"\x54\x24\xf5\x98\x13\x92\x15\xd3\x89\xb6\xb1\x2b\x86\xd5\x80\x14\x85\x7f\x2d\xda\xdb\x54\x0f\x51"[..]).unwrap();
            let z = &b"\xf6\x70\x53\xb9\x34\x45\x99\x85\xa3\x15\xcb\x01\x7b\xf0\x30\x28\x91\x79\x8d\x45\xd0\xe1\x95\x08"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x82\x47\x52\x96\x0c\x13\x07\xe5\xf1\x3a\x83\xda\x21\xc7\x99\x8c\xa8\xb5\xb0\x0b\x95\x49\xf6\xd0"[..],
                &b"\xbc\x52\xd9\x1e\x23\x43\x63\xbc\x32\xee\x0b\x67\x78\xf2\x5c\xd8\xc1\x84\x75\x10\xf4\x34\x8b\x94"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xfe\x94\x25\x15\x23\x7f\xff\xdd\x7b\x4e\xb5\xc6\x49\x09\xee\xe4\x85\x6a\x07\x6c\xdf\x12\xba\xe2"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xe6\x36\x9d\xf7\x9b\x20\x7b\x8b\x86\x79\xf7\xc8\x69\xcf\xc2\x64\x85\x9d\x1a\xb5\x5a\xa4\x01\xe8"[..],
                                                  &b"\x1f\x99\xc7\x1f\x80\x1a\x30\xb5\x2f\x74\xda\x6e\x5e\x6d\xbb\x62\xee\x4c\x5d\xa1\x09\x0c\xc0\x20"[..]).unwrap();
            let z = &b"\x75\x82\x29\x71\x19\x3e\xdd\x47\x2b\xf3\x01\x51\xa7\x82\x61\x9c\x55\xad\x0b\x27\x9c\x93\x03\xdd"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x10\xbb\x57\x02\x02\x91\x14\x19\x81\xf8\x33\xb4\x74\x9e\x56\x11\x03\x4b\x30\x8e\x84\x01\x1d\x21"[..],
                &b"\xe1\xca\xcd\x6b\x7b\xd1\x7e\xd8\xdd\xb5\x0b\x6a\xee\x06\x54\xc3\x5f\x2d\x0e\xdd\xc1\xcf\xfc\xf6"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x33\xfe\xd1\x04\x92\xaf\xa5\xbe\xa0\x33\x3c\x0a\xf1\x2c\xac\x94\x0c\x4d\x22\x24\x55\xbc\xd0\xfe"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xef\x0b\x28\xaf\xc4\x16\x37\xd7\x37\xf4\x2e\x4c\x8a\xac\xea\xdc\x84\xba\x2e\x0b\x84\x9c\xa1\x8c"[..],
                                                  &b"\x57\x79\x79\x42\xe5\x52\x17\x3b\xba\x17\xf7\x32\x78\xe0\x29\xf4\x23\x35\x06\x8b\xd7\x70\xdd\xf2"[..]).unwrap();
            let z = &b"\x67\xcb\xa2\xcb\xb6\x9e\xe7\x8b\xf1\xab\xaf\xb0\xe6\xfb\xe3\x3f\xa2\x09\x4c\x12\x8d\x59\x65\x2d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x51\x92\xfc\xe4\x18\x5a\x77\x58\xea\x1b\xc5\x6e\x0e\x4f\x4e\x8b\x2d\xce\x32\x34\x8d\x0d\xce\xd1"[..],
                &b"\x20\x98\x99\x81\xbe\xaa\xf0\x00\x6d\x88\xa9\x6e\x79\x71\xa2\xfa\x3a\x33\xba\x46\x04\x7f\xc7\xba"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xf3\x55\x7c\x5d\x70\xb4\xc7\x95\x49\x60\xc3\x35\x68\x77\x6a\xdb\xe8\xe4\x36\x19\xab\xe2\x6b\x13"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xd7\x01\x12\xc5\xf0\xf0\x84\x43\x86\x49\x4a\xc1\xad\x99\xdc\xe2\x21\x41\x34\x17\x6e\xbf\xb9\xaf"[..],
                                                  &b"\xd3\xc1\x87\xa0\x38\x51\x0a\xb3\x1d\x45\x9e\x2b\x7a\xf1\xa3\x80\xdd\x75\x76\xaf\x06\x26\x75\x48"[..]).unwrap();
            let z = &b"\xcf\x99\xa2\x77\x0a\x38\x6c\xa0\x13\x7d\x1e\xca\x0a\x22\x6e\x48\x42\x97\xac\x3c\x51\x3f\x36\x31"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x26\xd0\x19\xdb\xe2\x79\xea\xd0\x1e\xed\x14\x3a\x91\x60\x1a\xda\x26\xe2\xf4\x22\x25\xb1\xc6\x2b"[..],
                &b"\x6c\xa6\x53\xf0\x82\x72\xe0\x38\x6f\xc9\x42\x1f\xbd\x58\x00\x93\xd7\xae\x63\x01\xbc\xa9\x44\x76"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x58\x6c\xfb\xa1\xc6\xe8\x17\x66\xed\x52\x82\x8f\x17\x7b\x1b\xe1\x4e\xbb\xc5\xb8\x33\x48\xc3\x11"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x58\xb3\xc6\x3e\x56\xbe\xc9\xd6\x96\xbf\x9a\x88\xdf\x28\x73\x73\x83\x91\xf7\x63\x68\xaa\x2b\x49"[..],
                                                  &b"\x57\x76\x77\x3b\x26\x1f\xaf\x7b\xa2\xfd\xc4\xfe\x43\xb9\x2c\x0b\x1c\x7a\x2f\xd0\x54\xa4\x36\x50"[..]).unwrap();
            let z = &b"\x57\x63\x31\xe2\xb4\xfb\x38\xa1\x12\x81\x0e\x15\x29\x83\x4d\xe8\x30\x7f\xb0\xa0\xd2\x75\x68\x77"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x53\x9b\xc4\x0f\xe2\x0a\x0f\xb2\x67\x88\x8b\x64\x7b\x03\xea\xaf\x6e\xc2\x0c\x02\xa1\xe1\xf8\xc8"[..],
                &b"\x69\x09\x5e\x5b\xb7\xb4\xd4\x4c\x32\x78\xa7\xee\x6b\xec\xa3\x97\xc4\x52\x46\xda\x9a\x34\xc8\xbe"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xca\xd8\x10\x06\x03\xa4\xf6\x5b\xe0\x8d\x8f\xc8\xa1\xb7\xe8\x84\xc5\xff\x65\xde\xb3\xc9\x6d\x99"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xb7\xfc\xc0\xf5\x2c\x7a\x41\x1e\xdb\xed\x39\xe1\x0b\xf0\x2b\x6a\xe0\xf2\x66\x14\xc6\xb3\x25\xa2"[..],
                                                  &b"\x47\x48\x3b\x26\xeb\x67\x77\x6d\xe2\xb9\x3a\xb7\x11\x9d\x54\x47\x57\x37\x39\xe3\xd5\x5e\x72\xfb"[..]).unwrap();
            let z = &b"\x90\x2f\x45\x01\x91\x6a\x0d\xd9\x45\x55\x4c\x3a\x37\xb3\xd7\x80\xd3\x75\xa6\xda\x71\x31\x97\xc4"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x5d\x34\x3d\xdb\x96\x31\x8f\xb4\x79\x4d\x10\xf6\xc5\x73\xf9\x9f\xee\x5d\x0d\x57\xb9\x96\x25\x0f"[..],
                &b"\x99\xfb\xdf\x9d\x97\xdd\x88\xad\x41\x02\x35\xda\xc3\x6e\x5b\x92\xce\x28\x24\xb8\xe5\x87\xa8\x2c"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x1e\xdd\x87\x9c\xc5\xc7\x96\x19\xca\xe6\xc7\x3a\x69\x1b\xd5\xa0\x39\x5c\x0e\xf3\xb3\x56\xfc\xd2"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x6c\xe6\xad\xb2\xc3\x08\x08\xf5\x90\x04\x8c\x33\xdf\xfa\xd4\x52\x4e\xbf\x7a\x5f\xd3\x9b\x74\x7b"[..],
                                                  &b"\x49\x66\xbd\x2f\x3d\x00\x56\x9b\x4d\x4c\x04\x09\xfb\xd7\xa2\xdb\x75\x2f\x6d\x09\xbc\xa8\xc2\x5f"[..]).unwrap();
            let z = &b"\x46\xe4\xde\x33\x50\x54\xd4\x29\x86\x32\x18\xae\x33\x63\x6f\xc9\xb8\x9c\x62\x8b\x64\xb5\x06\xc7"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x8d\x3d\xb9\xbd\xce\x13\x7f\xfb\xfb\x89\x13\x88\xc3\x7d\xf6\xc0\xcb\xc9\x0a\xa5\xe5\x37\x62\x20"[..],
                &b"\x13\x5d\x30\xb5\xcb\x66\x0e\xef\x87\x64\xff\xc7\x44\xf1\x5c\x1b\x5d\x6d\xc0\x6b\xa4\x41\x6d\x37"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x46\x0e\x45\x22\x73\xfe\x18\x27\x60\x21\x87\xad\x3b\xeb\xee\x65\xcb\x84\x42\x3b\xb4\xf4\x75\x37"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xd1\xbd\x3a\x3e\xfa\xbf\x47\x67\xfe\x63\x80\xbd\xf0\xdb\xf4\x9d\x52\xd4\xcf\x0c\xbb\x89\x40\x4c"[..],
                                                  &b"\xc1\x50\xc2\xb4\xc8\xb3\xaa\x35\xf7\x65\xf8\x47\xe4\xf7\xf8\xfd\x87\x04\xd2\x41\xa1\x81\xee\x99"[..]).unwrap();
            let z = &b"\x1b\xfe\x9e\x5a\x20\xac\x7a\x38\xd8\xf6\x05\xb4\x25\xbb\x90\x30\xbe\x31\xef\x97\xc1\x01\xc7\x6c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x9e\x0a\x69\x49\x51\x9c\x7f\x5b\xe6\x8c\x04\x33\xc5\xfd\xf1\x30\x64\xaa\x13\xfb\x29\x48\x3d\xc3"[..],
                &b"\xe1\xc8\xba\x63\xe1\xf4\x71\xdb\x23\x18\x5f\x50\xd9\xc8\x71\xed\xea\x21\x25\x5b\x3a\x63\xb4\xb7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xb9\x70\x36\x50\x08\x45\x6f\x87\x58\xec\xc5\xa3\xb3\x3c\xf3\xae\x6a\x8d\x56\x81\x07\xa5\x21\x67"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xc1\xb8\x61\x0c\x8c\x63\xf8\xd4\xab\xda\x09\x3b\x9a\x11\xa5\x66\x04\x4b\xf6\x5c\x6f\xaa\x89\x99"[..],
                                                  &b"\xa5\xbc\x4b\x3c\xa0\x95\x38\x2e\x97\x38\xae\xe9\x5f\xe9\x47\x9b\x17\x87\x9b\x3a\xd5\x29\x55\x59"[..]).unwrap();
            let z = &b"\x0e\x8c\x49\x3a\x4a\xdc\x44\x5d\xc9\x28\x8a\x3b\x9b\x27\x25\x99\x22\x40\x54\x59\x2d\x72\x65\xb3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\xbe\x08\x82\x38\x90\x2e\x99\x39\xb3\xd0\x54\xee\xeb\x84\x92\xda\xf4\xbd\xcf\x09\xa2\xab\x77\xf1"[..],
                &b"\x58\xd6\x74\x9a\x3a\x92\x3d\xc8\x04\x40\xf2\x66\x1f\xd3\x5b\x65\x16\x17\xe6\x52\x94\xb4\x63\x75"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x59\xc1\x5b\x8a\x24\x64\xe4\x1d\xfe\x43\x71\xc7\xf7\xda\xdf\x47\x0a\xe4\x25\x54\x4f\x81\x13\xbd"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x1f\xe7\x76\xf7\x35\x67\xb6\xac\x0b\x0d\x67\x64\x16\x4d\xe6\xc5\xbe\x75\x1b\xa8\xd1\xff\x45\x5e"[..],
                                                  &b"\x4c\x16\x0b\xf3\x8a\xfb\x2b\x71\xf6\x84\x26\x16\x64\x11\x5c\xe8\x74\x55\x3e\x8b\x05\x94\x32\xd2"[..]).unwrap();
            let z = &b"\x0f\x19\x91\x08\x6b\x45\x5d\xed\x6a\x1c\x41\x46\xf7\xbf\x59\xfe\x9b\x49\x5d\xe5\x66\xeb\xc6\xbf"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\xbf\x5a\xe0\x50\x25\xe1\xbe\x61\x7e\x66\x6d\x87\xa4\x16\x83\x63\x87\x3d\x57\x61\xb3\x76\xb5\x03"[..],
                &b"\xe1\xe6\xe3\x8b\x37\x2b\x6b\xee\x0f\xf5\xb3\x50\x2d\x83\x73\x5e\x3b\x2c\x26\x82\x5e\x4f\x0f\xcc"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xa6\xe9\xb8\x85\xc6\x6b\x95\x9d\x1f\xc2\x70\x8d\x59\x1b\x6d\x32\x28\xe4\x9e\xb9\x8f\x72\x6d\x61"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x63\x2b\xb7\x65\x1d\xbf\x49\xdd\xe9\xdd\x12\x5d\x13\xfb\x23\x4e\x06\x61\x77\x23\xbe\xed\x3d\x1b"[..],
                                                  &b"\xf4\xad\x52\x09\x63\x84\x88\x39\x7c\x5f\x44\xf9\x94\xdd\x74\x79\x80\x7e\x79\xf4\x88\x7d\x2e\x71"[..]).unwrap();
            let z = &b"\xb3\x0f\x21\x27\xc3\x4d\xf3\x5a\xaa\x91\xdb\xf0\xbb\xe1\x57\x98\xe7\x99\xa0\x3e\xd1\x16\x98\xc1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x6c\xc4\xfe\xed\x84\xc7\xab\x0d\x09\x00\x5d\x66\x0e\xd3\x4d\xe6\x95\x5a\x94\x61\xc4\x13\x8d\x11"[..],
                &b"\x31\x22\x5f\x33\x86\x4e\xd4\x8d\xa0\x6f\xa4\x5a\x91\x3b\x46\xcf\x42\x55\x77\x42\xe3\x50\x85\xe6"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xbd\xb7\x54\x09\x6f\xfb\xfb\xd8\xb0\xf3\xcb\x04\x6c\xcb\x7c\xa1\x49\xc4\xe7\x19\x20\x67\xa3\xee"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xd9\xc0\x98\xd4\x21\xd7\x41\xf6\xfa\xab\x11\x6f\x3e\x47\x31\xd2\x8c\x55\x58\xe1\x9f\xe1\x12\xa1"[..],
                                                  &b"\x38\xd4\xdc\x48\xcc\xdb\x1d\x3e\xd8\xd3\x1f\xd0\x67\x84\xa4\xf8\x7a\x68\xae\xc1\xcb\xd5\xb0\x8f"[..]).unwrap();
            let z = &b"\x64\xa5\xc2\x46\x59\x9d\x3e\x81\x77\xa2\x40\x2a\x11\x10\xeb\x81\xe6\xc4\x56\xab\x4e\xdb\x51\x27"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x36\x15\x73\x15\xbe\xe7\xaf\xed\xde\xd5\x8c\x4e\x8b\xa1\x4d\x34\x21\xc4\x01\xe5\x11\x35\xbc\xc9"[..],
                &b"\x37\xc2\x97\xca\x70\x3f\x77\xc5\x2b\xb0\x62\xd8\xce\x97\x1d\xb8\x40\x97\xba\x0c\x75\x3a\x41\x8f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xd5\xbc\xf2\x53\x4d\xaf\xc3\xd9\x99\x64\xc7\xbd\x63\xab\x7b\xd1\x59\x99\xfe\x56\xdd\x96\x9c\x42"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xfd\xa1\xd5\xd2\x8d\x6f\xe0\xe7\x90\x9d\x6a\x8b\xaf\xa7\x82\x4d\xb5\x57\x2a\xb9\x2f\xfe\x7d\xe6"[..],
                                                  &b"\x13\x4a\x29\x7c\x1d\x9c\x8b\xba\xb2\x49\xab\xac\xd9\x51\xed\x11\xe5\xa9\x9f\x92\xe7\x99\x15\x72"[..]).unwrap();
            let z = &b"\x01\x7b\x8c\xa5\x3c\x82\xfa\xb1\x63\xda\x2a\xb7\x83\x96\x6a\x39\xe0\x61\xb3\x2c\x8c\xfa\x33\x4d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x98\x46\x4d\x47\xf0\x25\x6f\x82\x92\xe0\x27\xe8\xc9\x25\x82\xea\x77\xcf\x90\x51\xf5\xce\x8e\x5d"[..],
                &b"\x44\x95\x52\xef\x75\x78\xbe\x96\x23\x6f\xe5\xed\x9d\x06\x43\xc0\xbb\x6c\x5a\x91\x34\xb0\x10\x8d"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x43\xd4\xb9\xdf\x10\x53\xbe\x5b\x42\x68\x10\x4c\x02\x24\x4d\x3b\xf9\x59\x4b\x01\x0b\x46\xa8\xb2"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xc3\x02\x0b\x70\x91\x46\x3d\x78\x8f\x1f\x1d\x76\xf7\xcf\xee\xc8\x2e\xcd\xb3\xb7\xd9\x9c\x34\x5c"[..],
                                                  &b"\x9a\x77\x10\xd5\x17\x95\x91\xd8\xf3\xdf\x0a\xa1\x22\x30\x17\x68\xae\x7d\xb7\xee\xe2\xd7\xf5\x83"[..]).unwrap();
            let z = &b"\x34\x0e\xf3\xdb\x3d\xbe\xbd\xd9\x1c\x62\xc3\xd4\xe1\xa3\xda\x2c\x7c\x52\xa3\x33\x8b\x86\x52\x59"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x56\x3e\xb6\x6c\x33\x4c\xf6\xf1\x23\xbf\x04\xc7\x80\x3b\x48\xa3\x11\x02\x14\x23\x7e\x98\x3b\xf5"[..],
                &b"\x0f\x35\x11\x04\x81\x91\x99\xef\x07\xc9\xa6\x05\x1d\x20\x75\x8f\x3a\xf7\x90\x27\xea\x66\xa5\x3f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x94\xca\xc2\xc2\xca\x71\x47\x46\x40\x16\x70\xd9\x4e\xdb\xf3\xf6\x77\x86\x7b\x5a\x03\xbe\xe7\xad"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xb1\x85\x54\xa2\xe7\x43\xef\x0a\xa2\xf0\x40\x98\x7c\x4c\x45\x10\x04\xe0\x96\xdf\x3d\x80\xdd\xae"[..],
                                                  &b"\x6e\x3e\x2c\x61\x8f\x89\x6e\x36\xba\x62\x00\x77\x68\x4b\x70\xa0\x5f\xfb\x79\xbf\x5e\x6c\x76\x40"[..]).unwrap();
            let z = &b"\x21\x62\x14\x49\x21\xdf\x51\x03\xd0\xe6\xa6\x50\xfb\x13\xfd\x24\x6f\x47\x38\xd0\x89\x6c\xe9\x2f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x86\x82\x8c\x4a\xc9\x2b\x55\x07\x61\x8a\xec\x78\x73\xa1\xd4\xfc\x65\x43\xc5\xbe\x33\xcf\x30\x78"[..],
                &b"\xb2\x2c\xa7\x24\x37\x54\x5e\x10\xd6\xd4\xf0\x52\x42\x2e\xb8\x98\xb7\x37\xa4\xb8\x54\x3e\xe5\x50"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x2a\x3a\x9e\x33\xc8\xcc\x31\x07\xa9\xf9\x26\x5c\x3b\xde\xa1\x20\x65\x70\xe8\x6f\x92\xac\x70\x14"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xa7\xba\x38\xbe\x1b\xc6\x69\xdd\x23\xcc\xfc\xee\x06\x45\xb1\xf0\xdb\x8c\xf9\x42\xde\xaf\xae\xb6"[..],
                                                  &b"\xb8\x2d\xb7\x9d\x80\xcd\x0e\x37\xf2\x8d\x41\x63\xad\xc3\x89\xde\xe8\xfc\x77\x97\xb5\xc9\x83\x1b"[..]).unwrap();
            let z = &b"\x4c\x69\xe7\xfe\xed\x4b\x11\x15\x9a\xdf\xc1\x6a\x60\x47\xa9\x25\x72\xea\x44\xe0\x74\x0b\x23\xaf"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x67\x00\xa1\x02\x43\x77\x81\xa9\x58\x1d\xa2\xbc\x25\xce\xd5\xab\xf4\x19\xda\x91\xd3\xc8\x03\xdf"[..],
                &b"\x71\x39\x6c\x9c\xf0\x8b\xcd\x91\x85\x4e\x3e\x6e\x42\xd8\xc6\x57\xce\x0f\x27\xab\x77\xa9\xdc\x4b"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x4a\x6b\x78\xa9\x8a\xc9\x8f\xa8\xe9\x9a\x8e\xce\x08\xec\x02\x51\x12\x5f\x85\xc6\xfd\x0e\x28\x9b"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xe7\x69\xdb\xbc\xd5\xce\x2d\x83\x51\x4b\x76\x8d\x3d\x2d\x5a\xa0\xbc\xd8\xf6\x6a\xf1\x5f\x55\x00"[..],
                                                  &b"\x2f\xc6\xd0\xb0\x39\xe0\xf2\x8f\x74\xfb\xef\xfe\x9e\x88\x3d\x4d\xd7\x22\x96\xe4\xe9\x5c\xae\x71"[..]).unwrap();
            let z = &b"\x46\x07\x2a\xce\xfd\x67\xbf\xf5\x0d\xe3\x55\xca\x7a\x31\xfa\x6b\xe5\x9f\x26\xe4\x67\x58\x72\x59"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\xa8\x2f\x35\x4c\xf9\x7b\xee\x5d\x22\xdc\x6c\x07\x9f\x29\x02\xea\xd4\x4d\x96\xa8\xf6\x14\xf1\x78"[..],
                &b"\xa6\x54\xa9\xaa\x8a\x1a\x08\x02\xf2\xce\x0e\xe8\xa0\xf4\xeb\xe9\x6d\xee\x1b\x37\x46\x4b\x1f\xf2"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\xc5\xa6\x49\x1d\x78\x84\x4d\x66\x17\xef\x33\xbe\x6b\x8b\xd5\x4d\xa2\x21\x45\x08\x85\xd5\x95\x0f"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xdb\x1b\x24\xf7\x46\x6b\xc1\x54\xe9\xd7\xd2\xc3\xca\x52\xdc\xfe\x0b\xfc\x95\x63\xc5\xfd\xb6\xf3"[..],
                                                  &b"\x1c\x74\xfb\xbf\x5b\xd9\x99\x21\xf1\xa9\xa7\x44\xf8\xe1\xcf\x77\x0b\xd6\xa7\x6a\x77\x2b\x30\x03"[..]).unwrap();
            let z = &b"\xec\x55\x80\xea\xbc\xa9\xf3\x38\x9d\x2b\x42\x7d\xdf\x6e\x49\xe2\x6d\x62\x9a\xfd\x03\xfa\x76\x6e"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x3c\xec\x21\xb2\x86\x68\xa1\x2a\x2c\xf7\x8e\x1a\x8e\x55\xd0\xef\xe0\x65\x15\x2f\xff\xc3\x47\x18"[..],
                &b"\x10\x29\x55\x7b\xeb\xa4\xff\x19\x92\xbd\x21\xc2\x3c\xb4\x82\x5f\x6d\xae\x70\xe3\x31\x8f\xd1\xca"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x2b\xa2\x70\x3c\x5e\x23\xf6\x46\x3c\x5b\x88\xdc\x37\x29\x2f\xab\xd3\x39\x9b\x5e\x1f\xb6\x7c\x05"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\x75\x43\x14\x89\x06\xce\xf9\xb3\x7a\x71\xa7\xc0\x83\x63\xcd\xd3\xbb\xa5\x01\x42\xd6\x52\x41\xaa"[..],
                                                  &b"\x8b\x3a\x69\x73\xde\x8d\xc2\x71\xe2\x7c\x1e\xad\x1e\x96\x2f\xda\xae\x37\x10\xc7\x24\xda\xac\x38"[..]).unwrap();
            let z = &b"\x7f\x39\x29\xdd\x3c\xbf\x76\x73\xbc\x30\xd8\x59\xd9\x0b\x88\x03\x07\x47\x5f\x80\x06\x60\xea\x32"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp192r1>(
                &b"\x70\x82\x64\x47\x15\xb8\xb7\x31\xf8\x22\x8b\x51\x18\xe7\x27\x0d\x34\xd1\x81\xf3\x61\xa2\x21\xfc"[..],
                &b"\x46\x46\x49\xd6\xc8\x8c\xa8\x96\x14\x48\x8a\x1c\xc7\xb8\x44\x2b\xb4\x2f\x9f\xb3\x02\x0a\x3d\x76"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp192r1>(&b"\x83\x61\x18\xc6\x24\x8f\x88\x2e\x91\x47\x97\x6f\x76\x48\x26\xc1\xa2\x87\x55\xa6\x10\x29\x77\xd5"[..]).unwrap();
            let iut_pub = Point::new::<Secp192r1>(&b"\xfc\xd3\x45\xa9\x76\xc7\x20\xca\xaa\x97\xde\x66\x97\x22\x68\x25\x61\x5e\x12\x87\xa9\xef\xf6\x7e"[..],
                                                  &b"\x58\xea\x42\xed\xbe\xea\xfc\xa9\xff\x44\xcf\xd7\xf2\x9a\xbd\x2c\xbd\xe7\x62\x6d\x79\xe4\x22\xc9"[..]).unwrap();
            let z = &b"\x72\xe8\x8f\x3e\xa6\x7d\x46\xd4\x6d\xbf\x83\x92\x6e\x7e\x2a\x6b\x85\xb5\x45\x36\x74\x1e\x6d\x2c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }
    }

    #[test]
    fn nist_p_224() {
        {
            let cav_pub = Point::new::<Secp224r1>(&b"\xaf\x33\xcd\x06\x29\xbc\x7e\x99\x63\x20\xa3\xf4\x03\x68\xf7\x4d\xe8\x70\x4f\xa3\x7b\x8f\xab\x69\xab\xaa\xe2\x80"[..],
                                                  &b"\x88\x20\x92\xcc\xbb\xa7\x93\x0f\x41\x9a\x8a\x4f\x9b\xb1\x69\x78\xbb\xc3\x83\x87\x29\x99\x25\x59\xa6\xf2\xe2\xd7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x83\x46\xa6\x0f\xc6\xf2\x93\xca\x5a\x0d\x2a\xf6\x8b\xa7\x1d\x1d\xd3\x89\xe5\xe4\x08\x37\x94\x2d\xf3\xe4\x3c\xbd"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x8d\xe2\xe2\x6a\xdf\x72\xc5\x82\xd6\x56\x8e\xf6\x38\xc4\xfd\x59\xb1\x8d\xa1\x71\xbd\xf5\x01\xf1\xd9\x29\xe0\x48"[..],
                                                &b"\x4a\x68\xa1\xc2\xb0\xfb\x22\x93\x0d\x12\x05\x55\xc1\xec\xe5\x0e\xa9\x8d\xea\x84\x07\xf7\x1b\xe3\x6e\xfa\xc0\xde"[..]).unwrap();
            let z = &b"\x7d\x96\xf9\xa3\xbd\x3c\x05\xcf\x5c\xc3\x7f\xeb\x8b\x9d\x52\x09\xd5\xc2\x59\x74\x64\xde\xc3\xe9\x98\x37\x43\xe8"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x13\xbf\xcd\x4f\x8e\x94\x42\x39\x3c\xab\x8f\xb4\x6b\x9f\x05\x66\xc2\x26\xb2\x2b\x37\x07\x69\x76\xf0\x61\x7a\x46"[..],
                                                  &b"\xee\xb2\x42\x75\x29\xb2\x88\xc6\x3c\x2f\x89\x63\xc1\xe4\x73\xdf\x2f\xca\x6c\xaa\x90\xd5\x2e\x2f\x8d\xb5\x6d\xd4"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x04\x3c\xb2\x16\xf4\xb7\x2c\xdf\x76\x29\xd6\x37\x20\xa5\x4a\xee\x0c\x99\xeb\x32\xd7\x44\x77\xda\xc0\xc2\xf7\x3d"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x2f\x90\xf5\xc8\xea\xc9\xc7\xde\xcd\xbb\x97\xb6\xc2\xf7\x15\xab\x72\x5e\x4f\xe4\x0f\xe6\xd7\x46\xef\xbf\x4e\x1b"[..],
                                                &b"\x66\x89\x73\x51\x45\x4f\x92\x7a\x30\x9b\x26\x9c\x5a\x6d\x31\x33\x8b\xe4\xc1\x9a\x5a\xcf\xc3\x2c\xf6\x56\xf4\x5c"[..]).unwrap();
            let z = &b"\xee\x93\xce\x06\xb8\x9f\xf7\x20\x09\xe8\x58\xc6\x8e\xb7\x08\xe7\xbc\x79\xee\x03\x00\xf7\x3b\xed\x69\xbb\xca\x09"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x75\x6d\xd8\x06\xb9\xd9\xc3\x4d\x89\x96\x91\xec\xb4\x5b\x77\x1a\xf4\x68\xec\x00\x44\x86\xa0\xfd\xd2\x83\x41\x1e"[..],
                                                  &b"\x4d\x02\xc2\xca\x61\x7b\xb2\xc5\xd9\x61\x3f\x25\xdd\x72\x41\x3d\x22\x9f\xd2\x90\x15\x13\xaa\x29\x50\x4e\xee\xfb"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x5a\xd0\xdd\x6d\xba\xbb\x4f\x3c\x2e\xa5\xfe\x32\xe5\x61\xb2\xca\x55\x08\x14\x86\xdf\x2c\x7c\x15\xc9\x62\x2b\x08"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x5b\xca\x45\xd7\x93\xe7\xfe\x99\xa8\x43\x70\x4e\xd8\x38\x31\x5a\xb1\x4a\x5f\x62\x77\x50\x7e\x9b\xc3\x75\x31"[..],
                                                &b"\x43\xe9\xd4\x21\xe1\x48\x6a\xe5\x89\x3b\xfd\x23\xc2\x10\xe5\xc1\x40\xd7\xc6\xb1\xad\xa5\x9d\x84\x2c\x9a\x98\xde"[..]).unwrap();
            let z = &b"\x3f\xcc\x01\xe3\x4d\x44\x49\xda\x2a\x97\x4b\x23\xfc\x36\xf9\x56\x67\x54\x25\x9d\x39\x14\x97\x90\xcf\xa1\xeb\xd3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x0f\x53\x7b\xf1\xc1\x12\x2c\x55\x65\x6d\x25\xe8\xaa\x84\x17\xe0\xb4\x4b\x15\x26\xae\x05\x23\x14\x4f\x99\x21\xc4"[..],
                                                  &b"\xf7\x9b\x26\xd3\x0e\x49\x1a\x77\x36\x96\xcc\x2c\x79\xb4\xf0\x59\x6b\xc5\xb9\xee\xba\xf3\x94\xd1\x62\xfb\x86\x84"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x0a\xa6\xff\x55\xa5\xd8\x20\xef\xcb\x4e\x7d\x10\xb8\x45\xea\x3c\x9f\x9b\xc5\xdf\xf8\x61\x06\xdb\x85\x31\x8e\x22"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x2f\x96\x75\x41\x31\xe0\x96\x81\x98\xaa\x78\xfb\xe8\xc2\x01\xdc\x5f\x35\x81\xc7\x92\xde\x48\x73\x40\xd3\x24\x48"[..],
                                                &b"\x61\xe8\xa5\xcd\x79\x61\x52\x03\xb6\xd8\x9e\x94\x96\xf9\xe2\x36\xfe\x3b\x6b\xe8\x73\x1e\x74\x3d\x61\x55\x19\xc6"[..]).unwrap();
            let z = &b"\x49\x12\x96\x28\xb2\x3a\xfc\xef\x48\x13\x9a\x3f\x6f\x59\xff\x5e\x98\x11\xaa\x74\x6a\xa4\xff\x33\xc2\x4b\xb9\x40"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x2b\x36\x31\xd2\xb0\x61\x79\xb3\x17\x4a\x10\x0f\x7f\x57\x13\x1e\xee\xa8\x94\x7b\xe0\x78\x6c\x3d\xc6\x4b\x22\x39"[..],
                                                  &b"\x83\xde\x29\xae\x3d\xad\x31\xad\xc0\x23\x6c\x6d\xe7\xf1\x45\x61\xca\x2e\xa0\x83\xc5\x27\x0c\x78\xa2\xe6\xcb\xc0"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xef\xe6\xe6\xe2\x5a\xff\xaf\x54\xc9\x8d\x00\x2a\xbb\xc6\x32\x8d\xa1\x59\x40\x5a\x1b\x75\x2e\x32\xdc\x23\x95\x0a"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x35\x5e\x96\x29\x20\xbd\xe0\x43\x69\x5f\x6b\xff\xb4\xb3\x55\xc6\x3d\xa6\xf5\xde\x66\x5e\xd4\x6f\x2e\xc8\x17\xe2"[..],
                                                &b"\x74\x8e\x09\x53\x68\xf6\x2e\x1d\x36\x4e\xdd\x46\x17\x19\x79\x3b\x40\x4a\xdb\xda\xac\xbc\xad\xd8\x89\x22\xff\x37"[..]).unwrap();
            let z = &b"\xfc\xdc\x69\xa4\x05\x01\xd3\x08\xa6\x83\x96\x53\xa8\xf0\x43\x09\xec\x00\x23\x39\x49\x52\x29\x02\xff\xa5\xea\xc6"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x45\x11\x40\x3d\xe2\x90\x59\xf6\x9a\x47\x5c\x5a\x6a\x5f\x6c\xab\xed\x5d\x9f\x01\x44\x36\xa8\xcb\x70\xa0\x23\x38"[..],
                                                  &b"\x7d\x2d\x1b\x62\xaa\x04\x6d\xf9\x34\x0f\x9c\x37\xa0\x87\xa0\x6b\x32\xcf\x7f\x08\xa2\x23\xf9\x92\x81\x2a\x82\x8b"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x61\xcb\x29\x32\x52\x40\x01\xe5\xe9\xee\xed\x6d\xf7\xd9\xc8\x93\x5e\xe3\x32\x20\x29\xed\xd7\xaa\x8a\xcb\xfd\x51"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xd5\x0e\x4a\xda\xbf\xd9\x89\xd7\xdb\xc7\xcf\x40\x52\x54\x6c\xc7\xc4\x47\xa9\x76\x30\x43\x69\x97\xad\x4b\x95\x36"[..],
                                                &b"\x5b\xea\x50\x34\x73\xc5\xea\xef\x95\x52\xd4\x2c\x40\xb1\xf2\xf7\xca\x29\x27\x33\xb2\x55\xb9\xbb\xe1\xb1\x23\x37"[..]).unwrap();
            let z = &b"\x82\x7e\x90\x25\xcb\x62\xe0\xe8\x37\xc5\x96\x06\x3f\x3b\x9b\x5a\x0f\x7a\xfd\x8d\x87\x83\x20\x00\x86\xd6\x1e\xc1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x31\x4a\x0b\x26\xdd\x31\xc2\x48\x84\x5d\x7c\xc1\x7b\x61\xca\xd4\x60\x82\x59\xbe\xd8\x5a\x58\xd1\xf1\xff\xd3\x78"[..],
                                                  &b"\x66\xe4\xb3\x50\x35\x2e\x11\x9e\xec\xad\xa3\x82\x90\x7f\x36\x19\xfd\x74\x8e\xa7\x3a\xe4\x89\x9d\xfd\x49\x63\x02"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x8c\x7a\xce\x34\x71\x71\xf9\x2d\xef\x98\xd8\x45\x47\x5f\xc8\x2e\x1d\x14\x96\xda\x81\xee\x58\xf5\x05\xb9\x85\xfa"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xb1\xa8\xdc\xac\x89\xac\xa2\x79\x93\x20\xb4\x51\xdf\x1c\x7f\xf4\xd9\x75\x67\xab\xb6\x81\x41\xc0\xd9\x5f\xc2\xaa"[..],
                                                &b"\x35\x24\x95\x09\x02\xb1\x51\x0b\xdc\x98\x7d\x86\x0a\xfc\x27\xad\x87\x1c\xea\xea\x66\x93\x5a\xbd\x3c\x0a\x99\xa8"[..]).unwrap();
            let z = &b"\x33\x5b\xa5\x12\x28\xd9\x4a\xcb\xed\x85\x1c\xa7\x82\x1c\x80\x1d\x5c\xb1\xc7\x97\x5d\x7a\xa9\x0a\x71\x59\xf8\xfa"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\xab\xe6\x84\x3b\xee\xc2\xfd\x9e\x5f\xb6\x47\x30\xd0\xbe\x4d\x16\x54\x38\xce\x92\x2e\xd7\x5d\xd8\x0b\x46\x03\xe5"[..],
                                                  &b"\x6a\xfe\x86\x73\xa9\x6c\x4b\xa9\x90\x0a\xd8\x59\x95\xe6\x31\xe4\x36\xc6\xcc\x88\xa2\xc2\xb4\x7b\x7c\x48\x86\xb8"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x38\x2f\xeb\x9b\x9b\xa1\x0f\x18\x9d\x99\xe7\x1a\x89\xcd\xfe\x44\xcb\x55\x4c\xec\x13\xa2\x12\x84\x09\x77\xfb\x68"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xab\xb6\xf1\xe3\x77\x3f\xf8\xfc\x73\xae\xa2\xa0\xb1\x07\x80\x9c\xe7\x0a\xdc\xef\xed\x6e\x41\xfc\x5c\xb4\x30\x45"[..],
                                                &b"\xa9\x63\x89\x7a\xe9\x06\xc1\x0a\x05\x5e\xea\xdb\x97\xff\xdd\x6f\x74\x8d\x3e\x56\x21\xe5\xff\xf3\x04\xe4\x8b\xa7"[..]).unwrap();
            let z = &b"\x8c\x2e\x62\x75\x94\x20\x6b\x34\xf7\x35\x6d\x34\x26\xeb\x3d\x79\xf5\x18\xef\x84\x3f\xbe\x94\x01\x4c\xce\xac\xe3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x13\xcf\x9d\x6d\x2c\x9a\xae\x82\x74\xc2\x7d\x44\x6a\xfd\x0c\x88\x8f\xfd\xd5\x2a\xe2\x99\xa3\x59\x84\xd4\xf5\x27"[..],
                                                  &b"\xdc\xbe\xe7\x5b\x51\x57\x51\xf8\xee\x2a\xe3\x55\xe8\xaf\xd5\xde\x21\xc6\x2a\x93\x9a\x65\x07\xb5\x38\xcb\xc4\xaf"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xe0\xd6\x20\x35\x10\x1e\xf4\x87\xc4\x85\xc6\x0f\xb4\x50\x0e\xeb\xe6\xa3\x2e\xc6\x4d\xbe\x97\xdb\xe0\x23\x2c\x46"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x88\x53\x77\x35\xe9\xb2\x3e\x3e\x0e\x07\x6f\x13\x5a\x82\xd3\x3f\x9b\xff\xb4\x65\xf3\xab\xce\x83\x22\xa6\x2a\x62"[..],
                                                &b"\xb4\xc8\xc1\x23\x67\x31\x97\x87\x5c\x0b\xd1\x4e\xd0\x97\x60\x6d\x33\x0f\xba\x2b\x92\x00\xef\x65\xa4\x47\x64\xd3"[..]).unwrap();
            let z = &b"\x63\x2a\xbb\x66\x27\x28\xdb\xc9\x94\x50\x88\x73\xd5\xc5\x27\xca\x5e\xf9\x23\xc0\xd3\x1f\xa6\xc4\x7e\xf4\xc8\x25"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x96\x5b\x63\x7c\x0d\xfb\xc0\xcf\x95\x40\x35\x68\x6d\x70\xf7\xec\x30\x92\x9e\x66\x4e\x52\x1d\xba\xa2\x28\x06\x59"[..],
                                                  &b"\x82\xa5\x8f\xf6\x1b\xc9\x00\x19\xbb\xcb\xb5\x87\x5d\x38\x63\xdb\x0b\xc2\xa1\xfa\x34\xb0\xad\x4d\xe1\xa8\x3f\x99"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xb9\x6a\xde\x5b\x73\xba\x72\xaa\x8b\x6e\x4d\x74\xd7\xbf\x9c\x58\xe9\x62\xff\x78\xeb\x54\x22\x87\xc7\xb4\x4b\xa2"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x37\x68\x29\x26\xa5\x4f\x70\xa4\xc1\x74\x8f\x54\xd5\x0d\x5b\x00\x13\x8a\x05\x5f\x92\x4f\x2c\x65\xe5\xb0\xbb\xe4"[..],
                                                &b"\x59\x6a\xfe\xfc\xdd\x64\x0d\x29\x63\x50\x15\xb8\x9b\xdd\xdd\x1f\x8c\x27\x23\x68\x6d\x33\x2e\x7a\x06\xca\x87\x99"[..]).unwrap();
            let z = &b"\x34\x64\x11\x41\xaa\xb0\x5e\xf5\x8b\xd3\x76\xd6\x09\x34\x59\x01\xfb\x8f\x63\x47\x7c\x6b\xe9\x09\x7f\x03\x7f\x1f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x73\xcc\x64\x53\x72\xca\x2e\x71\x63\x7c\xda\x94\x3d\x81\x48\xf3\x38\x2a\xb6\xdd\x0f\x2e\x1a\x49\xda\x94\xe1\x34"[..],
                                                  &b"\xdf\x5c\x35\x5c\x23\xe6\xe2\x32\xeb\xc3\xbe\xe2\xab\x18\x73\xee\x0d\x83\xe3\x38\x2f\x8e\x6f\xe6\x13\xf6\x34\x3c"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xa4\x0d\x7e\x12\x04\x9c\x71\xe6\x52\x2c\x7f\xf2\x38\x42\x24\x06\x1c\x3a\x45\x70\x58\xb3\x10\x55\x76\x55\xb8\x54"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x39\x98\x01\x24\x3b\xfe\x0c\x2d\xa9\xb0\xa5\x3c\x8c\xa5\x7f\x2e\xee\x87\xaa\xa9\x4a\x8e\x4d\x5e\x02\x9f\x42\xca"[..],
                                                &b"\xaa\x49\xe6\xd4\xb4\x7c\xee\x7a\x5c\x4a\xb7\x1d\x5a\x67\xda\x84\xe0\xb9\xb4\x25\xce\x3e\x70\xda\x68\xc8\x89\xe7"[..]).unwrap();
            let z = &b"\x4f\x74\xac\x85\x07\x50\x1a\x32\xbf\xc5\xa7\x8d\x82\x71\xc2\x00\xe8\x35\x96\x6e\x18\x7e\x8d\x00\x01\x1a\x8c\x75"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x54\x65\x78\x21\x62\x50\x35\x4e\x44\x9e\x21\x54\x6d\xd1\x1c\xd1\xc5\x17\x42\x36\x73\x9a\xca\xd9\xce\x0f\x45\x12"[..],
                                                  &b"\xd2\xa2\x2f\xcd\x66\xd1\xab\xed\xc7\x67\x66\x83\x27\xc5\xcb\x9c\x59\x90\x43\x27\x62\x39\xcf\x3c\x85\x16\xaf\x24"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xad\x25\x19\xbc\x72\x4d\x48\x4e\x02\xa6\x9f\x05\x14\x9b\xb0\x47\x71\x4b\xf0\xf5\x98\x6f\xac\x2e\x22\x2c\xd9\x46"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xdf\x9c\x1e\x0e\xf1\x5e\x53\xb9\xf6\x26\xe2\xbe\x1c\xbe\x89\x36\x39\xc0\x6f\x3e\x04\x39\xee\x95\xd7\xd4\xb1\xe3"[..],
                                                &b"\x7a\x52\xa7\x38\x6a\xdd\xa2\x43\xef\xdf\x89\x41\x08\x5c\x84\xe3\x12\x39\xca\xb9\x2b\x80\x17\x33\x67\x48\x96\x5e"[..]).unwrap();
            let z = &b"\xad\x09\xc9\xae\x4d\x23\x24\xea\x81\xbb\x55\x5b\x20\x0d\x3c\x00\x3e\x22\xa6\x87\x0e\xe0\x3b\x52\xdf\x49\xe4\xde"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x1d\x46\xb1\xdc\x3a\x28\x12\x3c\xb5\x13\x46\xe6\x7b\xae\xc5\x64\x04\x86\x86\x78\xfa\xf7\xd0\xe8\xb2\xaf\xa2\x2a"[..],
                                                  &b"\x0e\xc9\xe6\x5e\xc9\x7e\x21\x83\x73\xe7\xfc\x11\x5c\x22\x74\xd5\xb8\x29\xa6\x0d\x93\xf7\x1e\x01\xd5\x81\x36\xc3"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x3d\x31\x2a\x9b\x9d\x8e\xd0\x91\x40\x90\x0b\xba\xc1\xe0\x95\x52\x7e\xbc\x9e\x3c\x64\x93\xbc\xf3\x66\x6e\x3a\x29"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xb4\xa0\x19\x8d\xc8\x81\x0e\x88\x44\x25\xb7\x50\x92\x8b\x0c\x96\x0c\x31\xf7\xa9\x96\x63\x40\x0b\x01\xa1\x79\xdf"[..],
                                                &b"\x81\x2b\x60\x1b\xfc\x07\x38\x24\x2c\x6f\x86\xf8\x30\xf2\x7a\xcd\x63\x2c\xa6\x18\xa0\xb5\x28\x0c\x9d\x57\x69\xf7"[..]).unwrap();
            let z = &b"\xef\x02\x9c\x28\xc6\x80\x64\xb8\xab\xd2\x96\x5a\x38\xc4\x04\xfb\x5e\x94\x4a\xce\x57\xe8\x63\x8d\xab\xa9\xd3\xcd"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x26\x6d\x03\x8c\xc7\xa4\xfe\x21\xf6\xc9\x76\x31\x8e\x82\x7b\x82\xbb\x5b\x8f\x74\x43\xa5\x52\x98\x13\x65\x06\xe0"[..],
                                                  &b"\xdf\x12\x3d\x98\xa7\xa2\x0b\xbd\xf3\x94\x3d\xf2\xe3\x56\x34\x22\xf8\xc0\xcf\x74\xd5\x3a\xaa\xbd\xd7\xc9\x73\xba"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x8c\xe0\x82\x2d\xc2\x4c\x15\x39\x95\x75\x5a\xc3\x50\x73\x7e\xf5\x06\x64\x1c\x7d\x75\x2b\x4f\x93\x00\xc6\x12\xed"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xdf\xc7\xec\x13\x76\x90\xcd\x6d\x12\xfd\xb2\xfd\x0b\x8c\x53\x14\x58\x21\x08\x76\x9c\x2b\x72\x2f\xfb\x39\x58"[..],
                                                &b"\x5e\xef\x3d\xa4\xba\x45\x81\x27\x34\x6b\xb6\x40\x23\x86\x8b\xdd\xb7\x55\x8a\x2e\xcf\xc8\x13\x64\x5f\x4c\xe9\xfe"[..]).unwrap();
            let z = &b"\xf8\x3c\x16\x66\x1d\xfc\xba\xd0\x21\xcc\x3b\x5a\x5a\xf5\x1d\x9a\x18\xdb\x46\x53\x86\x6b\x3f\xf9\x07\x87\xce\x3e"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\xeb\x0a\x09\xf7\xa1\xc2\x36\xa6\x1f\x59\x58\x09\xec\x56\x70\xef\xd9\x2e\x45\x98\xd5\xe6\x13\xe0\x92\xcd\xfd\xca"[..],
                                                  &b"\x50\x78\x7a\xe2\xf2\xf1\x5b\x88\xbc\x10\xf7\xb5\xf0\xae\xe1\x41\x83\x73\xf1\x61\x53\xae\xbd\x1f\xba\x54\x28\x8d"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x0f\xf9\xb4\x85\x32\x5a\xb7\x7f\x29\xe7\xbc\x37\x9f\xed\x74\xbf\xac\x85\x94\x82\xda\x0d\xee\x75\x28\xc1\x9d\xb2"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x7e\x60\x3e\x69\x76\xdb\x83\xc3\x60\x11\x50\x8f\xa6\x95\xd1\xb5\x15\x24\x9e\x2e\x54\xb4\x8f\xcb\xcf\xb9\x02\x47"[..],
                                                &b"\x01\x79\xa6\x00\xce\x86\xad\xfc\xa9\xb1\xb9\x31\xfa\x51\x73\xd6\x18\xda\x09\xe8\x41\x80\x3d\x19\xb0\x26\x42\x86"[..]).unwrap();
            let z = &b"\xf5\x12\x58\xc6\x3f\x23\x2e\x55\xa6\x6a\xa2\x5e\xbd\x59\x7b\x20\x18\xd1\x05\x2c\x02\xee\xb6\x38\x66\x75\x80\x05"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x6b\x2f\x6b\x18\xa5\x87\xf5\x62\xff\xc6\x1b\xd9\xb0\x04\x73\x22\x28\x69\x86\xa7\x8f\x1f\xd1\x39\xb8\x4f\x7c\x24"[..],
                                                  &b"\x70\x96\x90\x8e\x46\x15\x26\x6b\xe5\x9a\x53\xcd\x65\x55\x15\x05\x6f\xf9\x23\x70\xa6\x27\x1a\x5d\x38\x23\xd7\x04"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x19\xcf\x5f\xf6\x30\x64\x67\xf2\x8b\x9f\xe0\x67\x5a\x43\xc0\x58\x25\x52\xc8\xc1\x2e\x59\xce\x7c\x38\xf2\x92\xb1"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xfc\x20\xe9\x06\xe6\x09\xc1\x12\xcf\xc2\xe0\xfe\xa6\x30\x38\x82\xc5\xdb\x94\xe8\x7e\x02\x23\x73\xab\x2c\x08\x2a"[..],
                                                &b"\xae\xcd\xf1\xda\xa7\x17\x82\xbc\x5a\x26\xbb\xbd\x8d\x7e\x8a\x76\x49\x0e\x26\xab\xc1\x7d\xff\xc7\x74\xbd\x73\x41"[..]).unwrap();
            let z = &b"\x7f\xdc\x96\x9a\x18\x6f\xf1\x84\x29\xf2\xa2\x76\xda\xc4\x3b\xee\xa2\x11\x82\xd8\x2c\xe2\xe5\xa0\x87\x65\x52\xb1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x32\x81\x01\xba\x82\x6a\xcd\x75\xff\x9f\x34\xd5\x57\x4c\xe0\xdb\xc9\x2f\x70\x9b\xad\x8d\x7a\x33\xc4\x79\x40\xc1"[..],
                                                  &b"\xdf\x39\xf1\xea\x88\x48\x8c\x55\xd5\x53\x81\x60\x87\x8b\x9c\xed\x18\xa8\x87\xea\x26\x1d\xd7\x12\xd1\x40\x24\xff"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x90\xa1\x53\x68\xe3\x53\x2c\x0b\x1e\x51\xe5\x5d\x13\x94\x47\xc2\xc8\x9b\xc1\x60\x71\x9d\x69\x72\x91\xea\x7c\x14"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xc6\x83\x7d\x50\x6e\x97\x6d\xa7\xdb\x3a\xd1\x26\x7c\x35\x9d\xff\x2e\xa6\xfb\x0b\x7f\x7f\x8e\x77\x02\x4c\x59\xe9"[..],
                                                &b"\x67\xeb\x49\x1d\x2f\xc8\xa5\x30\xc4\x65\x25\xd2\xa8\xb2\xd7\xc1\xdf\x5f\xba\x1a\xe7\x40\xa4\x64\x9c\x68\x3e\xe6"[..]).unwrap();
            let z = &b"\x3d\x60\xab\x6d\xb2\xb3\xff\xe2\xd2\x9c\xcf\xf4\x6d\x05\x6e\x54\x23\x0c\xf3\x49\x82\xe2\x41\x55\x6e\xd2\x92\x0c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x81\xe3\x42\x70\x87\x1e\x2e\xbb\xd9\x41\x83\xf6\x17\xb4\xae\x15\xf0\x41\x6d\xd6\x34\xfe\x6e\x93\x4c\xf3\xc0"[..],
                                                  &b"\x3a\x1e\x9f\x38\xa7\xb9\x0b\x73\x17\xd2\x6b\x9f\x63\x11\x06\x3a\xb5\x8b\x26\x8c\xf4\x89\xb2\xe5\x03\x86\xd5\xd6"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x8e\x08\x38\xe0\x5e\x17\x21\x49\x10\x67\xe1\xca\xbc\x2e\x80\x51\xb2\x90\xe2\x61\x6e\xec\x42\x7b\x71\x21\x89\x7d"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xe9\x15\x0f\x77\x00\x75\x62\x60\x19\xe1\x8f\x95\x47\x3b\x71\xe6\x82\x80\x41\x79\x1d\x3f\x08\xd3\xfa\xee\xaa\x2b"[..],
                                                &b"\x47\x5f\x70\x73\x5e\xaa\xe5\x23\x08\xa3\xb7\x63\xdc\x88\xef\xe1\x8a\xb5\x90\xeb\xaf\xa0\x35\xf6\xe0\x8b\x00\x1c"[..]).unwrap();
            let z = &b"\x91\x16\xd7\x27\x86\xf4\xdb\x5d\xf7\xa8\xb4\x30\x78\xc6\xab\x91\x60\xd4\x23\x51\x3d\x35\xea\x5e\x25\x59\x30\x6d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x26\x23\x63\x2f\xdf\x0b\xd8\x56\x80\x5a\x69\xaa\x18\x6d\x41\x33\xef\x59\x04\xe1\xf6\x55\xa9\x72\xd6\x6c\xce\x07"[..],
                                                  &b"\x2c\xef\x97\x28\xdd\x06\xfb\x8b\x50\x15\x0f\x52\x9b\x69\x50\x76\xd4\x50\x79\x83\x91\x25\x85\xc8\x9b\xd0\x68\x2e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x38\x10\x6e\x93\xf1\x6a\x38\x1a\xdb\x1d\x72\xce\xe3\xda\x66\xae\x46\x2a\xd4\xbb\xfe\xa9\xec\xdf\x35\xd0\x81\x4e"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x7b\xe6\xc4\xc9\x17\x82\x9a\xb6\x57\xdd\x79\xe8\x63\x7d\x7a\xef\xd2\xf8\x1f\x0d\xe7\x65\x4d\x95\x7e\x97\x65\x8d"[..],
                                                &b"\x43\x0d\x22\xd9\xe8\x43\x83\x10\xf6\x1e\x0d\x43\xf2\x5f\xa3\xe3\x45\x85\xf4\x32\xba\xad\x27\xdb\x30\x21\xbf\x0d"[..]).unwrap();
            let z = &b"\x20\x7c\x53\xdc\xef\xac\x78\x9a\xaa\x02\x76\xd9\x20\x0b\x3a\x94\x0c\xe5\xf2\x29\x6f\x4c\xb2\xe8\x1a\x18\x5d\x3d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x8e\xe4\xd1\xdc\xc3\x1d\xee\x4b\xf6\xfe\x21\xca\x8a\x58\x77\x21\xd9\x10\xac\xfb\x12\x2c\x16\xc2\xa7\x7a\x81\x52"[..],
                                                  &b"\x4e\xbf\x32\x3f\xff\x04\xeb\x47\x70\x69\xa0\xac\x68\xb3\x45\xf6\xb1\xae\x13\x4e\xfc\x31\x94\x0e\x51\x3c\xb9\x9f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xe5\xd1\x71\x84\x31\xcf\x50\xf6\xcb\xd1\xbc\x80\x19\xfa\x16\x76\x2d\xfa\x12\xc9\x89\xe5\x99\x99\x77\xfb\x4e\xa2"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x2e\xa4\x96\x6e\x7f\x92\xed\x7f\x5c\xc6\x1f\xde\x79\x20\x45\xf6\x3b\x73\x1d\x6e\x7d\x0d\xe2\x57\x7f\x2d\x8e\xce"[..],
                                                &b"\x1c\x4a\x7b\x1e\xde\x6f\x83\x91\x62\x29\x2d\xf4\x24\xbe\x78\xe8\x17\x6f\xb6\xf9\x42\xa3\xc0\x23\x91\x70\x0f\x31"[..]).unwrap();
            let z = &b"\x10\xe4\x67\xda\x34\xf4\x8a\xd7\x07\x20\x05\xbc\xcd\x6d\xa1\xb2\xba\x3f\x71\xea\xfa\x1c\x39\x38\x42\xf9\x1d\x74"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x97\xdc\xbe\x6d\x28\x33\x58\x82\xa6\xd1\x93\xcc\x54\xa1\x06\x3d\xd0\x77\x5d\xc3\x28\x56\x53\x00\xbb\x99\xe6\x91"[..],
                                                  &b"\xda\xd1\x1d\xd5\xec\xe8\xcf\xd9\xf9\x7c\x9a\x52\x6e\x4a\x15\x06\xe6\x35\x59\x69\xee\x87\x82\x6f\xc3\x8b\xcd\x24"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x3d\x63\x56\x91\xb6\x2a\x9a\x92\x7c\x63\x39\x51\xc9\x36\x9c\x88\x62\xbd\x21\x19\xd3\x09\x70\xc2\x64\x47\x27\xd6"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x43\x8b\xbb\x98\x05\x17\xaf\xb2\x0b\xe1\xd6\x74\xe3\xac\x2b\x31\xce\xf0\x7a\x9b\x23\xfb\x8f\x6e\x38\xe0\xd6\xc0"[..],
                                                &b"\x0b\xe5\xf1\xc4\x7d\x58\xd2\x1b\x6e\xd2\x84\x23\xb3\x2f\x5a\x94\x75\x0d\xa4\x7e\xdc\xef\x33\xea\x79\x94\x2a\xfd"[..]).unwrap();
            let z = &b"\x82\xfd\x2f\x9c\x60\xc4\xf9\x99\xac\x00\xbb\xe6\x4b\xfc\x11\xda\x8f\xf8\xcd\xa2\xe4\x99\xfc\xed\x65\x23\x0b\xb1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\xce\x91\x26\xdd\x53\x97\x2d\xea\x1d\xe1\xd1\x1e\xfe\xf9\x00\xde\x34\xb6\x61\x85\x9c\x46\x48\xc5\xc0\xe5\x34\xf7"[..],
                                                  &b"\xe1\x13\xb6\xf2\xc1\x65\x9d\x07\xf2\x71\x6e\x64\xa8\x3c\x18\xbb\xce\x34\x4d\xd2\x12\x1f\xe8\x51\x68\xea\xe0\x85"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xac\xf3\xc8\x5b\xbd\xc3\x79\xf0\x2f\x5e\xa3\x6e\x7f\x0f\x53\x09\x5a\x9e\x70\x46\xa2\x86\x85\xa8\x65\x9b\xf7\x98"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xff\x75\x11\x21\x5c\x71\xd7\x96\xbd\x64\x6e\x84\x74\xbe\x44\x16\xb9\x16\x84\xce\x0d\x26\x9e\xf6\xf4\x22\x01\x3b"[..],
                                                &b"\xb7\xbf\x5e\x79\xb5\xa9\x39\x3b\xb9\xea\x42\xc0\xbd\xb2\xd3\xc2\xdc\x80\x6e\x1a\x73\x06\xaa\x58\xe4\xfd\xbe\xa5"[..]).unwrap();
            let z = &b"\x53\x0f\x7e\x7f\xc9\x32\x61\x3b\x29\xc9\x81\xf2\x61\xcb\x03\x6c\xba\x3f\x1d\xf3\x86\x4e\x0e\x1c\xba\x26\x85\xa2"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x84\x41\x99\x67\xd6\xcf\xad\x41\xe7\x5a\x02\xb6\xda\x60\x5a\x97\x94\x9a\x18\x3a\x97\xc3\x06\xc4\xb4\x6e\x66\xa5"[..],
                                                  &b"\x5c\xc9\xb2\x59\x71\x8b\x1b\xc8\xb1\x44\xfd\xe6\x33\xa8\x94\x61\x6f\xfd\x59\xa3\xa6\xd5\xd8\xe9\x42\xc7\xcb\xb7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xcf\xfd\x62\xcb\x00\xa0\xe3\x16\x3f\xbf\x2c\x39\x7f\xad\xc9\x61\x82\x10\xf8\x6b\x4f\x54\xa6\x75\x28\x73\x05\xf0"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x04\xbf\x4d\x94\x8f\x44\x30\xd1\x8b\x4e\xd6\xc9\x6d\xba\xf9\x81\xfa\x11\xa4\x03\xed\x16\x88\x7f\x06\x75\x49\x81"[..],
                                                &b"\x7c\x13\x26\xa9\xce\xf5\x1f\x79\xd4\xe7\x83\x03\xd6\x06\x4b\x45\x9f\x61\x25\x84\xac\x2f\xdf\x59\x3d\x7d\x5d\x84"[..]).unwrap();
            let z = &b"\x49\xf6\xfd\x01\x39\x24\x8e\xf4\xdf\x2d\xb0\x5d\x13\x19\xbd\x5b\x14\x89\xe2\x49\x82\x7a\x45\xa8\xa5\xf1\x24\x27"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x7c\x9c\xac\x35\x76\x80\x63\xc2\x82\x7f\x60\xa7\xf5\x13\x88\xf2\xa8\xf4\xb7\xf8\xcd\x73\x6b\xd6\xbc\x33\x74\x77"[..],
                                                  &b"\x29\xee\x6b\x84\x9c\x60\x25\xd5\x77\xdb\xcc\x55\xfb\xd1\x70\x18\xf4\xed\xbc\x2e\xf1\x05\xb0\x04\xd6\x25\x7b\xcd"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\x85\xf9\x03\xe4\x39\x43\xd1\x3c\x68\x93\x2e\x71\x0e\x80\xde\x52\xcb\xc0\xb8\xf1\xa1\x41\x8e\xa4\xda\x07\x92\x99"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\x97\x0a\x4a\x7e\x01\xd4\x18\x84\x97\xce\xb4\x69\x55\xeb\x1b\x84\x2d\x90\x85\x81\x9a\x9b\x92\x5c\x84\x52\x9d\x3d"[..],
                                                &b"\xdf\xa2\x52\x64\x80\xf8\x33\xea\x0e\xdb\xd2\x04\xe4\xe3\x65\xfe\xf3\x47\x28\x88\xfe\x7d\x96\x91\xc3\xeb\xc0\x9f"[..]).unwrap();
            let z = &b"\x8f\x7e\x34\xe5\x97\xae\x80\x93\xb9\x82\x70\xa7\x4a\x8d\xfc\xdb\xed\x45\x7f\x42\xf4\x3d\xf4\x87\xc5\x48\x71\x61"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp224r1>(&b"\x08\x5a\x76\x42\xad\x8e\x59\xb1\xa3\xe8\x72\x6a\x75\x47\xaf\xbe\xcf\xfd\xac\x1d\xab\x7e\x57\x23\x0c\x6a\x9d\xf4"[..],
                                                  &b"\xf9\x1c\x36\xd8\x81\xfe\x9b\x80\x47\xa3\x53\x07\x13\x55\x4a\x1a\xf4\xc2\x5c\x5a\x8e\x65\x4d\xcd\xcf\x68\x9f\x2e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp224r1>(&b"\xcc\xe6\x48\x91\xa3\xd0\x12\x9f\xee\x0d\x4a\x96\xcf\xbe\x7a\xc4\x70\xb8\x5e\x96\x75\x29\x05\x7c\xfa\x31\xa1\xd9"[..]).unwrap();
            let iut_pub = Point::new::<Secp224r1>(&b"\xa6\xb2\x96\x32\xdb\x94\xda\x21\x25\xdc\x1c\xf8\x0e\x03\x70\x26\x87\xb2\xac\xc1\x12\x20\x22\xfa\x21\x74\x76\x5a"[..],
                                                &b"\x61\x72\x3e\xdd\x73\xe1\x0d\xae\xd7\x37\x75\x27\x8f\x19\x58\xba\x56\xf1\xfc\x9d\x08\x5e\xbc\x2b\x64\xc8\x4f\xe5"[..]).unwrap();
            let z = &b"\x71\x95\x4e\x22\x61\xe8\x51\x0b\xe1\xa0\x60\x73\x36\x71\xd2\xe9\xd0\xa2\xd0\x12\xeb\x4e\x09\x55\x6d\x69\x7d\x2a"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }
    }

    #[test]
    fn nist_p_256() {
        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x70\x0c\x48\xf7\x7f\x56\x58\x4c\x5c\xc6\x32\xca\x65\x64\x0d\xb9\x1b\x6b\xac\xce\x3a\x4d\xf6\xb4\x2c\xe7\xcc\x83\x88\x33\xd2\x87"[..],
                                                  &b"\xdb\x71\xe5\x09\xe3\xfd\x9b\x06\x0d\xdb\x20\xba\x5c\x51\xdc\xc5\x94\x8d\x46\xfb\xf6\x40\xdf\xe0\x44\x17\x82\xca\xb8\x5f\xa4\xac"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x7d\x7d\xc5\xf7\x1e\xb2\x9d\xda\xf8\x0d\x62\x14\x63\x2e\xea\xe0\x3d\x90\x58\xaf\x1f\xb6\xd2\x2e\xd8\x0b\xad\xb6\x2b\xc1\xa5\x34"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xea\xd2\x18\x59\x01\x19\xe8\x87\x6b\x29\x14\x6f\xf8\x9c\xa6\x17\x70\xc4\xed\xbb\xf9\x7d\x38\xce\x38\x5e\xd2\x81\xd8\xa6\xb2\x30"[..],
                                                &b"\x28\xaf\x61\x28\x1f\xd3\x5e\x2f\xa7\x00\x25\x23\xac\xc8\x5a\x42\x9c\xb0\x6e\xe6\x64\x83\x25\x38\x9f\x59\xed\xfc\xe1\x40\x51\x41"[..]).unwrap();
            let z = &b"\x46\xfc\x62\x10\x64\x20\xff\x01\x2e\x54\xa4\x34\xfb\xdd\x2d\x25\xcc\xc5\x85\x20\x60\x56\x1e\x68\x04\x0d\xd7\x77\x89\x97\xbd\x7b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x80\x9f\x04\x28\x9c\x64\x34\x8c\x01\x51\x5e\xb0\x3d\x5c\xe7\xac\x1a\x8c\xb9\x49\x8f\x5c\xaa\x50\x19\x7e\x58\xd4\x3a\x86\xa7\xae"[..],
                                                  &b"\xb2\x9d\x84\xe8\x11\x19\x7f\x25\xeb\xa8\xf5\x19\x40\x92\xcb\x6f\xf4\x40\xe2\x6d\x44\x21\x01\x13\x72\x46\x1f\x57\x92\x71\xcd\xa3"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x38\xf6\x5d\x6d\xce\x47\x67\x60\x44\xd5\x8c\xe5\x13\x95\x82\xd5\x68\xf6\x4b\xb1\x60\x98\xd1\x79\xdb\xab\x07\x74\x1d\xd5\xca\xf5"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x11\x9f\x2f\x04\x79\x02\x78\x2a\xb0\xc9\xe2\x7a\x54\xaf\xf5\xeb\x9b\x96\x48\x29\xca\x99\xc0\x6b\x02\xdd\xba\x95\xb0\xa3\xf6\xd0"[..],
                                                &b"\x8f\x52\xb7\x26\x66\x4c\xac\x36\x6f\xc9\x8a\xc7\xa0\x12\xb2\x68\x2c\xbd\x96\x2e\x5a\xcb\x54\x46\x71\xd4\x1b\x94\x45\x70\x4d\x1d"[..]).unwrap();
            let z = &b"\x05\x7d\x63\x60\x96\xcb\x80\xb6\x7a\x8c\x03\x8c\x89\x0e\x88\x7d\x1a\xdf\xa4\x19\x5e\x9b\x3c\xe2\x41\xc8\xa7\x78\xc5\x9c\xda\x67"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xa2\x33\x9c\x12\xd4\xa0\x3c\x33\x54\x6d\xe5\x33\x26\x8b\x4a\xd6\x67\xde\xbf\x45\x8b\x46\x4d\x77\x44\x36\x36\x44\x0e\xe7\xfe\xc3"[..],
                                                  &b"\xef\x48\xa3\xab\x26\xe2\x02\x20\xbc\xda\x2c\x18\x51\x07\x68\x39\xda\xe8\x8e\xae\x96\x28\x69\xa4\x97\xbf\x73\xcb\x66\xfa\xf5\x36"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x1a\xcc\xfa\xf1\xb9\x77\x12\xb8\x5a\x6f\x54\xb1\x48\x98\x5a\x1b\xdc\x4c\x9b\xec\x0b\xd2\x58\xca\xd4\xb3\xd6\x03\xf4\x9f\x32\xc8"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xd9\xf2\xb7\x9c\x17\x28\x45\xbf\xdb\x56\x0b\xbb\x01\x44\x7c\xa5\xec\xc0\x47\x0a\x09\x51\x3b\x61\x26\x90\x2c\x6b\x4f\x8d\x10\x51"[..],
                                                &b"\xf8\x15\xef\x5e\xc3\x21\x28\xd3\x48\x78\x34\x76\x46\x78\x70\x2e\x64\xe1\x64\xff\x73\x15\x18\x5e\x23\xaf\xf5\xfa\xcd\x96\xd7\xbc"[..]).unwrap();
            let z = &b"\x2d\x45\x7b\x78\xb4\x61\x41\x32\x47\x76\x18\xa5\xb0\x77\x96\x5e\xc9\x07\x30\xa8\xc8\x1a\x1c\x75\xd6\xd4\xec\x68\x00\x5d\x67\xec"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xdf\x39\x89\xb9\xfa\x55\x49\x57\x19\xb3\xcf\x46\xdc\xcd\x28\xb5\x15\x3f\x78\x08\x19\x1d\xd5\x18\xef\xf0\xc3\xcf\xf2\xb7\x05\xed"[..],
                                                  &b"\x42\x22\x94\xff\x46\x00\x34\x29\xd7\x39\xa3\x32\x06\xc8\x75\x25\x52\xc8\xba\x54\xa2\x70\xde\xfc\x06\xe2\x21\xe0\xfe\xaf\x6a\xc4"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x20\x7c\x43\xa7\x9b\xfe\xe0\x3d\xb6\xf4\xb9\x44\xf5\x3d\x2f\xb7\x6c\xc4\x9e\xf1\xc9\xc4\xd3\x4d\x51\xb6\xc6\x5c\x4d\xb6\x93\x2d"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x24\x27\x7c\x33\xf4\x50\x46\x2d\xcb\x3d\x48\x01\xd5\x7b\x9c\xed\x05\x18\x8f\x16\xc2\x8e\xda\x87\x32\x58\x04\x8c\xd1\x60\x7e\x0d"[..],
                                                &b"\xc4\x78\x97\x53\xe2\xb1\xf6\x3b\x32\xff\x01\x4e\xc4\x2c\xd6\xa6\x9f\xac\x81\xdf\xe6\xd0\xd6\xfd\x4a\xf3\x72\xae\x27\xc4\x6f\x88"[..]).unwrap();
            let z = &b"\x96\x44\x12\x59\x53\x4b\x80\xf6\xae\xe3\xd2\x87\xa6\xbb\x17\xb5\x09\x4d\xd4\x27\x7d\x9e\x29\x4f\x8f\xe7\x3e\x48\xbf\x2a\x00\x24"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x41\x19\x2d\x28\x13\xe7\x95\x61\xe6\xa1\xd6\xf5\x3c\x8b\xc1\xa4\x33\xa1\x99\xc8\x35\xe1\x41\xb0\x5a\x74\xa9\x7b\x0f\xae\xb9\x22"[..],
                                                  &b"\x1a\xf9\x8c\xc4\x5e\x98\xa7\xe0\x41\xb0\x1c\xf3\x5f\x46\x2b\x75\x62\x28\x13\x51\xc8\xeb\xf3\xff\xa0\x2e\x33\xa0\x72\x2a\x13\x28"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x59\x13\x7e\x38\x15\x23\x50\xb1\x95\xc9\x71\x8d\x39\x67\x3d\x51\x98\x38\x05\x5a\xd9\x08\xdd\x47\x57\x15\x2f\xd8\x25\x5c\x09\xbf"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xa8\xc5\xfd\xce\x8b\x62\xc5\xad\xa5\x98\xf1\x41\xad\xb3\xb2\x6c\xf2\x54\xc2\x80\xb2\x85\x7a\x63\xd2\xad\x78\x3a\x73\x11\x5f\x6b"[..],
                                                &b"\x80\x6e\x1a\xaf\xec\x4a\xf8\x0a\x0d\x78\x6b\x3d\xe4\x53\x75\xb5\x17\xa7\xe5\xb5\x1f\xfb\x2c\x35\x65\x37\xc9\xe6\xef\x22\x7d\x4a"[..]).unwrap();
            let z = &b"\x19\xd4\x4c\x8d\x63\xe8\xe8\xdd\x12\xc2\x2a\x87\xb8\xcd\x4e\xce\x27\xac\xdd\xe0\x4d\xbf\x47\xf7\xf2\x75\x37\xa6\x99\x9a\x8e\x62"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x33\xe8\x20\x92\xa0\xf1\xfb\x38\xf5\x64\x9d\x58\x67\xfb\xa2\x8b\x50\x31\x72\xb7\x03\x55\x74\xbf\x8e\x5b\x71\x00\xa3\x05\x27\x92"[..],
                                                  &b"\xf2\xcf\x6b\x60\x1e\x0a\x05\x94\x5e\x33\x55\x50\xbf\x64\x8d\x78\x2f\x46\x18\x6c\x77\x2c\x0f\x20\xd3\xcd\x0d\x6b\x8c\xa1\x4b\x2f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xf5\xf8\xe0\x17\x46\x10\xa6\x61\x27\x79\x79\xb5\x8c\xe5\xc9\x0f\xee\x6c\x9b\x3b\xb3\x46\xa9\x0a\x71\x96\x25\x5e\x40\xb1\x32\xef"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x7b\x86\x1d\xcd\x28\x44\xa5\xa8\x36\x3f\x6b\x8e\xf8\xd4\x93\x64\x0f\x55\x87\x92\x17\x18\x9d\x80\x32\x6a\xad\x94\x80\xdf\xc1\x49"[..],
                                                &b"\xc4\x67\x5b\x45\xee\xb3\x06\x40\x5f\x6c\x33\xc3\x8b\xc6\x9e\xb2\xbd\xec\x9b\x75\xad\x5a\xf4\x70\x6a\xab\x84\x54\x3b\x9c\xc6\x3a"[..]).unwrap();
            let z = &b"\x66\x4e\x45\xd5\xbb\xa4\xac\x93\x1c\xd6\x5d\x52\x01\x7e\x4b\xe9\xb1\x9a\x51\x5f\x66\x9b\xea\x47\x03\x54\x2a\x2c\x52\x5c\xd3\xd3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x6a\x9e\x0c\x3f\x91\x6e\x4e\x31\x5c\x91\x14\x7b\xe5\x71\x68\x6d\x90\x46\x4e\x8b\xf9\x81\xd3\x4a\x90\xb6\x35\x3b\xca\x6e\xeb\xa7"[..],
                                                  &b"\x40\xf9\xbe\xad\x39\xc2\xf2\xbc\xc2\x60\x2f\x75\xb8\xa7\x3e\xc7\xbd\xff\xcb\xce\xad\x15\x9d\x01\x74\xc6\xc4\xd3\xc5\x35\x7f\x05"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x3b\x58\x9a\xf7\xdb\x03\x45\x9c\x23\x06\x8b\x64\xf6\x3f\x28\xd3\xc3\xc6\xbc\x25\xb5\xbf\x76\xac\x05\xf3\x54\x82\x88\x8b\x51\x90"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x9f\xb3\x8e\x2d\x58\xea\x1b\xaf\x76\x22\xe9\x67\x20\x10\x1c\xae\x3c\xde\x4b\xa6\xc1\xe9\xfa\x26\xd9\xb1\xde\x08\x99\x10\x28\x63"[..],
                                                &b"\xd5\x56\x1b\x90\x04\x06\xed\xf5\x08\x02\xdd\x7d\x73\xe8\x93\x95\xf8\xae\xd7\x2f\xba\x0e\x1d\x1b\x61\xfe\x1d\x22\x30\x22\x60\xf0"[..]).unwrap();
            let z = &b"\xca\x34\x2d\xaa\x50\xdc\x09\xd6\x1b\xe7\xc1\x96\xc8\x5e\x60\xa8\x0c\x5c\xb0\x49\x31\x74\x68\x20\xbe\x54\x8c\xdd\xe0\x55\x67\x9d"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xa9\xc0\xac\xad\xe5\x5c\x2a\x73\xea\xd1\xa8\x6f\xb0\xa9\x71\x32\x23\xc8\x24\x75\x79\x1c\xd0\xe2\x10\xb0\x46\x41\x2c\xe2\x24\xbb"[..],
                                                  &b"\xf6\xde\x0a\xfa\x20\xe9\x3e\x07\x84\x67\xc0\x53\xd2\x41\x90\x3e\xda\xd7\x34\xc6\xb4\x03\xba\x75\x8c\x2b\x5f\xf0\x4c\x9d\x42\x29"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xd8\xbf\x92\x9a\x20\xea\x74\x36\xb2\x46\x1b\x54\x1a\x11\xc8\x0e\x61\xd8\x26\xc0\xa4\xc9\xd3\x22\xb3\x1d\xd5\x4e\x7f\x58\xb9\xc8"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x20\xf0\x76\x31\xe4\xa6\x51\x2a\x89\xad\x48\x7c\x4e\x9d\x63\x03\x9e\x57\x9c\xb0\xd7\xa5\x56\xcb\x9e\x66\x1c\xd5\x9c\x1e\x7f\xa4"[..],
                                                &b"\x6d\xe9\x18\x46\xb3\xee\xe8\xa5\xec\x09\xc2\xab\x1f\x41\xe2\x1b\xd8\x36\x20\xcc\xdd\x1b\xdc\xe3\xab\x7e\xa6\xe0\x2d\xd2\x74\xf5"[..]).unwrap();
            let z = &b"\x35\xaa\x9b\x52\x53\x6a\x46\x1b\xfd\xe4\xe8\x5f\xc7\x56\xbe\x92\x8c\x7d\xe9\x79\x23\xf0\x41\x6c\x7a\x3a\xc8\xf8\x8b\x3d\x44\x89"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x94\xe9\x4f\x16\xa9\x82\x55\xff\xf2\xb9\xac\x0c\x95\x98\xaa\xc3\x54\x87\xb3\x23\x2d\x32\x31\xbd\x93\xb7\xdb\x7d\xf3\x6f\x9e\xb9"[..],
                                                  &b"\xd8\x04\x9a\x43\x57\x9c\xfa\x90\xb8\x09\x3a\x94\x41\x6c\xbe\xfb\xf9\x33\x86\xf1\x5b\x3f\x6e\x19\x0b\x6e\x34\x55\xfe\xdf\xe6\x9a"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x0f\x98\x83\xba\x0e\xf3\x2e\xe7\x5d\xed\x0d\x8b\xda\x39\xa5\x14\x6a\x29\xf1\xf2\x50\x7b\x3b\xd4\x58\xdb\xea\x0b\x2b\xb0\x5b\x4d"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xab\xb6\x1b\x42\x3b\xe5\xd6\xc2\x6e\x21\xc6\x05\x83\x2c\x91\x42\xdc\x1d\xfe\x5a\x5f\xff\x28\x72\x67\x37\x93\x6e\x6f\xbf\x51\x6d"[..],
                                                &b"\x73\x3d\x25\x13\xef\x58\xbe\xab\x20\x20\x90\x58\x6f\xac\x91\xbf\x0f\xee\x31\xe8\x0a\xb3\x34\x73\xab\x23\xa2\xd8\x9e\x58\xfa\xd6"[..]).unwrap();
            let z = &b"\x60\x5c\x16\x17\x8a\x9b\xc8\x75\xdc\xbf\xf5\x4d\x63\xfe\x00\xdf\x69\x9c\x03\xe8\xa8\x88\xe9\xe9\x4d\xfb\xab\x90\xb2\x5f\x39\xb4"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xe0\x99\xbf\x2a\x4d\x55\x74\x60\xb5\x54\x44\x30\xbb\xf6\xda\x11\x00\x4d\x12\x7c\xb5\xd6\x7f\x64\xab\x07\xc9\x4f\xcd\xf5\x27\x4f"[..],
                                                  &b"\xd9\xc5\x0d\xbe\x70\xd7\x14\xed\xb5\xe2\x21\xf4\xe0\x20\x61\x0e\xeb\x62\x70\x51\x7e\x68\x8c\xa6\x4f\xb0\xe9\x8c\x7e\xf8\xc1\xc5"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x2b\xee\xdb\x04\xb0\x5c\x69\x88\xf6\xa6\x75\x00\xbb\x81\x3f\xaf\x2c\xae\x0d\x58\x0c\x92\x53\xb6\x33\x9e\x4a\x33\x37\xbb\x6c\x08"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x3d\x63\xe4\x29\xcb\x5f\xa8\x95\xa9\x24\x71\x29\xbf\x4e\x48\xe8\x9f\x35\xd7\xb1\x1d\xe8\x15\x8e\xfe\xb3\xe1\x06\xa2\xa8\x73\x95"[..],
                                                &b"\x0c\xae\x9e\x47\x7e\xf4\x1e\x7c\x8c\x10\x64\x37\x9b\xb7\xb5\x54\xdd\xcb\xca\xe7\x9f\x98\x14\x28\x1f\x1e\x50\xf0\x40\x3c\x61\xf3"[..]).unwrap();
            let z = &b"\xf9\x6e\x40\xa1\xb7\x28\x40\x85\x4b\xb6\x2b\xc1\x3c\x40\xcc\x27\x95\xe3\x73\xd4\xe7\x15\x98\x0b\x26\x14\x76\x83\x5a\x09\x2e\x0b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xf7\x5a\x5f\xe5\x6b\xda\x34\xf3\xc1\x39\x62\x96\x62\x6e\xf0\x12\xdc\x07\xe4\x82\x58\x38\x77\x8a\x64\x5c\x82\x48\xcf\xf0\x16\x58"[..],
                                                  &b"\x33\xbb\xdf\x1b\x17\x72\xd8\x05\x9d\xf5\x68\xb0\x61\xf3\xf1\x12\x2f\x28\xa8\xd8\x19\x16\x7c\x97\xbe\x44\x8e\x3d\xc3\xfb\x0c\x3c"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x77\xc1\x5d\xcf\x44\x61\x0e\x41\x69\x6b\xab\x75\x89\x43\xef\xf1\x40\x93\x33\xe4\xd5\xa1\x1b\xbe\x72\xc8\xf6\xc3\x95\xe9\xf8\x48"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xad\x5d\x13\xc3\xdb\x50\x8d\xdc\xd3\x84\x57\xe5\x99\x14\x34\xa2\x51\xbe\xd4\x9c\xf5\xdd\xcb\x59\xcd\xee\x73\x86\x5f\x13\x8c\x9f"[..],
                                                &b"\x62\xce\xc1\xe7\x05\x88\xaa\x4f\xdf\xc7\xb9\xa0\x9d\xaa\x67\x80\x81\xc0\x4e\x12\x08\xb9\xd6\x62\xb8\xa2\x21\x4b\xf8\xe8\x1a\x21"[..]).unwrap();
            let z = &b"\x83\x88\xfa\x79\xc4\xba\xbd\xca\x02\xa8\xe8\xa3\x4f\x9e\x43\x55\x49\x76\xe4\x20\xa4\xad\x27\x3c\x81\xb2\x6e\x42\x28\xe9\xd3\xa3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x2d\xb4\x54\x0d\x50\x23\x07\x56\x15\x8a\xbf\x61\xd9\x83\x57\x12\xb6\x48\x6c\x74\x31\x21\x83\xcc\xef\xca\xef\x27\x97\xb7\x67\x4d"[..],
                                                  &b"\x62\xf5\x7f\x31\x4e\x3f\x34\x95\xdc\x4e\x09\x90\x12\xf5\xe0\xba\x71\x77\x0f\x96\x60\xa1\xea\xda\x54\x10\x4c\xdf\xde\x77\x24\x3e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x42\xa8\x3b\x98\x50\x11\xd1\x23\x03\xdb\x1a\x80\x0f\x26\x10\xf7\x4a\xa7\x1c\xdf\x19\xc6\x7d\x54\xce\x6c\x9e\xd9\x51\xe9\x09\x3e"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xab\x48\xca\xa6\x1e\xa3\x5f\x13\xf8\xed\x07\xff\xa6\xa1\x3e\x8d\xb2\x24\xdf\xec\xfa\xe1\xa7\xdf\x8b\x1b\xb6\xeb\xaf\x0c\xb9\x7d"[..],
                                                &b"\x12\x74\x53\x0c\xa2\xc3\x85\xa3\x21\x8b\xdd\xfb\xcb\xf0\xb4\x02\x4c\x9b\xad\xd5\x24\x3b\xff\x83\x4e\xbf\xf2\x4a\x86\x18\xdc\xcb"[..]).unwrap();
            let z = &b"\x72\x87\x7c\xea\x33\xcc\xc4\x71\x50\x38\xd4\xbc\xbd\xfe\x0e\x43\xf4\x2a\x9e\x2c\x0c\x3b\x01\x7f\xc2\x37\x0f\x4b\x9a\xcb\xda\x4a"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xcd\x94\xfc\x94\x97\xe8\x99\x07\x50\x30\x9e\x9a\x85\x34\xfd\x11\x4b\x0a\x6e\x54\xda\x89\xc4\x79\x61\x01\x89\x70\x41\xd1\x4e\xcb"[..],
                                                  &b"\xc3\xde\xf4\xb5\xfe\x04\xfa\xee\x0a\x11\x93\x22\x29\xff\xf5\x63\x63\x7b\xfd\xee\x0e\x79\xc6\xde\xea\xf4\x49\xf8\x54\x01\xc5\xc4"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xce\xed\x35\x50\x7b\x5c\x93\xea\xd5\x98\x91\x19\xb9\xba\x34\x2c\xfe\x38\xe6\xe6\x38\xba\x6e\xea\x34\x3a\x55\x47\x5d\xe2\x80\x0b"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x9a\x8c\xd9\xbd\x72\xe7\x17\x52\xdf\x91\x44\x0f\x77\xc5\x47\x50\x9a\x84\xdf\x98\x11\x4e\x7d\xe4\xf2\x6c\xdb\x39\x23\x4a\x62\x5d"[..],
                                                &b"\xd0\x7c\xfc\x84\xc8\xe1\x44\xfa\xb2\x83\x9f\x51\x89\xbb\x1d\x7c\x88\x63\x1d\x57\x9b\xbc\x58\x01\x2e\xd9\xa2\x32\x7d\xa5\x2f\x62"[..]).unwrap();
            let z = &b"\xe4\xe7\x40\x8d\x85\xff\x0e\x0e\x9c\x83\x80\x03\xf2\x8c\xdb\xd5\x24\x7c\xdc\xe3\x1f\x32\xf6\x24\x94\xb7\x0e\x5f\x1b\xc3\x63\x07"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x15\xb9\xe4\x67\xaf\x4d\x29\x0c\x41\x74\x02\xe0\x40\x42\x6f\xe4\xcf\x23\x6b\xae\x72\xba\xa3\x92\xed\x89\x78\x0d\xfc\xcd\xb4\x71"[..],
                                                  &b"\xcd\xf4\xe9\x17\x0f\xb9\x04\x30\x2b\x8f\xd9\x3a\x82\x0b\xa8\xcc\x7e\xd4\xef\xd3\xa6\xf2\xd6\xb0\x5b\x80\xb2\xff\x2a\xee\x4e\x77"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x43\xe0\xe9\xd9\x5a\xf4\xdc\x36\x48\x3c\xdd\x19\x68\xd2\xb7\xee\xb8\x61\x1f\xcc\xe7\x7f\x3a\x4e\x7d\x05\x9a\xe4\x3e\x50\x96\x04"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xf9\x89\xcf\x8e\xe9\x56\xa8\x2e\x7e\xbd\x98\x81\xcd\xbf\xb2\xfd\x94\x61\x89\xb0\x8d\xb5\x35\x59\xbc\x8c\xfd\xd4\x80\x71\xeb\x14"[..],
                                                &b"\x5e\xff\x28\xf1\xa1\x8a\x61\x6b\x04\xb7\xd3\x37\x86\x86\x79\xf6\xdd\x84\xf9\xa7\xb3\xd7\xb6\xf8\xaf\x27\x6c\x19\x61\x1a\x54\x1d"[..]).unwrap();
            let z = &b"\xed\x56\xbc\xf6\x95\xb7\x34\x14\x2c\x24\xec\xb1\xfc\x1b\xb6\x4d\x08\xf1\x75\xeb\x24\x3a\x31\xf3\x7b\x3d\x9b\xb4\x40\x7f\x3b\x96"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x49\xc5\x03\xba\x6c\x4f\xa6\x05\x18\x2e\x18\x6b\x5e\x81\x11\x3f\x07\x5b\xc1\x1d\xcf\xd5\x1c\x93\x2f\xb2\x1e\x95\x1e\xee\x2f\xa1"[..],
                                                  &b"\x8a\xf7\x06\xff\x09\x22\xd8\x7b\x3f\x0c\x5e\x4e\x31\xd8\xb2\x59\xae\xb2\x60\xa9\x26\x96\x43\xed\x52\x0a\x13\xbb\x25\xda\x59\x24"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xb2\xf3\x60\x0d\xf3\x36\x8e\xf8\xa0\xbb\x85\xab\x22\xf4\x1f\xc0\xe5\xf4\xfd\xd5\x4b\xe8\x16\x7a\x5c\x3c\xd4\xb0\x8d\xb0\x49\x03"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x69\xc6\x27\x62\x5b\x36\xa4\x29\xc3\x98\xb4\x5c\x38\x67\x7c\xb3\x5d\x8b\xeb\x1c\xf7\x8a\x57\x1e\x40\xe9\x9f\xe4\xea\xc1\xcd\x4e"[..],
                                                &b"\x81\x69\x01\x12\xb0\xa8\x8f\x20\xf7\x13\x6b\x28\xd7\xd4\x7e\x5f\xbc\x2a\xda\x3c\x8e\xdd\x87\x58\x9b\xc1\x9e\xc9\x59\x06\x37\xbd"[..]).unwrap();
            let z = &b"\xbc\x5c\x70\x55\x08\x9f\xc9\xd6\xc8\x9f\x83\xc1\xea\x1a\xda\x87\x9d\x99\x34\xb2\xea\x28\xfc\xf4\xe4\xa7\xe9\x84\xb2\x8a\xd2\xcf"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x19\xb3\x8d\xe3\x9f\xdd\x2f\x70\xf7\x09\x16\x31\xa4\xf7\x5d\x19\x93\x74\x0b\xa9\x42\x91\x62\xc2\xa4\x53\x12\x40\x16\x36\xb2\x9c"[..],
                                                  &b"\x09\xae\xd7\x23\x2b\x28\xe0\x60\x94\x17\x41\xb6\x82\x8b\xcd\xfa\x2b\xc4\x9c\xc8\x44\xf3\x77\x36\x11\x50\x4f\x82\xa3\x90\xa5\xae"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x40\x02\x53\x43\x07\xf8\xb6\x2a\x9b\xf6\x7f\xf6\x41\xdd\xc6\x0f\xef\x59\x3b\x17\xc3\x34\x12\x39\xe9\x5b\xdb\x3e\x57\x9b\xfd\xc8"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x5f\xe9\x64\x67\x13\x15\xa1\x8a\xa6\x8a\x2a\x6e\x3d\xd1\xfd\xe7\xe2\x3b\x8c\xe7\x18\x14\x71\xcf\xac\x43\xc9\x9e\x1a\xe8\x02\x62"[..],
                                                &b"\xd5\x82\x7b\xe2\x82\xe6\x2c\x84\xde\x53\x1b\x96\x38\x84\xba\x83\x2d\xb5\xd6\xb2\xc3\xa2\x56\xf0\xe6\x04\xfe\x7e\x6b\x8a\x7f\x72"[..]).unwrap();
            let z = &b"\x9a\x4e\x8e\x65\x7f\x6b\x0e\x09\x7f\x47\x95\x4a\x63\xc7\x5d\x74\xfc\xba\x71\xa3\x0d\x83\x65\x1e\x3e\x5a\x91\xaa\x7c\xcd\x83\x43"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x2c\x91\xc6\x1f\x33\xad\xfe\x93\x11\xc9\x42\xfd\xbf\xf6\xba\x47\x02\x0f\xef\xf4\x16\xb7\xbb\x63\xce\xc1\x3f\xaf\x9b\x09\x99\x54"[..],
                                                  &b"\x6c\xab\x31\xb0\x64\x19\xe5\x22\x1f\xca\x01\x4f\xb8\x4e\xc8\x70\x62\x2a\x1b\x12\xba\xb5\xae\x43\x68\x2a\xa7\xea\x73\xea\x08\xd0"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x4d\xfa\x12\xde\xfc\x60\x31\x90\x21\xb6\x81\xb3\xff\x84\xa1\x0a\x51\x19\x58\xc8\x50\x93\x9e\xd4\x56\x35\x93\x4b\xa4\x97\x91\x47"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\xc9\xb2\xb8\x49\x6f\x14\x40\xbd\x4a\x2d\x1e\x52\x75\x2f\xd3\x72\x83\x5b\x36\x48\x85\xe1\x54\xa7\xda\xc4\x92\x95\xf2\x81\xec\x7c"[..],
                                                &b"\xfb\xe6\xb9\x26\xa8\xa4\xde\x26\xcc\xc8\x3b\x80\x2b\x12\x12\x40\x07\x54\xbe\x25\xd9\xf3\xee\xaf\x00\x8b\x09\x87\x0a\xe7\x63\x21"[..]).unwrap();
            let z = &b"\x3c\xa1\xfc\x7a\xd8\x58\xfb\x1a\x6a\xba\x23\x25\x42\xf3\xe2\xa7\x49\xff\xc7\x20\x3a\x23\x74\xa3\xf3\xd3\x26\x7f\x1f\xc9\x7b\x78"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xa2\x8a\x2e\xdf\x58\x02\x56\x68\xf7\x24\xaa\xf8\x3a\x50\x95\x6b\x7a\xc1\xcf\xbb\xff\x79\xb0\x8c\x3b\xf8\x7d\xfd\x28\x28\xd7\x67"[..],
                                                  &b"\xdf\xa7\xbf\xff\xd4\xc7\x66\xb8\x6a\xbe\xaf\x5c\x99\xb6\xe5\x0c\xb9\xcc\xc9\xd9\xd0\x0b\x7f\xfc\x78\x04\xb0\x49\x1b\x67\xbc\x03"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x13\x31\xf6\xd8\x74\xa4\xed\x3b\xc4\xa2\xc6\xe9\xc7\x43\x31\xd3\x03\x97\x96\x31\x4b\xee\xe3\xb7\x15\x2f\xcd\xba\x55\x56\x30\x4e"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x59\xe1\xe1\x01\x52\x10\x46\xad\x9c\xf1\xd0\x82\xe9\xd2\xec\x7d\xd2\x25\x30\xcc\xe0\x64\x99\x1f\x1e\x55\xc5\xbc\xf5\xfc\xb5\x91"[..],
                                                &b"\x48\x2f\x4f\x67\x31\x76\xc8\xfd\xaa\x0b\xb6\xe5\x9b\x15\xa3\xe4\x74\x54\xe3\xa0\x42\x97\xd3\x86\x3c\x93\x38\xd9\x8a\xdd\x1f\x37"[..]).unwrap();
            let z = &b"\x1a\xaa\xbe\x7e\xe6\xe4\xa6\xfa\x73\x22\x91\x20\x24\x33\xa2\x37\xdf\x1b\x49\xbc\x53\x86\x6b\xfb\xe0\x0d\xb9\x6a\x0f\x58\x22\x4f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xa2\xef\x85\x7a\x08\x1f\x9d\x6e\xb2\x06\xa8\x1c\x4c\xf7\x8a\x80\x2b\xdf\x59\x8a\xe3\x80\xc8\x88\x6e\xcd\x85\xfd\xc1\xed\x76\x44"[..],
                                                  &b"\x56\x3c\x4c\x20\x41\x9f\x07\xbc\x17\xd0\x53\x9f\xad\xe1\x85\x5e\x34\x83\x95\x15\xb8\x92\xc0\xf5\xd2\x65\x61\xf9\x7f\xa0\x4d\x1a"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xdd\x5e\x9f\x70\xae\x74\x00\x73\xca\x02\x04\xdf\x60\x76\x3f\xb6\x03\x6c\x45\x70\x9b\xf4\xa7\xbb\x4e\x67\x14\x12\xfa\xd6\x5d\xa3"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x30\xb9\xdb\x2e\x2e\x97\x7b\xcd\xc9\x8c\xb8\x7d\xd7\x36\xcb\xd8\xe7\x85\x52\x12\x19\x25\xcf\x16\xe1\x93\x36\x57\xc2\xfb\x23\x14"[..],
                                                &b"\x6a\x45\x02\x88\x00\xb8\x12\x91\xbc\xe5\xc2\xe1\xfe\xd7\xde\xd6\x50\x62\x0e\xbb\xe6\x05\x0c\x6f\x3a\x7f\x0d\xfb\x46\x73\xab\x5c"[..]).unwrap();
            let z = &b"\x43\x0e\x6a\x4f\xba\x44\x49\xd7\x00\xd2\x73\x3e\x55\x7f\x66\xa3\xbf\x3d\x50\x51\x7c\x12\x71\xb1\xdd\xae\x11\x61\xb7\xac\x79\x8c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xcc\xd8\xa2\xd8\x6b\xc9\x2f\x2e\x01\xbc\xe4\xd6\x92\x2c\xf7\xfe\x16\x26\xae\xd0\x44\x68\x5e\x95\xe2\xee\xbd\x46\x45\x05\xf0\x1f"[..],
                                                  &b"\xe9\xdd\xd5\x83\xa9\x63\x5a\x66\x77\x77\xd5\xb8\xa8\xf3\x1b\x0f\x79\xeb\xa1\x2c\x75\x02\x34\x10\xb5\x4b\x85\x67\xdd\xdc\x0f\x38"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x5a\xe0\x26\xcf\xc0\x60\xd5\x56\x00\x71\x7e\x55\xb8\xa1\x2e\x11\x6d\x1d\x0d\xf3\x4a\xf8\x31\x97\x90\x57\x60\x7c\x2d\x9c\x2f\x76"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x46\xc9\xeb\xd1\xa4\xa3\xc8\xc0\xb6\xd5\x72\xb5\xdc\xfb\xa1\x24\x67\x60\x32\x08\xa9\xcb\x5d\x2a\xcf\xbb\x73\x3c\x40\xcf\x63\x91"[..],
                                                &b"\x46\xc9\x13\xa2\x7d\x04\x41\x85\xd3\x8b\x46\x7a\xce\x01\x1e\x04\xd4\xd9\xbb\xbb\x8c\xb9\xae\x25\xfa\x92\xaa\xf1\x5a\x59\x5e\x86"[..]).unwrap();
            let z = &b"\x1c\xe9\xe6\x74\x05\x29\x49\x9f\x98\xd1\xf1\xd7\x13\x29\x14\x7a\x33\xdf\x1d\x05\xe4\x76\x5b\x53\x9b\x11\xcf\x61\x5d\x69\x74\xd3"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xc1\x88\xff\xc8\x94\x7f\x73\x01\xfb\x7b\x53\xe3\x67\x46\x09\x7c\x21\x34\xbf\x9c\xc9\x81\xba\x74\xb4\xe9\xc4\x36\x1f\x59\x5e\x4e"[..],
                                                  &b"\xbf\x7d\x2f\x20\x56\xe7\x24\x21\xef\x39\x3f\x0c\x0f\x2b\x0e\x00\x13\x0e\x3c\xac\x4a\xbb\xcc\x00\x28\x61\x68\xe8\x5e\xc5\x50\x51"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xb6\x01\xac\x42\x5d\x5d\xbf\x9e\x17\x35\xc5\xe2\xd5\xbd\xb7\x9c\xa9\x8b\x3d\x5b\xe4\xa2\xcf\xd6\xf2\x27\x3f\x15\x0e\x06\x4d\x9d"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x7c\x9e\x95\x08\x41\xd2\x6c\x8d\xde\x89\x94\x39\x8b\x8f\x5d\x47\x5a\x02\x2b\xc6\x3d\xe7\x77\x3f\xcf\x8d\x55\x2e\x01\xf1\xba\x0a"[..],
                                                &b"\xcc\x42\xb9\x88\x5c\x9b\x3b\xee\x0f\x8d\x8c\x57\xd3\xa8\xf6\x35\x50\x16\xc0\x19\xc4\x06\x2f\xa2\x2c\xff\x2f\x20\x9b\x5c\xc2\xe1"[..]).unwrap();
            let z = &b"\x46\x90\xe3\x74\x3c\x07\xd6\x43\xf1\xbc\x18\x36\x36\xab\x2a\x9c\xb9\x36\xa6\x0a\x80\x21\x13\xc4\x9b\xb1\xb3\xf2\xd0\x66\x16\x60"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x31\x7e\x10\x20\xff\x53\xfc\xce\xf1\x8b\xf4\x7b\xb7\xf2\xdd\x77\x07\xfb\x7b\x7a\x75\x78\xe0\x4f\x35\xb3\xbe\xed\x22\x2a\x0e\xb6"[..],
                                                  &b"\x09\x42\x0c\xe5\xa1\x9d\x77\xc6\xfe\x1e\xe5\x87\xe6\xa4\x9f\xba\xf8\xf2\x80\xe8\xdf\x03\x3d\x75\x40\x33\x02\xe5\xa2\x7d\xb2\xae"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\xfe\xfb\x1d\xda\x18\x45\x31\x2b\x5f\xce\x6b\x81\xb2\xbe\x20\x5a\xf2\xf3\xa2\x74\xf5\xa2\x12\xf6\x6c\x0d\x9f\xc3\x3d\x7a\xe5\x35"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x38\xb5\x4d\xb8\x55\x00\xcb\x20\xc6\x10\x56\xed\xd3\xd8\x8b\x6a\x9d\xc2\x67\x80\xa0\x47\xf2\x13\xa6\xe1\xb9\x00\xf7\x65\x96\xeb"[..],
                                                &b"\x63\x87\xe4\xe5\x78\x15\x71\xe4\xeb\x8a\xe6\x29\x91\xa3\x3b\x5d\xc3\x33\x01\xc5\xbc\x7e\x12\x5d\x53\x79\x4a\x39\x16\x0d\x8f\xd0"[..]).unwrap();
            let z = &b"\x30\xc2\x26\x1b\xd0\x00\x4e\x61\xfe\xda\x2c\x16\xaa\x5e\x21\xff\xa8\xd7\xe7\xf7\xdb\xf6\xec\x37\x9a\x43\xb4\x8e\x4b\x36\xae\xb0"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x45\xfb\x02\xb2\xce\xb9\xd7\xc7\x9d\x9c\x2f\xa9\x3e\x9c\x79\x67\xc2\xfa\x4d\xf5\x78\x9f\x96\x40\xb2\x42\x64\xb1\xe5\x24\xfc\xb1"[..],
                                                  &b"\x5c\x6e\x8e\xcf\x1f\x7d\x30\x23\x89\x3b\x7b\x1c\xa1\xe4\xd1\x78\x97\x2e\xe2\xa2\x30\x75\x7d\xdc\x56\x4f\xfe\x37\xf5\xc5\xa3\x21"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x33\x4a\xe0\xc4\x69\x3d\x23\x93\x5a\x7e\x8e\x04\x3e\xbb\xde\x21\xe1\x68\xa7\xcb\xa3\xfa\x50\x7c\x9b\xe4\x1d\x76\x81\xe0\x49\xce"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x3f\x2b\xf1\x58\x9a\xbf\x30\x47\xbf\x3e\x54\xac\x9a\x95\x37\x9b\xff\x95\xf8\xf5\x54\x05\xf6\x4e\xca\x36\xa7\xee\xbe\x8f\xfc\xa7"[..],
                                                &b"\x52\x12\xa9\x4e\x66\xc5\xae\x9a\x89\x91\x87\x2f\x66\xa7\x27\x23\xd8\x0e\xc5\xb2\xe9\x25\x74\x5c\x45\x6f\x53\x71\x94\x3b\x3a\x06"[..]).unwrap();
            let z = &b"\x2a\xda\xe4\xa1\x38\xa2\x39\xdc\xd9\x3c\x24\x3a\x38\x03\xc3\xe4\xcf\x96\xe3\x7f\xe1\x4e\x6a\x9b\x71\x7b\xe9\x59\x99\x59\xb1\x1c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\xa1\x9e\xf7\xbf\xf9\x8a\xda\x78\x18\x42\xfb\xfc\x51\xa4\x7a\xff\x39\xb5\x93\x5a\x1c\x7d\x96\x25\xc8\xd3\x23\xd5\x11\xc9\x2d\xe6"[..],
                                                  &b"\xe9\xc1\x84\xdf\x75\xc9\x55\xe0\x2e\x02\xe4\x00\xff\xe4\x5f\x78\xf3\x39\xe1\xaf\xe6\xd0\x56\xfb\x32\x45\xf4\x70\x0c\xe6\x06\xef"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x2c\x4b\xde\x40\x21\x4f\xcc\x3b\xfc\x47\xd4\xcf\x43\x4b\x62\x9a\xcb\xe9\x15\x7f\x8f\xd0\x28\x25\x40\x33\x1d\xe7\x94\x2c\xf0\x9d"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x29\xc0\x80\x7f\x10\xcb\xc4\x2f\xb4\x5c\x99\x89\xda\x50\x68\x1e\xea\xd7\x16\xda\xa7\xb9\xe9\x1f\xd3\x2e\x06\x2f\x5e\xb9\x2c\xa0"[..],
                                                &b"\xff\x1d\x6d\x19\x55\xd7\x37\x6b\x2d\xa2\x4f\xe1\x16\x3a\x27\x16\x59\x13\x63\x41\xbc\x2e\xb1\x19\x5f\xc7\x06\xdc\x62\xe7\xf3\x4d"[..]).unwrap();
            let z = &b"\x2e\x27\x7e\xc3\x0f\x5e\xa0\x7d\x6c\xe5\x13\x14\x9b\x94\x79\xb9\x6e\x07\xf4\xb6\x91\x3b\x1b\x5c\x11\x30\x5c\x14\x44\xa1\xbc\x0b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp256r1>(&b"\x35\x6c\x5a\x44\x4c\x04\x9a\x52\xfe\xe0\xad\xeb\x7e\x5d\x82\xae\x5a\xa8\x30\x30\xbf\xff\x31\xbb\xf8\xce\x20\x96\xcf\x16\x1c\x4b"[..],
                                                  &b"\x57\xd1\x28\xde\x8b\x2a\x57\xa0\x94\xd1\xa0\x01\xe5\x72\x17\x3f\x96\xe8\x86\x6a\xe3\x52\xbf\x29\xcd\xda\xf9\x2f\xc8\x5b\x2f\x92"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp256r1>(&b"\x85\xa2\x68\xf9\xd7\x77\x2f\x99\x0c\x36\xb4\x2b\x0a\x33\x1a\xdc\x92\xb5\x94\x1d\xe0\xb8\x62\xd5\xd8\x9a\x34\x7c\xbf\x8f\xaa\xb0"[..]).unwrap();
            let iut_pub = Point::new::<Secp256r1>(&b"\x9c\xf4\xb9\x85\x81\xca\x17\x79\x45\x3c\xc8\x16\xff\x28\xb4\x10\x0a\xf5\x6c\xf1\xbf\x2e\x5b\xc3\x12\xd8\x3b\x6b\x1b\x21\xd3\x33"[..],
                                                &b"\x7a\x55\x04\xfc\xac\x52\x31\xa0\xd1\x2d\x65\x82\x18\x28\x48\x68\x22\x9c\x84\x4a\x04\xa3\x45\x0d\x6c\x73\x81\xab\xe0\x80\xbf\x3b"[..]).unwrap();
            let z = &b"\x1e\x51\x37\x3b\xd2\xc6\x04\x4c\x12\x9c\x43\x6e\x74\x2a\x55\xbe\x2a\x66\x8a\x85\xae\x08\x44\x1b\x67\x56\x44\x5d\xf5\x49\x38\x57"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }
    }

    #[test]
    fn nist_p_384() {
        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xa7\xc7\x6b\x97\x0c\x3b\x5f\xe8\xb0\x5d\x28\x38\xae\x04\xab\x47\x69\x7b\x9e\xaf\x52\xe7\x64\x59\x2e\xfd\xa2\x7f\xe7\x51\x32\x72\x73\x44\x66\xb4\x00\x09\x1a\xdb\xf2\xd6\x8c\x58\xe0\xc5\x00\x66"[..],
                                                  &b"\xac\x68\xf1\x9f\x2e\x1c\xb8\x79\xae\xd4\x3a\x99\x69\xb9\x1a\x08\x39\xc4\xc3\x8a\x49\x74\x9b\x66\x1e\xfe\xdf\x24\x34\x51\x91\x5e\xd0\x90\x5a\x32\xb0\x60\x99\x2b\x46\x8c\x64\x76\x6f\xc8\x43\x7a"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x3c\xc3\x12\x2a\x68\xf0\xd9\x50\x27\xad\x38\xc0\x67\x91\x6b\xa0\xeb\x8c\x38\x89\x4d\x22\xe1\xb1\x56\x18\xb6\x81\x8a\x66\x17\x74\xad\x46\x3b\x20\x5d\xa8\x8c\xf6\x99\xab\x4d\x43\xc9\xcf\x98\xa1"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x98\x03\x80\x7f\x2f\x6d\x2f\xd9\x66\xcd\xd0\x29\x0b\xd4\x10\xc0\x19\x03\x52\xfb\xec\x7f\xf6\x24\x7d\xe1\x30\x2d\xf8\x6f\x25\xd3\x4f\xe4\xa9\x7b\xef\x60\xcf\xf5\x48\x35\x5c\x01\x5d\xbb\x3e\x5f"[..],
                                                &b"\xba\x26\xca\x69\xec\x2f\x5b\x5d\x9d\xad\x20\xcc\x9d\xa7\x11\x38\x3a\x9d\xbe\x34\xea\x3f\xa5\xa2\xaf\x75\xb4\x65\x02\x62\x9a\xd5\x4d\xd8\xb7\xd7\x3a\x8a\xbb\x06\xa3\xa3\xbe\x47\xd6\x50\xcc\x99"[..]).unwrap();
            let z = &b"\x5f\x9d\x29\xdc\x5e\x31\xa1\x63\x06\x03\x56\x21\x36\x69\xc8\xce\x13\x2e\x22\xf5\x7c\x9a\x04\xf4\x0b\xa7\xfc\xea\xd4\x93\xb4\x57\xe5\x62\x1e\x76\x6c\x40\xa2\xe3\xd4\xd6\xa0\x4b\x25\xe5\x33\xf1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x30\xf4\x3f\xcf\x2b\x6b\x00\xde\x53\xf6\x24\xf1\x54\x30\x90\x68\x18\x39\x71\x7d\x53\xc7\xc9\x55\xd1\xd6\x9e\xfa\xf0\x34\x9b\x73\x63\xac\xb4\x47\x24\x01\x01\xcb\xb3\xaf\x66\x41\xce\x4b\x88\xe0"[..],
                                                  &b"\x25\xe4\x6c\x0c\x54\xf0\x16\x2a\x77\xef\xcc\x27\xb6\xea\x79\x20\x02\xae\x2b\xa8\x27\x14\x29\x9c\x86\x08\x57\xa6\x81\x53\xab\x62\xe5\x25\xec\x05\x30\xd8\x1b\x5a\xa1\x58\x97\x98\x1e\x85\x87\x57"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x92\x86\x0c\x21\xbd\xe0\x61\x65\xf8\xe9\x00\xc6\x87\xf8\xef\x0a\x05\xd1\x4f\x29\x0b\x3f\x07\xd8\xb3\xa8\xcc\x64\x04\x36\x6e\x5d\x51\x19\xcd\x6d\x03\xfb\x12\xdc\x58\xe8\x9f\x13\xdf\x9c\xd7\x83"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xea\x40\x18\xf5\xa3\x07\xc3\x79\x18\x0b\xf6\xa6\x2f\xd2\xce\xce\xeb\xee\xb7\xd4\xdf\x06\x3a\x66\xfb\x83\x8a\xa3\x52\x43\x41\x97\x91\xf7\xe2\xc9\xd4\x80\x3c\x93\x19\xaa\x0e\xb0\x3c\x41\x6b\x66"[..],
                                                &b"\x68\x83\x5a\x91\x48\x4f\x05\xef\x02\x82\x84\xdf\x64\x36\xfb\x88\xff\xeb\xab\xcd\xd6\x9a\xb0\x13\x3e\x67\x35\xa1\xbc\xfb\x37\x20\x3d\x10\xd3\x40\xa8\x32\x8a\x7b\x68\x77\x0c\xa7\x58\x78\xa1\xa6"[..]).unwrap();
            let z = &b"\xa2\x37\x42\xa2\xc2\x67\xd7\x42\x5f\xda\x94\xb9\x3f\x93\xbb\xcc\x24\x79\x1a\xc5\x1c\xd8\xfd\x50\x1a\x23\x8d\x40\x81\x2f\x4c\xbf\xc5\x9a\xac\x95\x20\xd7\x58\xcf\x78\x9c\x76\x30\x0c\x69\xd2\xff"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x1a\xef\xbf\xa2\xc6\xc8\xc8\x55\xa1\xa2\x16\x77\x45\x50\xb7\x9a\x24\xcd\xa3\x76\x07\xbb\x1f\x7c\xc9\x06\x65\x0e\xe4\xb3\x81\x6d\x68\xf6\xa9\xc7\x5d\xa6\xe4\x24\x2c\xeb\xfb\x66\x52\xf6\x51\x80"[..],
                                                  &b"\x41\x9d\x28\xb7\x23\xeb\xad\xb7\x65\x8f\xce\xbb\x9a\xd9\xb7\xad\xea\x67\x4f\x1d\xa3\xdc\x6b\x63\x97\xb5\x5d\xa0\xf6\x1a\x3e\xdd\xac\xb4\xac\xdb\x14\x44\x1c\xb2\x14\xb0\x4a\x08\x44\xc0\x2f\xa3"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x12\xcf\x6a\x22\x3a\x72\x35\x25\x43\x83\x0f\x3f\x18\x53\x0d\x5c\xb3\x7f\x26\x88\x0a\x0b\x29\x44\x82\xc8\xa8\xef\x8a\xfa\xd0\x9a\xa7\x8b\x7d\xc2\xf2\x78\x9a\x78\xc6\x6a\xf5\xd1\xcc\x55\x38\x53"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xfc\xfc\xea\x08\x5e\x8c\xf7\x4d\x0d\xce\xd1\x62\x0b\xa8\x42\x36\x94\xf9\x03\xa2\x19\xbb\xf9\x01\xb0\xb5\x9d\x6a\xc8\x1b\xaa\xd3\x16\xa2\x42\xba\x32\xbd\xe8\x5c\xb2\x48\x11\x9b\x85\x2f\xab\x66"[..],
                                                &b"\x97\x2e\x3c\x68\xc7\xab\x40\x2c\x58\x36\xf2\xa1\x6e\xd4\x51\xa3\x31\x20\xa7\x75\x0a\x60\x39\xf3\xff\x15\x38\x8e\xe6\x22\xb7\x06\x5f\x71\x22\xbf\x6d\x51\xae\xfb\xc2\x9b\x37\xb0\x34\x04\x58\x1b"[..]).unwrap();
            let z = &b"\x3d\x2e\x64\x0f\x35\x08\x05\xee\xd1\xff\x43\xb4\x0a\x72\xb2\xab\xed\x0a\x51\x8b\xce\xbe\x8f\x2d\x15\xb1\x11\xb6\x77\x32\x23\xda\x3c\x34\x89\x12\x1d\xb1\x73\xd4\x14\xb5\xbd\x5a\xd7\x15\x34\x35"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x8b\xc0\x89\x32\x6e\xc5\x5b\x9c\xf5\x9b\x34\xf0\xeb\x75\x4d\x93\x59\x6c\xa2\x90\xfc\xb3\x44\x4c\x83\xd4\xde\x3a\x56\x07\x03\x7e\xc3\x97\x68\x3f\x8c\xef\x07\xea\xb2\xfe\x35\x7e\xae\x36\xc4\x49"[..],
                                                  &b"\xd9\xd1\x6c\xe8\xac\x85\xb3\xf1\xe9\x45\x68\x52\x1a\xae\x53\x4e\x67\x13\x9e\x31\x0e\xc7\x26\x93\x52\x6a\xa2\xe9\x27\xb5\xb3\x22\xc9\x5a\x1a\x03\x3c\x22\x9c\xb6\x77\x0c\x95\x7c\xd3\x14\x8d\xd7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x8d\xd4\x80\x63\xa3\xa0\x58\xc3\x34\xb5\xcc\x7a\x4c\xe0\x7d\x02\xe5\xee\x6d\x8f\x1f\x3c\x51\xa1\x60\x09\x62\xcb\xab\x46\x26\x90\xae\x3c\xd9\x74\xfb\x39\xe4\x0b\x0e\x84\x3d\xaa\x0f\xd3\x2d\xe1"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xe3\x8c\x98\x46\x24\x81\x23\xc3\x42\x18\x61\xea\x4d\x32\x66\x9a\x7b\x5c\x3c\x08\x37\x6a\xd2\x81\x04\x39\x94\x94\xc8\x4f\xf5\xef\xa3\x89\x4a\xdb\x2c\x6c\xbe\x8c\x3c\x91\x3e\xf2\xee\xc5\xbd\x3c"[..],
                                                &b"\x9f\xa8\x40\x24\xa1\x02\x87\x96\xdf\x84\x02\x1f\x7b\x6c\x9d\x02\xf0\xf4\xbd\x1a\x61\x2a\x03\xcb\xf7\x5a\x0b\xee\xa4\x3f\xef\x8a\xe8\x4b\x48\xc6\x01\x72\xaa\xdf\x09\xc1\xad\x01\x6d\x0b\xf3\xce"[..]).unwrap();
            let z = &b"\x6a\x42\xcf\xc3\x92\xab\xa0\xbf\xd3\xd1\x7b\x7c\xcf\x06\x2b\x91\xfc\x09\xbb\xf3\x41\x76\x12\xd0\x2a\x90\xbd\xde\x62\xae\x40\xc5\x4b\xb2\xe5\x6e\x16\x7d\x6b\x70\xdb\x67\x00\x97\xeb\x8d\xb8\x54"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xeb\x95\x2e\x2d\x9a\xc0\xc2\x0c\x6c\xc4\x8f\xb2\x25\xc2\xad\x15\x4f\x53\xc8\x75\x0b\x00\x3f\xd3\xb4\xed\x8e\xd1\xdc\x0d\xef\xac\x61\xbc\xdd\xe0\x2a\x2b\xcf\xee\x70\x67\xd7\x5d\x34\x2e\xd2\xb0"[..],
                                                  &b"\xf1\x82\x82\x05\xba\xec\xe8\x2d\x1b\x26\x7d\x0d\x7f\xf2\xf9\xc9\xe1\x5b\x69\xa7\x2d\xf4\x70\x58\xa9\x7f\x38\x91\x00\x5d\x1f\xb3\x88\x58\xf5\x60\x3d\xe8\x40\xe5\x91\xdf\xa4\xf6\xe7\xd4\x89\xe1"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x84\xec\xe6\xcc\x34\x29\x30\x9b\xd5\xb2\x3e\x95\x97\x93\xed\x2b\x11\x1e\xc5\xcb\x43\xb6\xc1\x80\x85\xfc\xae\xa9\xef\xa0\x68\x5d\x98\xa6\x26\x2e\xe0\xd3\x30\xee\x25\x0b\xc8\xa6\x7d\x0e\x73\x3f"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x32\x22\x06\x3a\x29\x97\xb3\x02\xee\x60\xee\x19\x61\x10\x8f\xf4\xc7\xac\xf1\xc0\xef\x1d\x5f\xb0\xd1\x64\xb8\x4b\xce\x71\xc4\x31\x70\x5c\xb9\xae\xa9\xa4\x5f\x5d\x73\x80\x66\x55\xa0\x58\xbe\xe3"[..],
                                                &b"\xe6\x1f\xa9\xe7\xfb\xe7\xcd\x43\xab\xf9\x95\x96\xa3\xd3\xa0\x39\xe9\x9f\xa9\xdc\x93\xb0\xbd\xd9\xca\xd8\x19\x66\xd1\x7e\xea\xf5\x57\x06\x8a\xfa\x7c\x78\x46\x6b\xb5\xb2\x20\x32\xd1\x10\x0f\xa6"[..]).unwrap();
            let z = &b"\xce\x7b\xa4\x54\xd4\x41\x27\x29\xa3\x2b\xb8\x33\xa2\xd1\xfd\x2a\xe6\x12\xd4\x66\x7c\x3a\x90\x0e\x06\x92\x14\x81\x86\x13\x44\x7d\xf8\xc6\x11\xde\x66\xda\x20\x0d\xb7\xc3\x75\xcf\x91\x3e\x44\x05"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x44\x1d\x02\x9e\x24\x4e\xb7\x16\x8d\x64\x7d\x4d\xf5\x0d\xb5\xf4\xe4\x97\x4a\xb3\xfd\xaf\x02\x2a\xff\x05\x8b\x36\x95\xd0\xb8\xc8\x14\xcc\x88\xda\x62\x85\xdc\x6d\xf1\xac\x55\xc5\x53\x88\x50\x03"[..],
                                                  &b"\xe8\x02\x5a\xc2\x3a\x41\xd4\xb1\xea\x2a\xa4\x6c\x50\xc6\xe4\x79\x94\x6b\x59\xb6\xd7\x64\x97\xcd\x92\x49\x97\x7e\x0b\xfe\x4a\x62\x62\x62\x2f\x13\xd4\x2a\x3c\x43\xd6\x6b\xdb\xb3\x04\x03\xc3\x45"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x68\xfc\xe2\x12\x1d\xc3\xa1\xe3\x7b\x10\xf1\xdd\xe3\x09\xf9\xe2\xe1\x8f\xac\x47\xcd\x17\x70\x95\x14\x51\xc3\x48\x4c\xdb\x77\xcb\x13\x6d\x00\xe7\x31\x26\x05\x97\xcc\x28\x59\x60\x1c\x01\xa2\x5b"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x86\x8b\xe0\xe6\x94\x84\x18\x30\xe4\x24\xd9\x13\xd8\xe7\xd8\x6b\x84\xee\x10\x21\xd8\x2b\x0e\xcf\x52\x3f\x09\xfe\x89\xa7\x6c\x0c\x95\xc4\x9f\x2d\xfb\xcf\x82\x9c\x1e\x39\x70\x9d\x55\xef\xbb\x3b"[..],
                                                &b"\x91\x95\xeb\x18\x36\x75\xb4\x0f\xd9\x2f\x51\xf3\x77\x13\x31\x7e\x4a\x9b\x4f\x71\x5c\x8a\xb2\x2e\x07\x73\xb1\xbc\x71\xd3\xa2\x19\xf0\x5b\x81\x16\x07\x46\x58\xee\x86\xb5\x2e\x36\xf3\x89\x71\x16"[..]).unwrap();
            let z = &b"\xba\x69\xf0\xac\xdf\x3e\x1c\xa9\x5c\xaa\xac\x4e\xca\xf4\x75\xbb\xe5\x1b\x54\x77\x7e\xfc\xe0\x1c\xa3\x81\xf4\x53\x70\xe4\x86\xfe\x87\xf9\xf4\x19\xb1\x50\xc6\x1e\x32\x9a\x28\x6d\x1a\xa2\x65\xec"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x3d\x4e\x6b\xf0\x8a\x73\x40\x4a\xcc\xc1\x62\x98\x73\x46\x8e\x42\x69\xe8\x2d\x90\xd8\x32\xe5\x8a\xd7\x21\x42\x63\x9b\x5a\x05\x6a\xd8\xd3\x5c\x66\xc6\x0e\x81\x49\xfa\xc0\xc7\x97\xbc\xeb\x7c\x2f"[..],
                                                  &b"\x9b\x03\x08\xdc\x7f\x0e\x6d\x29\xf8\xc2\x77\xac\xbc\x65\xa2\x1e\x5a\xdb\x83\xd1\x1e\x68\x73\xbc\x0a\x07\xfd\xa0\x99\x7f\x48\x25\x04\x60\x2f\x59\xe1\x0b\xc5\xcb\x47\x6b\x83\xd0\xa4\xf7\x5e\x71"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xb1\x76\x4c\x54\x89\x7e\x7a\xae\x6d\xe9\xe7\x75\x1f\x2f\x37\xde\x84\x92\x91\xf8\x8f\x0f\x91\x09\x31\x55\xb8\x58\xd1\xcc\x32\xa3\xa8\x79\x80\xf7\x06\xb8\x6c\xc8\x3f\x92\x7b\xdf\xdb\xea\xe0\xbd"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xc3\x71\x22\x2f\xea\xa6\x77\x0c\x6f\x3e\xa3\xe0\xda\xc9\x74\x0d\xef\x4f\xcf\x82\x13\x78\xb7\xf9\x1f\xf9\x37\xc2\x1e\x04\x70\xf7\x0f\x3a\x31\xd5\xc6\xb2\x91\x21\x95\xf1\x09\x26\x94\x2b\x48\xae"[..],
                                                &b"\x04\x7d\x6b\x4d\x76\x51\x23\x56\x3f\x81\x11\x6b\xc6\x65\xb7\xb8\xcc\x62\x07\x83\x0d\x80\x5f\xd8\x4d\xa7\xcb\x80\x5a\x65\xba\xa7\xc1\x2f\xd5\x92\xd1\xb5\xb5\xe3\xe6\x5d\x96\x72\xa9\xef\x76\x62"[..]).unwrap();
            let z = &b"\x1a\x66\x88\xee\x1d\x6e\x59\x86\x5d\x8e\x3a\xda\x37\x78\x1d\x36\xbb\x0c\x27\x17\xee\xf9\x2e\x61\x96\x4d\x39\x27\xcb\x76\x5c\x29\x65\xea\x80\xf7\xf6\x3e\x58\xc3\x22\xba\x03\x97\xfa\xea\xf6\x2b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xf5\xf6\xbe\xf1\xd1\x10\xda\x03\xbe\x00\x17\xea\xc7\x60\xcc\x34\xb2\x4d\x09\x2f\x73\x6f\x23\x7b\xc7\x05\x4b\x38\x65\x31\x2a\x81\x3b\xcb\x62\xd2\x97\xfb\x10\xa4\xf7\xab\xf5\x47\x08\xfe\x2d\x3d"[..],
                                                  &b"\x06\xfd\xf8\xd7\xdc\x03\x2f\x4e\x10\x01\x0b\xf1\x9c\xbf\x61\x59\x32\x12\x52\xff\x41\x5f\xb9\x19\x20\xd4\x38\xf2\x4e\x67\xe6\x0c\x2e\xb0\x46\x32\x04\x67\x9f\xa3\x56\xaf\x44\xce\xa9\xc9\xeb\xf5"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xf0\xf7\xa9\x6e\x70\xd9\x8f\xd5\xa3\x0a\xd6\x40\x6c\xf5\x6e\xb5\xb7\x2a\x51\x0e\x9f\x19\x2f\x50\xe1\xf8\x45\x24\xdb\xf3\xd2\x43\x9f\x72\x87\xbb\x36\xf5\xaa\x91\x2a\x79\xde\xaa\xb4\xad\xea\x82"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x99\xc8\xc4\x1c\xb1\xab\x5e\x08\x54\xa3\x46\xe4\xb0\x8a\x53\x7c\x17\x06\xa6\x15\x53\x38\x7c\x8d\x94\x94\x3a\xb1\x51\x96\xd4\x0d\xba\xa5\x5b\x82\x10\xa7\x7a\x5d\x00\x91\x5f\x2c\x4e\xa6\x9e\xab"[..],
                                                &b"\x55\x31\x06\x5b\xdc\xf1\x7b\xfb\x3c\xb5\x5a\x02\xe4\x1a\x57\xc7\xf6\x94\xc3\x83\xad\x28\x9f\x90\x0f\xbd\x65\x6c\x22\x33\xa9\x3c\x92\xe9\x33\xe7\xa2\x6f\x54\xcb\xb5\x6f\x0a\xd8\x75\xc5\x1b\xb0"[..]).unwrap();
            let z = &b"\xd0\x6a\x56\x8b\xf2\x33\x6b\x90\xcb\xac\x32\x51\x61\xbe\x76\x95\xea\xcb\x22\x95\xf5\x99\x50\x0d\x78\x7f\x07\x26\x12\xac\xa3\x13\xee\x5d\x87\x4f\x80\x7d\xde\xf6\xc1\xf0\x23\xfe\x2b\x6e\x7c\xd0"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x7c\xde\xc7\x7e\x07\x37\xea\x37\xc6\x7b\x89\xb7\x13\x7f\xe3\x88\x18\x01\x0f\x44\x64\x43\x8e\xe4\xd1\xd3\x5a\x0c\x48\x8c\xad\x3f\xde\x2f\x37\xd0\x08\x85\xd3\x6d\x3b\x79\x5b\x9f\x93\xd2\x3a\x67"[..],
                                                  &b"\x28\xc4\x2e\xe8\xd6\x02\x7c\x56\xcf\x97\x9b\xa4\xc2\x29\xfd\xb0\x1d\x23\x49\x44\xf8\xac\x43\x36\x50\x11\x2c\x3c\xf0\xf0\x28\x44\xe8\x88\xa3\x56\x9d\xfe\xf7\x82\x8a\x8a\x88\x45\x89\xaa\x05\x5e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x9e\xfb\x87\xdd\xc6\x1d\x43\xc4\x82\xba\x66\xe1\xb1\x43\xae\xf6\x78\xfb\xd0\xd1\xbe\xbc\x20\x00\x94\x1f\xab\xe6\x77\xfe\x5b\x70\x6b\xf7\x8f\xce\x36\xd1\x00\xb1\x7c\xc7\x87\xea\xd7\x4b\xbc\xa2"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x4c\x34\xef\xee\x8f\x0c\x95\x56\x5d\x20\x65\xd1\xbb\xac\x2a\x2d\xd2\x5a\xe9\x64\x32\x0e\xb6\xbc\xce\xdc\x5f\x3a\x9b\x42\xa8\x81\xa1\xaf\xca\x1b\xb6\xb8\x80\x58\x4f\xa2\x7b\x01\xc1\x93\xcd\x92"[..],
                                                &b"\xd8\xfb\x01\xdb\xf7\xcd\x0a\x38\x68\xc2\x6b\x95\x1f\x39\x3c\x3c\x56\xc2\x85\x8c\xee\x90\x1f\x77\x93\xff\x5d\x27\x19\x25\xd1\x3a\x41\xf8\xe5\x24\x09\xf4\xeb\xa1\x99\x0f\x33\xac\xb0\xba\xc6\x69"[..]).unwrap();
            let z = &b"\xbb\x3b\x1e\xda\x9c\x65\x60\xd8\x2f\xf5\xbe\xe4\x03\x33\x9f\x1e\x80\x34\x23\x38\xa9\x91\x34\x48\x53\xb5\x6b\x24\xf1\x09\xa4\xd9\x4b\x92\xf6\x54\xf0\x42\x5e\xdd\x4c\x20\x59\x03\xd7\x58\x61\x04"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x8e\xee\xa3\xa3\x19\xc8\xdf\x99\xfb\xc2\x9c\xb5\x5f\x24\x3a\x72\x0d\x95\x50\x95\x15\xee\x5c\xc5\x87\xa5\xc5\xae\x22\xfb\xbd\x00\x9e\x62\x6d\xb3\xe9\x11\xde\xf0\xb9\x9a\x4f\x7a\xe3\x04\xb1\xba"[..],
                                                  &b"\x73\x87\x7d\xc9\x4d\xb9\xad\xdd\xc0\xd9\xa4\xb2\x4e\x89\x76\xc2\x2d\x73\xc8\x44\x37\x0e\x1e\xe8\x57\xf8\xd1\xb1\x29\xa3\xbd\x5f\x63\xf4\x0c\xaf\x3b\xd0\x53\x3e\x38\xa5\xf5\x77\x70\x74\xff\x9e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xd7\x87\xa5\x7f\xde\x22\xec\x65\x6a\x0a\x52\x5c\xf3\xc7\x38\xb3\x0d\x73\xaf\x61\xe7\x43\xea\x90\x89\x3e\xcb\x2d\x7b\x62\x2a\xdd\x2f\x94\xee\x25\xc2\x17\x14\x67\xaf\xb0\x93\xf3\xf8\x4d\x00\x18"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x17\x15\x46\x92\x3b\x87\xb2\xcb\xba\xd6\x64\xf0\x1c\xe9\x32\xbf\x09\xd6\xa6\x11\x81\x68\x67\x84\x46\xbf\xa9\xf0\x93\x86\x08\xcb\x46\x67\xa9\x8f\x4e\xc8\xac\x14\x62\x28\x5c\x25\x08\xf7\x48\x62"[..],
                                                &b"\xfa\x41\xcb\x4d\xb6\x8a\xe7\x1f\x1f\x8a\x3e\x89\x39\xdc\x52\xc2\xde\xc6\x1a\x83\xc9\x83\xbe\xb2\xa0\x2b\xaf\x29\xec\x49\x27\x80\x88\x88\x2e\xd0\xcf\x56\xc7\x4b\x5c\x17\x3b\x55\x2c\xcf\x63\xcf"[..]).unwrap();
            let z = &b"\x1e\x97\xb6\x0a\xdd\x7c\xb3\x5c\x74\x03\xdd\x88\x4c\x0a\x75\x79\x5b\x76\x83\xff\xf8\xb4\x9f\x9d\x86\x72\xa8\x20\x6b\xfd\xcf\x0a\x10\x6b\x87\x68\xf9\x83\x25\x8c\x74\x16\x74\x22\xe4\x4e\x4d\x14"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xa7\x21\xf6\xa2\xd4\x52\x74\x11\x83\x4b\x13\xd4\xd3\xa3\x3c\x29\xbe\xb8\x3a\xb7\x68\x24\x65\xc6\xcb\xaf\x66\x24\xac\xa6\xea\x58\xc3\x0e\xb0\xf2\x9d\xd8\x42\x88\x66\x95\x40\x0d\x72\x54\xf2\x0f"[..],
                                                  &b"\x14\xba\x6e\x26\x35\x51\x09\xad\x35\x12\x93\x66\xd5\xe3\xa6\x40\xae\x79\x85\x05\xa7\xfa\x55\xa9\x6a\x36\xb5\xda\xd3\x3d\xe0\x04\x74\xf6\x67\x0f\x52\x22\x14\xdd\x79\x52\x14\x0a\xb0\xa7\xeb\x68"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x83\xd7\x0f\x7b\x16\x4d\x9f\x4c\x22\x7c\x76\x70\x46\xb2\x0e\xb3\x4d\xfc\x77\x8f\x53\x87\xe3\x2e\x83\x4b\x1e\x6d\xae\xc2\x0e\xdb\x8c\xa5\xbb\x41\x92\x09\x3f\x54\x3b\x68\xe6\xae\xb7\xce\x78\x8b"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x57\xcd\x77\x0f\x3b\xbc\xbe\x0c\x78\xc7\x70\xea\xb0\xb1\x69\xbc\x45\xe1\x39\xf8\x63\x78\xff\xae\x1c\x2b\x16\x96\x67\x27\xc2\xf2\xeb\x72\x45\x72\xb8\xf3\xeb\x22\x8d\x13\x0d\xb4\xff\x86\x2c\x63"[..],
                                                &b"\x7e\xc5\xc8\x81\x3b\x68\x55\x58\xd8\x3e\x92\x4f\x14\xbc\x71\x9f\x6e\xb7\xae\x0c\xbb\x2c\x47\x42\x27\xc5\xbd\xa8\x86\x37\xa4\xf2\x6c\x64\x81\x79\x29\xaf\x99\x95\x92\xda\x6f\x78\x74\x90\x33\x2f"[..]).unwrap();
            let z = &b"\x10\x23\x47\x88\x40\xe5\x47\x75\xbf\xc6\x92\x93\xa3\xcf\x97\xf5\xbc\x91\x47\x26\x45\x5c\x66\x53\x8e\xb5\x62\x3e\x21\x8f\xee\xf7\xdf\x4b\xef\xa2\x3e\x09\xd7\x71\x45\xad\x57\x7d\xb3\x2b\x41\xf9"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xd8\x82\xa8\x50\x5c\x2d\x5c\xb9\xb8\x85\x1f\xc6\x76\x67\x7b\xb0\x08\x76\x81\xad\x53\xfa\xce\xba\x17\x38\x28\x6b\x45\x82\x75\x61\xe7\xda\x37\xb8\x80\x27\x6c\x65\x6c\xfc\x38\xb3\x2a\xde\x84\x7e"[..],
                                                  &b"\x34\xb3\x14\xbd\xc1\x34\x57\x56\x54\x57\x3c\xff\xaf\x40\x44\x5d\xa2\xe6\xaa\xf9\x87\xf7\xe9\x13\xcd\x4c\x30\x91\x52\x30\x58\x98\x4a\x25\xd8\xf2\x1d\xa8\x32\x61\x92\x45\x6c\x6a\x0f\xa5\xf6\x0c"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x8f\x55\x8e\x05\x81\x8b\x88\xed\x38\x3d\x5f\xca\x96\x2e\x53\x41\x3d\xb1\xa0\xe4\x63\x7e\xda\x19\x4f\x76\x19\x44\xcb\xea\x11\x4a\xb9\xd5\xda\x17\x5a\x7d\x57\x88\x25\x50\xb0\xe4\x32\xf3\x95\xa9"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x9a\x2f\x57\xf4\x86\x7c\xe7\x53\xd7\x2b\x0d\x95\x19\x5d\xf6\xf9\x6c\x1f\xae\x93\x4f\x60\x2e\xfd\x7b\x6a\x54\x58\x2f\x55\x6c\xfa\x53\x9d\x89\x00\x5c\xa2\xed\xac\x08\xad\x9b\x72\xdd\x1f\x60\xba"[..],
                                                &b"\xd9\xb9\x4e\xe8\x2d\xa9\xcc\x60\x1f\x34\x60\x44\x99\x8b\xa3\x87\xae\xe5\x64\x04\xdc\x6e\xcc\x8a\xb2\xb5\x90\x44\x33\x19\xd0\xb2\xb6\x17\x6f\x9d\x0e\xac\x2d\x44\x67\x8e\xd5\x61\x60\x7d\x09\xa9"[..]).unwrap();
            let z = &b"\x6a\xd6\xb9\xdc\x8a\x6c\xf0\xd3\x69\x1c\x50\x1c\xbb\x96\x78\x67\xf6\xe4\xbb\xb7\x64\xb6\x0d\xbf\xf8\xfc\xff\x3e\xd4\x2d\xbb\xa3\x9d\x63\xcf\x32\x5b\x4b\x40\x78\x85\x84\x95\xdd\xee\x75\xf9\x54"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x81\x5c\x9d\x77\x3d\xbf\x5f\xb6\xa1\xb8\x67\x99\x96\x62\x47\xf4\x00\x6a\x23\xc9\x2e\x68\xc5\x5e\x9e\xaa\x99\x8b\x17\xd8\x83\x2d\xd4\xd8\x4d\x92\x7d\x83\x1d\x4f\x68\xda\xc6\x7c\x64\x88\x21\x9f"[..],
                                                  &b"\xe7\x92\x69\x94\x8b\x26\x11\x48\x45\x60\xfd\x49\x0f\xee\xc8\x87\xcb\x55\xef\x99\xa4\xb5\x24\x88\x0f\xa7\x49\x9d\x6a\x07\x28\x3a\xae\x2a\xfa\x33\xfe\xab\x97\xde\xca\x40\xbc\x60\x6c\x4d\x87\x64"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x0f\x5d\xee\x0a\xff\xa7\xbb\xf2\x39\xd5\xdf\xf3\x29\x87\xeb\xb7\xcf\x84\xfc\xce\xed\x64\x3e\x1d\x3c\x62\xd0\xb3\x35\x2a\xec\x23\xb6\xe5\xac\x7f\xa4\x10\x5c\x8c\xb2\x61\x26\xad\x2d\x18\x92\xcb"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x23\x34\x6b\xdf\xbc\x9d\x7c\x7c\x73\x6e\x02\xbd\xf6\x07\x67\x1f\xf6\x08\x2f\xdd\x27\x33\x4a\x8b\xc7\x5f\x3b\x23\x68\x1e\xbe\x61\x4d\x05\x97\xdd\x61\x4f\xae\x58\x67\x7c\x83\x5a\x9f\x0b\x27\x3b"[..],
                                                &b"\x82\xba\x36\x29\x0d\x2f\x94\xdb\x41\x47\x9e\xb4\x5a\xb4\xea\xf6\x79\x28\xa2\x31\x51\x38\xd5\x9e\xec\xc9\xb5\x28\x5d\xfd\xdd\x67\x14\xf7\x75\x57\x21\x6e\xa4\x4c\xc6\xfc\x11\x9d\x82\x43\xef\xaf"[..]).unwrap();
            let z = &b"\xcc\x9e\x06\x35\x66\xd4\x6b\x35\x7b\x3f\xca\xe2\x18\x27\x37\x73\x31\xe5\xe2\x90\xa3\x6e\x60\xcd\x7c\x39\x10\x2b\x82\x8a\xe0\xb9\x18\xdc\x5a\x02\x21\x6b\x07\xfe\x6f\x19\x58\xd8\x34\xe4\x24\x37"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x1c\x0e\xed\xa7\xa2\xbe\x00\x0c\x5b\xdc\xda\x04\x78\xae\xd4\xdb\x73\x3d\x2a\x9e\x34\x12\x24\x37\x91\x23\xad\x84\x70\x30\xf2\x9e\x3b\x16\x8f\xa1\x8e\x89\xa3\xc0\xfb\xa2\xa6\xce\x1c\x28\xfc\x3b"[..],
                                                  &b"\xec\x8c\x1c\x83\xc1\x18\xc4\xdb\xea\x94\x27\x18\x69\xf2\xd8\x68\xeb\x65\xe8\xb4\x4e\x21\xe6\xf1\x4b\x0f\x4d\x9b\x38\xc0\x68\xda\xef\xa2\x71\x14\x25\x5b\x9a\x41\xd0\x84\xcc\x4a\x1a\xd8\x54\x56"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x03\x7b\x63\x3b\x5b\x8b\xa8\x57\xc0\xfc\x85\x65\x68\x68\x23\x2e\x2f\xeb\xf5\x95\x78\x71\x83\x91\xb8\x1d\xa8\x54\x1a\x00\xbf\xe5\x3c\x30\xae\x04\x15\x18\x47\xf2\x74\x99\xf8\xd7\xab\xad\x8c\xf4"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x88\x78\xac\x8a\x94\x7f\x7d\x5c\xb2\xb4\x7a\xad\x24\xfb\xb8\x21\x0d\x86\x12\x65\x85\x39\x9a\x28\x71\xf8\x4a\xa9\xc5\xfd\xe3\x07\x4a\xe5\x40\xc6\xbf\x82\x27\x5c\xa8\x22\xd0\xfe\xb8\x62\xbc\x74"[..],
                                                &b"\x63\x2f\x5c\xd2\xf9\x00\xc2\x71\x1c\x32\xf8\x93\x07\x28\xeb\x64\x7d\x31\xed\xd8\xd6\x50\xf9\x65\x4e\x7d\x33\xe5\xed\x1b\x47\x54\x89\xd0\x8d\xaa\x30\xd8\xcb\xcb\xa6\xbf\xc3\xb6\x0d\x9b\x5a\x37"[..]).unwrap();
            let z = &b"\xde\xff\x7f\x03\xbd\x09\x86\x5b\xaf\x94\x5e\x73\xed\xff\x6d\x51\x22\xc0\x3f\xb5\x61\xdb\x87\xde\xc8\x66\x2e\x09\xbe\xd4\x34\x0b\x28\xa9\xef\xe1\x18\x33\x7b\xb7\xd3\xd4\xf7\xf5\x68\x63\x5f\xf9"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xc9\x5c\x18\x5e\x25\x6b\xf9\x97\xf3\x0b\x31\x15\x48\xae\x7f\x76\x8a\x38\xde\xe4\x3e\xee\xef\x43\x08\x3f\x30\x77\xbe\x70\xe2\xbf\x39\xac\x1d\x4d\xaf\x36\x0c\x51\x4c\x8c\x6b\xe6\x23\x44\x3d\x1a"[..],
                                                  &b"\x3e\x63\xa6\x63\xea\xf7\x5d\x8a\x76\x5a\xb2\xb9\xa3\x55\x13\xd7\x93\x3f\xa5\xe2\x64\x20\xa5\x24\x45\x50\xec\x6c\x3b\x6f\x03\x3b\x96\xdb\x2a\xca\x3d\x6a\xc6\xaa\xb0\x52\xce\x92\x95\x95\xae\xa5"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xe3\xd0\x71\x06\xbe\xdc\xc0\x96\xe7\xd9\x16\x30\xff\xd3\x09\x4d\xf2\xc7\x85\x9d\xb8\xd7\xed\xbb\x2e\x37\xb4\xac\x47\xf4\x29\xa6\x37\xd0\x6a\x67\xd2\xfb\xa3\x38\x38\x76\x4e\xf2\x03\x46\x49\x91"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xe7\x4a\x1a\x2b\x85\xf1\xcb\xf8\xdb\xbd\xf0\x50\xcf\x1a\xff\x8a\xcb\x02\xfd\xa2\xfb\x65\x91\xf9\xd3\xcf\xe4\xe7\x9d\x0a\xe9\x38\xa9\xc1\x48\x3e\x7b\x75\xf8\xdb\x24\x50\x5d\x65\x06\x5c\xdb\x18"[..],
                                                &b"\x17\x73\xee\x59\x18\x22\xf7\xab\xaa\x85\x6a\x1a\x60\xbc\x0a\x52\x03\x54\x8d\xbd\x1c\xb5\x02\x54\x66\xef\xf8\x48\x1b\xd0\x76\x14\xea\xa0\x4a\x16\xc3\xdb\x76\x90\x59\x13\xe9\x72\xa5\xb6\xb5\x9d"[..]).unwrap();
            let z = &b"\xc8\xb1\x03\x8f\x73\x5a\xd3\xbb\x3e\x46\x37\xc3\xe4\x7e\xab\x48\x76\x37\x91\x1a\x6b\x79\x50\xa4\xe4\x61\x94\x83\x29\xd3\x92\x3b\x96\x9e\x5d\xb6\x63\x67\x56\x23\x61\x1a\x45\x7f\xcd\xa3\x5a\x71"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x34\x97\x23\x8a\x7e\x6a\xd1\x66\xdf\x2d\xac\x03\x9a\xa4\xda\xc8\xd1\x7a\xa9\x25\xe7\xc7\x63\x1e\xb3\xb5\x6e\x3a\xaa\x1c\x54\x5f\xcd\x54\xd2\xe5\x98\x58\x07\x91\x0f\xb2\x02\xb1\xfc\x19\x1d\x2a"[..],
                                                  &b"\xa4\x9e\x5c\x48\x7d\xcc\x7a\xa4\x0a\x8f\x23\x4c\x97\x94\x46\x04\x0d\x91\x74\xe3\xad\x35\x7d\x40\x4d\x77\x65\x18\x31\x95\xae\xd3\xf9\x13\x64\x1b\x90\xc8\x1a\x30\x6e\xbf\x0d\x89\x13\x86\x13\x16"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xf3\xf9\xb0\xc6\x5a\x49\xa5\x06\x63\x2c\x8a\x45\xb1\x0f\x66\xb5\x31\x6f\x9e\xeb\x06\xfa\xe2\x18\xf2\xda\x62\x33\x3f\x99\x90\x51\x17\xb1\x41\xc7\x60\xe8\x97\x4e\xfc\x4a\xf1\x05\x70\x63\x57\x91"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xa4\xad\x77\xaa\x7d\x86\xe5\x36\x11\x18\xa6\xb9\x21\x71\x0c\x82\x07\x21\x21\x07\x12\xf4\xc3\x47\x98\x5f\xde\xe5\x8a\xa4\xef\xfa\x1e\x28\xbe\x80\xa1\x7b\x12\x0b\x13\x9f\x96\x30\x0f\x89\xb4\x9b"[..],
                                                &b"\x1d\xdf\x22\xe0\x7e\x03\xf1\x56\x0d\x8f\x45\xa4\x80\x09\x45\x60\xdb\xa9\xfa\xe7\xf9\x53\x11\x30\xc1\xb5\x7e\xbb\x95\x98\x24\x96\x52\x4f\x31\xd3\x79\x77\x93\x39\x6f\xa8\x23\xf2\x2b\xdb\x43\x28"[..]).unwrap();
            let z = &b"\xd3\x37\xea\xa3\x2b\x9f\x71\x6b\x87\x47\xb0\x05\xb9\x7a\x55\x3c\x59\xda\xb0\xc5\x1d\xf4\x1a\x2d\x49\x03\x9c\xda\xe7\x05\xaa\x75\xc7\xb9\xe7\xbc\x0b\x6a\x0e\x8c\x57\x8c\x90\x2b\xc4\xff\xf2\x3e"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x90\xa3\x47\x37\xd4\x5b\x1a\xa6\x5f\x74\xe0\xbd\x06\x59\xbc\x11\x8f\x8e\x4b\x77\x4b\x76\x19\x44\xff\xa6\x57\x3c\x6d\xf4\xf4\x1d\xec\x0d\x11\xb6\x97\xab\xd9\x34\xd3\x90\x87\x1d\x4b\x45\x32\x40"[..],
                                                  &b"\x9b\x59\x07\x19\xbb\x33\x07\xc1\x49\xa7\x81\x7b\xe3\x55\xd6\x84\x89\x3a\x30\x77\x64\xb5\x12\xee\xff\xe0\x7c\xb6\x99\xed\xb5\xa6\xff\xbf\x8d\x60\x32\xe6\xc7\x9d\x5e\x93\xe9\x42\x12\xc2\xaa\x4e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x59\xfc\xe7\xfa\xd7\xde\x28\xba\xc0\x23\x06\x90\xc9\x57\x10\xc7\x20\xe5\x28\xf9\xa4\xe5\x4d\x3a\x6a\x8c\xd5\xfc\x5c\x5f\x21\x63\x70\x31\xce\x1c\x5b\x4e\x3d\x39\x64\x7d\x8d\xcb\x9b\x79\x46\x64"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x9c\x43\xbf\x97\x1e\xdf\x09\x40\x28\x76\xee\x74\x20\x95\x38\x1f\x78\xb1\xbd\x3a\xa3\x9b\x51\x32\xaf\x75\xdb\xfe\x7e\x98\xbd\x78\xbd\xe1\x0f\xe2\xe9\x03\xc2\xb6\x37\x9e\x1d\xee\xe1\x75\xa1\xb0"[..],
                                                &b"\xa6\xc5\x8e\xce\xa5\xa4\x77\xbb\x01\xbd\x54\x3b\x33\x9f\x1c\xc4\x9f\x13\x71\xa2\xcd\xa4\xd4\x6e\xb4\xe5\x3e\x25\x05\x97\x94\x23\x51\xa9\x96\x65\xa1\x22\xff\xea\x9b\xde\x06\x36\xc3\x75\xda\xf2"[..]).unwrap();
            let z = &b"\x32\xd2\x92\xb6\x95\xa4\x48\x8e\x42\xa7\xb7\x92\x2e\x1a\xe5\x37\xd7\x6a\x3d\x21\xa0\xb2\xe3\x68\x75\xf6\x0e\x9f\x6d\x3e\x87\x79\xc2\xaf\xb3\xa4\x13\xb9\xdd\x79\xae\x18\xe7\x0b\x47\xd3\x37\xc1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xdd\xa5\x46\xac\xfc\x8f\x90\x3d\x11\xe2\xe3\x92\x06\x69\x63\x6d\x44\xb2\x06\x8a\xeb\x66\xff\x07\xaa\x26\x6f\x00\x30\xe1\x53\x5b\x0e\xd0\x20\x3c\xb8\xa4\x60\xac\x99\x0f\x13\x94\xfa\xf2\x2f\x1d"[..],
                                                  &b"\x15\xbb\xb2\x59\x79\x13\x03\x5f\xaa\xdf\x41\x34\x76\xf4\xc7\x0f\x72\x79\x76\x9a\x40\xc9\x86\xf4\x70\xc4\x27\xb4\xee\x49\x62\xab\xdf\x81\x73\xbb\xad\x81\x87\x47\x72\x92\x5f\xd3\x2f\x0b\x15\x9f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x3e\x49\xfb\xf9\x50\xa4\x24\xc5\xd8\x02\x28\xdc\x4b\xc3\x5e\x9f\x6c\x6c\x0c\x1d\x04\x44\x09\x98\xda\x0a\x60\x9a\x87\x75\x75\xdb\xe4\x37\xd6\xa5\xce\xda\xa2\xdd\xd2\xa1\xa1\x7f\xd1\x12\xad\xed"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x5a\x94\x95\x94\x22\x8b\x1a\x3d\x6f\x59\x9e\xb3\xdb\x0d\x06\x07\x0f\xbc\x55\x1c\x65\x7b\x58\x23\x4b\xa1\x64\xce\x3f\xe4\x15\xfa\x5f\x3e\xb8\x23\xc0\x8d\xc2\x9b\x8c\x34\x12\x19\xc7\x7b\x6b\x3d"[..],
                                                &b"\x2b\xaa\xd4\x47\xc8\xc2\x90\xcf\xed\x25\xed\xd9\x03\x1c\x41\xd0\xb7\x69\x21\x45\x73\x27\xf4\x2d\xb3\x11\x22\xb8\x1f\x33\x7b\xbf\x0b\x10\x39\xec\x83\x0c\xe9\x06\x1a\x37\x61\x95\x3c\x75\xe4\xa8"[..]).unwrap();
            let z = &b"\x12\x20\xe7\xe6\xca\xd7\xb2\x5d\xf9\x8e\x5b\xbd\xcc\x6c\x0b\x65\xca\x6c\x2a\x50\xc5\xff\x6c\x41\xdc\xa7\x1e\x47\x56\x46\xfd\x48\x96\x15\x97\x9c\xa9\x2f\xb4\x38\x9a\xea\xde\xfd\xe7\x9a\x24\xf1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x78\x8b\xe2\x33\x6c\x52\xf4\x45\x4d\x63\xee\x94\x4b\x1e\x49\xbf\xb6\x19\xa0\x83\x71\x04\x8e\x6d\xa9\x2e\x58\x4e\xae\x70\xbd\xe1\xf1\x71\xc4\xdf\x37\x8b\xd1\xf3\xc0\xab\x03\x04\x8a\x23\x78\x02"[..],
                                                  &b"\x46\x73\xeb\xd8\xdb\x60\x4e\xaf\x41\x71\x17\x48\xba\xb2\x96\x8a\x23\xca\x44\x76\xce\x14\x4e\x72\x82\x47\xf0\x8a\xf7\x52\x92\x91\x57\xb5\x83\x0f\x1e\x26\x06\x74\x66\xbd\xfa\x8b\x65\x14\x5a\x33"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x50\xcc\xc1\xf7\x07\x6e\x92\xf4\x63\x8e\x85\xf2\xdb\x98\xe0\xb4\x83\xe6\xe2\x20\x4c\x92\xbd\xd4\x40\xa6\xde\xea\x04\xe3\x7a\x07\xc6\xe7\x27\x91\xc1\x90\xad\x4e\x4e\x86\xe0\x1e\xfb\xa8\x42\x69"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x75\x6c\x07\xdf\x0c\xe3\x2c\x83\x9d\xac\x9f\xb4\x73\x3c\x9c\x28\xb7\x01\x13\xa6\x76\xa7\x05\x7c\x38\xd2\x23\xf2\x2a\x3a\x90\x95\xa8\xd5\x64\x65\x3a\xf5\x28\xe0\x4c\x7e\x18\x24\xbe\x4a\x65\x12"[..],
                                                &b"\x17\xc2\xce\x69\x62\xcb\xd2\xa2\xe0\x66\x29\x7b\x39\xd5\x7d\xd9\xbb\x46\x80\xf0\x19\x1d\x39\x0f\x70\xb4\xe4\x61\x41\x9b\x29\x72\xce\x68\xad\x46\x12\x7f\xdd\xa6\xc3\x91\x95\x77\x4e\xa8\x6d\xf3"[..]).unwrap();
            let z = &b"\x79\x3b\xb9\xcd\x22\xa9\x3c\xf4\x68\xfa\xf8\x04\xa3\x8d\x12\xb7\x8c\xb1\x21\x89\xec\x67\x9d\xdd\x2e\x9a\xa2\x1f\xa9\xa5\xa0\xb0\x49\xab\x16\xa2\x35\x74\xfe\x04\xc1\xc3\xc0\x23\x43\xb9\x1b\xeb"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\xd0\x9b\xb8\x22\xeb\x99\xe3\x80\x60\x95\x47\x47\xc8\x2b\xb3\x27\x8c\xf9\x6b\xbf\x36\xfe\xce\x34\x00\xf4\xc8\x73\x83\x8a\x40\xc1\x35\xeb\x3b\xab\xb9\x29\x3b\xd1\x00\x1b\xf3\xec\xde\xe7\xbf\x26"[..],
                                                  &b"\xd4\x16\xdb\x6e\x1b\x87\xbb\xb7\x42\x77\x88\xa3\xb6\xc7\xa7\xab\x2c\x16\x5b\x1e\x36\x6f\x96\x08\xdf\x51\x20\x37\x58\x4f\x21\x3a\x64\x8d\x47\xf1\x6a\xc3\x26\xe1\x9a\xae\x97\x2f\x63\xfd\x76\xc9"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x06\xf1\x32\xb7\x1f\x74\xd8\x7b\xf9\x98\x57\xe1\xe4\x35\x0a\x59\x4e\x5f\xe3\x55\x33\xb8\x88\x55\x2c\xec\xcb\xc0\xd8\x92\x3c\x90\x2e\x36\x14\x1d\x76\x91\xe2\x86\x31\xb8\xbc\x9b\xaf\xe5\xe0\x64"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x2a\x3c\xc6\xb8\xff\x5c\xde\x92\x6e\x7e\x3a\x18\x9a\x1b\xd0\x29\xc9\xb5\x86\x35\x1a\xf8\x83\x8f\x4f\x20\x1c\xb8\xf4\xb7\x0e\xf3\xb0\xda\x06\xd3\x52\xc8\x0f\xc2\x6b\xaf\x8f\x42\xb7\x84\x45\x9e"[..],
                                                &b"\xbf\x99\x85\x96\x01\x76\xda\x6d\x23\xc7\x45\x2a\x29\x54\xff\xcb\xbc\xb2\x42\x49\xb4\x30\x19\xa2\xa0\x23\xe0\xb3\xda\xbd\x46\x1f\x19\xad\x3e\x77\x5c\x36\x4f\x3f\x11\xad\x49\xf3\x09\x94\x00\xd3"[..]).unwrap();
            let z = &b"\x01\x2d\x19\x1c\xf7\x40\x4a\x52\x36\x78\xc6\xfc\x07\x5d\xe8\x28\x5b\x24\x37\x20\xa9\x03\x04\x77\x08\xbb\x33\xe5\x01\xe0\xdb\xee\x5b\xcc\x40\xd7\xc3\xef\x6c\x6d\xa3\x9e\xa2\x4d\x83\x0d\xa1\xe8"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x13\x74\x12\x62\xed\xe5\x86\x1d\xad\x71\x06\x3d\xfd\x20\x4b\x91\xea\x1d\x3b\x7c\x63\x1d\xf6\x8e\xb9\x49\x96\x95\x27\xd7\x9a\x1d\xc5\x92\x95\xef\x7d\x2b\xca\x67\x43\xe8\xcd\x77\xb0\x4d\x1b\x58"[..],
                                                  &b"\x0b\xaa\xea\xdc\x7e\x19\xd7\x4a\x8a\x04\x45\x1a\x13\x5f\x1b\xe1\xb0\x2f\xe2\x99\xf9\xdc\x00\xbf\xdf\x20\x1e\x83\xd9\x95\xc6\x95\x0b\xcc\x1c\xb8\x9d\x6f\x7b\x30\xbf\x54\x65\x6b\x9a\x4d\xa5\x86"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x12\x04\x8e\xbb\x43\x31\xec\x19\xa1\xe2\x3f\x1a\x2c\x77\x3b\x66\x4c\xcf\xe9\x0a\x28\xbf\xb8\x46\xfc\x12\xf8\x1d\xff\x44\xb7\x44\x3c\x77\x64\x71\x64\xbf\x1e\x9e\x67\xfd\x2c\x07\xa6\x76\x62\x41"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xbc\x18\x83\x6b\xc7\xa9\xfd\xf5\x4b\x53\x52\xf3\x7d\x75\x28\xab\x8f\xa8\xec\x54\x4a\x8c\x61\x80\x51\x1c\xbf\xdd\x49\xcc\xe3\x77\xc3\x9e\x34\xc0\x31\xb5\x24\x0d\xc9\x98\x05\x03\xed\x2f\x26\x2c"[..],
                                                &b"\x80\x86\xcb\xe3\x38\x19\x10\x80\xf0\xb7\xa1\x6c\x7a\xfc\x4c\x7b\x03\x26\xf9\xac\x66\xf5\x85\x52\xef\x4b\xb9\xd2\x4d\xe3\x42\x9e\xd5\xd3\x27\x7e\xd5\x8f\xcf\x48\xf2\xb5\xf6\x13\x26\xbe\xc6\xc6"[..]).unwrap();
            let z = &b"\xad\x0f\xd3\xdd\xff\xe8\x88\x4b\x92\x63\xf3\xc1\x5f\xe1\xf0\x7f\x2a\x5a\x22\xff\xdc\x7e\x96\x70\x85\xee\xa4\x5f\x0c\xd9\x59\xf2\x0f\x18\xf5\x22\x76\x3e\x28\xbc\xc9\x25\xe4\x96\xa5\x2d\xda\x98"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x9e\x22\xcb\xc1\x86\x57\xf5\x16\xa8\x64\xb3\x7b\x78\x33\x48\xb6\x6f\x1a\xa9\x62\x6c\xd6\x31\xf4\xfa\x1b\xd3\x2a\xd8\x8c\xf1\x1d\xb5\x20\x57\xc6\x60\x86\x0d\x39\xd1\x1f\xbf\x02\x4f\xab\xd4\x44"[..],
                                                  &b"\x6b\x0d\x53\xc7\x96\x81\xc2\x81\x16\xdf\x71\xe9\xce\xe7\x4f\xd5\x6c\x8b\x7f\x04\xb3\x9f\x11\x98\xcc\x72\x28\x4e\x98\xbe\x95\x62\xe3\x59\x26\xfb\x4f\x48\xa9\xfb\xec\xaf\xe7\x29\x30\x9e\x8b\x6f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x34\xd6\x1a\x69\x9c\xa5\x76\x16\x9f\xcd\xc0\xcc\x7e\x44\xe4\xe1\x22\x1d\xb0\xfe\x63\xd1\x68\x50\xc8\x10\x40\x29\xf7\xd4\x84\x49\x71\x4b\x98\x84\x32\x8c\xae\x18\x99\x78\x75\x4a\xb4\x60\xb4\x86"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x86\x7f\x81\x10\x4c\xcd\x6b\x16\x3a\x79\x02\xb6\x70\xef\x40\x60\x42\xcb\x0c\xce\x7d\xcd\xc6\x3d\x1d\xfc\x91\xb2\xc4\x0e\x3c\xdf\x75\x95\x83\x4b\xf9\xec\xeb\x79\x84\x9f\x16\x36\xfc\x84\x62\xfc"[..],
                                                &b"\x9d\x4b\xde\x8e\x87\x5e\xc4\x96\x97\xd2\x58\xd1\xd5\x94\x65\xf8\x43\x1c\x6f\x55\x31\xe1\xc5\x9e\x9f\x9e\xbe\x3c\xf1\x64\xa8\xd9\xce\x10\xa1\x2f\x19\x79\x28\x3a\x95\x9b\xad\x24\x4d\xd8\x38\x63"[..]).unwrap();
            let z = &b"\xdc\x4c\xa3\x92\xdc\x15\xe2\x01\x85\xf2\xc6\xa8\xea\x5e\xc3\x1d\xfc\x96\xf5\x61\x53\xa4\x73\x94\xb3\x07\x2b\x13\xd0\x01\x5f\x5d\x4a\xe1\x3b\xeb\x3b\xed\x54\xd6\x58\x48\xf9\xb8\x38\x3e\x6c\x95"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x2d\xb5\xda\x5f\x94\x0e\xaa\x88\x4f\x4d\xb5\xec\x21\x39\xb0\x46\x9f\x38\xe4\xe6\xfb\xbc\xc5\x2d\xf1\x5c\x0f\x7c\xf7\xfc\xb1\x80\x8c\x74\x97\x64\xb6\xbe\x85\xd2\xfd\xc5\xb1\x6f\x58\xad\x5d\xc0"[..],
                                                  &b"\x22\xe8\xb0\x2d\xcf\x33\xe1\xb5\xa0\x83\x84\x95\x45\xf8\x4a\xd5\xe4\x3f\x77\xcb\x71\x54\x6d\xbb\xac\x0d\x11\xbd\xb2\xee\x20\x2e\x9d\x38\x72\xe8\xd0\x28\xc0\x89\x90\x74\x6c\x5e\x1d\xde\x99\x89"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\xdc\x60\xfa\x87\x36\xd7\x02\x13\x5f\xf1\x6a\xab\x99\x2b\xb8\x8e\xac\x39\x7f\x59\x72\x45\x6c\x72\xec\x44\x73\x74\xd0\xd8\xce\x61\x15\x38\x31\xbf\xc8\x6a\xd5\xa6\xeb\x5b\x60\xbf\xb9\x6a\x86\x2c"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\xb6\x9b\xee\xde\x85\xd0\xf8\x29\xfe\xc1\xb8\x93\xcc\xb9\xc3\xe0\x52\xff\x69\x2e\x13\xb9\x74\x53\x7b\xc5\xb0\xf9\xfe\xaf\x7b\x22\xe8\x4f\x03\x23\x16\x29\xb2\x48\x66\xbd\xb4\xb8\xcf\x90\x89\x14"[..],
                                                &b"\x66\xf8\x5e\x2b\xfc\xab\xa2\x84\x32\x85\xb0\xe1\x4e\xbc\x07\xef\x7d\xaf\xff\x8b\x42\x44\x16\xfe\xe6\x47\xb5\x98\x97\xb6\x19\xf2\x0e\xed\x95\xa6\x32\xe6\xa4\x20\x6b\xf7\xda\x42\x9c\x04\xc5\x60"[..]).unwrap();
            let z = &b"\xd7\x65\xb2\x08\x11\x2d\x2b\x9e\xd5\xad\x10\xc4\x04\x6e\x2e\x3b\x0d\xbf\x57\xc4\x69\x32\x95\x19\xe2\x39\xac\x28\xb2\x5c\x7d\x85\x2b\xf7\x57\xd5\xde\x0e\xe2\x71\xca\xdd\x02\x1d\x86\xcf\xd3\x47"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x32\x96\x47\xba\xa3\x54\x22\x4e\xb4\x41\x48\x29\xc5\x36\x8c\x82\xd7\x89\x3b\x39\x80\x4e\x08\xcb\xb2\x18\x0f\x45\x9b\xef\xc4\xb3\x47\xa3\x89\xa7\x0c\x91\xa2\x3b\xd9\xd3\x0c\x83\xbe\x52\x95\xd3"[..],
                                                  &b"\xcc\x8f\x61\x92\x3f\xad\x2a\xa8\xe5\x05\xd6\xcf\xa1\x26\xb9\xfa\xbd\x5a\xf9\xdc\xe2\x90\xb7\x56\x60\xef\x06\xd1\xca\xa7\x36\x81\xd0\x60\x89\xc3\x3b\xc4\x24\x6b\x3a\xa3\x0d\xbc\xd2\x43\x5b\x12"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x6f\xa6\xa1\xc7\x04\x73\x09\x87\xaa\x63\x4b\x05\x16\xa8\x26\xab\xa8\xc6\xd6\x41\x1d\x3a\x4c\x89\x77\x2d\x7a\x62\x61\x02\x56\xa2\xe2\xf2\x89\xf5\xc3\x44\x0b\x0e\xc1\xe7\x0f\xa3\x39\xe2\x51\xce"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x53\xde\x1f\xc1\x32\x8e\x8d\xe1\x4a\xec\xab\x29\xad\x8a\x40\xd6\xb1\x37\x68\xf8\x6f\x7d\x29\x84\x33\xd2\x0f\xec\x79\x1f\x86\xf8\xbc\x73\xf3\x58\x09\x8b\x25\x6a\x29\x8b\xb4\x88\xde\x25\x7b\xf4"[..],
                                                &b"\xac\x28\x94\x4f\xd2\x7f\x17\xb8\x29\x46\xc0\x4c\x66\xc4\x1f\x00\x53\xd3\x69\x2f\x27\x5d\xa5\x5c\xd8\x73\x9a\x95\xbd\x8c\xd3\xaf\x2f\x96\xe4\xde\x95\x9e\xa8\x34\x4d\x89\x45\x37\x59\x05\x85\x8b"[..]).unwrap();
            let z = &b"\xd3\x77\x88\x50\xae\xb5\x88\x04\xfb\xe9\xdf\xe6\xf3\x8b\x9f\xa8\xe2\x0c\x2c\xa4\xe0\xde\xc3\x35\xaa\xfc\xec\xa0\x33\x3e\x3f\x24\x90\xb5\x3c\x0c\x1a\x14\xa8\x31\xba\x37\xc4\xb9\xd7\x4b\xe0\xf2"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp384r1>(&b"\x29\xd8\xa3\x6d\x22\x20\x0a\x75\xb7\xae\xa1\xbb\x47\xcd\xfc\xb1\xb7\xfd\x66\xde\x96\x70\x41\x43\x47\x28\xab\x5d\x53\x3a\x06\x0d\xf7\x32\x13\x06\x00\xfe\x6f\x75\x85\x2a\x87\x1f\xb2\x93\x8e\x39"[..],
                                                  &b"\xe1\x9b\x53\xdb\x52\x83\x95\xde\x89\x7a\x45\x10\x89\x67\x71\x5e\xb8\xcb\x55\xc3\xfc\xbf\x23\x37\x93\x72\xc0\x87\x3a\x05\x8d\x57\x54\x4b\x10\x2e\xcc\xe7\x22\xb2\xcc\xab\xb1\xa6\x03\x77\x4f\xd5"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp384r1>(&b"\x74\xad\x83\x86\xc1\xcb\x2c\xa0\xfc\xde\xb3\x1e\x08\x69\xbb\x3f\x48\xc0\x36\xaf\xe2\xef\x11\x0c\xa3\x02\xbc\x8b\x91\x0f\x62\x1c\x9f\xcc\x54\xce\xc3\x2b\xb8\x9e\xc7\xca\xa8\x4c\x7b\x8e\x54\xa8"[..]).unwrap();
            let iut_pub = Point::new::<Secp384r1>(&b"\x27\xa3\xe8\x3c\xfb\x9d\x51\x22\xe7\x31\x29\xd8\x01\x61\x58\x57\xda\x7c\xc0\x89\xcc\xcc\x9c\x54\xab\x30\x32\xa1\x9e\x0a\x0a\x9f\x67\x73\x46\xe3\x7f\x08\xa0\xb3\xed\x8d\xa6\xe5\xdd\x69\x10\x63"[..],
                                                &b"\x8d\x60\xe4\x4a\xa5\xe0\xfd\x30\xc9\x18\x45\x67\x96\xaf\x37\xf0\xe4\x19\x57\x90\x16\x45\xe5\xc5\x96\xc6\xd9\x89\xf5\x85\x9b\x03\xa0\xbd\x7d\x1f\x4e\x77\x93\x6f\xff\x3c\x74\xd2\x04\xe5\x38\x8e"[..]).unwrap();
            let z = &b"\x81\xe1\xe7\x15\x75\xbb\x45\x05\x49\x8d\xe0\x97\x35\x01\x86\x43\x0a\x62\x42\xfa\x6c\x57\xb8\x5a\x5f\x98\x4a\x23\x37\x11\x23\xd2\xd1\x42\x4e\xef\xbf\x80\x42\x58\x39\x2b\xc7\x23\xe4\xef\x1e\x35"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }
    }

    #[test]
    fn nist_p_521() {
        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xdf\x27\x7c\x15\x21\x08\x34\x9b\xc3\x4d\x53\x9e\xe0\xcf\x06\xb2\x4f\x5d\x35\x00\x67\x7b\x44\x45\x45\x3c\xcc\x21\x40\x94\x53\xaa\xfb\x8a\x72\xa0\xbe\x9e\xbe\x54\xd1\x22\x70\xaa\x51\xb3\xab\x7f\x31\x6a\xa5\xe7\x4a\x95\x1c\x5e\x53\xf7\x4c\xd9\x5f\xc2\x9a\xee\x7a"[..],
                                                  &b"\x01\x3d\x52\xf3\x3a\x9f\x3c\x14\x38\x4d\x15\x87\xfa\x8a\xbe\x7a\xed\x74\xbc\x33\x74\x9a\xd9\xc5\x70\xb4\x71\x77\x64\x22\xc7\xd4\x50\x5d\x9b\x0a\x96\xb3\xbf\xac\x04\x1e\x4c\x6a\x69\x90\xae\x7f\x70\x0e\x5b\x4a\x66\x40\x22\x91\x12\xde\xaf\xa0\xcd\x8b\xb0\xd0\x89\xb0"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x81\x6f\x19\xc1\xfb\x10\xef\x94\xd4\xa1\xd8\x1c\x15\x6e\xc3\xd1\xde\x08\xb6\x67\x61\xf0\x3f\x06\xee\x4b\xb9\xdc\xeb\xbb\xfe\x1e\xaa\x1e\xd4\x9a\x6a\x99\x08\x38\xd8\xed\x31\x8c\x14\xd7\x4c\xc8\x72\xf9\x5d\x05\xd0\x7a\xd5\x0f\x62\x1c\xeb\x62\x0c\xd9\x05\xcf\xb8"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xd4\x56\x15\xed\x5d\x37\xfd\xe6\x99\x61\x0a\x62\xcd\x43\xba\x76\xbe\xdd\x8f\x85\xed\x31\x00\x5f\xe0\x0d\x64\x50\xfb\xbd\x10\x12\x91\xab\xd9\x6d\x49\x45\xa8\xb5\x7b\xc7\x3b\x3f\xe9\xf4\x67\x11\x05\x30\x9e\xc9\xb6\x87\x9d\x05\x51\xd9\x30\xda\xc8\xba\x45\xd2\x55"[..],
                                                &b"\x01\x42\x53\x32\x84\x4e\x59\x2b\x44\x0c\x00\x27\x97\x2a\xd1\x52\x64\x31\xc0\x67\x32\xdf\x19\xcd\x46\xa2\x42\x17\x2d\x4d\xd6\x7c\x2c\x8c\x99\xdf\xc2\x2e\x49\x94\x9a\x56\xcf\x90\xc6\x47\x36\x35\xce\x82\xf2\x5b\x33\x68\x2f\xb1\x9b\xc3\x3b\xd9\x10\xed\x8c\xe3\xa7\xfa"[..]).unwrap();
            let z = &b"\x0b\x39\x20\xac\x83\x0a\xde\x81\x2c\x8f\x96\x80\x5d\xa2\x23\x6e\x00\x2a\xcb\xbf\x13\x59\x6a\x9a\xb2\x54\xd4\x4d\x0e\x91\xb6\x25\x5e\xbf\x12\x29\xf3\x66\xfb\x5a\x05\xc5\x88\x4e\xf4\x60\x32\xc2\x6d\x42\x18\x92\x73\xca\x4e\xfa\x4c\x3d\xb6\xbd\x12\xa6\x85\x37\x59"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x92\xdb\x31\x42\x56\x4d\x27\xa5\xf0\x00\x6f\x81\x99\x08\xfb\xa1\xb8\x50\x38\xa5\xbc\x25\x09\x90\x6a\x49\x7d\xaa\xc6\x7f\xd7\xae\xe0\xfc\x2d\xab\xa4\xe4\x33\x4e\xea\xef\x0e\x00\x19\x20\x4b\x47\x1c\xd8\x80\x24\xf8\x21\x15\xd8\x14\x9c\xc0\xcf\x4f\x7c\xe1\xa4\xd5"[..],
                                                  &b"\x01\x6b\xad\x06\x23\xf5\x17\xb1\x58\xd9\x88\x18\x41\xd2\x57\x1e\xfb\xad\x63\xf8\x5c\xbe\x2e\x58\x19\x60\xc5\xd6\x70\x60\x1a\x67\x60\x27\x26\x75\xa5\x48\x99\x62\x17\xe4\xab\x2b\x8e\xbc\xe3\x1d\x71\xfc\xa6\x3f\xcc\x3c\x08\xe9\x1c\x1d\x8e\xdd\x91\xcf\x6f\xe8\x45\xf8"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\x2f\x2e\x0c\x6d\x9e\x9d\x11\x7c\xeb\x97\x23\xbc\xed\x02\xeb\x3d\x4e\xeb\xf5\xfe\xea\xf8\xee\x01\x13\xcc\xd8\x05\x7b\x13\xdd\xd4\x16\xe0\xb7\x42\x80\xc2\xd0\xba\x8e\xd2\x91\xc4\x43\xbc\x1b\x14\x1c\xaf\x8a\xfb\x3a\x71\xf9\x7f\x57\xc2\x25\xc0\x3e\x1e\x4d\x42\xb0"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x71\x7f\xcb\x3d\x4a\x40\xd1\x03\x87\x1e\xde\x04\x4d\xc8\x03\xdb\x50\x8a\xaa\x4a\xe7\x4b\x70\xb9\xfb\x8d\x8d\xfd\x84\xbf\xec\xfa\xd1\x78\x71\x87\x96\x98\xc2\x92\xd2\xfd\x5e\x17\xb4\xf9\x34\x36\x36\xc5\x31\xa4\xfa\xc6\x8a\x35\xa9\x36\x65\x54\x6b\x9a\x87\x86\x79"[..],
                                                &b"\xf3\xd9\x6a\x86\x37\x03\x69\x93\xab\x5d\x24\x45\x00\xff\xf9\xd2\x77\x21\x12\x82\x6f\x64\x36\x60\x3d\x3e\xb2\x34\xa4\x4d\x5c\x4e\x5c\x57\x72\x34\x67\x9c\x4f\x9d\xf7\x25\xee\x5b\x91\x18\xf2\x3d\x8a\x58\xd0\xcc\x01\x09\x6d\xaf\x70\xe8\xdf\xec\x01\x28\xbd\xc2\xe8"[..]).unwrap();
            let z = &b"\x6b\x38\x0a\x6e\x95\x67\x92\x77\xcf\xee\x4e\x83\x53\xbf\x96\xef\x2a\x1e\xbd\xd0\x60\x74\x9f\x2f\x04\x6f\xe5\x71\x05\x37\x40\xbb\xcc\x9a\x0b\x55\x79\x0b\xc9\xab\x56\xc3\x20\x8a\xa0\x5d\xdf\x74\x6a\x10\xa3\xad\x69\x4d\xaa\xe0\x0d\x98\x0d\x94\x4a\xab\xc6\xa0\x8f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\xfd\xd4\x0d\x9e\x9d\x97\x40\x27\xcb\x3b\xae\x68\x21\x62\xea\xc1\x32\x8a\xd6\x1b\xc4\x35\x3c\x45\xbf\x5a\xfe\x76\xbf\x60\x7d\x28\x94\xc8\xcc\xe2\x36\x95\xd9\x20\xf2\x46\x4f\xda\x47\x73\xd4\x69\x3b\xe4\xb3\x77\x35\x84\x69\x1b\xdb\x03\x29\xb7\xf4\xc8\x6c\xc2\x99"[..],
                                                  &b"\x34\xce\xac\x6a\x3f\xef\x1c\x3e\x1c\x49\x4b\xfe\x8d\x87\x2b\x18\x38\x32\x21\x9a\x7e\x14\xda\x41\x4d\x4e\x34\x74\x57\x36\x71\xec\x19\xb0\x33\xbe\x83\x1b\x91\x54\x35\x90\x59\x25\xb4\x49\x47\xc5\x92\x95\x99\x45\xb4\xeb\x7c\x95\x1c\x3b\x9c\x8c\xf5\x25\x30\xba\x23"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\xe5\x48\xa7\x9d\x8b\x05\xf9\x23\xb9\x82\x5d\x11\xb6\x56\xf2\x22\xe8\xcb\x98\xb0\xf8\x9d\xe1\xd3\x17\x18\x4d\xc5\xa6\x98\xf7\xc7\x11\x61\xee\x7d\xc1\x1c\xd3\x1f\x4f\x4f\x8a\xe3\xa9\x81\xe1\xa3\xe7\x8b\xde\xbb\x97\xd7\xc2\x04\xb9\x26\x1b\x4e\xf9\x2e\x09\x18\xe0"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x0c\xe8\x00\x21\x7e\xd2\x43\xdd\x10\xa7\x9a\xd7\x3d\xf5\x78\xaa\x8a\x3f\x91\x94\xaf\x52\x8c\xd1\x09\x4b\xbf\xee\x27\xa3\xb5\x48\x1a\xd5\x86\x2c\x88\x76\xc0\xc3\xf9\x12\x94\xc0\xab\x3a\xa8\x06\xd9\x02\x0c\xba\xa2\xed\x72\xb7\xfe\xcd\xc5\xa0\x9a\x6d\xad\x6f\x32"[..],
                                                &b"\x01\x54\x3c\x9a\xb4\x5b\x12\x46\x92\x32\x91\x8e\x21\xd5\xa3\x51\xf9\xa4\xb9\xcb\xf9\xef\xb2\xaf\xcc\x40\x2f\xa9\xb3\x16\x50\xbe\xc2\xd6\x41\xa0\x5c\x44\x0d\x35\x33\x1c\x08\x93\xd1\x1f\xb1\x31\x51\x33\x59\x88\xb3\x03\x34\x13\x01\xa7\x3d\xc5\xf6\x1d\x57\x4e\x67\xd9"[..]).unwrap();
            let z = &b"\xfb\xbc\xd0\xb8\xd0\x53\x31\xfe\xf6\x08\x6f\x22\xa6\xcc\xe4\xd3\x57\x24\xab\x7a\x2f\x49\xdd\x84\x58\xd0\xbf\xd5\x7a\x0b\x8b\x70\xf2\x46\xc1\x7c\x44\x68\xc0\x76\x87\x4b\x0d\xff\x7a\x03\x36\x82\x3b\x19\xe9\x8b\xf1\xce\xc0\x5e\x4b\xef\xfb\x05\x91\xf9\x77\x13\xc6"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x98\xd9\x9d\xee\x08\x16\x55\x0e\x84\xdb\xfc\xed\x7e\x88\x13\x7f\xdd\xcf\x58\x1a\x72\x5a\x45\x50\x21\x11\x5f\xe4\x9f\x8d\xc3\xcf\x23\x3c\xd9\xea\x0e\x6f\x03\x9d\xc7\x91\x9d\xa9\x73\xcd\xce\xac\xa2\x05\xda\x39\xe0\xbd\x98\xc8\x06\x25\x36\xc4\x7f\x25\x8f\x44\xb5"[..],
                                                  &b"\xcd\x22\x5c\x87\x97\x37\x1b\xe0\xc4\x29\x7d\x2b\x45\x77\x40\x10\x0c\x77\x41\x41\xd8\xf2\x14\xc2\x3b\x61\xaa\x2b\x6c\xd4\x80\x6b\x9b\x70\x72\x2a\xa4\x96\x5f\xb6\x22\xf4\x2b\x73\x91\xe2\x7e\x5e\xc2\x1c\x56\x79\xc5\xb0\x6b\x59\x12\x73\x72\x99\x7d\x42\x1a\xdc\x1e"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\xc8\xaa\xe9\x4b\xb1\x0b\x8c\xa4\xf7\xbe\x57\x7b\x4f\xb3\x2b\xb2\x38\x10\x32\xc4\x94\x2c\x24\xfc\x2d\x75\x3e\x7c\xc5\xe4\x7b\x48\x33\x89\xd9\xf3\xb9\x56\xd2\x0e\xe9\x00\x1b\x1e\xef\x9f\x23\x54\x5f\x72\xc5\x60\x21\x40\x04\x68\x39\xe9\x63\x31\x3c\x3d\xec\xc8\x64"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\x06\xa1\x4e\x2e\xe8\xff\x97\x0a\xa8\xab\x0c\x79\xb9\x7a\x33\xbb\xa2\x95\x8e\x07\x0b\x75\xb9\x47\x36\xb7\x7b\xbe\x3f\x77\x73\x24\xfa\x52\x87\x27\x71\xaa\x88\xa6\x3a\x9e\x84\x90\xc3\x37\x8d\xf4\xdc\x76\x0c\xd1\x4d\x62\xbe\x70\x07\x79\xdd\x1a\x43\x77\x94\x36\x56"[..],
                                                &b"\x23\x66\xce\x39\x41\xe0\xb2\x84\xb1\xaa\x81\x21\x5d\x0d\x3b\x97\x78\xfc\xe2\x3c\x8c\xd1\xe4\xed\x6f\xa0\xab\xf6\x21\x56\xc9\x1d\x4b\x3e\xb5\x59\x99\xc3\x47\x1b\xed\x27\x5e\x9e\x60\xe5\xaa\x9d\x69\x0d\x31\x0b\xfb\x15\xc9\xc5\xbb\xd6\xf5\xe9\xeb\x39\x68\x2b\x74"[..]).unwrap();
            let z = &b"\x01\x45\xcf\xa3\x8f\x25\x94\x35\x16\xc9\x6a\x5f\xd4\xbf\xeb\xb2\xf6\x45\xd1\x05\x20\x11\x7a\xa5\x19\x71\xef\xf4\x42\x80\x8a\x23\xb4\xe2\x3c\x18\x7e\x63\x9f\xf9\x28\xc3\x72\x5f\xbd\x1c\x0c\x2a\xd0\xd4\xae\xb2\x07\xbc\x1a\x6f\xb6\xcb\x6d\x46\x78\x88\xdc\x04\x4b\x3c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x7a\xe1\x15\xad\xaa\xf0\x41\x69\x1a\xb6\xb7\xfb\x8c\x92\x1f\x99\xd8\xed\x32\xd2\x83\xd6\x70\x84\xe8\x0b\x9a\xd9\xc4\x0c\x56\xcd\x98\x38\x9f\xb0\xa8\x49\xd9\xec\xf7\x26\x8c\x29\x7b\x6f\x93\x40\x61\x19\xf4\x0e\x32\xb5\x77\x3e\xd2\x5a\x28\xa9\xa8\x5c\x4a\x75\x88"[..],
                                                  &b"\x01\xa2\x8e\x00\x4e\x37\xee\xae\xfe\x1f\x4d\xbb\x71\xf1\x87\x86\x96\x14\x1a\xf3\xa1\x0a\x96\x91\xc4\xed\x93\x48\x72\x14\x64\x3b\x76\x1f\xa4\xb0\xfb\xee\xb2\x47\xcf\x6d\x3f\xba\x7a\x60\x69\x75\x36\xad\x03\xf4\x9b\x80\xa9\xd1\xcb\x07\x96\x73\x65\x49\x77\xc5\xfa\x94"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x9b\x0a\xf1\x37\xc9\x69\x6c\x75\xb7\xe6\xdf\x7b\x73\x15\x6b\xb2\xd4\x5f\x48\x2e\x5a\x42\x17\x32\x4f\x47\x8b\x10\xce\xb7\x6a\xf0\x97\x24\xcf\x86\xaf\xa3\x16\xe7\xf8\x99\x18\xd3\x1d\x54\x82\x4a\x5c\x33\x10\x7a\x48\x3c\x15\xc1\x5b\x96\xed\xc6\x61\x34\x0b\x1c\x0e"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x74\x8c\xdb\xb8\x75\xd3\x5f\x4b\xcc\xb6\x2a\xbe\x20\xe8\x2d\x32\xe4\xc1\x4d\xc2\xfe\xb5\xb8\x7d\xa2\xd0\xcc\xb1\x1c\x9b\x6d\x4b\x77\x37\xb6\xc4\x6f\x0d\xfb\x4d\x89\x6e\x2d\xb9\x2f\xcf\x53\xcd\xbb\xae\x2a\x40\x4c\x0b\xab\xd5\x64\xad\x7a\xde\xac\x62\x73\xef\xa3"[..],
                                                &b"\x01\x98\x4a\xca\xb8\xd8\xf1\x73\x32\x3d\xe0\xbb\x60\x27\x4b\x22\x88\x71\x60\x93\x73\xbb\x22\xa1\x72\x87\xe9\xde\xc7\x49\x58\x73\xab\xc0\x9a\x89\x15\xb5\x4c\x84\x55\xc8\xe0\x2f\x65\x4f\x60\x2e\x23\xa2\xbb\xd7\xa9\xeb\xb7\x4f\x30\x09\xbd\x65\xec\xc6\x50\x81\x4c\xc0"[..]).unwrap();
            let z = &b"\x5c\x57\x21\xe9\x6c\x27\x33\x19\xfd\x60\xec\xc4\x6b\x59\x62\xf6\x98\xe9\x74\xb4\x29\xf2\x8f\xe6\x96\x2f\x4a\xc6\x56\xbe\x2e\xb8\x67\x4c\x4a\xaf\xc0\x37\xea\xb4\x8e\xce\x61\x29\x53\xb1\xe8\xd8\x61\x01\x6b\x6a\xd0\xc7\x98\x05\x78\x4c\x67\xf7\x3a\xda\x96\xf3\x51"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x25\x88\x11\x5e\x6f\x7f\x7b\xdc\xfd\xf5\x7f\x03\xb1\x69\xb4\x79\x75\x8b\xaa\xfd\xaf\x56\x9d\x04\x13\x59\x87\xb2\xce\x61\x64\xc0\x2a\x57\x68\x5e\xb5\x27\x6b\x5d\xae\x62\x95\xd3\xfe\x90\x62\x0f\x38\xb5\x53\x5c\x6d\x22\x60\xc1\x73\xe6\x1e\xb8\x88\xca\x92\x02\x03"[..],
                                                  &b"\x01\x54\x2c\x16\x9c\xf9\x7c\x25\x96\xfe\x2d\xdd\x84\x8a\x22\x2e\x36\x7c\x5f\x7e\x62\x67\xeb\xc1\xbc\xd9\xab\x5d\xcf\x49\x15\x8f\x1a\x48\xe4\xaf\x29\xa8\x97\xb7\xe6\xa8\x20\x91\xc2\xdb\x87\x4d\x8e\x7a\xbf\x0f\x58\x06\x46\x91\x34\x41\x54\xf3\x96\xdb\xae\xd1\x88\xb6"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\xe4\x8f\xaa\xce\xe6\xde\xc8\x3f\xfc\xde\x94\x4c\xf6\xbd\xf4\xce\x4b\xae\x72\x74\x78\x88\xeb\xaf\xee\x45\x5b\x1e\x91\x58\x49\x71\xef\xb4\x91\x27\x97\x6a\x52\xf4\x14\x29\x52\xf7\xc2\x07\xec\x02\x65\xf2\xb7\x18\xcf\x3e\xad\x96\xea\x4f\x62\xc7\x52\xe4\xf7\xac\xd3"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\x0e\xb1\xb4\xd9\x17\x2b\xcc\x23\xf4\xf2\x0c\xc9\x56\x0f\xc5\x49\x28\xc3\xf3\x4e\xa6\x1c\x00\x39\x1d\xc7\x66\xc7\x6e\xd9\xfa\x60\x84\x49\x37\x7d\x1e\x4f\xad\xd1\x23\x60\x25\x41\x73\x30\xb4\xb9\x10\x86\x70\x4a\xce\x3e\x4e\x64\x84\xc6\x06\xe2\xa9\x43\x47\x8c\x86"[..],
                                                &b"\x01\x49\x41\x38\x64\x06\x98\x25\xee\x1d\x08\x28\xda\x9f\x4a\x97\x71\x30\x05\xe9\xbd\x1a\xdb\xc3\xb3\x8c\x5b\x94\x69\x00\x72\x1a\x96\x0f\xe9\x6a\xd2\xc1\xb3\xa4\x4f\xe3\xde\x91\x56\x13\x6d\x44\xcb\x17\xcb\xc2\x41\x57\x29\xbb\x78\x2e\x16\xbf\xe2\xde\xb3\x06\x9e\x43"[..]).unwrap();
            let z = &b"\x01\x73\x6d\x97\x17\x42\x9b\x4f\x41\x2e\x90\x3f\xeb\xe2\xf9\xe0\xff\xfd\x81\x35\x5d\x6c\xe2\xc0\x6f\xf3\xf6\x6a\x3b\xe1\x5c\xee\xc6\xe6\x5e\x30\x83\x47\x59\x3f\x00\xd7\xf3\x35\x91\xda\x40\x43\xc3\x07\x63\xd7\x27\x49\xf7\x2c\xdc\xee\xbe\x82\x5e\x4b\x34\xec\xd5\x70"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x69\x49\x1d\x55\xbd\x09\x04\x9f\xdf\x4c\x2a\x53\xa6\x60\x48\x0f\xee\x4c\x03\xa0\x53\x86\x75\xd1\xcd\x09\xb5\xbb\xa7\x8d\xac\x48\x54\x3e\xf1\x18\xa1\x17\x3b\x3f\xbf\x8b\x20\xe3\x9c\xe0\xe6\xb8\x90\xa1\x63\xc5\x0f\x96\x45\xb3\xd2\x1d\x1c\xbb\x3b\x60\xa6\xff\xf4"[..],
                                                  &b"\x83\x49\x4b\x2e\xba\x76\x91\x0f\xed\x33\xc7\x61\x80\x45\x15\x01\x1f\xab\x50\xe3\xb3\x77\xab\xd8\xa8\xa0\x45\xd8\x86\xd2\x23\x8d\x2c\x26\x8a\xc1\xb6\xec\x88\xbd\x71\xb7\xba\x78\xe2\xc3\x3c\x15\x2e\x4b\xf7\xda\x5d\x56\x5e\x4a\xcb\xec\xf5\xe9\x2c\x7a\xd6\x62\xbb"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\xc2\x9a\xa2\x23\xea\x8d\x64\xb4\xa1\xed\xa2\x7f\x39\xd3\xbc\x98\xea\x01\x48\xdd\x98\xc1\xcb\xe5\x95\xf8\xfd\x2b\xfb\xde\x11\x9c\x9e\x01\x7a\x50\xf5\xd1\xfc\x12\x1c\x08\xc1\xce\xf3\x1b\x75\x88\x59\x55\x6e\xb3\xe0\xe0\x42\xd8\xdd\x6a\xaa\xc5\x7a\x05\xca\x61\xe3"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x15\x11\xc8\x48\xef\x60\xd5\x41\x9a\x98\xd1\x02\x04\xdb\x0f\xe5\x82\x24\x12\x43\x70\x06\x1b\xcf\xa4\xe9\x24\x9d\x50\x61\x8c\x56\xbf\x37\x22\x47\x1b\x25\x9f\x38\x26\x3b\xb7\xb2\x80\xd2\x3c\xaf\x2a\x1e\xe8\x73\x7f\x93\x71\xcd\xb2\x73\x2c\xdc\x95\x83\x69\x93\x0c"[..],
                                                &b"\x01\xd4\x61\x68\x1a\xe6\xd8\xc4\x9b\x4c\x5f\x4d\x60\x16\x14\x3f\xb1\xbd\x74\x91\x57\x3e\x3e\xd0\xe6\xc4\x8b\x82\xe8\x21\x64\x4f\x87\xf8\x2f\x0e\x5f\x08\xfd\x16\xf1\xf9\x8f\xa1\x75\x86\x20\x0a\xb0\x2e\xd8\xc6\x27\xb3\x5c\x3f\x27\x61\x7e\xc5\xfd\x92\xf4\x56\x20\x3f"[..]).unwrap();
            let z = &b"\x01\x8f\x2a\xe9\x47\x6c\x77\x17\x26\xa7\x77\x80\x20\x8d\xed\xfe\xfa\x20\x54\x88\x99\x6b\x18\xfe\xcc\x50\xbf\xd4\xc1\x32\x75\x3f\x57\x66\xb2\xcd\x74\x4a\xfa\x99\x18\x60\x6d\xe2\xe0\x16\xef\xfc\x63\x62\x2e\x90\x29\xe7\x6d\xc6\xe3\xf0\xc6\x9f\x7a\xec\xed\x56\x5c\x2c"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x84\x15\xf5\xbb\xd0\xee\xe3\x87\xd6\xc0\x9d\x0e\xf8\xac\xaf\x29\xc6\x6d\xb4\x5d\x6b\xa1\x01\x86\x0a\xe4\x5d\x3c\x60\xe1\xe0\xe3\xf7\x24\x7a\x46\x26\xa6\x0f\xdd\x40\x49\x65\xc3\x56\x6c\x79\xf6\x44\x9e\x85\x6c\xe0\xbf\x94\x61\x9f\x97\xda\x8d\xa2\x4b\xd2\xcf\xb6"[..],
                                                  &b"\xfd\xd7\xc5\x9c\x58\xc3\x61\xbc\x50\xa7\xa5\xd0\xd3\x6f\x72\x3b\x17\xc4\xf2\xad\x2b\x03\xc2\x4d\x42\xdc\x50\xf7\x4a\x8c\x46\x5a\x0a\xfc\x46\x83\xf1\x0f\xab\x84\x65\x2d\xfe\x9e\x92\x8c\x26\x26\xb5\x45\x64\x53\xe1\x57\x3f\xf6\x0b\xe1\x50\x74\x67\xd4\x31\xfb\xb2"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x28\x69\x2b\xe2\xbf\x5c\x4b\x48\x93\x98\x46\xfb\x3d\x5b\xce\x74\x65\x4b\xb2\x64\x6e\x15\xf8\x38\x9e\x23\x70\x8a\x1a\xfa\xdf\x56\x15\x11\xea\x0d\x99\x57\xd0\xb5\x34\x53\x81\x9d\x60\xfb\xa8\xf6\x5a\x18\xf7\xb2\x9d\xf0\x21\xb1\xbb\x01\xcd\x16\x32\x93\xac\xc3\xcc"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\xcf\xdc\x10\xc7\x99\xf5\xc7\x9c\xb6\x93\x0a\x65\xfb\xa3\x51\x74\x8e\x07\x56\x79\x93\xe5\xe4\x10\xef\x4c\xac\xc4\xcd\x8a\x25\x78\x49\x91\xeb\x46\x74\xe4\x10\x50\xf9\x30\xc7\x19\x0a\xc8\x12\xb9\x24\x5f\x48\xa7\x97\x3b\x65\x8d\xaf\x40\x88\x22\xfe\x5b\x85\xf6\x68"[..],
                                                &b"\x01\x80\xd9\xdd\xfc\x9a\xf7\x7b\x9c\x4a\x6f\x02\xa8\x34\xdb\x15\xe5\x35\xe0\xb3\x84\x5b\x2c\xce\x30\x38\x83\x01\xb5\x1c\xec\xbe\x32\x76\x30\x7e\xf4\x39\xb5\xc9\xe6\xa7\x2d\xc2\xd9\x4d\x87\x9b\xc3\x95\x05\x2d\xbb\x4a\x57\x87\xd0\x6e\xfb\x28\x02\x10\xfb\x8b\xe0\x37"[..]).unwrap();
            let z = &b"\x01\x05\xa3\x46\x98\x8b\x92\xed\x8c\x7a\x25\xce\x4d\x79\xd2\x1b\xc8\x6c\xfc\xc7\xf9\x9c\x6c\xd1\x9d\xbb\x4a\x39\xf4\x8a\xb9\x43\xb7\x9e\x4f\x06\x47\x34\x8d\xa0\xb8\x0b\xd8\x64\xb8\x5c\x6b\x8d\x92\x53\x6d\x6a\xa5\x44\xdc\x75\x37\xa0\x0c\x85\x8f\x8b\x66\x31\x9e\x25"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xc7\x21\xee\xa8\x05\xa5\xcb\xa2\x9f\x34\xba\x57\x58\x77\x5b\xe0\xcf\x61\x60\xe6\xc0\x87\x23\xf5\xab\x17\xbf\x96\xa1\xff\x2b\xd9\x42\x79\x61\xa4\xf3\x4b\x07\xfc\x0b\x14\xca\x4b\x2b\xf6\x84\x5d\xeb\xd5\xa8\x69\xf1\x24\xeb\xfa\x7a\xa7\x2f\xe5\x65\x05\x0b\x7f\x18"[..],
                                                  &b"\xb6\xe8\x9e\xb0\xe1\xdc\xf1\x81\x23\x6f\x7c\x54\x8f\xd1\xa8\xc1\x6b\x25\x8b\x52\xc1\xa9\xbf\xd3\xfe\x8f\x22\x84\x1b\x26\x76\x32\x65\xf0\x74\xc4\xcc\xf2\xd6\x34\xae\x97\xb7\x01\x95\x6f\x67\xa1\x10\x06\xc5\x2d\x97\x19\x7d\x92\xf5\x85\xf5\x74\x8b\xc2\x67\x2e\xeb"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\x19\x4d\x1e\xe6\x13\xf5\x36\x6c\xbc\x44\xb5\x04\xd2\x1a\x0c\xf6\x71\x5e\x20\x9c\xd3\x58\xf2\xdd\x5f\x3e\x71\xcc\x0d\x67\xd0\xe9\x64\x16\x8c\x42\xa0\x84\xeb\xda\x74\x6f\x98\x63\xa8\x6b\xac\xff\xc8\x19\xf1\xed\xf1\xb8\xc7\x27\xcc\xfb\x30\x47\x24\x0a\x57\xc4\x35"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\x6b\xd1\x5c\x8a\x58\xd3\x66\xf7\xf2\xb2\xf2\x98\xcc\x87\xb7\x48\x5e\x9e\xe7\x0d\x11\xd1\x24\x48\xb8\x37\x7c\x0a\x82\xc7\x62\x6f\x67\xaf\xf7\xf9\x7b\xe7\xa3\x54\x6b\xf4\x17\xee\xed\xdf\x75\xa9\x3c\x13\x01\x91\xc8\x41\x08\x04\x2e\xa2\xfc\xa1\x7f\xd3\xf8\x0d\x14"[..],
                                                &b"\x01\x56\x05\x02\xd0\x4b\x74\xfc\xe1\x74\x3a\xab\x47\x7a\x9d\x1e\xac\x93\xe5\x22\x69\x81\xfd\xb9\x7a\x74\x78\xce\x4c\xe5\x66\xff\x72\x43\x93\x12\x84\xfa\xd8\x50\xb0\xc2\xbc\xae\x0d\xdd\x2d\x97\x79\x01\x60\xc1\xa2\xe7\x7c\x3e\xd6\xc9\x5e\xcc\x44\xb8\x9e\x26\x37\xfc"[..]).unwrap();
            let z = &b"\x45\x31\xb3\xd2\xc6\xcd\x12\xf2\x16\x04\xc8\x61\x0e\x67\x23\xdb\xf4\xda\xf8\x0b\x5a\x45\x9d\x6b\xa5\x81\x43\x97\xd1\xc1\xf7\xa2\x1d\x7c\x11\x4b\xe9\x64\xe2\x73\x76\xaa\xeb\xe3\xa7\xbc\x3d\x6a\xf7\xa7\xf8\xc7\xbe\xfb\x61\x1a\xfe\x48\x7f\xf0\x32\x92\x1f\x75\x0f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xc3\x58\x23\xe4\x40\xa9\x36\x3a\xb9\x8d\x9f\xc7\xa7\xbc\x0c\x05\x32\xdc\x79\x77\xa7\x91\x65\x59\x9b\xf1\xa9\xcc\x64\xc0\x0f\xb3\x87\xb4\x2c\xca\x36\x52\x86\xe8\x43\x03\x60\xbf\xad\x36\x43\xbc\x31\x35\x4e\xda\x50\xdc\x93\x6c\x32\x9e\xcd\xb6\x09\x05\xc4\x0f\xcb"[..],
                                                  &b"\xd9\xe7\xf4\x33\x53\x1e\x44\xdf\x4f\x6d\x51\x42\x01\xcb\xaa\xbb\x06\xba\xdd\x67\x83\xe0\x11\x11\x72\x6d\x81\x55\x31\xd2\x33\xc5\xcd\xb7\x22\x89\x3f\xfb\xb2\x02\x72\x59\xd5\x94\xde\x77\x43\x88\x09\x73\x81\x20\xc6\xf7\x83\x93\x4f\x92\x6c\x3f\xb6\x9b\x40\xc4\x09"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\xfd\x90\xe3\xe4\x16\xe9\x8a\xa3\xf2\xb6\xaf\xa7\xf3\xbf\x36\x8e\x45\x1a\xd9\xca\x5b\xd5\x4b\x5b\x14\xae\xe2\xed\x67\x23\xdd\xe5\x18\x1f\x50\x85\xb6\x81\x69\xb0\x9f\xbe\xc7\x21\x37\x2c\xcf\x6b\x28\x47\x13\xf9\xa6\x35\x6b\x8d\x56\x0a\x8f\xf7\x8c\xa3\x73\x7c\x88"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\xeb\xea\x1b\x10\xd3\xe3\xb9\x71\xb7\xef\xb6\x9f\xc8\x78\xde\x11\xc7\xf4\x72\xe4\xe4\xd3\x84\xc3\x1b\x8d\x62\x88\xd8\x07\x15\x17\xac\xad\xe9\xb3\x97\x96\xc7\xaf\x51\x63\xbc\xf7\x1a\xed\xa7\x77\x53\x3f\x38\x2c\x6c\xf0\xa4\xd9\xbb\xb9\x38\xc8\x5f\x44\xb7\x80\x37"[..],
                                                &b"\x01\x6b\x0e\x3e\x19\xc2\x99\x6b\x2c\xbd\x1f\xf6\x47\x30\xe7\xca\x90\xed\xca\x19\x84\xf9\xb2\x95\x13\x33\x53\x5e\x57\x48\xba\xa3\x4a\x99\xf6\x1f\xf4\xd5\xf8\x12\x07\x9e\x0f\x01\xe8\x77\x89\xf3\x4e\xfd\xad\x80\x98\x01\x5e\xe7\x4a\x4f\x84\x6d\xd1\x90\xd1\x6d\xc6\xe1"[..]).unwrap();
            let z = &b"\x01\x00\xc8\x93\x59\x69\x07\x7b\xae\x0b\xa8\x9e\xf0\xdf\x81\x61\xd9\x75\xec\x58\x70\xac\x81\x1a\xe7\xe6\x5c\xa5\x39\x4e\xfb\xa4\xf0\x63\x3d\x41\xbf\x79\xea\x5e\x5b\x94\x96\xbb\xd7\xaa\xe0\x00\xb0\x59\x4b\xaa\x82\xef\x8f\x24\x4e\x69\x84\xae\x87\xae\x1e\xd1\x24\xb7"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x09\x30\x57\xfb\x86\x2f\x2a\xd2\xe8\x2e\x58\x1b\xae\xb3\x32\x4e\x7b\x32\x94\x6f\x2b\xa8\x45\xa9\xbe\xee\xd8\x7d\x69\x95\xf5\x49\x18\xec\x66\x19\xb9\x93\x19\x55\xd5\xa8\x9d\x4d\x74\xad\xf1\x04\x6b\xb3\x62\x19\x2f\x2e\xf6\xbd\x3e\x3d\x2d\x04\xdd\x1f\x87\x05\x4a"[..],
                                                  &b"\xaa\x3f\xb2\x44\x83\x35\xf6\x94\xe3\xcd\xa4\xae\x0c\xc7\x1b\x1b\x2f\x2a\x20\x6f\xa8\x02\xd7\x26\x2f\x19\x98\x3c\x44\x67\x4f\xe1\x53\x27\xac\xaa\xc1\xfa\x40\x42\x4c\x39\x5a\x65\x56\xcb\x81\x67\x31\x25\x27\xfa\xe5\x86\x5e\xcf\xfc\x14\xbb\xdc\x17\xda\x78\xcd\xcf"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x90\x12\xec\xfd\xad\xc8\x5c\xed\x63\x0a\xfe\xa5\x34\xcd\xc8\xe9\xd1\xab\x8b\xe5\xf3\x75\x3d\xcf\x5f\x2b\x09\xb4\x0e\xda\x66\xfc\x68\x58\x54\x9b\xc3\x6e\x6f\x8d\xf5\x59\x98\xcf\xa9\xa0\x70\x3a\xec\xf6\xc4\x27\x99\xc2\x45\x01\x10\x64\xf5\x30\xc0\x9d\xb9\x83\x69"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x23\x4e\x32\xbe\x0a\x90\x71\x31\xd2\xd1\x28\xa6\x47\x7e\x0c\xac\xeb\x86\xf0\x24\x79\x74\x5e\x0f\xe2\x45\xcb\x33\x2d\xe6\x31\xc0\x78\x87\x11\x60\x48\x2e\xee\xf5\x84\xe2\x74\xdf\x7f\xa4\x12\xce\xa3\xe1\xe9\x1f\x71\xec\xba\x87\x81\xd9\x20\x5d\x48\x38\x63\x41\xad"[..],
                                                &b"\x01\xcf\x86\x45\x5b\x09\xb1\xc0\x05\xcf\xfb\xa8\xd7\x62\x89\xa3\x75\x96\x28\xc8\x74\xbe\xea\x46\x2f\x51\xf3\x0b\xd5\x81\xe3\x80\x31\x34\x30\x7d\xed\xbb\x77\x1b\x33\x34\xee\x15\xbe\x2e\x24\x2c\xd7\x9c\x34\x07\xd2\xf5\x89\x35\x45\x6c\x69\x41\xdd\x9b\x6d\x15\x5a\x46"[..]).unwrap();
            let z = &b"\x01\x7f\x36\xaf\x19\x30\x38\x41\xd1\x3a\x38\x9d\x95\xec\x0b\x80\x1c\x7f\x9a\x67\x9a\x82\x31\x46\xc7\x5c\x17\xbc\x44\x25\x6e\x9a\xd4\x22\xa4\xf8\xb3\x1f\x14\x64\x7b\x2c\x7d\x31\x7b\x93\x3f\x7c\x29\x46\xc4\xb8\xab\xd1\xd5\x6d\x62\x0f\xab\x1b\x5f\xf1\xa3\xad\xc7\x1f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x83\x19\x2e\xd0\xb1\xcb\x31\xf7\x58\x17\x79\x49\x37\xf6\x6a\xd9\x1c\xf7\x45\x52\xcd\x51\x0c\xed\xb9\xfd\x64\x13\x10\x42\x2a\xf5\xd0\x9f\x22\x1c\xad\x24\x9e\xe8\x14\xd1\x6d\xd7\xac\x84\xde\xd9\xea\xcd\xc2\x83\x40\xfc\xfc\x9c\x0c\x06\xab\xe3\x0a\x2f\xc2\x8c\xd8"[..],
                                                  &b"\x22\x12\xed\x86\x8c\x9b\xa0\xfb\x2c\x91\xe2\xc3\x9b\xa9\x39\x96\xa3\xe4\xeb\xf4\x5f\x28\x52\xd0\x92\x8c\x48\x93\x0e\x87\x5c\xc7\xb4\x28\xd0\xe7\xf3\xf4\xd5\x03\xe5\xd6\x0c\x68\xcb\x49\xb1\x3c\x24\x80\xcd\x48\x6b\xed\x92\x00\xca\xdd\xad\xdf\xe4\xff\x8e\x35\x62"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\xb5\xff\x84\x7f\x8e\xff\x20\xb8\x8c\xfa\xd4\x2c\x06\xe5\x8c\x37\x42\xf2\xf8\xf1\xfd\xfd\x64\xb5\x39\xba\x48\xc2\x59\x26\x92\x6b\xd5\xe3\x32\xb4\x56\x49\xc0\xb1\x84\xf7\x72\x55\xe9\xd5\x8f\xe8\xaf\xa1\xa6\xd9\x68\xe2\xcb\x1d\x46\x37\x77\x71\x20\xc7\x65\xc1\x28"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\xde\x3d\xc9\x26\x3b\xc8\xc4\x96\x9d\xc6\x84\xbe\x0e\xec\x54\xbe\xfd\x9a\x9f\x3d\xba\x19\x4d\x86\x58\xa7\x89\x34\x1b\xf0\xd7\x8d\x84\xda\x67\x35\x22\x7c\xaf\xaf\x09\x35\x19\x51\x69\x11\x97\x57\x3c\x8c\x36\x0a\x11\xe5\x28\x57\x12\xb8\xbb\xdf\x5a\xc9\x1b\x97\x7c"[..],
                                                &b"\x81\x2d\xe5\x8c\xd0\x95\xec\x2e\x5a\x9b\x24\x7e\xb3\xed\x41\xd8\xbe\xf6\xae\xac\xe1\x94\xa7\xa0\x5b\x65\xaa\x5d\x28\x9f\xbc\x9b\x17\x70\xec\x84\xbb\x6b\xe0\xc2\xc6\x4c\xc3\x7c\x1d\x54\xa7\xf5\xd7\x13\x77\xa9\xad\xbe\x20\xf2\x6f\x6f\x2b\x54\x4a\x82\x1e\xa8\x31"[..]).unwrap();
            let z = &b"\x06\x2f\x9f\xc2\x9a\xe1\xa6\x8b\x2e\xe0\xdc\xf9\x56\xcb\xd3\x8c\x88\xae\x5f\x64\x5e\xaa\x54\x6b\x00\xeb\xe8\x7a\x72\x60\xbf\x72\x4b\xe2\x0d\x34\xb9\xd0\x20\x76\x65\x5c\x93\x3d\x05\x6b\x21\xe3\x04\xc2\x4d\xdb\x1d\xed\xf1\xdd\x76\xde\x61\x1f\xc4\xa2\x34\x03\x36"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xa8\x9b\x63\x6a\x93\xe5\xd2\xba\x6c\x22\x92\xbf\x23\x03\x3a\x84\xf0\x6a\x3a\xc1\x22\x0e\xa7\x1e\x80\x6a\xfb\xe0\x97\xa8\x04\xcc\x67\xe9\xba\xa5\x14\xcf\xb6\xc1\x2c\x91\x94\xbe\x30\x21\x2b\xf7\xaa\xe7\xfd\xf6\xd3\x76\xc2\x12\xf0\x55\x4e\x65\x64\x63\xff\xab\x7e"[..],
                                                  &b"\x01\x82\xef\xca\xf7\x0f\xc4\x12\xd3\x36\x60\x2e\x01\x4d\xa4\x72\x56\xa0\xb6\x06\xf2\xad\xdc\xce\x80\x53\xbf\x81\x7a\xc8\x65\x6b\xb4\xe4\x2f\x14\xc8\xcb\xf2\xa6\x8f\x48\x8a\xb3\x5d\xcd\xf6\x40\x56\x27\x1d\xee\x1f\x60\x6a\x44\x0b\xa4\xbd\x4e\x5a\x11\xb8\xb8\xe5\x4f"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\x1a\x63\x47\xd4\xe8\x01\xc9\x19\x23\x48\x83\x54\xcc\x53\x3e\x7e\x35\xfd\xdf\x81\xff\x0f\xb7\xf5\x6b\xb0\x72\x6e\x0c\x29\xee\x5d\xcd\xc5\xf3\x94\xba\x54\xcf\x57\x26\x90\x48\xaa\xb6\xe0\x55\x89\x5c\x8d\xa2\x4b\x8b\x06\x39\xa7\x42\x31\x43\x90\xcc\x04\x19\x0e\xd6"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xfe\x30\x26\x7f\x33\xba\x5c\xde\xfc\x25\xcb\xb3\xc9\x32\x0d\xad\x9c\xcb\x1d\x7d\x37\x66\x44\x62\x0c\xa4\xfa\xde\xe5\x62\x6a\x3c\xed\xe2\x5a\xd2\x54\x62\x4d\xef\x72\x7a\x70\x48\xf7\x14\x5f\x76\x16\x2a\xa9\x80\x42\xf9\xb1\x23\xb2\x07\x6f\x8e\x8c\xf5\x9b\x3f\xdf"[..],
                                                &b"\x11\x45\xdc\x66\x31\x95\x3b\x6e\x29\x45\xe9\x43\x01\xd6\xcb\xb0\x98\xfe\x4b\x04\xf7\xee\x9b\x09\x41\x1d\xf1\x04\xdc\x82\xd7\xd7\x9e\xc4\x6a\x01\xed\x0f\x2d\x3e\x7d\xb6\xeb\x68\x06\x94\xbd\xeb\x10\x7c\x10\x78\xae\xc6\xca\xbd\x9e\xbe\xe3\xd3\x42\xfe\x7e\x54\xdf"[..]).unwrap();
            let z = &b"\x01\x28\xab\x09\xbf\xec\x54\x06\x79\x9e\x61\x0f\x77\x2b\xa1\x7e\x89\x22\x49\xfa\x8e\x0e\x7b\x18\xa0\x4b\x91\x97\x03\x4b\x25\x0b\x48\x29\x4f\x18\x67\xfb\x96\x41\x51\x8f\x92\x76\x60\x66\xa0\x7a\x8b\x91\x7b\x0e\x76\x87\x9e\x10\x11\xe5\x1c\xcb\xd9\xf5\x40\xc5\x4d\x4f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x72\x00\xb3\xf1\x6a\x68\xcb\xae\xd2\xbf\x78\xba\x8c\xdd\xfb\x6c\xff\xac\x26\x2b\xba\x00\xfb\xc2\x5f\x9d\xc7\x2a\x07\xce\x59\x37\x29\x04\x89\x9f\x36\x4c\x44\xcb\x26\x4c\x09\x7b\x64\x7d\x44\x12\xbe\xe3\xe5\x19\x89\x2d\x53\x4d\x91\x29\xf8\xa2\x8f\x75\x00\xfe\xe7"[..],
                                                  &b"\xba\xba\x8d\x67\x2a\x4f\x4a\x3b\x63\xde\x48\xb9\x6f\x56\xe1\x8d\xf5\xd6\x8f\x7d\x70\xd5\x10\x98\x33\xf4\x37\x70\xd6\x73\x2e\x06\xb3\x9a\xd6\x0d\x93\xe5\xb4\x3d\xb8\x78\x9f\x1e\xc0\xab\xa4\x72\x86\xa3\x9e\xa5\x84\x23\x5a\xce\xa7\x57\xdb\xf1\x3d\x53\xb5\x83\x64"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x22\xb6\xd2\xa2\x2d\x71\xdf\xaa\x81\x1d\x2d\x9f\x9f\x31\xfb\xed\x27\xf2\xe1\xf3\xd2\x39\x53\x8d\xdf\x3e\x4c\xc8\xc3\x9a\x33\x02\x66\xdb\x25\xb7\xbc\x0a\x97\x04\xf1\x7b\xde\x7f\x35\x92\xbf\x5f\x1f\x2d\x4b\x56\x01\x3a\xac\xc3\xd8\xd1\xbc\x02\xf0\x0d\x31\x46\xcc"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xba\x38\xcf\xbf\x9f\xd2\x51\x8a\x3f\x61\xd4\x35\x49\xe7\xa6\xa6\xd2\x8b\x2b\xe5\x7f\xfd\x3e\x0f\xac\xeb\x63\x6b\x34\xed\x17\xe0\x44\xa9\xf2\x49\xda\xe8\xfc\x13\x2e\x93\x7e\x2d\x93\x49\xcd\x2e\xd7\x7b\xb1\x04\x9c\xeb\x69\x2a\x2e\xc5\xb1\x7a\xd6\x15\x02\xa6\x4c"[..],
                                                &b"\x1e\xc9\x1d\x30\x58\x57\x3f\xa6\xc0\x56\x4a\x02\xa1\xa0\x10\x16\x0c\x31\x3b\xc7\xc7\x35\x10\xdc\x98\x3e\x54\x61\x68\x2b\x5b\xe0\x0d\xbc\xe7\xe2\xc6\x82\xad\x73\xf2\x9c\xa8\x22\xcd\xc1\x11\xf6\x8f\xab\xe3\x3a\x7b\x38\x4a\x64\x83\x42\xc3\xcd\xb9\xf0\x50\xbc\xdb"[..]).unwrap();
            let z = &b"\x01\x01\xe4\x62\xe9\xd9\x15\x99\x68\xf6\x44\x0e\x95\x6f\x11\xdc\xf2\x22\x7a\xe4\xae\xa8\x16\x67\x12\x2b\x6a\xf9\x23\x9a\x29\x1e\xb5\xd6\xcf\x5a\x40\x87\xf3\x58\x52\x5f\xca\xcf\xa4\x6b\xb2\xdb\x01\xa7\x5a\xf1\xba\x51\x9b\x2d\x31\xda\x33\xed\xa8\x7a\x9d\x56\x57\x48"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x4e\xfd\x5d\xbd\x2f\x97\x9e\x38\x31\xce\x98\xf8\x23\x55\xd6\xca\x14\xa5\x75\x78\x42\x87\x58\x82\x99\x0a\xb8\x5a\xb9\xb7\x35\x2d\xd6\xb9\xb2\xf4\xea\x9a\x1e\x95\xc3\x88\x0d\x65\xd1\xf3\x60\x2f\x9c\xa6\x53\xdc\x34\x6f\xac\x85\x86\x58\xd7\x56\x26\xf4\xd4\xfb\x08"[..],
                                                  &b"\x61\xcf\x15\xdb\xda\xa7\xf3\x15\x89\xc9\x84\x00\x37\x3d\xa2\x84\x50\x6d\x70\xc8\x9f\x07\x4e\xd2\x62\xa9\xe2\x81\x40\x79\x6b\x72\x36\xc2\xee\xf9\x90\x16\x08\x5e\x71\x55\x2f\xf4\x88\xc7\x2b\x73\x39\xfe\xfb\x79\x15\xc3\x84\x59\xcb\x20\xab\x85\xae\xc4\xe4\x50\x52"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x5b\xac\xff\xf2\x68\xac\xf6\x55\x3c\x3c\x58\x3b\x46\x4e\xa3\x6a\x1d\x35\xe2\xb2\x57\xa5\xd4\x9e\xb3\x41\x9d\x5a\x09\x50\x87\xc2\xfb\x4d\x15\xcf\x5b\xf5\xaf\x81\x6d\x0f\x3f\xf7\x58\x64\x90\xcc\xd3\xdd\xc1\xa9\x8b\x39\xce\x63\x74\x9c\x62\x88\xce\x0d\xbd\xac\x7d"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x36\xe4\x88\xda\x75\x81\x47\x2a\x9d\x8e\x62\x8c\x58\xd6\xad\x72\x73\x11\xb7\xe6\xa3\xf6\xae\x33\xa8\x54\x4f\x34\xb0\x92\x80\x24\x90\x20\xbe\x71\x96\x91\x6f\xaf\xd9\x0e\x2e\xc5\x4b\x66\xb5\x46\x8d\x23\x61\xb9\x9b\x56\xfa\x00\xd7\xac\x37\xab\xb8\xc6\xf1\x66\x53"[..],
                                                &b"\x01\x1e\xdb\x9f\xb8\xad\xb6\xa4\x3f\x4f\x5f\x5f\xdc\x14\x21\xc9\xfe\x04\xfc\x8b\xa4\x6c\x9b\x66\x33\x4e\x3a\xf9\x27\xc8\xbe\xfb\x43\x07\x10\x4f\x29\x9a\xce\xc4\xe3\x0f\x81\x2d\x93\x45\xc9\x72\x0d\x19\x86\x9d\xbf\xff\xd4\xca\x3e\x7d\x27\x13\xeb\x5f\xc3\xf4\x26\x15"[..]).unwrap();
            let z = &b"\x01\x41\xd6\xa4\xb7\x19\xab\x67\xea\xf0\x4a\x92\xc0\xa4\x1e\x2d\xda\x78\xf4\x35\x4f\xb9\x0b\xdc\x35\x20\x2c\xc7\x69\x9b\x9b\x04\xd4\x96\x16\xf8\x22\x55\xde\xbf\x7b\xbe\xc0\x45\xae\x58\xf9\x82\xa6\x69\x05\xfc\xfa\xe6\x9d\x68\x97\x85\xe3\x8c\x86\x8e\xb4\xa2\x7e\x7b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x29\x89\x1d\xe0\xcf\x3c\xf8\x2e\x8c\x2c\xf1\xbf\x90\xbb\x29\x6f\xe0\x0a\xb0\x8c\xa4\x5b\xb7\x89\x2e\x0e\x22\x7a\x50\x4f\xdd\x05\xd2\x38\x1a\x44\x48\xb6\x8a\xdf\xf9\xc4\x15\x3c\x87\xea\xcb\x78\x33\x0d\x8b\xd5\x25\x15\xf9\xf9\xa0\xb5\x8e\x85\xf4\x46\xbb\x4e\x10"[..],
                                                  &b"\x9e\xdd\x67\x96\x96\xd3\xd1\xd0\xef\x32\x7f\x20\x03\x83\x25\x3f\x64\x13\x68\x3d\x9e\x4f\xcc\x87\xbb\x35\xf1\x12\xc2\xf1\x10\x09\x8d\x15\xe5\x70\x1d\x7c\xee\xe4\x16\x29\x1f\xf5\xfe\xd8\x5e\x68\x7f\x72\x73\x88\xb9\xaf\xe2\x6a\x4f\x6f\xee\xd5\x60\xb2\x18\xe6\xbb"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x8e\x2c\x93\xc5\x42\x38\x76\x22\x3a\x63\x7c\xad\x36\x7c\x85\x89\xda\x69\xa2\xd0\xfc\x68\x61\x2f\x31\x92\x3a\xe5\x02\x19\xdf\x24\x52\xe7\xcc\x92\x61\x5b\x67\xf1\x7b\x57\xff\xd2\xf5\x2b\x19\x15\x4b\xb4\x0d\x77\x15\x33\x64\x20\xfd\xe2\xe8\x9f\xee\x24\x4f\x59\xdc"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xfa\x3b\x35\x11\x8d\x6c\x42\x25\x70\xf7\x24\xa2\x6f\x90\xb2\x83\x3b\x19\x23\x91\x74\xce\xa0\x81\xc5\x31\x33\xf6\x4d\xb6\x0d\x69\x40\xea\x12\x61\x29\x9c\x04\xc1\xf4\x58\x7c\xdb\x0c\x4c\x39\x61\x64\x79\xc1\xbb\x0c\x14\x67\x99\xa1\x18\x03\x2d\xcf\x98\xf8\x99\xc0"[..],
                                                &b"\x69\xf0\x40\x22\x90\x06\x15\x1f\xa3\x2b\x51\xf6\x79\xc8\x81\x6f\x7c\x17\x50\x6b\x40\x38\x09\xdc\x77\xcd\x58\xa2\xae\xc4\x30\xd9\x4d\x13\xb6\xc9\x16\xde\x99\xf3\x55\xaa\x45\xfc\xfb\xc6\x85\x3d\x68\x6c\x71\xbe\x49\x6a\x06\x7d\x24\xbf\xae\xa4\x81\x8f\xc5\x1f\x75"[..]).unwrap();
            let z = &b"\x34\x5e\x26\xe0\xab\xb1\xaa\xc1\x2b\x75\xf3\xa9\xcf\x41\xef\xe1\xc3\x36\x39\x6d\xff\xa4\xa0\x67\xa4\xc2\xcf\xeb\x87\x8c\x68\xb2\xb0\x45\xfa\xa4\xe5\xb4\xe6\xfa\x46\x78\xf5\xb6\x03\xc3\x51\x90\x3b\x14\xbf\x9a\x6a\x70\xc4\x39\x25\x71\x99\xa6\x40\x89\x0b\x61\xd1"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xa3\xc2\x02\x40\xe5\x9f\x5b\x7a\x3e\x17\xc2\x75\xd2\x31\x4b\xa1\x74\x12\x10\xad\x58\xb7\x10\x36\xf8\xc8\x3c\xc1\xf6\xb0\xf4\x09\xdf\xdd\x91\x13\xe9\x4b\x67\xec\x39\xc3\x29\x14\x26\xc2\x3f\xfc\xc4\x47\x05\x46\x70\xd2\x90\x8f\xf8\xfe\x67\xdc\x23\x06\x03\x4c\x5c"[..],
                                                  &b"\x01\xd2\x82\x5b\xfd\x3a\xf8\xb1\xe1\x32\x05\x78\x0c\x13\x7f\xe9\x38\xf8\x4f\xde\x40\x18\x8e\x61\xea\x02\xce\xad\x81\xba\xdf\xdb\x42\x5c\x29\xf7\xd7\xfb\x03\x24\xde\xba\xdc\x10\xbb\xb9\x3d\xe6\x8f\x62\xc3\x50\x69\x26\x82\x83\xf5\x26\x58\x65\xdb\x57\xa7\x9f\x7b\xf7"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x04\xd4\x9d\x39\xd4\x0d\x81\x11\xbf\x16\xd2\x8c\x59\x36\x55\x43\x26\xb1\x97\x35\x3e\xeb\xbc\xf4\x75\x45\x39\x3b\xc8\xd3\xaa\xf9\x8f\x14\xf5\xbe\x70\x74\xbf\xb3\x8e\x6c\xc9\x7b\x98\x97\x54\x07\x4d\xad\xdb\x30\x45\xf4\xe4\xce\x74\x56\x69\xfd\xb3\xec\x0d\x5f\xa8"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x01\x2e\xc2\x26\xd0\x50\xce\x07\xc7\x9b\x3d\xf4\xd0\xf0\x89\x1f\x9f\x7a\xdf\x46\x2e\x8c\x98\xdb\xc1\xa2\xa1\x4f\x5e\x53\xa3\xf5\xad\x89\x44\x33\x58\x7c\xc4\x29\xa8\xbe\x9e\xa1\xd8\x4f\xa3\x3b\x18\x03\x69\x0d\xae\x04\xda\x72\x18\xd3\x00\x26\x15\x7f\xc9\x95\xcf\x52"[..],
                                                &b"\x48\x37\xdf\xbf\x34\x26\xf5\x7b\x5c\x79\x32\x69\x13\x0a\xbb\x9a\x38\xf6\x18\x53\x22\x11\x93\x11\x54\xdb\x4e\xeb\x9a\xed\xe8\x8e\x57\x29\x0f\x84\x2e\xa0\xf2\xea\x9a\x5f\x74\xc6\x20\x3a\x39\x20\xfe\x4e\x30\x5f\x61\x18\xf6\x76\xb1\x54\xe1\xd7\x5b\x9c\xb5\xeb\x88"[..]).unwrap();
            let z = &b"\x6f\xe9\xde\x6f\xb8\xe6\x72\xe7\xfd\x15\x0f\xdc\x5e\x61\x7f\xab\xb0\xd4\x39\x06\x35\x4c\xcf\xd2\x24\x75\x7c\x72\x76\xf7\xa1\x01\x00\x91\xb1\x7e\xd0\x72\x07\x4f\x8d\x10\xa5\xec\x97\x1e\xb3\x5a\x5c\xb7\x07\x66\x03\xb7\xbc\x38\xd4\x32\xcb\xc0\x59\xf8\x0f\x94\x88"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x7e\x2d\x13\x8f\x28\x32\xe3\x45\xae\x8f\xf6\x59\x57\xe4\x0e\x5e\xc7\x16\x3f\x01\x6b\xdf\x6d\x24\xa2\x24\x3d\xaa\x63\x1d\x87\x8a\x4a\x16\x78\x39\x90\xc7\x22\x38\x21\x30\xf9\xe5\x1f\x0c\x1b\xd6\xff\x5a\xc9\x67\x80\xe4\x8b\x68\xf5\xde\xc9\x5f\x42\xe6\x14\x4b\xb5"[..],
                                                  &b"\xb0\xde\x5c\x89\x67\x91\xf5\x28\x86\xb0\xf0\x99\x13\xe2\x6e\x78\xdd\x0b\x69\x79\x8f\xc4\xdf\x6d\x95\xe3\xca\x70\x8e\xcb\xcb\xcc\xe1\xc1\x89\x5f\x55\x61\xbb\xab\xaa\xe3\x72\xe9\xe6\x7e\x6e\x1a\x3b\xe6\x0e\x19\xb4\x70\xcd\xf6\x73\xec\x1f\xc3\x93\xd3\x42\x6e\x20"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\x1a\x5d\x1c\xc7\x9c\xd2\xbf\x73\xea\x10\x6f\x0e\x60\xa5\xac\xe2\x20\x81\x3b\x53\xe2\x7b\x73\x98\x64\x33\x4a\x07\xc0\x33\x67\xef\xda\x7a\x46\x19\xfa\x6e\xef\x3a\x97\x46\x49\x22\x83\xb3\xc4\x45\x61\x0a\x02\x3a\x9c\xc4\x9b\xf4\x59\x11\x40\x38\x4f\xca\x5c\x8b\xb5"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xeb\x07\xc7\x33\x2e\xed\xb7\xd3\x03\x60\x59\xd3\x5f\x7d\x22\x88\xd4\x37\x7d\x5f\x42\x33\x7a\xd3\x96\x40\x79\xfb\x12\x0c\xcd\x4c\x8b\xd3\x84\xb5\x85\x62\x10\x55\x21\x70\x23\xac\xd9\xa9\x4f\xcb\x3b\x96\x5b\xfb\x39\x46\x75\xe7\x88\xad\xe4\x1a\x1d\xe7\x3e\x62\x0c"[..],
                                                &b"\x49\x1a\x83\x5d\xe2\xe6\xe7\xde\xb7\xe0\x90\xf4\xa1\x1f\x2c\x46\x0c\x0b\x1f\x3d\x5e\x94\xee\x8d\x75\x10\x14\xdc\x72\x07\x84\xfd\x3b\x54\x50\x0c\x86\xeb\xae\xf1\x84\x29\xf0\x9e\x8e\x87\x6d\x5d\x15\x38\x96\x8a\x03\x0d\x77\x15\xdd\xe9\x9f\x0d\x8f\x06\xe2\x9d\x59"[..]).unwrap();
            let z = &b"\x01\xe4\xe7\x59\xec\xed\xce\x10\x13\xba\xf7\x3e\x6f\xcc\x0b\x92\x45\x1d\x03\xbd\xd5\x04\x89\xb7\x88\x71\xc3\x33\x11\x49\x90\xc9\xba\x6a\x9b\x2f\xc7\xb1\xa2\xd9\xa1\x79\x4c\x1b\x60\xd9\x27\x9a\xf6\xf1\x46\xf0\xbb\xfb\x06\x83\x14\x04\x03\xbf\xa4\xcc\xdb\x52\x4a\x29"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x11\x8c\x36\x02\x22\x09\xb1\xaf\x8e\xba\xd1\xa1\x2b\x56\x6f\xc4\x87\x44\x57\x6e\x11\x99\xfe\x80\xde\x1c\xdf\x85\x1c\xdf\x03\xe5\xb9\x09\x1a\x8f\x7e\x07\x9e\x83\xb7\xf8\x27\x25\x9b\x69\x1d\x0c\x22\xee\x29\xd6\xbd\xf7\x3e\xc7\xbb\xfd\x74\x6f\x2c\xd9\x7a\x35\x7d"[..],
                                                  &b"\xda\x5f\xf4\x90\x45\x48\xa3\x42\xe2\xe7\xba\x6a\x1f\x4e\xe5\xf8\x40\x41\x1a\x96\xcf\x63\xe6\xfe\x62\x2f\x22\xc1\x3e\x61\x4e\x0a\x84\x7c\x11\xa1\xab\x3f\x1d\x12\xcc\x85\x0c\x32\xe0\x95\x61\x4c\xa8\xf7\xe2\x72\x14\x77\xb4\x86\xe9\xff\x40\x37\x29\x77\xc3\xf6\x5c"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\x0c\x90\x8c\xaf\x1b\xe7\x4c\x61\x6b\x62\x5f\xc8\xc1\xf5\x14\x44\x6a\x6a\xec\x83\xb5\x93\x71\x41\xd6\xaf\xbb\x0a\x8c\x76\x66\xa7\x74\x6f\xa1\xf7\xa6\x66\x4a\x21\x23\xe8\xcd\xf6\xcd\x8b\xf8\x36\xc5\x6d\x3c\x0e\xbd\xcc\x98\x0e\x43\xa1\x86\xf9\x38\xf3\xa7\x8a\xe7"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x31\x89\x0f\x4c\x7a\xbe\xc3\xf7\x23\x36\x22\x85\xd7\x7d\x26\x36\xf8\x76\x81\x7d\xb3\xbb\xc8\x8b\x01\xe7\x73\x59\x7b\x96\x9f\xf6\xf0\x13\xea\x47\x0c\x85\x4a\xb4\xa7\x73\x90\x04\xeb\x8c\xbe\xa6\x9b\x82\xdd\xf3\x6a\xca\xdd\x40\x68\x71\x79\x8e\xcb\x2a\xc3\xaa\x7f"[..],
                                                &b"\xd8\xb4\x29\xae\x32\x50\x26\x6b\x96\x43\xc0\xc7\x65\xa6\x0d\xc1\x01\x55\xbc\x25\x31\xcf\x86\x27\x29\x6f\x49\x78\xb6\x64\x0a\x9e\x60\x0e\x19\xd0\x03\x7d\x58\x50\x3f\xa8\x07\x99\x54\x6a\x81\x4d\x74\x78\xa5\x50\xaa\x90\xe5\xeb\xeb\x05\x25\x27\xfa\xae\xae\x5d\x08"[..]).unwrap();
            let z = &b"\x01\x63\xc9\x19\x1d\x65\x10\x39\xa5\xfe\x98\x5a\x0e\xea\x1e\xba\x01\x8a\x40\xab\x19\x37\xfc\xd2\xb6\x12\x20\x82\x0e\xe8\xf2\x30\x2e\x97\x99\xf6\xed\xfc\x3f\x51\x74\xf3\x69\xd6\x72\xd3\x77\xea\x89\x54\xa8\xd0\xc8\xb8\x51\xe8\x1a\x56\xfd\xa9\x52\x12\xa6\x57\x8f\x0e"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x78\x0e\xdf\xf1\xca\x1c\x03\xcf\xbe\x59\x3e\xdc\x6c\x04\x9b\xcb\x28\x60\x29\x4a\x92\xc3\x55\x48\x9d\x9a\xfb\x2e\x70\x20\x75\xad\xe1\xc9\x53\x89\x5a\x45\x62\x30\xa0\xcd\xe9\x05\xde\x4a\x3f\x38\x57\x3d\xbf\xcc\xcd\x67\xad\x6e\x7e\x93\xf0\xb5\x58\x1e\x92\x6a\x5d"[..],
                                                  &b"\xa5\x48\x19\x62\xc9\x16\x29\x62\xe7\xf0\xeb\xde\xc9\x36\x93\x5d\x0e\xaa\x81\x3e\x82\x26\xd4\x0d\x7f\x61\x19\xbf\xd9\x40\x60\x23\x80\xc8\x67\x21\xe6\x1d\xb1\x83\x0f\x51\xe1\x39\xf2\x10\x00\x0b\xce\xc0\xd8\xed\xd3\x9e\x54\xd7\x3a\x9a\x12\x9f\x95\xcd\x5f\xa9\x79"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x01\xb3\x7d\x6b\x72\x88\xde\x67\x13\x60\x42\x5d\x3e\x5a\xc1\xcc\xb2\x18\x15\x07\x9d\x8d\x73\x43\x1e\x9b\x74\xa6\xf0\xe7\xae\x00\x4a\x35\x75\x75\xb1\x1a\xd6\x66\x42\xce\x8b\x77\x55\x93\xeb\xa9\xd9\x8b\xf2\x5c\x75\xef\x0b\x4d\x3a\x20\x98\xbb\xc6\x41\xf5\x9a\x2b\x77"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x18\x9a\x5e\xe3\x4d\xe7\xe3\x5a\xef\xea\xee\xf9\x22\x0c\x18\x07\x1b\x4c\x29\xa4\xc3\xbd\x9d\x95\x44\x58\xbd\x3e\x82\xa7\xa3\x4d\xa3\x4c\xff\x55\x79\xb8\x10\x1c\x06\x5b\x1f\x2f\x52\x7c\xf4\x58\x15\x01\xe2\x8e\xf5\x67\x18\x73\xe6\x52\x67\x73\x3d\x00\x35\x20\xaf"[..],
                                                &b"\x01\xeb\x4b\xc5\x0a\x7b\x4d\x45\x99\xd7\xe3\xfa\x77\x3d\xdb\x9e\xb2\x52\xc9\xb3\x42\x28\x72\xe5\x44\xbd\xf7\x5c\x7b\xf6\x0f\x51\x66\xdd\xc1\x1e\xb0\x8f\xa7\xc3\x08\x22\xda\xba\xee\x37\x3a\xb4\x68\xeb\x2d\x92\x2e\x48\x4e\x2a\x52\x7f\xff\x2e\xbb\x80\x4b\x7d\x9a\x37"[..]).unwrap();
            let z = &b"\x01\x5d\x61\x3e\x26\x7a\x36\x34\x2e\x0d\x12\x5c\xda\xd6\x43\xd8\x0d\x97\xed\x06\x00\xaf\xb9\xe6\xb9\x54\x5c\x9e\x64\xa9\x8c\xc6\xda\x7c\x5a\xaa\x3a\x8d\xa0\xbd\xd9\xdd\x3b\x97\xe9\x78\x82\x18\xa8\x0a\xba\xfc\x10\x6e\xf0\x65\xc8\xf1\xc4\xe1\x11\x9e\xf5\x8d\x29\x8b"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\x6d\xac\xff\xa1\x83\xe5\x30\x30\x83\xa3\x34\xf7\x65\xde\x72\x4e\xc5\xec\x94\x02\x02\x6d\x47\x97\x88\x4a\x98\x28\xa0\xd3\x21\xa8\xcf\xac\x74\xab\x73\x7f\xe2\x0a\x7d\x6b\xef\xcf\xc7\x3b\x6a\x35\xc1\xc7\xb0\x1d\x37\x3e\x31\xab\xc1\x92\xd4\x8a\x42\x41\xa3\x58\x03"[..],
                                                  &b"\x01\x1e\x53\x27\xca\xc2\x2d\x30\x5e\x71\x56\xe5\x59\x17\x6e\x19\xbe\xe7\xe4\xf2\xf5\x9e\x86\xf1\xa9\xd0\xb6\x60\x3b\x6a\x7d\xf1\x06\x9b\xde\x63\x87\xfe\xb7\x15\x87\xb8\xff\xce\x5b\x26\x6e\x1b\xae\x86\xde\x29\x37\x8a\x34\xe5\xc7\x4b\x67\x24\xc4\xd4\x0a\x71\x99\x23"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\xf2\x66\x1a\xc7\x62\xf6\x0c\x5f\xff\x23\xbe\x5d\x96\x9c\xcd\x4e\xc6\xf9\x8e\x4e\x72\x61\x8d\x12\xbd\xcd\xb9\xb4\x10\x21\x62\x33\x37\x88\xc0\xba\xe5\x9f\x91\xcd\xfc\x17\x2c\x7a\x16\x81\xee\x44\xd9\x6a\xb2\x13\x5a\x6e\x5f\x34\x15\xeb\xbc\xd5\x51\x65\xb1\xaf\xb0"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xa8\xe2\x5a\x69\x02\xd6\x87\xb4\x78\x7c\xdc\x94\xc3\x64\xac\x7c\xec\xc5\xc4\x95\x48\x3e\xd3\x63\xdc\x0a\xa9\x5e\xe2\xbd\x73\x9c\x4c\x4d\x46\xb1\x70\x06\xc7\x28\xb0\x76\x35\x0d\x7d\x7e\x54\xc6\x82\x2f\x52\xf4\x71\x62\xa2\x51\x09\xaa\xab\xa6\x90\xca\xb6\x96\xec"[..],
                                                &b"\x01\x68\xd2\xf0\x8f\xe1\x9e\x4d\xc9\xee\x7a\x19\x5b\x03\xc9\xf7\xfe\x66\x76\xf9\xf5\x20\xb6\x27\x05\x57\x50\x4e\x72\xca\x43\x94\xa2\xc6\x91\x86\x25\xe1\x5a\xc0\xc5\x1b\x8f\x95\xcd\x56\x01\x23\x65\x3f\xb8\xe8\xee\x6d\xb9\x61\xe2\xc4\xc6\x2c\xc5\x4e\x92\xe2\xa2\xa9"[..]).unwrap();
            let z = &b"\x01\x4d\x60\x82\xa3\xb5\xce\xd1\xab\x8c\xa2\x65\xa8\x10\x6f\x30\x21\x46\xc4\xac\xb8\xc3\x0b\xb1\x4a\x4c\x99\x1e\x3c\x82\xa9\x73\x12\x88\xbd\xb9\x1e\x0e\x85\xbd\xa3\x13\x91\x2d\x06\x38\x4f\xc4\x4f\x21\x53\xfb\x13\x50\x6f\xa9\xcf\x43\xc9\xaa\xb5\x75\x09\x88\xc9\x43"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\xa0\x91\x42\x1d\x37\x03\xe3\xb3\x41\xe9\xf1\xe7\xd5\x8f\x8c\xf7\xbd\xbd\x17\x98\xd0\x01\x96\x7b\x80\x1d\x1c\xec\x27\xe6\x05\xc5\x80\xb2\x38\x7c\x1c\xb4\x64\xf5\x5c\xe7\xac\x80\x33\x41\x02\xab\x03\xcf\xb8\x6d\x88\xaf\x76\xc9\xf4\x12\x9c\x01\xbe\xdd\x3b\xbf\xc4"[..],
                                                  &b"\x8c\x9c\x57\x7a\x8e\x6f\xc4\x46\x81\x5e\x9d\x40\xba\xa6\x60\x25\xf1\x5d\xae\x28\x5f\x19\xeb\x66\x8e\xe6\x0a\xe9\xc9\x8e\x7e\xcd\xbf\x2b\x2a\x68\xe2\x29\x28\x05\x9f\x67\xdb\x18\x80\x07\x16\x1d\x3e\xcf\x39\x7e\x08\x83\xf0\xc4\xeb\x7e\xaf\x78\x27\xa6\x22\x05\xcc"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\xf4\x30\xca\x12\x61\xf0\x96\x81\xa9\x28\x2e\x9e\x97\x0a\x92\x34\x22\x7b\x1d\x5e\x58\xd5\x58\xc3\xcc\x6e\xff\x44\xd1\xbd\xf5\x3d\xe1\x6a\xd5\xee\x2b\x18\xb9\x2d\x62\xfc\x79\x58\x61\x16\xb0\xef\xc1\x5f\x79\x34\x0f\xb7\xea\xf5\xce\x6c\x44\x34\x1d\xcf\x8d\xde\x27"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x6c\x1d\x9b\x5e\xca\x87\xde\x1f\xb8\x71\xa0\xa3\x2f\x80\x7c\x72\x5a\xdc\xcd\xe9\xb3\x96\x74\x53\xa7\x13\x47\xd6\x08\xf0\xc0\x30\xcd\x09\xe3\x38\xcd\xec\xbf\x4a\x02\x01\x5b\xc8\xa6\xe8\xd3\xe2\x59\x5f\xe7\x73\xff\xc2\xfc\x4e\x4a\x55\xd0\xb1\xa2\xcc\x00\x32\x3b"[..],
                                                &b"\x01\x14\x1b\x21\x09\xe7\xf4\x98\x1c\x95\x2a\xa8\x18\xa2\xb9\xf6\xf5\xc4\x1f\xec\xcd\xb7\xa7\xa4\x5b\x9b\x4b\x67\x29\x37\x77\x1b\x00\x8c\xae\x5f\x93\x4d\xfe\x3f\xed\x10\xd3\x83\xab\x1f\x38\x76\x9c\x92\xce\x88\xd9\xbe\x54\x14\x81\x7e\xcb\x07\x3a\x31\xab\x36\x8c\xcb"[..]).unwrap();
            let z = &b"\x20\xc0\x07\x47\xcb\x8d\x49\x2f\xd4\x97\xe0\xfe\xc5\x46\x44\xbf\x02\x7d\x41\x8a\xb6\x86\x38\x1f\x10\x97\x12\xa9\x9c\xab\xe3\x28\xb9\x74\x3d\x22\x25\x83\x6f\x9a\xd6\x6e\x5d\x7f\xed\x1d\xe2\x47\xe0\xda\x92\xf6\x0d\x5b\x31\xf9\xe4\x76\x72\xe5\x7f\x71\x05\x98\xf4"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x4f\x38\x81\x66\x81\x77\x12\x89\xce\x0c\xb8\x3a\x5e\x29\xa1\xab\x06\xfc\x91\xf7\x86\x99\x4b\x23\x70\x8f\xf0\x8a\x08\xa0\xf6\x75\xb8\x09\xae\x99\xe9\xf9\x96\x7e\xb1\xa4\x9f\x19\x60\x57\xd6\x9e\x50\xd6\xde\xdb\x4d\xd2\xd9\xa8\x1c\x02\xbd\xcc\x8f\x7f\x51\x84\x60"[..],
                                                  &b"\x9e\xfb\x24\x4c\x8b\x91\x08\x7d\xe1\xee\xd7\x66\x50\x0f\x0e\x81\x53\x07\x52\xd4\x69\x25\x6e\xf7\x9f\x6b\x96\x5d\x8a\x22\x32\xa0\xc2\xdb\xc4\xe8\xe1\xd0\x92\x14\xba\xb3\x84\x85\xbe\x6e\x35\x7c\x42\x00\xd0\x73\xb5\x2f\x04\xe4\xa1\x6f\xc6\xf5\x24\x71\x87\xae\xcb"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\x5d\xc3\x3a\xed\xa0\x3c\x2e\xb2\x33\x01\x4e\xe4\x68\xdf\xf7\x53\xb7\x2f\x73\xb0\x09\x91\x04\x3e\xa3\x53\x82\x8a\xe6\x9d\x4c\xd0\xfa\xde\xda\x7b\xb2\x78\xb5\x35\xd7\xc5\x74\x06\xff\x2e\x6e\x47\x3a\x5a\x4f\xf9\x8e\x90\xf9\x0d\x6d\xad\xd2\x51\x00\xe8\xd8\x56\x66"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\xc8\x25\xba\x30\x73\x73\xce\xc8\xdd\x24\x98\xee\xf8\x2e\x21\xfd\x98\x62\x16\x8d\xbf\xeb\x83\x59\x39\x80\xca\x9f\x82\x87\x53\x33\x89\x9f\xe9\x4f\x13\x7d\xaf\x1c\x41\x89\xeb\x50\x29\x37\xc3\xa3\x67\xea\x79\x51\xed\x8b\x0f\x33\x77\xfc\xdf\x29\x22\x02\x1d\x46\xa5"[..],
                                                &b"\x01\x6b\x8a\x25\x40\xd5\xe6\x54\x93\x88\x8b\xc3\x37\x24\x9e\x67\xc0\xa6\x87\x74\xf3\xe8\xd8\x1e\x3b\x45\x74\xa0\x12\x51\x65\xf0\xbd\x58\xb8\xaf\x9d\xe7\x4b\x35\x83\x25\x39\xf9\x5c\x3c\xd9\xf1\xb7\x59\x40\x85\x60\xaa\x68\x51\xae\x3a\xc7\x55\x53\x47\xb0\xd3\xb1\x3b"[..]).unwrap();
            let z = &b"\xc2\xbf\xaf\xcd\x7f\xbd\x3e\x2f\xd1\xc7\x50\xfd\xea\x61\xe7\x0b\xd4\x78\x7a\x7e\x68\x46\x8c\x57\x4e\xe9\x9e\xbc\x47\xee\xde\xf0\x64\xe8\x94\x4a\x73\xbc\xb7\x91\x3d\xba\xb5\xd9\x3d\xca\x66\x0d\x21\x6c\x55\x36\x22\x36\x27\x94\xf7\xa2\xac\xc7\x10\x22\xbd\xb1\x6f"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }

        {
            let cav_pub = Point::new::<Secp521r1>(&b"\x01\xa3\x20\x99\xb0\x2c\x0b\xd8\x53\x71\xf6\x0b\x0d\xd2\x08\x90\xe6\xc7\xaf\x04\x8c\x81\x79\x89\x0f\xda\x30\x8b\x35\x9d\xbb\xc2\xb7\xa8\x32\xbb\x8c\x65\x26\xc4\xaf\x99\xa7\xea\x3f\x0b\x3c\xb9\x6a\xe1\xeb\x76\x84\x13\x27\x95\xc4\x78\xad\x6f\x96\x2e\x4a\x6f\x44\x6d"[..],
                                                  &b"\x01\x76\x27\x35\x7b\x39\xe9\xd7\x63\x2a\x13\x70\xb3\xe9\x3c\x1a\xfb\x5c\x85\x1b\x91\x0e\xb4\xea\xd0\xc9\xd3\x87\xdf\x67\xcd\xe8\x50\x03\xe0\xe4\x27\x55\x2f\x1c\xd0\x90\x59\xaa\xd0\x26\x2e\x23\x5c\xce\x5f\xba\x8c\xed\xc4\xfd\xc1\x46\x3d\xa7\x6d\xcd\x4b\x6d\x1a\x46"[..]).unwrap();
            let iut_priv = Scalar::new::<Secp521r1>(&b"\xdf\x14\xb1\xf1\x43\x2a\x7b\x0f\xb0\x53\x96\x5f\xd8\x64\x3a\xfe\xe2\x6b\x24\x51\xec\xb6\xa8\xa5\x3a\x65\x5d\x5f\xbe\x16\xe4\xc6\x4c\xe8\x64\x72\x25\xeb\x11\xe7\xfd\xcb\x23\x62\x74\x71\xdf\xfc\x5c\x25\x23\xbd\x2a\xe8\x99\x57\xcb\xa3\xa5\x7a\x23\x93\x3e\x5a\x78"[..]).unwrap();
            let iut_pub = Point::new::<Secp521r1>(&b"\x4e\x85\x83\xbb\xbb\x2e\xcd\x93\xf0\x71\x4c\x33\x2d\xff\x5a\xb3\xbc\x63\x96\xe6\x2f\x3c\x56\x02\x29\x66\x43\x29\xba\xa5\x13\x8c\x3b\xb1\xc3\x64\x28\xab\xd4\xe2\x3d\x17\xfc\xb7\xa2\xcf\xcc\x22\x4b\x2e\x73\x4c\x89\x41\xf6\xf1\x21\x72\x2d\x7b\x6b\x94\x15\x45\x76"[..],
                                                &b"\x01\xcf\x08\x74\xf2\x04\xb0\x36\x3f\x02\x08\x64\x67\x2f\xad\xbf\x87\xc8\x81\x1e\xb1\x47\x75\x8b\x25\x4b\x74\xb1\x4f\xae\x74\x21\x59\xf0\xf6\x71\xa0\x18\x21\x2b\xbf\x25\xb8\x51\x9e\x12\x6d\x4c\xad\x77\x8c\xff\xf5\x0d\x28\x8f\xd3\x9c\xeb\x0c\xac\x63\x5b\x17\x5e\xc0"[..]).unwrap();
            let z = &b"\x01\xaa\xf2\x4e\x5d\x47\xe4\x08\x0c\x18\xc5\x5e\xa3\x55\x81\xcd\x8d\xa3\x0f\x1a\x07\x95\x65\x04\x5d\x20\x08\xd5\x1b\x12\xd0\xab\xb4\x41\x1c\xda\x7a\x07\x85\xb1\x5d\x14\x9e\xd3\x01\xa3\x69\x70\x62\xf4\x2d\xa2\x37\xaa\x7f\x07\xe0\xaf\x3f\xd0\x0e\xb1\x80\x0d\x9c\x41"[..];

            let got_pub = point_mul_g(&iut_priv);
            assert_eq!(got_pub.as_bytes(), iut_pub.as_bytes());

            let got_z = point_mul(&iut_priv, &cav_pub).unwrap();
            assert_eq!(&got_z.as_bytes().0[..], z);
        }
    }
}
