<?php
/**
 * Vacation_Driver_forwards:: implements the Vacation_Driver API for ftp
 * driven dot-forward compliant mail servers.
 *
 * $Horde: vacation/lib/Driver/forwards.php,v 1.43.2.6 2009/01/06 15:28:08 jan Exp $
 *
 * Copyright 2001-2009 The Horde Project (http://www.horde.org/)
 *
 * See the enclosed file LICENSE for license information (BSD). If you
 * did not receive this file, see http://www.horde.org/licenses/bsdl.php.
 *
 * @author  Eric Rostetter <eric.rostetter@physics.utexas.edu>
 * @author  Jan Schneider <jan@horde.org>
 * @package Vacation
 */
class Vacation_Driver_forwards extends Vacation_Driver {

    /**
     * The FTP stream we open via the VFS class.
     *
     * @var VFS_ftp
     */
    var $_vfs;

    /**
     * The current vacation details.
     *
     * @var array
     */
    var $_details = null;

    /**
     * Sets up vacation notices for a user.
     *
     * @param string $password  The password for the user.
     * @param string $message   The text of the vacation notice.
     * @param string $subject   The subject of the vacation notice.
     * @param string $from      The From: address of the vacation notice.
     * @param string $alias     The alias email address.
     */
    function setVacation($password, $message, $subject, $from, $alias = '')
    {
        // Make sure the configuration file is correct
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // We need to find out what type of database file to use
        $dbfile = VACATION_BASE . '/files/empty.'
            . $this->getParam('dbtype') . '.bin';

        // Connect to the server.
        if (is_a($connected = $this->_connect($password), 'PEAR_Error')) {
            return $connected;
        }

        // Run any pre-checks.
        $result = $this->_setVacationPreCheck();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        // Build the message.
        $full_message = $this->_buildMessage($message, $subject, $from);

        // Set up the vacation specific files first.
        $status = $this->_vfs->writeData('', '.vacation.msg', $full_message);
        if (is_a($status, 'PEAR_Error')) {
            return $status;
        }
        foreach (array('.vacation.pag' => $dbfile,
                       '.vacation.dir' => $dbfile,
                       '.vacation.db'  => $dbfile) as $file => $content) {
            $status = $this->_vfs->write('', $file, $content);
            if (is_a($status, 'PEAR_Error')) {
                return $status;
            }
        }

        // Create the vacation file.
        $status = $this->_createVacationFile($alias);
        if (is_a($status, 'PEAR_Error')) {
            return $status;
        }

        // Update the current details.
        $this->_details = array('vacation' => 'Y',
                                'message' => $message,
                                'subject' => $subject,
                                'from' => $from);
    }

    /**
     * Runs any code required before creating the vacation file.
     *
     * @see Vacation_Driver_qmail::
     */
    function _setVacationPreCheck()
    {
    }

    /**
     * Creates the main vacation file.
     *
     * @param string $alias  The alias email address.
     */
    function _createVacationFile($alias)
    {
        // Parse the email address and alias address passed in.
        $my_email = $this->_makeEmailAddress($this->_user);
        if (is_a($my_email, 'PEAR_Error')) {
            return $my_email;
        }
        if (!empty($alias)) {
            $alias_list = preg_split('/[,\s]+/', $alias);
            foreach ($alias_list as $i => $elt) {
                $addr = $this->_makeEmailAddress($elt);
                if (is_a($addr, 'PEAR_Error')) {
                    return $addr_email;
                }
                $alias_list[$i] = '-a ' . escapeshellarg($addr);
            }
            $alias = join(' ', $alias_list);
        } else {
            $alias = '';
        }

        // Now set up the .forward file
        if (!empty($alias) && ($alias != $my_email)) {
           $contents = "\\$my_email, \"|"
               . $GLOBALS['conf']['vacation']['path']
               . " $alias $my_email\"";
        } else {
           $contents = "\\$my_email, \"|"
               . $GLOBALS['conf']['vacation']['path']
               . " $my_email\"";
        }
        $status = $this->_vfs->writeData('', '.forward', $contents);
        if (is_a($status, 'PEAR_Error')) {
            return $status;
        }

        // Try to change the permissions, but ignore any errors.
        $this->_vfs->changePermissions('', '.forward', '0600');
    }

    /**
     * Removes any existing vacation notices.
     *
     * @param string $password  The password of the user.
     */
    function unsetVacation($password)
    {
        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password),
                 'PEAR_Error')) {
            return $connected;
        }

        $status = $this->_deleteVacationFile();
        if (is_a($status, 'PEAR_Error')) {
            return $status;
        }
        $status = $this->_vfs->deleteFile('', '.vacation.msg');
        if (is_a($status, 'PEAR_Error')) {
            return $status;
        }

        // We could, but don't, check for errors on these. They are
        // more-or-less harmless without the above two files.
        $this->_vfs->deleteFile('', '.vacation.pag');
        $this->_vfs->deleteFile('', '.vacation.dir');
        $this->_vfs->deleteFile('', '.vacation.db');

        // Update the current details.
        $this->_details = null;
    }

    /**
     * Deletes the main vacation file.
     */
    function _deleteVacationFile()
    {
        $status = $this->_vfs->deleteFile('', '.forward');
        if (is_a($status, 'PEAR_Error')) {
            $status->message .= '  '
                . _("Maybe you didn't have a vacation notice installed?");
        }
        return $status;
    }

    /**
     * Retrieves the current vacation details for the user.
     *
     * @param string $password  The password for user.
     *
     * @return array  Vacation details or PEAR_Error on failure.
     */
    function _getUserDetails($password)
    {
        if (!is_null($this->_details)) {
            return $this->_details;
        }

        if (is_a($checked = $this->_checkConfig(), 'PEAR_Error')) {
            return $checked;
        }

        // Connect to the server.
        if (is_a($connected = $this->_connect($password),
                 'PEAR_Error')) {
            return $connected;
        }

        if ($this->_vfs->exists('', '.vacation.msg')) {
            $message = $this->_vfs->read('', '.vacation.msg');
            if (is_a($message, 'PEAR_Error')) {
                return $message;
            }
            $this->_details = $this->_parseMessage($message);
        }

        if ($this->_vfs->exists('', '.forward')) {
            $forward = $this->_vfs->read('', '.forward');
            if (is_a($forward, 'PEAR_Error')) {
                return $forward;
            }
            $details = $this->_parseForward($forward);
            if ($details['vacation']['set']) {
                // Driver.php wants output in y/n format:
                $this->_details['vacation'] = 'Y';
                $this->_details['alias'] = $details['vacation']['alias'];
            } else {
                $this->_details['vacation'] = 'N';
            }
        } else {
            $this->_details['vacation'] = false;
        }

        return $this->_details;
    }

    /**
     * Checks if the realm has a specific configuration. If not, tries to fall
     * back on the default configuration. If still not a valid configuration
     * then returns an error.
     */
    function _checkConfig()
    {
        // If no host config for the realm, then we fall back to the default
        // realm.
        if (empty($this->_params[$this->_realm]['host'])) {
            $this->_realm = 'default';
        }

        // If still no host/port, then we have a misconfigured module.
        if (empty($this->_params[$this->_realm]['host']) ||
            empty($this->_params[$this->_realm]['port']) ) {
            return PEAR::raiseError(_("The vacation application is not properly configured."));
        }
    }

    /**
     * Parses a string from the .forward file.
     *
     * @param string $str   The string from the .forward file.
     *
     * @return mixed  The contents of the file in an array
     */
    function _parseForward($str)
    {
        require_once 'Horde/MIME.php';
        $adrlist = MIME::rfc822Explode($str, ',');
        foreach ($adrlist as $adr) {
            $adr = trim($adr);
            if ($adr == '\\' . $this->_user) {
                // This matches the way the forwards module writes the user
                // name.
                $content['forward']['metoo'] = true;
            } elseif (preg_match('/\|.*vacation\s*(-a\s+(.*))?\s+(.+)/',
                                 $adr, $matches)) {
                // This matches the way the vacation module writes vacation
                // command.
                $content['vacation']['alias'] = $matches[2];
            } elseif (!empty($adr)) {
                // This matches everything else.
                $buffer[] = $adr;
            }
        }
        if ($content['forward']['set'] = isset($buffer) && is_array($buffer)) {
            $content['forward']['receivers'] = implode(', ', $buffer);
        }
        $content['vacation']['set'] = isset($content['vacation']) &&
            is_array($content['vacation']);

        return $content;
    }

    /**
     * Connects to the FTP server.
     *
     * @param string $password  The password to connect with.
     */
    function _connect($password)
    {
        if ($this->_vfs) {
            return;
        }

        // Build the FTP array to pass to VFS.
        $args = array('hostspec' => $this->_params[$this->_realm]['host'],
                      'port' => $this->_params[$this->_realm]['port'],
                      'pasv' => $this->_params[$this->_realm]['pasv'],
                      'ssl' => $this->_params[$this->_realm]['ssl'],
                      'username' => $this->_user,
                      'password' => $password);

        // Create the VFS ftp driver.
        require_once 'VFS.php';
        $vfs = &VFS::singleton('ftp', $args);
        if (is_a($vfs, 'PEAR_Error')) {
            return $vfs;
        }
        $this->_vfs = &$vfs;

        // Try to login with the username/password, no realm.
        return $this->_vfs->checkCredentials();
    }

}
