/* Copyright (C) 2001, 2007 United States Government as represented by
   the Administrator of the National Aeronautics and Space Administration.
   All Rights Reserved.
 */
package gov.nasa.worldwind.servers.wms.formats;

import gov.nasa.worldwind.formats.dds.DDSConverter;

import javax.imageio.ImageIO;
import javax.imageio.ImageWriter;
import javax.imageio.stream.MemoryCacheImageOutputStream;
import javax.imageio.stream.ImageOutputStream;
import java.io.*;
import java.awt.image.BufferedImage;
import java.util.Iterator;

/**
 * @author brownrigg
 * @version $Id$
 */

abstract public class ImageFormatter {

    abstract public BufferedImage toIntermediateForm() throws IOException;

    public InputStream asPng() throws IOException {
        return intermediateToPng(toIntermediateForm());
    }

    public InputStream asDDS() throws IOException {
        return intermediateToDDS(toIntermediateForm());
    }

    protected InputStream intermediateToPng(BufferedImage image) throws IOException {
        Iterator iter = ImageIO.getImageWritersByFormatName("png");
        ImageWriter writer = (ImageWriter) iter.next();
        ByteArrayOutputStream imageBytes = new ByteArrayOutputStream();
        ImageOutputStream ios = new MemoryCacheImageOutputStream(imageBytes);
        writer.setOutput(ios);
        writer.write(image);
        imageBytes.close();
        return new ByteArrayInputStream(imageBytes.toByteArray());
    }

    protected InputStream intermediateToDDS(BufferedImage image) throws IOException {
        byte[] imageBytes = DDSConverter.convertToDxt3(image).array();
        return new ByteArrayInputStream(imageBytes);
    }

    public InputStream getStreamFromMimeType(String mimeType) throws IOException {
        if (SupportedFormats.PNG.equals(mimeType))
            return asPng();
        if (SupportedFormats.DDS.equals(mimeType))
            return asDDS();
        throw new IOException("ImageFormatter: unsupport image type: " + mimeType);
    }
    
    static public ImageFormatter getFormatterFromMimeType(String mimeType, File sourceFile) {
        if (SupportedFormats.PNG.equals(mimeType))
            return new PNGImageFormatter(sourceFile);
        return null;
    }

    // ----------------- Supported Output Formats ----------------------------

    private enum SupportedFormats {
        PNG("image/png"),
        DDS("image/dds");

        SupportedFormats(String mimeType) {
            this.mimeType = mimeType;
        }

        public boolean equals(String that) {
            return this.mimeType.equalsIgnoreCase(that);
        }

        String mimeType;
    }

    static public boolean isSupportedType(String mimeType) {
        for (SupportedFormats f : formats)
            if (f.equals(mimeType))
                return true;
        return false;
    }

    static private SupportedFormats[] formats = SupportedFormats.class.getEnumConstants();
}
