# Load init .reg script during WINEPREFIX initialisation.
# USAGE: regedit_initial
regedit_initial() {
	local script_content
	script_content=$(regedit_initial_content)

	# Return early if there is no script to load during prefix initilisation
	if [ -z "$script_content" ]; then
		return 0
	fi

	local game_id
	game_id=$(game_id)
	cat <<- EOF
	# Load initial registry keys.
	printf 'Loading initial registry keys…\\n'
	(
	    ## regedit fails to find the script if called with env --chdir.
	    cd "\${WINEPREFIX}/drive_c/${game_id}"
	    \$(regedit_command) registry-scripts/init.reg
	)

	EOF
}

# Print the content of the init .reg file.
# USAGE: regedit_initial_content
regedit_initial_content() {
	local script_content
	script_content=$(context_value 'WINE_REGISTRY_INIT')

	# Return early if there is no initial registry key to load.
	if [ -z "$script_content" ]; then
		return 0
	fi

	cat <<- EOF
	Windows Registry Editor Version 5.00

	$script_content
	EOF
}

# Write the init .reg script setting the initial keys for the given package.
# USAGE: wine_registry_script_write $package
## TODO: This should be renamed to regedit_initial_write,
##       but that can only be done in sync with game scripts overriding this function.
##       Maybe this can be handled with a temporary copy of the function,
##       and one or the other called based on the compatibility level.
wine_registry_script_write() {
	## Identifier of the package that should include that .reg script
	local package="$1"

	local script_content
	script_content=$(regedit_initial_content)
	## Return early if there is no script to write.
	if [ -z "$script_content" ]; then
		return 0
	fi

	local package_path path_game_data script_directory script_path
	package_path=$(package_path "$package")
	path_game_data=$(path_game_data)
	script_directory="${package_path}${path_game_data}/registry-scripts"
	script_path="${script_directory}/init.reg"
	mkdir --parents "$script_directory"
	printf '%s' "$script_content" |
		iconv --from-code=UTF-8 --to-code=UTF-16 --output="$script_path"
}

# WINE launcher - Set environment for registry keys persistent storage
# USAGE: wine_launcher_regedit_environment
wine_launcher_regedit_environment() {
	# Return early if no persistent registry keys are listed
	if [ -z "${WINE_REGEDIT_PERSISTENT_KEYS:-}" ]; then
		return 0
	fi

	local game_id
	game_id=$(game_id)
	cat <<- EOF
	# Set environment for registry keys persistent storage

	REGEDIT_DUMPS_WINEPREFIX_PATH="\${WINEPREFIX}/drive_c/${game_id}/wine/regedit"
	REGEDIT_PERSISTENT_KEYS='$WINE_REGEDIT_PERSISTENT_KEYS'

	EOF
	cat <<- 'EOF'
	## Convert registry key name to file path
	regedit_convert_key_to_path() {
	    printf '%s.reg' "${1##*\\}" |
	        tr '[:upper:]' '[:lower:]'
	}

	EOF
}

# WINE launcher - Store registry keys in a persistent path
# USAGE: wine_launcher_regedit_store
wine_launcher_regedit_store() {
	# Return early if no persistent registry keys are listed
	if [ -z "${WINE_REGEDIT_PERSISTENT_KEYS:-}" ]; then
		return 0
	fi

	## TODO: env --chdir could be used instead of cd in a subshell.
	cat <<- 'EOF'
	# Store registry keys in a persistent path

	while read -r registry_key; do
	    if [ -z "$registry_key" ]; then
	        continue
	    fi
	    registry_dump="${REGEDIT_DUMPS_WINEPREFIX_PATH}/$(regedit_convert_key_to_path "$registry_key")"
	    registry_dump_directory=$(dirname "$registry_dump")
	    mkdir --parents "$registry_dump_directory"
	    printf 'Dumping registry key in "%s".\n' "$registry_dump"
	    $(regedit_command) -E "$registry_dump" "$registry_key"
	done << EOL
	$(printf '%s' "$REGEDIT_PERSISTENT_KEYS")
	EOL
	unset registry_key registry_dump registry_dump_directory

	mkdir --parents "$PATH_WINE_REGISTRY"
	(
	    cd "$REGEDIT_DUMPS_WINEPREFIX_PATH"
	    find . -type f \
	        -exec cp --force --parents --target-directory="$PATH_WINE_REGISTRY" {} +
	)

	EOF
}

# WINE launcher - Load registry keys from persistent dumps
# USAGE: wine_launcher_regedit_load
wine_launcher_regedit_load() {
	# Return early if no persistent registry keys are listed
	if [ -z "${WINE_REGEDIT_PERSISTENT_KEYS:-}" ]; then
		return 0
	fi

	## TODO: env --chdir could be used instead of cd in a subshell.
	cat <<- 'EOF'
	# Load registry keys from persistent dumps

	if [ -e "$PATH_WINE_REGISTRY" ]; then
	    mkdir --parents "$REGEDIT_DUMPS_WINEPREFIX_PATH"
	    (
	        cd "$PATH_WINE_REGISTRY"
	        find . -type f \
	            -exec cp --force --parents --target-directory="$REGEDIT_DUMPS_WINEPREFIX_PATH" {} +
	    )
	fi
	while read -r registry_key; do
	    if [ -z "$registry_key" ]; then
	        continue
	    fi
	    registry_dump="${REGEDIT_DUMPS_WINEPREFIX_PATH}/$(regedit_convert_key_to_path "$registry_key")"
	    if [ -e "$registry_dump" ]; then
	        printf 'Loading registry key from "%s".\n' "$registry_dump"
	        $(regedit_command) "$registry_dump"
	    fi
	done << EOL
	$(printf '%s' "$REGEDIT_PERSISTENT_KEYS")
	EOL
	unset registry_key registry_dump

	EOF
}

