/*******************************************************************************
 *  PROJECT: Agave
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2006 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/
#include <string>
#include <fstream>
#include <list>

#include <glib/gstdio.h>

#include "palette-selector.h"
#include "gcs-conf.h"
#include "core/log-stream.h"

namespace gcs
{
    namespace Widgets
    {
        PaletteSelector::PaletteSelector(const vector<string>& search_dirs) :
            m_refListStore(Gtk::ListStore::create(m_columns))
        {
            set_model(m_refListStore);
            pack_start(m_columns.m_columnName); // only display name, not path

            LOG("Looking for installed palette files...");

            typedef std::vector<Glib::Dir::iterator::value_type> filelist_t;
            filelist_t files;

            for (vector<string>::const_iterator dir_iter = search_dirs.begin();
                    dir_iter != search_dirs.end(); ++dir_iter)
            {
                // read the directory to see what palette files are there
                try
                {
                    LOG("Looking for palettes in " << *dir_iter);
                    Glib::Dir dir(*dir_iter);
                    // there is no path information -- we need to prepend the path
                    // information to each file
                    for (Glib::Dir::iterator i = dir.begin(); i != dir.end(); ++i)
                    {
                        LOG("Found " << *i);
                        files.push_back(*dir_iter + "/" + *i);
                    }
                }
                catch (Glib::Error& e)
                {
                    g_warning("Can't open palette directory: %s", (*dir_iter).c_str());
                }
            }

            // now check to see if the files are valid palette files
            for (filelist_t::iterator iter = files.begin(); iter != files.end(); ++iter)
            {
                std::string filename = *iter;
                std::ifstream infile(filename.c_str());
                if (infile.is_open())
                {
                    try
                    {
                        LOG("Checking " << filename);
                        shared_ptr<pp::Palette> palette(new pp::Palette());
                        palette->parse(infile);
                        // add valid files to the ComboBox
                        Gtk::TreeModel::iterator treeiter = m_refListStore->append();
                        (*treeiter)[m_columns.m_columnName] = palette->name();
                        (*treeiter)[m_columns.m_columnFile] = filename;
                        (*treeiter)[m_columns.m_columnPalette] = palette;
                        LOG("Found palette file: " << filename);
                    }
                    catch(pp::ParseError& e)
                    {
                        //it wasn't a valid palette, move on
                        LOG("Invalid palette file: " << filename);
                    }
                }
            }

            // default to selecting the first palette in the list (if there are
            // any in the list)
            Gtk::TreeModel::iterator first_palette = m_refListStore->children().begin();
            if (first_palette != m_refListStore->children().end())
            {
                set_active(first_palette);
            }
        }

        PaletteSelector::~PaletteSelector()
        {
        }


        shared_ptr<pp::Palette> PaletteSelector::get_palette(void) const
        {
            Gtk::TreeModel::iterator active_iter = get_active();
            if (active_iter)
            {
                return active_iter->get_value(m_columns.m_columnPalette);
            }
            else
            {
                return shared_ptr<pp::Palette>();
            }
        }


        std::string PaletteSelector::get_palette_file(void) const
        {
            return get_active()->get_value(m_columns.m_columnFile);
        }


        // returns a boolean value specifying whether the currently active
        // selection was changed or not
        bool PaletteSelector::set_palette_file(const std::string& filename)
        {
            for (Gtk::TreeModel::iterator iter =
                    m_refListStore->children().begin(); iter !=
                    m_refListStore->children().end(); ++iter)
            {
                LOG (iter->get_value(m_columns.m_columnFile) << " == " << filename << "?")
                if (iter->get_value(m_columns.m_columnFile) == filename)
                {
                    if (iter != get_active())
                    {
                        set_active(iter);
                        return true;    // palette found, requires update
                    }
                    LOG("Palette file already selected");
                    return false;    // palette already set, no need for update
                }
            }
            g_warning("Palette not found: %s", filename.c_str());
            return false;   // palette was not found
        }

    } // namespace Widgets
} // namespace gcs
