# /u/sy/beebe/emacs/isbn-el-to-bibclean-isbn.awk, Sun Nov  7 12:03:37 1999
# Edit by Nelson H. F. Beebe <beebe@sunev.math.utah.edu>
# ======================================================================
# Convert data in isbn.el to data for bibclean's isbn.c file.
#
# Usage:
#	gawk -f isbn-el-to-bibclean-isbn.awk isbn.el >foo.c
#
# [11-Mar-2018] -- change (const char NULL) to "" in all but final 
#		   countries field to match logic changes in bibclean
#		   to remove all memory leaks
# [25-Nov-2000] -- add forgotten dimension MAX_ISBN_RANGE; its absence
#		   caused a core dump in bibclean when a .bibclean.isbn
#		   file was supplied
# [07-Nov-1999] -- original version
# ======================================================================
BEGIN			{ print_begin_group() }

/^[ \t]*[(]cond/	{ In_Cond = 1 }

/^[ \t]*[(]t *$/	{ In_Cond = 0 }

/^[ \t]*$/		{ new_group() }

/^[ \t]*;; +[(][(]isbn-in-rangep/	{ next } # ignore commented-out function calls

/^[ \t]*;; +/		{ add_group_name($0) }

/[(][(]isbn-in-rangep/	{ add_range($2,$3) }

END			{ print_group(); print_end_group() }

# ======================================================================

function add_group_name(s)
{
    if (In_Cond)
    {
	## Reset Group_Name at each new region, so as to discard the
	## commentary about the source of the data
	if (s ~ "Region name:")
	    Group_Name = ""
	gsub("[ \t]*;; +","",s)
	Group_Name = ((Group_Name == "") ? s : (Group_Name "\\t" s))
    }
}


function add_range(begin,end)
{
    gsub("[()]","",begin)
    gsub("[()]","",end)
    Begin_Range = begin
    End_Range = end
    if (In_Cond)
	print_group()
}


function country_language(s)
{
    # Reduce "abc-xyz" to "abc" and return it
    gsub("-.*$","",s)
    return (s)
}


function new_group()
{
    if (Group_Name != "")
    {
	print_group()
	Group_Name = ""
	Begin_Range = ""
	End_Range = ""
    }
}


function print_begin_group()
{
    "date" | getline current_date_and_time
    close("date")

    print "/**********************************************************************/"
    print "/**********************************************************************/"
    print "/** WARNING: Do NOT edit this file.  It was created automatically    **/"
    print "/** from the master ISBN prefix list on " current_date_and_time " **/"
    print "/** by                                                               **/"
    print "/**                                                                  **/"
    print "/**        awk -f isbn-el-to-bibclean-isbn.awk isbn.el >this-file    **/"
    print "/**                                                                  **/"
    print "/** Update the bibclean ISBN initialization file, .bibclean.isbn,    **/"
    print "/** instead!                                                         **/"
    print "/**********************************************************************/"
    print "/**********************************************************************/"
    print ""
    print "static ISBN_range_t ISBN_range[MAX_ISBN_RANGE] ="
    print "{"
    print "    /*******************************************************************"
    print "    NB: We intentionally include the hyphen that separates the"
    print "    countrygroupnumber from the publishernumber, in order to improve"
    print "    readability, even though this complicates the processing in"
    print "    in_ISBN_range().  Given the position of that hyphen, we automatically"
    print "    compute the required hyphen positions."
    print "    *******************************************************************/"
}


function print_end_group()
{
    print ""
    print ""
    print "     /* The list is ended by a triple of NULL pointers */"
    print "     { (const char*)NULL,	(const char*)NULL, (const char*)NULL}"
    print "};"
}


function print_group( new_block)
{
    if ((Group_Name != "") && (Begin_Range != "") && (End_Range != ""))
    {
	Country_Language = country_language(Begin_Range)
	new_block = (Last_Group_Name != Group_Name) || \
		    (Last_Country_Language != Country_Language)
	printf("%s     { %-25s%-16s%s},\n", \
	       (new_block ? "\n\n" : ""), \
	       (Begin_Range ","), \
	       (End_Range ","), \
	       (new_block ? ("\"" Group_Name "\"") : "\"\""))
	Last_Group_Name = Group_Name
	Last_Country_Language = Country_Language
	Begin_Range = ""
	End_Range = ""
    }
}
