// CamiTK includes
#include "ComputeNormals.h"
#include <Application.h>
#include <MeshComponent.h>
#include <Log.h>
#include <Property.h>

// Vtk includes
#include <vtkPolyDataNormals.h>
#include <vtkCallbackCommand.h>
#include <vtkDataSetAttributes.h>
#include <vtkPointData.h>
#include <vtkCellData.h>

using namespace camitk;

ComputeNormals::ComputeNormals ( camitk::ActionExtension* ext ) : Action ( ext ) {
    this->setName ( "Compute Normals" );
    this->setDescription ( QString ( tr ( "Compute normals of surface mesh." ) ) );
    this->setComponent ( "MeshComponent" );
    this->setFamily ( "Basic Mesh" );
    this->addTag ( tr ( "normal" ) );
    this->addTag ( tr ( "surface" ) );
}

ComputeNormals::~ComputeNormals() {

}

Action::ApplyStatus ComputeNormals::apply() {
    // set waiting cursor and status bar
    QApplication::setOverrideCursor ( QCursor ( Qt::WaitCursor ) );
    Application::showStatusBarMessage ( tr ( "Compute Curvatures..." ) );
    Application::resetProgressBar();
    vtkSmartPointer<vtkCallbackCommand> progressCallback = vtkSmartPointer<vtkCallbackCommand>::New();
    progressCallback->SetCallback ( &Application::vtkProgressFunction );

    // target must be a polydata
    MeshComponent *mesh = dynamic_cast<MeshComponent*> ( getTargets().first() );

    if ( !mesh ) {
        CAMITK_DEBUG ( "ComputeNormals", "apply", "mesh " +
                       getTargets().first()->getName().toStdString() + " is not a MeshComponent" )
        return ABORTED;
    }

    vtkPolyData* pData = vtkPolyData::SafeDownCast ( mesh->getPointSet() );
    pData->AddObserver ( vtkCommand::ProgressEvent, progressCallback );

    if ( !pData ) {
        CAMITK_DEBUG ( "ComputeNormals", "apply", "the mesh point set of " +
                       getTargets().first()->getName().toStdString() + " is not a polydata" )
        return ABORTED;
    }

    CAMITK_INFO ( "ComputeNormals", "apply", "Compute normals for mesh " +
                  getTargets().first()->getName().toStdString() )

    // compute normals
    vtkSmartPointer<vtkPolyDataNormals> polyNormals = vtkSmartPointer<vtkPolyDataNormals>::New();
    polyNormals->SetInputData ( pData );
    polyNormals->AddObserver ( vtkCommand::ProgressEvent, progressCallback );
    polyNormals->Update();

    mesh->addPointData ( "Normals", polyNormals->GetOutput()->GetPointData()->GetScalars ( "Normals" ) );

    // refresh restore the normal cursor and progress bar
    Application::refresh();
    Application::resetProgressBar();
    Application::showStatusBarMessage ( "" );
    QApplication::restoreOverrideCursor();
    return SUCCESS;
}
