/*
 * Author: Andrei Zavada <johnhommer@gmail.com>
 *
 * License: GPL-2+
 *
 * Initial version: 2008-12-02
 *
 * Expression parser
 */


#include <cstdlib>
#include <cstdio>
#include <array>
#include <iostream>

#include "config.h"

#include "exprparser.hh"


using namespace std;


const char* const Stilton::__exparse_error_strings[] = {
	"",
	"Missing operand",
	"Unbalanced parentheses",
	"Unparsable value or missing operator",
	"Unary operator used as binary",
	"Undefined variable",
	"Non-lvalue in assignment",
	"varlist is NULL"
};




enum TOperator {
	OP_VOID = -1,
	OP_NEG,
	OP_UNARYMINUS,
	OP_MULT,	OP_DIV,
	OP_ADD,		OP_SUBTRACT,
	OP_LT,		OP_GT,
	OP_ASSIGN,

	OP_LE,		OP_GE,		OP_EQ,
};

struct SOp {
	char	literal[4];
	int	prio;
	bool	assoc_ltr,
		is_binary;

	SOp( const char *l, int p, bool a, bool b)
	      : prio (p), assoc_ltr (a), is_binary (b)
		{ strncpy( literal, l, 3); }

	bool isat( const char *where)
		{ return (strncmp( where, literal, strlen( literal)) == 0); }
};


#define n_ops 12

inline namespace {
array<SOp, n_ops> Ops = {
	{
		SOp("!", 1, false, false),
		SOp("-", 1, false, false),
		SOp("*", 3, true, true),	SOp("/", 3, true, true),
		SOp("+", 5, true, true),	SOp("-", 5, true, true),
		SOp("<", 7, true, true),	SOp(">", 7, true, true),
		SOp("=", 9, false, true),

		SOp("<=", 7, true, true),	SOp(">=", 7, true, true),	SOp("==", 7, true, true)
	}
};
} // inline namespace



Stilton::TExprParserError
Stilton::CExpression::
_do_parse( const char *str, double& parsed, list<SVariable> *varlist)
{
	if ( !str ) {
		parsed = 0;
		return status = EXPARSE_OK;
	}

	parsed = NAN;
	_var = "";

	string	workbuf( str);
	char	*p = &workbuf[0];

	p += strspn( p, " \t");
	if ( !*p ) {
		parsed = 0;
		return status = EXPARSE_EMPTY;
	}

	char	*expr1 = p,
		*expr2 = nullptr;
	TExprParserError subexpr_retval;

      // determine subexpressions, if any, at top level
	int	level = 0;
	char	*tl_op_at = nullptr;
	TOperator
		tl_op = OP_VOID;
	bool	last_token_was_operator = true;

//	cerr << "\nPARSE \"" << p << "\"\n";
	while ( *p ) {
		if ( *p == eol_comment_delim ) {
			*p = '\0';
			break;
		}
		if      ( *p == '(' )	level++;
		else if ( *p == ')' )	level--;

		if ( level < 0 )
			return status = EXPARSE_UNBALANCED;
		if ( level > 0 || isspace( *p) )
			goto end_detect;

	      // detect exponent (e-4)
		if ( strncasecmp( p, "e-", 2) == 0 ) {
			p++;
			goto end_detect;
		}
	      // serve the case of unary -: part one
		if ( *p == '-' && last_token_was_operator ) {
			char *pp = p;
			while ( pp > &workbuf[0] && !isspace(*pp) )  pp--; // pp++;
//			cerr << "  (checking \"" << pp << "\"";
			char *tp;
			if ( strtod( pp, &tp) )
				;
			if ( tp > p ) { // we have indeed read a number
//				cerr << "  parsed a number up to \"" << tp<< "\")\n";
				p = tp - 1;
				last_token_was_operator = false;
				goto end_detect;
			}
//			cerr << " not a number)\n";
		}

		int o;
		for ( o = n_ops-1; o >= 0; o-- ) // check for multibyte operators first (those are at end)
			if ( Ops[o].isat( p) ) {
				char *pp = p;
				p += strlen( Ops[o].literal) - 1; // anticipate general p++

				if ( o == OP_SUBTRACT && last_token_was_operator ) {
//					cerr << "override\n";
					o = OP_UNARYMINUS;
				} else
					if ( !last_token_was_operator && !Ops[o].is_binary ) {
//					cerr << " ...at \"" << pp << "\" with op " << Ops[o].literal << endl;
						if ( !silent ) fprintf( stderr, "Unary %s used after an operand\n", Ops[o].literal);
						return status = EXPARSE_UNASSOC;
					}

				if ( tl_op == OP_VOID ||
				     (Ops[o].assoc_ltr && Ops[tl_op].prio <= Ops[o].prio) ||
				     (!Ops[o].assoc_ltr && Ops[tl_op].prio < Ops[o].prio) ) {
//					cerr << "current tlop: " << Ops[o].literal << endl;
					tl_op_at = pp;
					tl_op = (TOperator)o;
				}
				last_token_was_operator = true;
				goto end_detect;
			}

		last_token_was_operator = false;

	end_detect:
		p++;
	}
//	cerr << "tlop is " << Ops[tl_op].literal << endl;

	if ( level > 0 ) {
		if ( !silent ) fprintf( stderr, "Expression lacks some `)''\n");
		return status = EXPARSE_UNBALANCED;
	}

	list<SVariable>::iterator V;

	if ( tl_op != OP_VOID ) {
		*tl_op_at = '\0';
		expr2 = tl_op_at + strlen( Ops[tl_op].literal);
		double opd1, opd2;

//		cerr << "parsing [" << expr1 << "] "<< Ops[tl_op].literal << " [" << expr2 << "]\n";

	      // second subexpr must always be good
		subexpr_retval = _do_parse( expr2, opd2, varlist);
		if ( subexpr_retval )
			return status = subexpr_retval;

	      // first subexpr must be empty, but only in the case of OP_NEG
		subexpr_retval = _do_parse( expr1, opd1, varlist);

		switch ( subexpr_retval ) {
		case EXPARSE_OK:
			break;
		case EXPARSE_EMPTY:
			if ( !Ops[tl_op].is_binary ) {
//				cerr << "was a unary op\n";
				break;
			} else
				return subexpr_retval;
		case EXPARSE_UNDEFVAR:
			if ( tl_op == OP_ASSIGN )
				break;
			else {
				// have it reported here (in deeper _do_parse where it is flagged), we don't know yet
				// if an undefined var is going to be defined
				if ( !silent ) fprintf( stderr, "Undefined variable `%s'\n", strtok( expr1, " \t"));
				return status = subexpr_retval;
			}
		      break;
		default:
			return subexpr_retval;
		}

		switch ( tl_op ) {
		case OP_VOID:	break;
		case OP_UNARYMINUS:	parsed = -opd2;		break;
		case OP_ADD:		parsed = opd1 + opd2;	break;
		case OP_SUBTRACT:	parsed = opd1 - opd2;	break;
		case OP_MULT:		parsed = opd1 * opd2;	break;
		case OP_DIV:		parsed = opd1 / opd2;	break;
		case OP_LT:		parsed = opd1 < opd2;	break;
		case OP_LE:		parsed = opd1 <= opd2;	break;
		case OP_GT:		parsed = opd1 > opd2;	break;
		case OP_GE:		parsed = opd1 >= opd2;	break;
		case OP_EQ:		parsed = opd1 == opd2;	break;
		case OP_NEG:		parsed = !opd2;		break;
		case OP_ASSIGN:
			if ( !varlist ) {
				if ( !silent ) fprintf( stderr, "Variable assignment reqires a user varlist\n");
				return status = EXPARSE_VARLISTNULL;
			}
			if ( _var == "" ) {
				if ( !silent ) fprintf( stderr, "Non-lvalue in assignment\n");
				return status = EXPARSE_NONLVAL;
			}
			parsed = opd2;
			for ( V = varlist->begin(); V != varlist->end(); V++ )
				if ( strcmp( V->name, _var.c_str()) == 0 ) { // _var has been cached by a previous call to _do_parse
					V->value = opd2;
					toplevel_op = tl_op;
					return status = EXPARSE_OK;
				}
			varlist->push_back( SVariable( _var.c_str(), opd2));
		    break;
		}
		toplevel_op = tl_op;
		return status = EXPARSE_OK;
	}

      // single expression, possibly in parentheses
	if ( *expr1 == '(' ) {
		*strrchr( ++expr1, ')') = '\0';  // parentheses have been checked in the by-char parser loop above
		return _do_parse( expr1, parsed, varlist);
	}

      // bare expression
	expr1 = strtok( expr1, " \t");
	char *tailp;
	parsed = strtod( expr1, &tailp);
	if ( tailp == nullptr || strspn( tailp, " \t\n\r;") == strlen( tailp) )   // digits followed by whitespace
		return status = EXPARSE_OK;

	if ( tailp == expr1 && varlist ) { // no digits at front: check if that's a variable
		for ( V = varlist->begin(); V != varlist->end(); V++ ) {
			if ( strcmp( V->name, expr1) == 0 ) {
				parsed = V->value;
				_var = V->name;
				return status = EXPARSE_OK;
			}
		}
		_var = expr1;  // possibly to be assigned in caller; parsed remains NAN
		return status = EXPARSE_UNDEFVAR;
	}

      // some digits followed by rubbish
	return status = EXPARSE_BAD;
}


// EOF
