# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Custom renderers for package archives."""

from collections.abc import Mapping
from typing import Any

from pgpy import PGPKey
from rest_framework.renderers import BaseRenderer

from debusine.assets.asset_utils import ExportablePGPKeyring


class SigningKeysRenderer(BaseRenderer):
    """A renderer for OpenPGP signing keys."""

    media_type = "application/pgp-keys"
    format = "signing-keys"

    def render(
        self,
        data: Any,
        accepted_media_type: str | None = None,  # noqa: U100
        renderer_context: Mapping[str, Any] | None = None,  # noqa: U100
    ) -> str:
        """Render an ASCII-armored public key block."""
        assert isinstance(data, dict)
        return str(
            ExportablePGPKeyring(
                *(PGPKey.from_blob(key)[0] for key in data["signing_keys"])
            )
        )
