/* -*- mode: c++; c-basic-offset: 2; indent-tabs-mode: nil; -*-
 *  vim:expandtab:shiftwidth=2:tabstop=2:smarttab:
 *
 *  Copyright (C) 2008 Sun Microsystems, Inc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */


#pragma once

#include <drizzled/memory/sql_alloc.h>
#include <drizzled/sql_list.h>
#include <drizzled/natural_join_column.h>
#include <drizzled/item/field.h>

namespace drizzled {

/*
  Iterator over the fields of a generic table reference.
*/

class Field_iterator : public memory::SqlAlloc
{
public:
  virtual ~Field_iterator() {}
  virtual void set(TableList *)= 0;
  virtual void next()= 0;
  virtual bool end_of_fields() const= 0;              /* Return true at end of list */
  virtual const char *name() const= 0;
  virtual Item *create_item(Session *)= 0;
  virtual Field *field()= 0;
};


/*
  Iterator over the fields of a base table, view with temporary
  table, or subquery.
*/

class Field_iterator_table : public Field_iterator
{
  Field **ptr;
public:
  Field_iterator_table() :ptr(0) {}
  void set(TableList *table);
  void set_table(Table *table);
  void next() { ptr++; }
  bool end_of_fields() const { return *ptr == 0; }
  const char *name() const;
  Item *create_item(Session *session);
  Field *field() { return *ptr; }
};


/* Iterator over the fields of a merge view. */

/*
  Field_iterator interface to the list of materialized fields of a
  NATURAL/USING join.
*/

class Field_iterator_natural_join: public Field_iterator
{
  List<Natural_join_column>::iterator column_ref_it;
  Natural_join_column *cur_column_ref;
public:
  Field_iterator_natural_join() : cur_column_ref(NULL) {}
  void set(TableList *table);
  void next();
  bool end_of_fields() const { return not cur_column_ref; }
  const char *name() const { return cur_column_ref->name(); }
  Item *create_item(Session *session) { return cur_column_ref->create_item(session); }
  Field *field() { return cur_column_ref->field(); }
  Natural_join_column *column_ref() { return cur_column_ref; }
};


/*
  Generic iterator over the fields of an arbitrary table reference.

  DESCRIPTION
    This class unifies the various ways of iterating over the columns
    of a table reference depending on the type of SQL entity it
    represents. If such an entity represents a nested table reference,
    this iterator encapsulates the iteration over the columns of the
    members of the table reference.

  IMPLEMENTATION
    The implementation assumes that all underlying NATURAL/USING table
    references already contain their result columns and are linked into
    the list TableList::next_name_resolution_table.
*/

class Field_iterator_table_ref: public Field_iterator
{
  TableList *table_ref, *first_leaf, *last_leaf;
  Field_iterator_table        table_field_it;
  Field_iterator_natural_join natural_join_it;
  Field_iterator *field_it;
  void set_field_iterator();
public:
  Field_iterator_table_ref() :field_it(NULL) {}
  void set(TableList *table);
  void next();
  bool end_of_fields() const
  { return (table_ref == last_leaf && field_it->end_of_fields()); }
  const char *name() const { return field_it->name(); }
  const char *table_name();
  const char *db_name();
  Item *create_item(Session *session) { return field_it->create_item(session); }
  Field *field() { return field_it->field(); }
  Natural_join_column *get_or_create_column_ref(TableList *parent_table_ref);
  Natural_join_column *get_natural_column_ref();
};

} /* namespace drizzled */

