/*******************************************************************
 * Fritz Fun                                                       *
 * Created by Jan-Michael Brummer                                  *
 * All parts are distributed under the terms of GPLv2. See COPYING *
 *******************************************************************/

/**
 * \file csv.c
 * \brief Import functions for CSV, Person and Caller structure functions
 */

#include <ffgtk.h>

/** Global person list */
GList *psPersonsList = NULL;
/** Global caller list */
GList *psCallerList = NULL;
static gint nImported = 0;
static GHashTable *psAreaCodesTable = NULL;

/**
 * \brief Add info to hashtable
 * \param psTable hash table
 * \param nKey hash table key
 * \param pnValue key value
 */
void AddInfo( GHashTable *psTable, gint nKey, const gchar *pnValue ) {
	g_hash_table_insert( psTable, GINT_TO_POINTER( nKey ), ( gchar * ) pnValue );
}

/**
 * \brief Adds a new person into global person list, check for duplicates
 * \param psTable hash table containing person information
 * \param bNew new person?
 * \return created person
 */
struct sPerson *AddPerson( GHashTable *psTable, gboolean bNew ) {
	GList *psList;
	struct sPerson *psPerson = NULL;
	const gchar *pnId = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_ID ) );
	const gchar *pnDisplayName = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_DISPLAY_NAME ) );
	const gchar *pnFirstName = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_FIRST_NAME ) );
	const gchar *pnLastName = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_LAST_NAME ) );
	const gchar *pnCompany = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_COMPANY ) );
	const gchar *pnPrivatePhone = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_PHONE ) );
	const gchar *pnPrivateFax = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_FAX ) );
	const gchar *pnPrivateMobile = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_MOBILE ) );
	const gchar *pnPrivateStreet = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_STREET ) );
	const gchar *pnPrivateCity = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_CITY ) );
	const gchar *pnPrivateZipCode = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_ZIPCODE ) );
	const gchar *pnPrivateCountry = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_PRIVATE_COUNTRY ) );
	const gchar *pnBusinessPhone = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_PHONE ) );
	const gchar *pnBusinessFax = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_FAX ) );
	const gchar *pnBusinessStreet = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_STREET ) );
	const gchar *pnBusinessCity = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_CITY ) );
	const gchar *pnBusinessZipCode = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_ZIPCODE ) );
	const gchar *pnBusinessCountry = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_BUSINESS_COUNTRY ) );
	const gchar *pnTitle = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_TITLE ) );
	const gchar *pnCategory = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_CATEGORY ) );
	GdkPixbuf *psImage = g_hash_table_lookup( psTable, GINT_TO_POINTER( PERSON_IMAGE ) );

	if ( ( pnFirstName == NULL || strlen( pnFirstName ) < 1 ) &&
		( pnLastName == NULL || strlen( pnLastName ) < 1 ) ) {
		return NULL;
	}

	for ( psList = psPersonsList; psList != NULL; psList = psList -> next ) {
		psPerson = psList -> data;

		if ( psPerson != NULL ) {
			if ( !strcmp( psPerson -> pnId, pnId ) ) {
				if ( psPerson -> pnBusinessPhone ) {
					g_free( psPerson -> pnBusinessPhone );
				}
				psPerson -> pnBusinessPhone = pnBusinessPhone ? normalizeNumber( pnBusinessPhone ) : strdup( "" );

				if ( psPerson -> pnPrivatePhone ) {
					g_free( psPerson -> pnPrivatePhone );
				}
				psPerson -> pnPrivatePhone = pnPrivatePhone ? normalizeNumber( pnPrivatePhone ) : strdup( "" );

				if ( psPerson -> pnPrivateFax ) {
					g_free( psPerson -> pnPrivateFax );
				}
				psPerson -> pnPrivateFax = pnPrivateFax ? normalizeNumber( pnPrivateFax ) : strdup( "" );

				if ( psPerson -> pnPrivateMobile ) {
					g_free( psPerson -> pnPrivateMobile );
				}
				psPerson -> pnPrivateMobile = pnPrivateMobile ? normalizeNumber( pnPrivateMobile ) : strdup( "" );

				if ( psPerson -> pnPrivateStreet ) {
					g_free( psPerson -> pnPrivateStreet );
				}
				psPerson -> pnPrivateStreet = pnPrivateStreet ? strdup( pnPrivateStreet ) : strdup( "" );

				if ( psPerson -> pnPrivateCity ) {
					g_free( psPerson -> pnPrivateCity );
				}
				psPerson -> pnPrivateCity = pnPrivateCity ? strdup( pnPrivateCity ) : strdup( "" );

				if ( psPerson -> pnCategory ) {
					g_free( psPerson -> pnCategory );
				}
				psPerson -> pnCategory = pnCategory ? strdup( pnCategory ) : strdup( "" );
				return psPerson;
			}
		}
	}

	psPerson = g_malloc0( sizeof( struct sPerson ) );

	if ( psPerson != NULL ) {
		psPerson -> pnId = pnId ? strdup( pnId ) : strdup( "" );
		psPerson -> pnTitle = pnTitle ? strdup( pnTitle ) : strdup( "" );
		psPerson -> pnFirstName = pnFirstName ? strdup( pnFirstName ) : strdup( "" );
		psPerson -> pnLastName = pnLastName ? strdup( pnLastName ) : strdup( "" );
		psPerson -> pnDisplayName = pnDisplayName ? convert_utf8( pnDisplayName ) : strdup( "" );
		psPerson -> pnCompany = pnCompany ? strdup( pnCompany ) : strdup( "" );
		psPerson -> pnPrivatePhone = pnPrivatePhone ? normalizeNumber( pnPrivatePhone ) : strdup( "" );
		psPerson -> pnPrivateFax = pnPrivateFax ? normalizeNumber( pnPrivateFax ) : strdup( "" );
		psPerson -> pnPrivateMobile = pnPrivateMobile ? normalizeNumber( pnPrivateMobile ) : strdup( "" );
		psPerson -> pnPrivateStreet = pnPrivateStreet ? strdup( pnPrivateStreet ) : strdup( "" );
		psPerson -> pnPrivateCity = pnPrivateCity ? strdup( pnPrivateCity ) : strdup( "" );
		psPerson -> pnPrivateZipCode = pnPrivateZipCode ? strdup( pnPrivateZipCode ) : strdup( "" );
		psPerson -> pnPrivateCountry = pnPrivateCountry ? strdup( pnPrivateCountry ) : strdup( "" );
		psPerson -> pnBusinessPhone = pnBusinessPhone ? normalizeNumber( pnBusinessPhone ) : strdup( "" );
		psPerson -> pnBusinessFax = pnBusinessFax ? normalizeNumber( pnBusinessFax ) : strdup( "" );
		psPerson -> pnBusinessStreet = pnBusinessStreet ? strdup( pnBusinessStreet ) : strdup( "" );
		psPerson -> pnBusinessCity = pnBusinessCity ? strdup( pnBusinessCity ) : strdup( "" );
		psPerson -> pnBusinessZipCode = pnBusinessZipCode ? strdup( pnBusinessZipCode ) : strdup( "" );
		psPerson -> pnBusinessCountry = pnBusinessCountry ? strdup( pnBusinessCountry ) : strdup( "" );
		psPerson -> pnCategory = pnCategory ? strdup( pnCategory ) : strdup( "" );
		psPerson -> psImage = psImage;
		psPerson -> nFlags = bNew == TRUE ? PERSON_FLAG_NEW : PERSON_FLAG_UNCHANGED;

		psPersonsList = g_list_append( psPersonsList, psPerson );
	}

	return psPerson;
}

/**
  * \brief Find person by name
  * \param pnName full name
  * \return person pointer or NULL on error
  */
struct sPerson *findPerson( const char *pnName ) {
	GList *psList = psPersonsList;

	if ( pnName == NULL ) {
		return NULL;
	}

	while ( psList != NULL ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL && !strcmp( psPerson -> pnDisplayName, pnName ) ) {
			return psPerson;
		}

		psList = psList -> next;
	}

	return NULL;
}

/**
  * \brief Find person by number
  * \param pnNumber full number
  * \return person structure
  */
struct sPerson *findPersonByNumber( const char *pnNumber ) {
	GList *psList = psPersonsList;
	gchar *pnFullNumber;
	gchar *pnFullNumber2;
	struct sPerson *psFoundPerson = NULL;

	if ( pnNumber == NULL || strlen( pnNumber ) <= 0 ) {
		return NULL;
	}

	pnFullNumber = fullNumber( pnNumber, true );
	if ( pnFullNumber == NULL ) {
		return NULL;
	}

	while ( psList != NULL ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL ) {
			pnFullNumber2 = fullNumber( psPerson -> pnPrivatePhone, true );
			if ( pnFullNumber2 != NULL && !strcmp( pnFullNumber2, pnFullNumber ) ) {
				psFoundPerson = psPerson;
				g_free( pnFullNumber2 );
				break;
			}
			g_free( pnFullNumber2 );
			pnFullNumber2 = fullNumber( psPerson -> pnPrivateFax, true );
			if ( pnFullNumber2 != NULL && !strcmp( pnFullNumber2, pnFullNumber ) ) {
				psFoundPerson = psPerson;
				g_free( pnFullNumber2 );
				break;
			}
			g_free( pnFullNumber2 );
			pnFullNumber2 = fullNumber( psPerson -> pnPrivateMobile, true );
			if ( pnFullNumber2 != NULL && !strcmp( pnFullNumber2, pnFullNumber ) ) {
				psFoundPerson = psPerson;
				g_free( pnFullNumber2 );
				break;
			}
			g_free( pnFullNumber2 );
			pnFullNumber2 = fullNumber( psPerson -> pnBusinessPhone, true );
			if ( pnFullNumber2 != NULL && !strcmp( pnFullNumber2, pnFullNumber ) ) {
				psFoundPerson = psPerson;
				g_free( pnFullNumber2 );
				break;
			}
			g_free( pnFullNumber2 );
			pnFullNumber2 = fullNumber( psPerson -> pnBusinessFax, true );
			if ( pnFullNumber2 != NULL && !strcmp( pnFullNumber2, pnFullNumber ) ) {
				psFoundPerson = psPerson;
				g_free( pnFullNumber2 );
				break;
			}
			g_free( pnFullNumber2 );
		}

		psList = psList -> next;
	}

	g_free( pnFullNumber );

	return psFoundPerson;
}

/**
  * \brief Remove person by name
  * \param pnName full name
  */
void removePerson( const char *pnName ) {
	GList *psList = psPersonsList;

	if ( pnName == NULL ) {
		return;
	}

	while ( psList != NULL ) {
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL && !strcmp( psPerson -> pnDisplayName, pnName ) ) {
			psPersonsList = g_list_delete_link( psPersonsList, psList );
			break;
		}

		psList = psList -> next;
	}
}

/**
 * \brief Free global person list
 */
void freePersons( void ) {
	GList *psList;

	while ( psPersonsList != NULL ) {
		psList = psPersonsList;
		struct sPerson *psPerson = psList -> data;

		if ( psPerson != NULL ) {
			g_free( psPerson -> pnId );
			g_free( psPerson -> pnTitle );
			g_free( psPerson -> pnFirstName );
			g_free( psPerson -> pnLastName );
			g_free( psPerson -> pnDisplayName );
			g_free( psPerson -> pnCompany );
			g_free( psPerson -> pnPrivatePhone );
			g_free( psPerson -> pnPrivateFax );
			g_free( psPerson -> pnPrivateMobile );
			g_free( psPerson -> pnPrivateStreet );
			g_free( psPerson -> pnPrivateCity );
			g_free( psPerson -> pnPrivateZipCode );
			g_free( psPerson -> pnPrivateCountry );
			g_free( psPerson -> pnBusinessPhone );
			g_free( psPerson -> pnBusinessFax );
			g_free( psPerson -> pnBusinessStreet );
			g_free( psPerson -> pnBusinessCity );
			g_free( psPerson -> pnBusinessZipCode );
			g_free( psPerson -> pnBusinessCountry );
		}

		psPersonsList = g_list_delete_link( psPersonsList, psList );
	}
}

/**
 * \brief Add a new call entry to global caller list
 * \param pnType type of call (in/out/missed)
 * \param pnDateTime date and time information
 * \param pnName caller name
 * \param pnNumber caller number
 * \param pnLocalName which phone
 * \param pnLocalNumber local phone number called
 * \param pnDuration duration of talk
 * \param pnProfile profile name
 */
void AddCall( const char *pnType, const char *pnDateTime, const char *pnName, const char *pnNumber, const char *pnLocalName, const char *pnLocalNumber, const char *pnDuration, const char *pnProfile ) {
	GList *psList = NULL;
	struct sCaller *psCaller = NULL;

	for ( psList = psCallerList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psCaller = psList -> data;

		if ( ( psCaller -> nType == atoi( pnType ) ) && ( strcmp( psCaller -> pnDateTime, pnDateTime ) == 0 ) && ( strcmp( psCaller -> pnLocalNumber, pnLocalNumber ) == 0 ) ) {
			/* Call already exists */
			return;
		}
	}

	psCaller = g_malloc( sizeof( struct sCaller ) );

	if ( psCaller != NULL ) {
		psCaller -> nType = atoi( pnType );
		psCaller -> pnDateTime = strdup( pnDateTime );
		psCaller -> pnName = strdup( pnName );
		psCaller -> pnNumber = strdup( pnNumber );
		psCaller -> pnLocalName = strdup( pnLocalName );
		psCaller -> pnLocalNumber = strdup( pnLocalNumber );
		psCaller -> pnDuration = strdup( pnDuration );
		psCaller -> pnProfile = strdup( pnProfile );

		psCallerList = g_list_append( psCallerList, psCaller );
	}
}

/**
 * \brief Free global caller list
 */
void freeCallers( void ) {
	GList *psList;

	while ( psCallerList != NULL ) {
		psList = psCallerList;
		struct sCaller *psCaller = psList -> data;

		g_free( psCaller -> pnDateTime );
		g_free( psCaller -> pnName );
		g_free( psCaller -> pnNumber );
		g_free( psCaller -> pnLocalName );
		g_free( psCaller -> pnLocalNumber );
		g_free( psCaller -> pnDuration );
		g_free( psCaller -> pnProfile );
		g_free( psCaller );
		psCallerList = g_list_delete_link( psCallerList, psList );
	}
}

/**
 * \brief Parse the old fritzbox csv format
 * \param pnLine input line
 * \param pnSeparator separator work
 * \param pnProfileName profile name
 */
static void parseFritzBoxCsv( char *pnLine, char *pnSeparator, char *pnProfileName ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return;
	}

	while ( ppnSplit[ nCount++ ] != NULL );
	if ( nCount == 7 ) {
		AddCall( ppnSplit[ 0 ], ppnSplit[ 1 ], "", ppnSplit[ 2 ], ppnSplit[ 3 ], ppnSplit[ 4 ], ppnSplit[ 5 ], pnProfileName );
	}

	g_strfreev( ppnSplit );
}

/**
 * \brief Parse the new fritzbox csv format
 * \param pnLine input line
 * \param pnSeparator separator work
 * \param pnProfileName profile name
 */
static void parseFritzBoxNewCsv( char *pnLine, char *pnSeparator, char *pnProfileName ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gchar *pnName;
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return;
	}

	while ( ppnSplit[ nCount++ ] != NULL );
	if ( nCount == 8 ) {
		pnName = g_convert( ppnSplit[ 2 ], -1, "UTF-8", "iso-8859-1", NULL, NULL, NULL );

		AddCall( ppnSplit[ 0 ], ppnSplit[ 1 ], pnName, ppnSplit[ 3 ], ppnSplit[ 4 ], ppnSplit[ 5 ], ppnSplit[ 6 ], pnProfileName );

		g_free( pnName );
	}

	g_strfreev( ppnSplit );
}

/**
 * \brief Parse ffgtk csv file
 * \param pnLine input line
 * \param pnSeparator work separator
 */
static void parseFfgtkCsv( char *pnLine, char *pnSeparator ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return;
	}

	while ( ppnSplit[ nCount++ ] != NULL );

	if ( nCount == 7 ) {
		if (
			( strlen( ppnSplit[ 1 ] ) == 0 ) &&
			( strlen( ppnSplit[ 2 ] ) == 0 ) &&
			( strlen( ppnSplit[ 3 ] ) == 0 ) &&
			( strlen( ppnSplit[ 4 ] ) == 0 ) &&
			( strlen( ppnSplit[ 5 ] ) == 0 ) ) {
			//printf( "No phone number present for contact!\n" );
		} else {
			GHashTable *psTable = g_hash_table_new( NULL, NULL );

			gchar *pnTmp = g_strdup_printf( "%d", nImported++ );
			AddInfo( psTable, PERSON_ID, pnTmp );
			AddInfo( psTable, PERSON_DISPLAY_NAME, ppnSplit[ 0 ] );
			AddInfo( psTable, PERSON_BUSINESS_PHONE, ppnSplit[ 1 ] );
			AddInfo( psTable, PERSON_PRIVATE_PHONE, ppnSplit[ 2 ] );
			AddInfo( psTable, PERSON_PRIVATE_FAX, ppnSplit[ 3 ] );
			AddInfo( psTable, PERSON_PRIVATE_MOBILE, ppnSplit[ 4 ] );

			AddPerson( psTable, FALSE );
			g_free( pnTmp );
			g_hash_table_destroy( psTable );
		}
	} else {
		printf( "Invalid ffgtk CSV format! (%d)\n", nCount );
	}

	g_strfreev( ppnSplit );
}

/**
 * \brief Parse ffgtk2 csv file
 * \param pnLine input line
 * \param pnSeparator work separator
 */
static void parseFfgtk2Csv( char *pnLine, char *pnSeparator ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return;
	}

	while ( ppnSplit[ nCount++ ] != NULL );

	if ( nCount == 19 ) {
		GHashTable *psTable = g_hash_table_new( NULL, NULL );

		gchar *pnTmp = g_strdup_printf( "%d", nImported++ );
		AddInfo( psTable, PERSON_ID, pnTmp );
		AddInfo( psTable, PERSON_TITLE, ppnSplit[ 0 ] );
		AddInfo( psTable, PERSON_FIRST_NAME, ppnSplit[ 1 ] );
		AddInfo( psTable, PERSON_LAST_NAME, ppnSplit[ 2 ] );
		AddInfo( psTable, PERSON_DISPLAY_NAME, ppnSplit[ 3 ] );
		AddInfo( psTable, PERSON_COMPANY, ppnSplit[ 4 ] );
		AddInfo( psTable, PERSON_PRIVATE_PHONE, ppnSplit[ 5 ] );
		AddInfo( psTable, PERSON_PRIVATE_FAX, ppnSplit[ 6 ] );
		AddInfo( psTable, PERSON_PRIVATE_MOBILE, ppnSplit[ 7 ] );
		AddInfo( psTable, PERSON_PRIVATE_STREET, ppnSplit[ 8 ] );
		AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, ppnSplit[ 9 ] );
		AddInfo( psTable, PERSON_PRIVATE_CITY, ppnSplit[ 10 ] );
		AddInfo( psTable, PERSON_PRIVATE_COUNTRY, ppnSplit[ 11 ] );
		AddInfo( psTable, PERSON_BUSINESS_PHONE, ppnSplit[ 12 ] );
		AddInfo( psTable, PERSON_BUSINESS_FAX, ppnSplit[ 13 ] );
		AddInfo( psTable, PERSON_BUSINESS_STREET, ppnSplit[ 14 ] );
		AddInfo( psTable, PERSON_BUSINESS_ZIPCODE, ppnSplit[ 15 ] );
		AddInfo( psTable, PERSON_BUSINESS_CITY, ppnSplit[ 16 ] );
		AddInfo( psTable, PERSON_BUSINESS_COUNTRY, ppnSplit[ 17 ] );

		AddPerson( psTable, FALSE );
		g_free( pnTmp );
		g_hash_table_destroy( psTable );
	} else {
		printf( "Invalid ffgtk2 CSV format! (%d)\n", nCount );
	}

	g_strfreev( ppnSplit );
}

/**
 * \brief Parse jfritz csv file
 * \param pnLine input line
 * \param pnSeparator work separator
 */
static void parseJfritzCsv( char *pnLine, char *pnSeparator ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return;
	}

	/* Count number of entries and remove first and last " */
	while ( ppnSplit[ nCount ] != NULL ) {
		int nLen = strlen( ppnSplit[ nCount ] );

		strncpy( ppnSplit[ nCount ], ppnSplit[ nCount ] + 1, nLen - 1 );
		ppnSplit[ nCount ][ nLen - 2 ] = 0;

		nCount++;
	}

	if ( nCount == 17 ) {
		if ( strlen( ppnSplit[ 1 ] ) > 0 || strlen( ppnSplit[ 2 ] ) > 0 ) {
			GHashTable *psTable = g_hash_table_new( NULL, NULL );
			gchar *pnDisplayName = NULL;

			gchar *pnTmp = g_strdup_printf( "%d", nImported++ );
			AddInfo( psTable, PERSON_ID, pnTmp );
			/* Private - NOT SUPPORTED */
			/* Last Name */
			AddInfo( psTable, PERSON_LAST_NAME, ppnSplit[ 1 ] );
			/* First Name */
			AddInfo( psTable, PERSON_FIRST_NAME, ppnSplit[ 2 ] );
			if ( strlen( ppnSplit[ 2 ] ) <= 0 ) {
				pnDisplayName = g_strdup_printf( "%s", ppnSplit[ 1 ] );
			} else {
				pnDisplayName = g_strdup_printf( "%s %s", ppnSplit[ 2 ], ppnSplit[ 1 ] );
			}
			AddInfo( psTable, PERSON_DISPLAY_NAME, pnDisplayName );
			/* Company */
			AddInfo( psTable, PERSON_COMPANY, ppnSplit[ 3 ] );
			/* Street */
			AddInfo( psTable, PERSON_PRIVATE_STREET, ppnSplit[ 4 ] );
			/* Zip Code */
			AddInfo( psTable, PERSON_PRIVATE_ZIPCODE, ppnSplit[ 5 ] );
			/* City */
			AddInfo( psTable, PERSON_PRIVATE_CITY, ppnSplit[ 6 ] );
			/* E-Mail - NOT SUPPORTED */
			/* Picture - NOT SUPPORTED */
			/* Home */
			AddInfo( psTable, PERSON_PRIVATE_PHONE, ppnSplit[ 9 ] );
			/* Mobile */
			AddInfo( psTable, PERSON_PRIVATE_MOBILE, ppnSplit[ 10 ] );
			/* HomeZone - NOT SUPPORTED */
			/* Business */
			AddInfo( psTable, PERSON_BUSINESS_PHONE, ppnSplit[ 12 ] );
			/* Other - TREAT LIKE BUSINESS FAX */
			AddInfo( psTable, PERSON_BUSINESS_FAX, ppnSplit[ 13 ] );
			/* Fax */
			AddInfo( psTable, PERSON_PRIVATE_FAX, ppnSplit[ 14 ] );
			/* SIP - NOT SUPPORTED */
			/* MAIN - NOT SUPPORTED */

			AddPerson( psTable, TRUE );
			g_free( pnDisplayName );
			g_free( pnTmp );
			g_hash_table_destroy( psTable );
		}
	} else {
		printf( "Invalid jfritz CSV format! (%d)\n", nCount );
	}

	g_strfreev( ppnSplit );
}

/**
 * \brief Parse speedport csv file
 * \param pnLine input line
 * \param pnSeparator work separator
 * \param nType call type (in/out)
 * \param pnProfileName profile name
 * \return nType
 */
static int parseSpeedportCsv( char *pnLine, char *pnSeparator, int nType, char *pnProfileName ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	if ( ppnSplit == NULL ) {
		return nType;
	}

	while ( ppnSplit[ nCount++ ] != NULL );

	if ( nCount == 1 ) {
		return 3;
	}

	if ( nCount != 6 ) {
		return nType;
	}

	AddCall(
		nType == 1 ? strcmp( ppnSplit[ 4 ], "0:00" ) == 0 ? "2" : "1" : "3",
		ppnSplit[ 0 ],
		"",
		ppnSplit[ 1 ],
		ppnSplit[ 2 ],
		ppnSplit[ 3 ],
		ppnSplit[ 4 ],
	    pnProfileName );

	g_strfreev( ppnSplit );

	return nType;
}

/**
 * \brief Parse areacodes csv file
 * \param pnLine input line
 * \param pnSeparator work separator
 */
static void parseAreaCodesCsv( char *pnLine, char *pnSeparator ) {
	gchar **ppnSplit = g_strsplit( pnLine, pnSeparator, -1 );
	gint nCount = 0;

	while ( ppnSplit[ nCount++ ] != NULL );

	if ( nCount < 2 ) {
		g_strfreev( ppnSplit );
		return;
	}

	if ( psAreaCodesTable == NULL ) {
		psAreaCodesTable = g_hash_table_new( NULL, NULL );
	}

	g_hash_table_insert( psAreaCodesTable, ( gpointer ) atol( ppnSplit[ 0 ] ), g_strdup( ppnSplit[ 1 ] ) );

	g_strfreev( ppnSplit );
}

/**
 * \brief Get City by Number
 * \param pnNumber phone number
 * \return city name or NULL
 */
gchar *getCityByNumber( gchar *pnNumber ) {
	static int nLoad = 0;
	int nLen = strlen( pnNumber );
	gchar *pnRet = NULL;
	gchar *pnSubString = NULL;
	gchar *pnNewNumber = NULL;

	/* ATM: Only work in germany */
	if ( strncmp( pnNumber, "0049", 4 ) != 0 ) {
		return NULL;
	}

	pnNewNumber = g_strdup_printf( "0%s", pnNumber + 4 );

	if ( nLoad == 0 ) {
		gchar *pnAreaCodes = g_build_filename( getDirectory( PKGDATADIR ), "areacodes_germany.csv", NULL );
		parseCsvFile( pnAreaCodes, getActiveProfile() -> pnName );
		g_free( pnAreaCodes );
		nLoad++;
	}

	nLen = strlen( pnNewNumber );
	/* Check for a 3 char match */
	if ( nLen >= 3 && pnRet == NULL ) {
		pnSubString = getSubString( pnNewNumber, 0, 3 );
		if ( pnSubString != NULL ) {
			pnRet = g_hash_table_lookup( psAreaCodesTable, ( gpointer ) atol( pnSubString ) );
			g_free( pnSubString );
		}
	}
	/* Check for a 4 char match */
	if ( nLen >= 4 && pnRet == NULL ) {
		pnSubString = getSubString( pnNewNumber, 0, 4 );
		if ( pnSubString != NULL ) {
			pnRet = g_hash_table_lookup( psAreaCodesTable, ( gpointer ) atol( pnSubString ) );
			g_free( pnSubString );
		}
	}
	/* Check for a 5 char match */
	if ( nLen >= 5 && pnRet == NULL ) {
		pnSubString = getSubString( pnNewNumber, 0, 5 );
		if ( pnSubString != NULL ) {
			pnRet = g_hash_table_lookup( psAreaCodesTable, ( gpointer ) atol( pnSubString ) );
			g_free( pnSubString );
		}
	}

	g_free( pnNewNumber );

	return pnRet;
}

/**
 * \brief Parse CSV data and add new entrys to global person or caller list
 * \param pnData csv data to parse
 * \param pnProfileName profile name
 * \return error code
 */
int parseCsvData( char *pnData, char *pnProfileName ) {
	char *pnSep = strdup( ";" );
	gchar **ppnLines = NULL;
	char *pnPos;
	char bIsNewFirmware = 0;
	char bIsFfgtk = 0;
	char bIsFfgtk2 = 0;
	char bIsOldFirmware = 0;
	char bIsSpeedport = 0;
	char bIsJfritz = 0;
	char bIsAreaCode = 0;
	int nType = 1;
	gint nIndex = 0;

	nImported = 0;

	ppnLines = g_strsplit( pnData, "\n", -1 );

	pnPos = strstr( ppnLines[ nIndex ], "sep=" );
	if ( pnPos != NULL ) {
		pnSep = g_malloc( strlen( pnPos ) - 4 + 1 );
		strcpy( pnSep, pnPos + 4 );
		nIndex++;
	}

	if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_FRITZBOX, strlen( EXPORT_CSV_FRITZBOX ) ) == 0 ) {
		bIsOldFirmware = 1;
	} else if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_FRITZBOX_NEWFIRMWARE, strlen( EXPORT_CSV_FRITZBOX_NEWFIRMWARE ) ) == 0 ) {
		bIsNewFirmware = 1;
	} else if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_FRITZBOX_NEWFIRMWARE_EN, strlen( EXPORT_CSV_FRITZBOX_NEWFIRMWARE_EN ) ) == 0 ) {
		bIsNewFirmware = 1;
	} else if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_FFGTK, strlen( EXPORT_CSV_FFGTK ) ) == 0 ) {
		bIsFfgtk = 1;
	} else if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_FFGTK2, strlen( EXPORT_CSV_FFGTK2 ) ) == 0 ) {
		bIsFfgtk2 = 1;
	} else if ( strncmp( ppnLines[ nIndex ], EXPORT_CSV_SPEEDPORT, strlen( EXPORT_CSV_SPEEDPORT ) ) == 0 ) {
		bIsSpeedport = 1;
	} else if ( strncmp( ppnLines[ nIndex ], IMPORT_CSV_JFRITZ, strlen( IMPORT_CSV_JFRITZ ) ) == 0 ) {
		bIsJfritz = 1;
	} else if ( strncmp( ppnLines[ nIndex ], AREACODES, strlen( AREACODES ) ) == 0 ) {
		bIsAreaCode = 1;
	} else {
		Debug( KERN_DEBUG, "CSV-Header %s\n", ppnLines[ nIndex ] );
	}

	while ( ppnLines[ ++nIndex ] != NULL ) {
		if ( bIsOldFirmware == 1 ) {
			parseFritzBoxCsv( ppnLines[ nIndex ], pnSep, pnProfileName );
		} else if ( bIsNewFirmware == 1 ) {
			parseFritzBoxNewCsv( ppnLines[ nIndex ], pnSep, pnProfileName );
		} else if ( bIsFfgtk == 1 ) {
			parseFfgtkCsv( ppnLines[ nIndex ], pnSep );
		} else if ( bIsFfgtk2 == 1 ) {
			parseFfgtk2Csv( ppnLines[ nIndex ], pnSep );
		} else if ( bIsSpeedport == 1 ) {
			nType = parseSpeedportCsv( ppnLines[ nIndex ], pnSep, nType, pnProfileName );
		} else if ( bIsJfritz == 1 ) {
			parseJfritzCsv( ppnLines[ nIndex ], pnSep );
		} else if ( bIsAreaCode == 1 ) {
			parseAreaCodesCsv( ppnLines[ nIndex ], pnSep );
		}
	}

	g_free( pnSep );
	g_strfreev( ppnLines );

	return 0;
}

/**
 * \brief Parse CSV file pnName and add new entrys to global person or caller list
 * \param pnName file name to parse
 * \param pnProfileName profile name
 * \return error code
 */
int parseCsvFile( const char *pnName, char *pnProfileName ) {
	gchar *pnData = loadData( ( char * ) pnName, NULL );
	int nError = -1;

	if ( pnData == NULL ) {
		return nError;
	}

	nError = parseCsvData( pnData, pnProfileName );
	g_free( pnData );

	return nError;
}

/**
 * \brief Save caller list to disk
 * \return error code
 */
int saveCallerList( void ) {
	GList *psList = NULL;
	struct sCaller *psCaller = NULL;
	FILE *psFile = NULL;

	Debug( KERN_DEBUG, "Save list (%s)\n", getCallerList( getActiveProfile() ) );
	psFile = fopen( getCallerList( getActiveProfile() ), "w+" );
	if ( psFile == NULL ) {
		Debug( KERN_DEBUG, "fail\n" );
		return -1;
	}

	fprintf( psFile, "sep=;\n" );
	fprintf( psFile, "Typ;Datum;Name;Rufnummer;Nebenstelle;Eigene Rufnummer;Dauer\n" );

	for ( psList = psCallerList; psList != NULL && psList -> data != NULL; psList = psList -> next ) {
		psCaller = psList -> data;

		if ( psCaller -> nType == 4 || psCaller -> nType == 5 ) {
			continue;
		}

		if ( !strcmp( psCaller -> pnProfile, getActiveProfile() -> pnName ) ) {
			gchar *pnName = g_convert( psCaller -> pnName, -1, "iso-8859-1", "UTF-8", NULL, NULL, NULL );
			fprintf( psFile, "%d;%s;%s;%s;%s;%s;%s\n",
				    psCaller -> nType,
				    psCaller -> pnDateTime,
				    pnName,
				    psCaller -> pnNumber,
				    psCaller -> pnLocalName,
				    psCaller -> pnLocalNumber,
				    psCaller -> pnDuration );
			g_free( pnName );
		}
	}

	fclose( psFile );

	return 0;
}
