/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_common.h>
#include <globus_gss_assist.h>
#include "misc.h"
#include "auth.h"
#include <stdio.h>
#include <ctype.h>
#include <string.h>

extern int	loglevel;

typedef struct {
  char	*name;
  int	bit;
} PERM;

static PERM	perms[] = {
  { "all", 0xFFFF },
  { "admin", P_ADMIN },
  { "lrc_read", P_LRCREAD },
  { "lrc_update", P_LRCUPDATE },
  { "rli_read", P_RLIREAD },
  { "rli_update", P_RLIUPDATE },
  { "stats", P_STATS },
};
#define NUMPERMS	(sizeof(perms) / sizeof(PERM))

/*
 * Find entry in acllist for dn.  First attempt to map dn to a local user
 * using a gridmap file, if successful search acllist for the local user.
 * Otherwise search acllist for dn.  Searching is done using regular
 * expressions.
 */
int
auth_getperms(ACL *acllist, char *dn)

{
  char	*localuser;

  if (globus_gss_assist_gridmap(dn, &localuser) != 0)
    localuser = NULL;
  while (acllist) {
    if (localuser && regexec(&acllist->re, localuser, 0, NULL, 0) == 0)
      break;
    else if (regexec(&acllist->re, dn, 0, NULL, 0) == 0)
      break;
    acllist = acllist->nxt;
  }
  if (loglevel > 1)
    logit(LOG_DEBUG, "auth_getperms(%s): localuser %s perms %X",
	  dn, localuser ? localuser : "-", acllist ? acllist->perms : 0);
  if (localuser)
    globus_libc_free(localuser);

  return acllist ? acllist->perms : 0;
}

ACL *
auth_newacl(ACL *acllist, char *val)

{
  ACL	*p;
  ACL	*q;
  char	*cp;
  char	*s;
  int	done;
  int	i;

  for (cp = val; *cp && *cp != ':'; cp++);
  if (!*cp) {
    logit(LOG_WARNING, "auth_newacl: No DN specified for %s", val);
    return acllist;
  }
  for (i = -1; isspace((int) cp[i]); i--); /* Trim trailing whitespace	*/
  cp[i+1] = '\0';			/* and colon			*/
  cp++;
  if ((p = (ACL *) globus_libc_malloc(sizeof(ACL))) == NULL) {
    logit(LOG_WARNING, "auth_newacl: No memory for ACL entry %s", val);
    return acllist;
  }
  if (!(p->dn = globus_libc_strdup(val))) {
    logit(LOG_WARNING, "auth_newacl: No memory for DN %s", val);
    globus_libc_free(p);
    return acllist;
  }
  if (regcomp(&p->re, p->dn, REG_EXTENDED|REG_NOSUB) != 0) {
    logit(LOG_WARNING, "auth_newacl: No memory for RE %s", p->dn);
    globus_libc_free(p->dn);
    globus_libc_free(p);
    return acllist;
  }
  p->perms = 0;
  done = 0;
  while (!done) {		/* Get permission bits		*/
    while (isspace((int) *cp))
      cp++;
    s = cp;
    while (*cp && !isspace((int) *cp))
      cp++;
    if (*cp)
      *cp++ = '\0';
    else
      done++;
    if (!*s)			/* Ignore trailing whitespace	*/
      break;
    for (i = 0; i < NUMPERMS; i++) {
      if (strcmp(s, perms[i].name) == 0) {
	p->perms |= perms[i].bit;
	break;
      }
    }
    if (i >= NUMPERMS)
      logit(LOG_WARNING, "auth_newacl: Unknown permission %s", s);
  }
  p->nxt = NULL;
  if (!acllist)
    return p;
  for (q = acllist; q->nxt; q = q->nxt);
  q->nxt = p;
  return acllist;
}

ACL *
auth_freeacl(ACL *acl)

{
  ACL	*p;

  while ((p = acl)) {
    acl = acl->nxt;
    globus_libc_free(p->dn);
    regfree(&p->re);
    globus_libc_free(p);
  }
  return NULL;
}

char *
auth_perms2s(int p)

{
  static char	buf[1000];
  int		i;
  int		j;

  if (p == 0xFFFF)
    return "all";
  *buf = '\0';
  j = 0;
  for (i = 0; i < NUMPERMS; i++)
    if ((p & perms[i].bit) == perms[i].bit) {
      if (j) {
	globus_libc_sprintf(&buf[j], " %s", perms[i].name);
	j += strlen(perms[i].name) + 1;
      } else {
	strcpy(buf, perms[i].name);
	j = strlen(perms[i].name);
      }
    }
  return buf;
}
