/*
 * Copyright 1999-2006 University of Chicago
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <globus_config.h>
#include <globus_io.h>
#include "misc.h"
#include "event.h"
#include <stdio.h>
#include <time.h>

/*
 * Periodic events (softstate updates, idle session timeout checks, etc)
 * are queued in EVENT structs and processed by event_process().  Events
 * are run in separate threads to avoid interfering with normal RLS
 * request processing.
 */

typedef struct event_ {
  void			*(*fun)(void *a);
  int			*interval;
  time_t		when;
  EVENTCONTROL		ec;
  struct event_		*nxt;
} EVENT;

extern globus_mutex_t	commonmtx;
extern globus_cond_t	commoncond;
static EVENT		*eventqueue = NULL;
static globus_mutex_t	eventmtx;

static void		reschedevent();
static void		freeevent(EVENT *e);

int
event_init()

{
  int	r;

  if ((r = globus_mutex_init(&eventmtx, GLOBUS_NULL)) != 0) {
    logit(LOG_WARNING, "event_init: mutex_init: %d", r);
    return 0;
  }
  return 1;
}

void
event_queue(void *(*fun)(void *a), int *interval)

{
  EVENT		*e;
  EVENT		*p;
  EVENT		*q;
  time_t	now;
  pthread_t	thr;
  int		rc;

  if ((e = (EVENT *) globus_libc_malloc(sizeof(EVENT))) == NULL) {
    logit(LOG_WARNING, "event_queue: No memory");
    return;
  }
  e->fun = fun;
  now = time(0);
  e->interval = interval;
  e->when = now + *interval;
  globus_mutex_init(&e->ec.mtx, GLOBUS_NULL);
  globus_cond_init(&e->ec.cond, GLOBUS_NULL);
  e->ec.flags = EF_RUNNING;

  if ((rc = pthread_create(&thr, GLOBUS_NULL, e->fun, &e->ec)) != 0)
    logit(LOG_WARNING, "event_process: pthread_create: %s",
	  globus_libc_system_error_string(rc));
  if ((rc = pthread_detach(thr)) != 0)
    logit(LOG_WARNING, "event_process: pthread_detach: %s",
	  globus_libc_system_error_string(rc));

  globus_mutex_lock(&eventmtx);
  for (q = NULL, p = eventqueue; p; q = p, p = p->nxt)
    if (e->when < p->when)
      break;
  if (q) {
    e->nxt = q->nxt;
    q->nxt = e;
  } else {
    e->nxt = eventqueue;
    eventqueue = e;
    reschedevent();
  }
  globus_mutex_unlock(&eventmtx);
}

void
event_process()

{
  time_t	now;
  EVENT		*e;
  EVENT		*p;
  EVENT		*q;

  now = time(0);
  globus_mutex_lock(&eventmtx);
  while (eventqueue) {
    if (eventqueue->when > now)
      break;
    e = eventqueue;
    eventqueue = eventqueue->nxt;

    if (!(e->ec.flags & EF_RUNNING)) {
      freeevent(e);
      continue;
    }

    globus_mutex_lock(&e->ec.mtx);
    e->ec.flags |= EF_RUN;
    globus_cond_signal(&e->ec.cond);
    globus_mutex_unlock(&e->ec.mtx);

    /*
     * Requeue event.
     */
    e->when = now + *e->interval;
    for (q = NULL, p = eventqueue; p; q = p, p = p->nxt)
      if (e->when < p->when)
	break;
    if (q) {
      e->nxt = q->nxt;
      q->nxt = e;
    } else {
      e->nxt = eventqueue;
      eventqueue = e;
    }
  }
  globus_mutex_unlock(&eventmtx);
  if (eventqueue)
    reschedevent();
}

void
event_cancel(void *(*fun)(void *a))

{
  EVENT	*p;
  EVENT	*q;

  globus_mutex_lock(&eventmtx);
  for (q = NULL, p = eventqueue; p; q = p, p = p->nxt)
    if (p->fun == fun)
      break;
  if (!p) {
    logit(LOG_DEBUG, "event_cancel: Can't find event %X", fun);
    globus_mutex_unlock(&eventmtx);
    return;
  }

  if (q)
    q->nxt = p->nxt;
  else {
    if ((eventqueue = p->nxt))
      reschedevent();
  }
  globus_mutex_unlock(&eventmtx);

  globus_mutex_lock(&p->ec.mtx);
  p->ec.flags |= EF_EXIT;
  globus_cond_signal(&p->ec.cond);
  while (p->ec.flags & EF_RUNNING)
    globus_cond_wait(&p->ec.cond, &p->ec.mtx);
  globus_mutex_unlock(&p->ec.mtx);

  freeevent(p);
}

/*
 * Check if event is queued.
 */
int
event_exists(void *(*fun)(void *a))

{
  EVENT	*e;

  globus_mutex_lock(&eventmtx);
  for (e = eventqueue; e; e = e->nxt)
    if (e->fun == fun)
      break;
  globus_mutex_unlock(&eventmtx);
  return e ? 1 : 0;
}

/*
 * Change event time to now so it is processed immediately.
 */
void
event_now(void *(*fun)(void *a))

{
  EVENT	*p;
  EVENT	*q;

  globus_mutex_lock(&eventmtx);
  for (q = NULL, p = eventqueue; p; q = p, p = p->nxt)
    if (p->fun == fun)
      break;
  if (!p) {
    logit(LOG_WARNING, "event_now: Can't find event");
    globus_mutex_unlock(&eventmtx);
    return;
  }
  p->when = time(0);
  if (q) {
    q->nxt = p->nxt;
    p->nxt = eventqueue;
    eventqueue = p;
  }
  globus_mutex_unlock(&eventmtx);
}

int
event_when()

{
  if (eventqueue)
    return eventqueue->when;
  return -1;
}

static void
reschedevent()

{
  globus_mutex_lock(&commonmtx);
  globus_cond_signal(&commoncond);
  globus_mutex_unlock(&commonmtx);
}

static void
freeevent(EVENT *e)

{
  globus_cond_destroy(&e->ec.cond);
  globus_mutex_destroy(&e->ec.mtx);
  globus_libc_free(e);
}
