/*
    This file is part of KolabAdmin.

    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#include "connection.h"
#include "user_filtermodel.h"

#include "user_listingmodel.h"

void UserListingModel::doQuery( const QString &userFilter )
{
  QString filter = QString( "(&(objectClass=kolabInetOrgPerson)%1(uid=*)(mail=*)(sn=*))" ).arg( userFilter );

  QStringList attributes;
  attributes << "givenName" << "sn" << "uid" << "mail" << "kolabDeleteflag";

  mData = Connection::self()->search( QString(), QLdap::Sub, filter, attributes, QLatin1String( "givenName" ) );

  emit layoutChanged();
}

int UserListingModel::columnCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return 4;
}

int UserListingModel::rowCount( const QModelIndex &parent ) const
{
  if ( parent.isValid() )
    return 0;

  return mData.entries().count();
}

QVariant UserListingModel::data( const QModelIndex &index, int role ) const
{
  if ( !index.isValid() )
    return QVariant();

  const QLdapEntry entry = mData.entries()[ index.row() ];

  if ( role == Qt::DisplayRole ) {
    switch ( index.column() ) {
      case 0:
        {
          QStringList parts = entry.dn().split( "," );
          if ( parts.contains( "cn=groups" ) )
            return QString( "G" );
          else if ( parts.contains( "cn=resources" ) )
            return QString( "R" );
          else if ( parts.contains( "cn=internal" ) )
            return QString( "I" );
          else
            return QString( "U" );
        }
        break;
      case 1:
        return QString( "%1 %2" ).arg( entry.value( "givenName" ), entry.value( "sn" ) );
        break;
      case 2:
        return entry.value( "mail" );
        break;
      case 3:
        return entry.value( "uid" );
        break;
      default:
        return QVariant();
    }
  } else if ( role == HasModifyActionRole ) {
    return true;
  } else if ( role == HasDeleteActionRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return true;
    else
      return false;
  } else if ( role == ModifyActionLabelRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return tr( "Modify" );
    else
      return tr( "User deleted, awaiting cleanup..." );
  } else if ( role == DeleteActionLabelRole ) {
    return tr( "Delete" );
  } else if ( role == ModifyActionIdRole ) {
    if ( entry.values( "kolabDeleteflag" ).isEmpty() )
      return mData.entries()[ index.row() ].dn();
    else
      return QString();
  } else if ( role == DeleteActionIdRole ) {
    return mData.entries()[ index.row() ].dn();
  }

  return QVariant();
}

QVariant UserListingModel::headerData( int section, Qt::Orientation, int role ) const
{
  if ( role != Qt::DisplayRole )
    return QVariant();

  switch ( section ) {
    case 0:
      return tr( "Type" );
      break;
    case 1:
      return tr( "Name" );
      break;
    case 2:
      return tr( "Email" );
      break;
    case 3:
      return tr( "UID" );
      break;
    default:
      return QVariant();
  }
}

FilterModel* UserListingModel::filterModel()
{
  return new UserFilterModel( this );
}
