/****************************************************************************/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     HB_precon.c                                                    */
/*                                                                          */
/* description:  hierachical basis preconditioning, including higher        */
/*               order                                                      */
/*               and also the BPX preconditioner                            */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

/*---8<---------------------------------------------------------------------*/
/*---  to do: something is wrong in BPX for p > 2                        ---*/
/*--------------------------------------------------------------------->8---*/

#include "alberta.h"

typedef struct
{ 
  const FE_SPACE      *fe_space;
  const DOF_SCHAR_VEC *bound_dv;
  int                 use_get_bound;

  int                 info;
  int                 high_degree;
  int                 mg_levels;
  int                 size;
  U_CHAR              *dof_level;
  U_CHAR              *local_dof;
  DOF                 (*dof_parent)[N_VERTICES_MAX];
  DOF                 *sort_dof, *sort_dof_invers;
  int                 *dofs_per_level;
  const S_CHAR        *bound;
  REAL                (*ipol)[N_VERTICES_MAX]; /* for high_degree interpol. */

  /* BPX section: */
  REAL                *g;
  REAL                diam;
} HB_DATA;


static void exit_HB_precon(void *precon_data);

/****************************************************************************/

typedef struct hb_traverse_data {
  int n0_vert, max_level, max_dof_level;
  const int *n_dof, *n0_dof, *node;
  int *local_dof_sort;
  HB_DATA *hb_data;
} HB_TRAVERSE_DATA;

static void max_level_fct(const EL_INFO *el_info, void *data)
{
  HB_TRAVERSE_DATA *ud = (HB_TRAVERSE_DATA *)data;
  int dof, dof0, dof1, level = (int)(el_info->level);
  int dim = el_info->mesh->dim;
  EL  *el   = el_info->el;

  ud->max_level = MAX(level, ud->max_level);

  if (!(IS_LEAF_EL(el))) {
    dof   = el->child[0]->dof[N_VERTICES(dim)-1][ud->n0_vert];
    dof0  = ud->hb_data->dof_parent[dof][0] = el->dof[0][ud->n0_vert];
    dof1  = ud->hb_data->dof_parent[dof][1] = el->dof[1][ud->n0_vert];
    level = 1 + MAX(ud->hb_data->dof_level[dof0], ud->hb_data->dof_level[dof1]);
    ud->hb_data->dof_level[dof] = level; 
    ud->max_dof_level = MAX(level, ud->max_dof_level);
  }
}


/* high_degree_fct(): set level of all non-vertex dofs to highest level,  */
/*                    and save parent data                                */

static void high_degree_fct(const EL_INFO *el_info, void *data)
{
  FUNCNAME("high_degree_fct");
  HB_TRAVERSE_DATA *ud = (HB_TRAVERSE_DATA *)data;
  int  i, j, k, m, n, nod0, n0, dof;
  int  dim = el_info->mesh->dim;
  EL   *el   = el_info->el;
  DOF  vertex_dof[N_VERTICES_MAX];
  
  DEBUG_TEST_EXIT(IS_LEAF_EL(el), "Non-leaf element???\n");

  for (k=0; k<N_VERTICES(dim); k++)
    vertex_dof[k] = el->dof[k][ud->n0_vert];
  m = N_VERTICES(dim);
  
  if ((n = ud->n_dof[CENTER]) > 0) {
    nod0 = ud->node[CENTER];
    n0 = ud->n0_dof[CENTER];
    for (j=0; j<n; j++) {
      dof = el->dof[nod0][n0+j];
      /*  	  MSG("center dof %2d: level %d\n", dof, max_dof_level); */
      ud->hb_data->dof_level[dof] = ud->max_dof_level;
      for (k=0; k<N_VERTICES(dim); k++)
	ud->hb_data->dof_parent[dof][k] = vertex_dof[k];
      ud->hb_data->local_dof[dof] = ud->local_dof_sort[m++];
    }
  }
  
  if(dim > 1 && (n = ud->n_dof[EDGE]) > 0) {
    nod0 = ud->node[EDGE];
    n0 = ud->n0_dof[EDGE];
    for (i=0; i<N_EDGES(dim); i++)
      for (j=0; j<n; j++) {
	dof = el->dof[nod0+i][n0+j];
	if (!ud->hb_data->local_dof[dof]) {
	  /* MSG("edge dof   %2d: level %d\n", dof, max_dof_level); */
	  ud->hb_data->dof_level[dof] = ud->max_dof_level;
	  for (k=0; k<N_VERTICES(dim); k++)
	    ud->hb_data->dof_parent[dof][k] = vertex_dof[k];
	  ud->hb_data->local_dof[dof] = ud->local_dof_sort[m];
	}
	m++;
      }
    
  }
  
  if (dim == 3 && (n = ud->n_dof[FACE]) > 0) {
    nod0 = ud->node[FACE];
    n0 = ud->n0_dof[FACE];
    for (i=0; i < N_FACES_3D; i++)
      for (j=0; j<n; j++) {
	dof = el->dof[nod0+i][n0+j];
	if (!ud->hb_data->local_dof[dof]) {
	  /*  	    MSG("face dof   %2d: level %d\n", dof, max_dof_level); */
	  ud->hb_data->dof_level[dof] = ud->max_dof_level;
	  for (k=0; k < N_VERTICES_3D; k++)
	      ud->hb_data->dof_parent[dof][k] = vertex_dof[k];
	  ud->hb_data->local_dof[dof] = ud->local_dof_sort[m];
	}
	m++;
      }
  }
  
    
  DEBUG_TEST_EXIT(m == ud->hb_data->fe_space->bas_fcts->n_bas_fcts,
	      "m <> n_bas_fcts: %d  %d\n",
	      m, ud->hb_data->fe_space->bas_fcts->n_bas_fcts);
  
  return;
}


/****************************************************************************/

/* NOTE: This terrible hack is definitely something we could do better...   */
static DOF *el_hat_dofs[N_VERTICES_MAX+N_EDGES_MAX+N_FACES_MAX+1];

static MESH dummy_mesh[4] = {{0},{nil,1},{nil,2},{nil,3}};

static EL el_hat =
  { 
    {nil, nil},  /* children */
    el_hat_dofs,
    0    /* ... */
  };

#if DIM_OF_WORLD == 1
static const EL_INFO el_hat_info[2] = {
  { nil,
    {{0.0}, {0.0}, {0.0}, {0.0}},
    nil,
    nil, nil,
    0,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 1,
    {{0.0}, {1.0}},
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  }};
#elif DIM_OF_WORLD == 2
static const EL_INFO el_hat_info[3] = {
  { nil,
    {{0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
    nil,
    nil, nil,
    0,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 1,
    {{0.0, 0.0}, {1.0, 0.0}, {0.0, 0.0}, {0.0, 0.0}},
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 2,
    { {1.0, 0.0}, {0.0, 1.0}, {0.0, 0.0} },
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  }};
#elif DIM_OF_WORLD == 3
static const EL_INFO el_hat_info[4] = {
  { nil,
    {{0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}},
    nil,
    nil, nil,
    0,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 1,
    {{0.0, 0.0, 0.0}, {1.0, 0.0, 0.0}, {0.0, 0.0, 0.0}, {0.0, 0.0, 0.0}},
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 2,
    { {1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 0.0} },
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  },
  { dummy_mesh + 3,
    { {1.0, 0.0, 0.0}, {0.0, 1.0, 0.0}, {0.0, 0.0, 1.0}, {0.0, 0.0, 0.0} },
    nil,
    nil, nil,
    FILL_COORDS,
    { 0 }  /*---  and the rest is initialized with 0s  ---*/
  }};
#endif


static REAL lambda_0(const REAL_D x) { return(x[0]);}
static REAL lambda_1(const REAL_D x) { return(x[1]);}
static REAL lambda_2(const REAL_D x) { return(x[2]);}

static REAL lambda_0_1d(const REAL_D x) { return(1.0-x[0]);}
static REAL lambda_2_2d(const REAL_D x) { return(1.0-x[0]-x[1]);}
static REAL lambda_3_3d(const REAL_D x) { return(1.0-x[0]-x[1]-x[2]);}

static REAL (*lambda[4][N_VERTICES_MAX])(const REAL_D)
		= {{nil, nil, nil, nil},
		   {lambda_0_1d,lambda_0,nil,nil},
		   {lambda_0,lambda_1,lambda_2_2d,nil},
		   {lambda_0,lambda_1,lambda_2,lambda_3_3d}};

/****************************************************************************/
/*  init_HB_precon(matrix, boundary, info);    initialize HB_DATA structure */
/*  fills    (void *)(HB_DATA *), if ok,                                    */
/*  returns   nil, if anything goes wrong.                                  */
/****************************************************************************/

static int init_HB_BPX_precon(void *precon_data, int BPX)
{
  FUNCNAME("init_HB_BPX_precon");
  static HB_TRAVERSE_DATA td[1];
  HB_DATA         *data = (HB_DATA *)precon_data;
  const FE_SPACE  *fe_space;
  const DOF_ADMIN *admin;
  int             i, j, k, m, size, info, dim;
  int             *tmp_per_level = nil;
  int             n_bas_fcts;
  const REAL      *one_ipol;
  DOF             *tmp_dofs, *tmp_dof_ptr;
  const DOF       *dof_indices;

  if (!data) 
  {
      ERROR("no precon_data\n");
      return false;
  }
  if (!(data->fe_space)) 
  {
      ERROR("no precon_data->fe_space\n");
      return false;
  }
  
  info     = data->info;
  fe_space = data->fe_space;
  if (!fe_space || !(fe_space->admin) || !(fe_space->bas_fcts))
  {
    MSG("no fe_space or admin or bas_fcts.\n");
    return false;
  }

  if (fe_space->bas_fcts->n_dof[VERTEX] != 1)
  {
    MSG("sorry, only for FE spaces with n_dof[VERTEX]==1.\n");
    return false;
  }

  admin = fe_space->admin;
  size  = admin->size_used;
  dim   = fe_space->mesh->dim;

  n_bas_fcts = fe_space->bas_fcts->n_bas_fcts;
  data->high_degree = (n_bas_fcts > N_VERTICES(dim));
  if (data->high_degree)
  {
    INFO(info, 1,"use high degree version\n");
  }

  data->mg_levels = 0;
  data->dofs_per_level = nil;

  if (data->use_get_bound)
  {
    DOF_SCHAR_VEC  dsv = {nil, nil, "HB/BPX bound", 0, nil, nil, nil};

    dsv.fe_space = data->fe_space;
    dsv.size     = size;
    dsv.vec      = MEM_ALLOC(size, S_CHAR);

    dirichlet_bound(nil, nil, nil, &dsv);
    data->bound = dsv.vec;
  }
  else if (data->bound_dv)
  {
    data->bound = data->bound_dv->vec;
  }
  else
  {
    data->bound = nil;
  }

  data->ipol = nil;
  
  data->dof_level       = MEM_ALLOC(2*size, U_CHAR);
  data->local_dof       = data->dof_level       + size;
  
  data->sort_dof        = MEM_ALLOC((N_VERTICES_MAX+2)*size, DOF);
  data->sort_dof_invers = data->sort_dof        + size;
  data->dof_parent      = (DOF (*)[N_VERTICES_MAX])(data->sort_dof_invers + size);
  data->size            = size;

  FOR_ALL_DOFS(admin,
	       data->dof_level[dof] = 0;
	       for (j=0; j < N_VERTICES_MAX;j++) data->dof_parent[dof][j] = -1;
	       data->local_dof[dof] = 0;
	       );

  td->n0_vert = admin->n0_dof[VERTEX];
  td->max_level = td->max_dof_level = 0;
  td->hb_data = data;
  mesh_traverse(fe_space->mesh, -1, CALL_EVERY_EL_PREORDER, max_level_fct, td);
  data->mg_levels = (td->max_level + dim - 1) / dim + 1;

  TEST_EXIT(data->mg_levels == (td->max_dof_level+1),
	    "mg_levels %d != max_dof_level %d + 1\n",
	    data->mg_levels, td->max_dof_level);


  if (data->high_degree) {     /* add level for fine-grid non-vertex DOFs */
    data->mg_levels++;
    data->ipol = (REAL (*)[N_VERTICES_MAX])
      (MEM_ALLOC(N_VERTICES_MAX*n_bas_fcts, REAL));

    td->local_dof_sort = MEM_ALLOC(n_bas_fcts, int);
    for (j=0; j<n_bas_fcts; j++) td->local_dof_sort[j] = j;  /* ???? */
    
    tmp_dof_ptr = tmp_dofs = MEM_ALLOC(fe_space->mesh->n_dof_el, DOF);
    m = 0;
    for (i=0; i<N_VERTICES(dim); i++) {
      el_hat_dofs[fe_space->mesh->node[VERTEX]+i] = tmp_dof_ptr;
      for (j=0; j<admin->n_dof[VERTEX]; j++)
	tmp_dof_ptr[admin->n0_dof[VERTEX]+j] = m++;
      tmp_dof_ptr += fe_space->mesh->n_dof[VERTEX];
    }

    if (fe_space->mesh->n_dof[CENTER]) {
      el_hat_dofs[fe_space->mesh->node[CENTER]] = tmp_dof_ptr;
      for (j=0; j<admin->n_dof[CENTER]; j++)
	tmp_dof_ptr[admin->n0_dof[CENTER]+j] = m++;
    }

    if(dim > 1 && fe_space->mesh->n_dof[EDGE])
      for (i=0; i < N_EDGES(dim); i++) {
	el_hat_dofs[fe_space->mesh->node[EDGE]+i] = tmp_dof_ptr;
	for (j=0; j<admin->n_dof[EDGE]; j++)
	  tmp_dof_ptr[admin->n0_dof[EDGE]+j] = m++;
	tmp_dof_ptr += fe_space->mesh->n_dof[EDGE];
      }

    if (dim == 3 && fe_space->mesh->n_dof[FACE])
      for (i=0; i<N_FACES_3D; i++) {
	el_hat_dofs[fe_space->mesh->node[FACE]+i] = tmp_dof_ptr;
	for (j=0; j<admin->n_dof[FACE]; j++)
	  tmp_dof_ptr[admin->n0_dof[FACE]+j] = m++;
	tmp_dof_ptr += fe_space->mesh->n_dof[FACE];
      }

    TEST_EXIT(m==n_bas_fcts,"m != n_bas_fcts: %d %d\n", m, n_bas_fcts);
    
    dof_indices = fe_space->bas_fcts->get_dof_indices(&el_hat, admin, nil);
    for (i = 0; i<fe_space->bas_fcts->n_bas_fcts; i++)
      td->local_dof_sort[dof_indices[i]] = i;

#if 0
    print_int_vec("dof_indices   ", dof_indices, n_bas_fcts);
    print_int_vec("local_dof_sort", td->local_dof_sort, n_bas_fcts);
#endif

    MEM_FREE(tmp_dofs, fe_space->mesh->n_dof_el, DOF);

    for (i=0; i<N_VERTICES(dim); i++) {
      one_ipol =
	fe_space->bas_fcts->interpol(&(el_hat_info[dim]), 0, nil,
				     lambda[dim][i], nil, nil);
      for (j=0; j<n_bas_fcts; j++)
	data->ipol[j][i] = one_ipol[j];
    }

   
    td->max_dof_level++;
    td->n0_dof = admin->n0_dof;
    td->n_dof  = admin->n_dof;
    td->node   = fe_space->mesh->node;
    
    mesh_traverse(fe_space->mesh, -1, CALL_LEAF_EL, high_degree_fct, td);    

    MEM_FREE(td->local_dof_sort, n_bas_fcts, int);

    if (info > 3) 
    {
      for (i=0; i<n_bas_fcts; i++)
      {
	MSG("ipol[%2d]", i); PRINT_REAL_VEC("", data->ipol[i],N_VERTICES(dim));
      }
    }
    
#if 0
    FOR_ALL_DOFS(admin,
		 if (data->local_dof[dof])
		   MSG("dof %3d: local_dof=%2d\n", dof, data->local_dof[dof]);
		 );
#endif
  }
  
  if (data->mg_levels < 2) {
/*     exit_HB_precon(data); */
    return false;
  }
 
  data->dofs_per_level = MEM_ALLOC(data->mg_levels, int);
  tmp_per_level  = MEM_ALLOC(data->mg_levels, int);
  
  for (i = 0; i < data->mg_levels; i++) data->dofs_per_level[i] = 0;
  FOR_ALL_DOFS(admin, data->dofs_per_level[data->dof_level[dof]]++);
  if (info > 3) {
    MSG("dofs_per_level:");
    for (i = 0; i < data->mg_levels; i++)
      print_msg(" %d", data->dofs_per_level[i]);
    print_msg("\n");
  }
  for (i = 1; i < data->mg_levels; i++) {
    tmp_per_level[i]   = data->dofs_per_level[i-1];
    data->dofs_per_level[i] += data->dofs_per_level[i-1];
  }
  tmp_per_level[0] = 0;      /* pointers for filling the sort vectors */

  if (info > 3) {
    MSG("dofs_per_level accumulated:");
    for (i = 0; i < data->mg_levels; i++)
      print_msg(" %d", data->dofs_per_level[i]);
    print_msg("\n");
  }

#if 0
  for (i = 0; i < data->dofs_per_level[data->mg_levels-1]; i++)
    MSG("dof_parent[%3d] = (%3d,%3d,%3d), lev=%2d (%2d,%2d,%3d)\n",
	i, 
	data->dof_parent[i][0], data->dof_parent[i][1], data->dof_parent[i][2],
	data->dof_level[i], 
	data->dof_level[data->dof_parent[i][0]],
	data->dof_level[data->dof_parent[i][1]],
	data->dof_level[data->dof_parent[i][2]]);
#endif


/* build sort_dof[] and sort_dof_invers[] vectors                           */

  FOR_ALL_DOFS(admin,
	       j = data->dof_level[dof];
	       k = tmp_per_level[j]++;
	       data->sort_dof[k] = dof;
	       data->sort_dof_invers[dof] = k;
    );

#if 0
  for (i = 0; i < data->dofs_per_level[data->mg_levels-1]; i++)
  {
    j = data->sort_dof[i];
    MSG("sort[%3d]: dof=%3d, lev=%2d; invers[%3d]=%3d\n",
	i, j, data->dof_level[j], j, data->sort_dof_invers[j]);
  }
  /* WAIT; */
#endif

  MEM_FREE(tmp_per_level, data->mg_levels, int);

  if (BPX) 
  {
    data->g = MEM_CALLOC(DIM_OF_WORLD*data->size, REAL);
    data->diam = fe_space->mesh->diam[0];
    for (i=1; i<DIM_OF_WORLD; i++)
      data->diam = MAX(data->diam, fe_space->mesh->diam[i]);
  }
  else
  {
    data->g = nil;
    data->diam = 0;
  }

  return true;
}


/****************************************************************************/
/****************************************************************************/

static void exit_HB_BPX_precon(void *vdata, int BPX)
{
  FUNCNAME("exit_HB_BPX_precon");
  HB_DATA *data = (HB_DATA *)vdata;
  int      dim;

  if (!data) 
  {
    MSG("no data ???\n");
    return;
  }

  dim = data->fe_space->mesh->dim;

  if (BPX) 
  {
    DEBUG_TEST(data->g, "no g entry in BPX data???\n");
    if (data->g) MEM_FREE(data->g, DIM_OF_WORLD*data->size, REAL);
  }
  else 
  {
    DEBUG_TEST(!data->g, "g entry in HB data???\n");
    if (data->g) MEM_FREE(data->g, DIM_OF_WORLD*data->size, REAL);
  }
  
  MEM_FREE(data->dofs_per_level, data->mg_levels, int);

  if (data->high_degree) 
  {
    MEM_FREE(data->ipol, N_VERTICES_MAX*data->fe_space->bas_fcts->n_bas_fcts,
	     REAL);
  }

  MEM_FREE(data->sort_dof, (N_VERTICES_MAX+2)*data->size, DOF);
  MEM_FREE(data->dof_level, 2*data->size, U_CHAR);

  if (data->use_get_bound)
  {
    MEM_FREE(data->bound, data->size, S_CHAR);
  }

  MEM_FREE(data, 1, HB_DATA);

  return;
}

/****************************************************************************/
static int init_HB_precon(void *precon_data)
{
  return init_HB_BPX_precon(precon_data, 0);
}
static void exit_HB_precon(void *precon_data)
{
  exit_HB_BPX_precon(precon_data, 0);
}
/****************************************************************************/

static void HB_precon_s(void *vdata, int n, REAL *r)
{
  FUNCNAME("HB_precon_s");
  int     i, idof, level, last, jdof, k, level1, dim;
  HB_DATA *data;

  data = (HB_DATA *)vdata;
  if (!data) {
    MSG("no data ???\n");
    return;
  }
  dim = data->fe_space->mesh->dim;

  if (n > data->size) {
    MSG("n > data->size ???\n");
    return;
  }

  if (data->mg_levels < 2)                                /* nothing to do */
  {
    return;
  }

/* transposed basis transformation (run over refined levels only)          */

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      jdof = data->local_dof[i];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[data->dof_parent[idof][k]] <= INTERIOR)
	    r[data->dof_parent[idof][k]] += data->ipol[jdof][k] * r[idof];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	  r[data->dof_parent[idof][k]] += data->ipol[jdof][k] * r[idof];
	}
      }
    }
    level1 = data->mg_levels - 2;
  } 
  else {
    level1 = data->mg_levels - 1;
  }
  


  for (level = level1; level > 0; level--)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (data->bound) {
	if (data->bound[data->dof_parent[idof][0]] <= INTERIOR)
	  r[data->dof_parent[idof][0]] += 0.5 * r[idof];
	if (data->bound[data->dof_parent[idof][1]] <= INTERIOR)
	  r[data->dof_parent[idof][1]] += 0.5 * r[idof];
      }
      else {
	r[data->dof_parent[idof][0]] += 0.5 * r[idof];
	r[data->dof_parent[idof][1]] += 0.5 * r[idof];
      }
    }
  }

/* basis transformation (run over refined levels only) */

  for (level = 1; level <= level1; level++)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	r[idof] += 0.5 * (r[data->dof_parent[idof][0]]
			  + r[data->dof_parent[idof][1]]);
      }
    }
  }

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	jdof = data->local_dof[i];
	for (k = 0; k < N_VERTICES(dim); k++)
	  r[idof] += data->ipol[jdof][k] * r[data->dof_parent[idof][k]];
      }
    }
  }

  return;
}

/****************************************************************************/

static void HB_precon_d(void *vdata, int n, REAL *r)
{
  FUNCNAME("HB_precon_d");
  int     i,j, idof, level, last, jdof, k, level1, dim;
  HB_DATA *data;
  REAL_D  *rd = (REAL_D *)r;

  data = (HB_DATA *)vdata;
  if (!data) {
    MSG("no data ???\n");
    return;
  }
  dim = data->fe_space->mesh->dim;

  if (n > DIM_OF_WORLD*data->size) {
    MSG("n > data->DIM_OF_WORLD*size ???\n");
    return;
  }

  if (data->mg_levels < 2)                                /* nothing to do */
  {
    return;
  }

/* transposed basis transformation (run over refined levels only)          */

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      jdof = data->local_dof[i];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[data->dof_parent[idof][k]] <= INTERIOR)
	    for (j=0; j<DIM_OF_WORLD; j++)
	      rd[data->dof_parent[idof][k]][j]
		+= data->ipol[jdof][k] * rd[idof][j];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	  for (j=0; j<DIM_OF_WORLD; j++)
	    rd[data->dof_parent[idof][k]][j]
	      += data->ipol[jdof][k] * rd[idof][j];
	}
      }
    }
    level1 = data->mg_levels - 2;
  } 
  else {
    level1 = data->mg_levels - 1;
  }
  

  for (level = level1; level > 0; level--)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (data->bound) {
	if (data->bound[data->dof_parent[idof][0]] <= INTERIOR)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    rd[data->dof_parent[idof][0]][j] += 0.5 * rd[idof][j];
	if (data->bound[data->dof_parent[idof][1]] <= INTERIOR)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    rd[data->dof_parent[idof][1]][j] += 0.5 * rd[idof][j];
      }
      else {
	for (j=0; j<DIM_OF_WORLD; j++) {
	  rd[data->dof_parent[idof][0]][j] += 0.5 * rd[idof][j];
	  rd[data->dof_parent[idof][1]][j] += 0.5 * rd[idof][j];
	}
      }
    }
  }

/* basis transformation (run over refined levels only) */

  for (level = 1; level <= level1; level++)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	for (j=0; j<DIM_OF_WORLD; j++) {
	  rd[idof][j] += 0.5 * (rd[data->dof_parent[idof][0]][j]
			       + rd[data->dof_parent[idof][1]][j]);
	}
      }
    }
  }

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	jdof = data->local_dof[i];
	for (k = 0; k < N_VERTICES(dim); k++)
	  for (j=0; j<DIM_OF_WORLD; j++) {
	    rd[idof][j]
	      += data->ipol[jdof][k] * rd[data->dof_parent[idof][k]][j];
	  }
      }
    }
  }
  
  return;
}

/****************************************************************************/
/****************************************************************************/

const PRECON *get_HB_precon_s(const FE_SPACE *fe_space,
			      const DOF_SCHAR_VEC *bound, 
			      int use_get_bound, int info)
{
    PRECON  *precon;
    HB_DATA *data;
    
    if (bound && (bound->fe_space != fe_space)) {
	ERROR("different fe spaces ?\n");
	return(nil);
    }

    precon = MEM_CALLOC(1, PRECON);
    data   = MEM_CALLOC(1, HB_DATA);
    
    data->fe_space = fe_space;
    if (!(data->bound_dv = bound))
      data->use_get_bound = use_get_bound;

    data->info     = info;

    precon->precon_data = data;
    precon->init_precon = init_HB_precon;
    precon->precon      = HB_precon_s;
    precon->exit_precon = exit_HB_precon;

    return(precon);
}

const PRECON *get_HB_precon_d(const FE_SPACE *fe_space,
			      const DOF_SCHAR_VEC *bound, 
			      int use_get_bound, int info)
{
    PRECON  *precon;
    HB_DATA *data;
    
    if (bound && (bound->fe_space != fe_space)) {
	ERROR("different fe spaces ?\n");
	return(nil);
    }

    precon = MEM_CALLOC(1, PRECON);
    data   = MEM_CALLOC(1, HB_DATA);
    
    data->fe_space = fe_space;
    if (!(data->bound_dv = bound))
      data->use_get_bound = use_get_bound;

    data->info     = info;

    precon->precon_data = data;
    precon->init_precon = init_HB_precon;
    precon->precon      = HB_precon_d;
    precon->exit_precon = exit_HB_precon;

    return(precon);
}

/****************************************************************************/
/****************************************************************************/
/**   BPX  preconditioner                                                  **/
/****************************************************************************/
/****************************************************************************/

static int init_BPX_precon(void *precon_data)
{
  return init_HB_BPX_precon(precon_data, 1);
}

void exit_BPX_precon(void *precon_data)
{
  exit_HB_BPX_precon(precon_data, 1);
}
/****************************************************************************/

static void BPX_precon_s(void *vdata, int n, REAL *h)
{
  FUNCNAME("BPX_precon_s");
  int     i, idof, level, first = 0, last, jdof, k, level1, *dof_parent, dim;
  HB_DATA *data;
  REAL    *g, *ipol;

  data = (HB_DATA *)vdata;
  if (!data) {
    MSG("no data ???\n");
    return;
  }
  dim = data->fe_space->mesh->dim;

  if (n > data->size) {
    MSG("n > data->size ???\n");
    return;
  }

  if (data->mg_levels < 2)                                /* nothing to do */
  {
    return;
  }

  g = data->g;
  DEBUG_TEST_EXIT(g, "no g vec in HB_DATA\n");

  /* copy h to g */
  for (i=0; i<data->size; i++)  g[i] = h[i];
  
  /* Loop over all non-macro levels */

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];

    /* inverse basis transformation on high degree level for h */

    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      dof_parent = data->dof_parent[idof];
      jdof = data->local_dof[i];
      ipol = data->ipol[jdof];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[data->dof_parent[idof][k]] <= INTERIOR)
	    h[idof] -= ipol[k] * h[dof_parent[k]];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	  h[idof] -= ipol[k] * h[dof_parent[k]];
	}
      }
    }

    /* transposed basis transformation on high degree level for g */

    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      dof_parent = data->dof_parent[idof];
      jdof = data->local_dof[i];
      ipol = data->ipol[jdof];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[dof_parent[k]] <= INTERIOR)
	    g[dof_parent[k]] += ipol[k] * g[idof];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	  g[dof_parent[k]] += ipol[k] * g[idof];
	}
      }
    }

    /* add up to degree1 level */

    last = data->dofs_per_level[data->mg_levels - 2];
    if (data->bound) {
      for (i = 0; i < last; i++) {
	idof = data->sort_dof[i];
	if (data->bound[idof] <= 0)
	  h[idof] += g[idof];
      }
    } else 
    {
      for (i = first; i < last; i++) 
      {
	idof = data->sort_dof[i];
	h[idof] += g[idof];
      }
    }
    
    level1 = data->mg_levels - 2;
  } else {
    level1 = data->mg_levels - 1;
  }

  for (level = level1; level > 0; level--)
  {
    last = data->dofs_per_level[level];

    /* inverse basis transformation on level k (only) for h */
    for (i = data->dofs_per_level[level-1]; i < last; i++) {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	h[idof] -= 0.5 * (h[data->dof_parent[idof][0]]
			  + h[data->dof_parent[idof][1]]);
      }
    }
    

    /* transposed basis transformation on level k-1 for g */
    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (data->bound) {
	if (data->bound[data->dof_parent[idof][0]] <= INTERIOR)
	  g[data->dof_parent[idof][0]] += 0.5 * g[idof];
	if (data->bound[data->dof_parent[idof][1]] <= INTERIOR)
	  g[data->dof_parent[idof][1]] += 0.5 * g[idof];
      }
      else {
	g[data->dof_parent[idof][0]] += 0.5 * g[idof];
	g[data->dof_parent[idof][1]] += 0.5 * g[idof];
      }
    }
  
    /* add up to level k-1 */


    last = data->dofs_per_level[level-1];
    if (data->bound) {
      for (i = 0; i < last; i++) {
	idof = data->sort_dof[i];
	if (data->bound[idof] <= 0)
	  h[idof] += g[idof];
      }
    } else 
    {
      for (i = first; i < last; i++) 
      {
	idof = data->sort_dof[i];
	h[idof] += g[idof];
      }
    }
  }
  
  /* basis transformation (all refined levels) */

  for (level = 1; level <= level1; level++)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	h[idof] += 0.5 * (h[data->dof_parent[idof][0]]
			  + h[data->dof_parent[idof][1]]);
      }
    }
  }

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	dof_parent = data->dof_parent[idof];
	jdof = data->local_dof[i];
	ipol = data->ipol[jdof];
	for (k = 0; k < N_VERTICES(dim); k++)
	  h[idof] += ipol[k] * h[dof_parent[k]];
      }
    }
  }

  return;
}

/****************************************************************************/
static void BPX_precon_d(void *vdata, int n, REAL *hh)
{
  FUNCNAME("BPX_precon_d");
  int     i, j, k, idof, level, first = 0, last, jdof, level1, dim;
  HB_DATA *data;
  REAL_D  *g;
  REAL_D  *h = (REAL_D *)hh;

  data = (HB_DATA *)vdata;
  if (!data) {
    MSG("no data ???\n");
    return;
  }
  dim = data->fe_space->mesh->dim;

  if (n > data->size*DIM_OF_WORLD) {
    MSG("n > data->size*DIM_OF_WORLD ???\n");
    return;
  }

  if (data->mg_levels < 2)                                /* nothing to do */
  {
    return;
  }

  g = (REAL_D *)data->g;
  DEBUG_TEST_EXIT(g, "no g vec in HB_DATA\n");

  /* copy h to g */
  for (i=0; i<data->size; i++)
    for (j=0; j<DIM_OF_WORLD; j++)
      g[i][j] = h[i][j];
  
  /* Loop over all non-macro levels */

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];

    /* inverse basis transformation on high degree level for h */

    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      jdof = data->local_dof[i];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[data->dof_parent[idof][k]] <= INTERIOR)
	    for (j=0; j<DIM_OF_WORLD; j++)
	      h[idof][j]
		-= data->ipol[jdof][k] * h[data->dof_parent[idof][k]][j];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	  for (j=0; j<DIM_OF_WORLD; j++)
	    h[idof][j]
	      -= data->ipol[jdof][k] * h[data->dof_parent[idof][k]][j];
	}
      }
    }

    /* transposed basis transformation on high degree level for g */

    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      jdof = data->local_dof[i];
      if (data->bound) {
	for (k=0; k<N_VERTICES(dim); k++) {
	  if (data->bound[data->dof_parent[idof][k]] <= INTERIOR)
	    for (j=0; j<DIM_OF_WORLD; j++)
	      g[data->dof_parent[idof][k]][j] 
		+= data->ipol[jdof][k] * g[idof][j];
	}
      }
      else {
	for (k=0; k<N_VERTICES(dim); k++) {
	    for (j=0; j<DIM_OF_WORLD; j++)
	      g[data->dof_parent[idof][k]][j]
		+= data->ipol[jdof][k] * g[idof][j];
	}
      }
    }

    /* add up to degree1 level */

    last = data->dofs_per_level[data->mg_levels - 2];
    if (data->bound) {
      for (i = 0; i < last; i++) {
	idof = data->sort_dof[i];
	if (data->bound[idof] <= 0)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    h[idof][j] += g[idof][j];
      }
    } else {
      for (i = first; i < last; i++) {
	idof = data->sort_dof[i];
	for (j=0; j<DIM_OF_WORLD; j++)
	  h[idof][j] += g[idof][j];
      }
    }
    
    level1 = data->mg_levels - 2;
  } else {
    level1 = data->mg_levels - 1;
  }

  for (level = level1; level > 0; level--)
  {
    last = data->dofs_per_level[level];

    /* inverse basis transformation on level k (only) for h */
    for (i = data->dofs_per_level[level-1]; i < last; i++) {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	for (j=0; j<DIM_OF_WORLD; j++)
	  h[idof][j] -= 0.5 * (h[data->dof_parent[idof][0]][j]
			       + h[data->dof_parent[idof][1]][j]);
      }
    }
    

    /* transposed basis transformation on level k-1 for g */
    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (data->bound) {
	if (data->bound[data->dof_parent[idof][0]] <= INTERIOR)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    g[data->dof_parent[idof][0]][j] += 0.5 * g[idof][j];
	if (data->bound[data->dof_parent[idof][1]] <= INTERIOR)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    g[data->dof_parent[idof][1]][j] += 0.5 * g[idof][j];
      }
      else {
	for (j=0; j<DIM_OF_WORLD; j++) {
	  g[data->dof_parent[idof][0]][j] += 0.5 * g[idof][j];
	  g[data->dof_parent[idof][1]][j] += 0.5 * g[idof][j];
	}
      }
    }
  
    /* add up to level k-1 */


    last = data->dofs_per_level[level-1];
    if (data->bound) {
      for (i = 0; i < last; i++) {
	idof = data->sort_dof[i];
	if (data->bound[idof] <= 0)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    h[idof][j] += g[idof][j];
      }
    } else {
      for (i = first; i < last; i++) {
	idof = data->sort_dof[i];
	for (j=0; j<DIM_OF_WORLD; j++)
	  h[idof][j] += g[idof][j];
      }
    }

  }
  
  /* basis transformation (all refined levels) */

  for (level = 1; level <= level1; level++)
  {
    last = data->dofs_per_level[level];

    for (i = data->dofs_per_level[level-1]; i < last; i++)
    {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	for (j=0; j<DIM_OF_WORLD; j++)
	  h[idof][j] += 0.5 * (h[data->dof_parent[idof][0]][j]
			       + h[data->dof_parent[idof][1]][j]);
      }
    }
  }

  if (data->high_degree) {
    last = data->dofs_per_level[data->mg_levels - 1];
    for (i = data->dofs_per_level[data->mg_levels - 2]; i < last; i++)  {
      idof = data->sort_dof[i];
      if (!(data->bound && data->bound[idof] >= DIRICHLET)) {
	jdof = data->local_dof[i];
	for (k = 0; k < N_VERTICES(dim); k++)
	  for (j=0; j<DIM_OF_WORLD; j++)
	    h[idof][j]
	      += data->ipol[jdof][k] * h[data->dof_parent[idof][k]][j];
      }
    }
  }
  return;
}

/****************************************************************************/

const PRECON *get_BPX_precon_s(const FE_SPACE *fe_space,
			       const DOF_SCHAR_VEC *bound, 
			       int use_get_bound, int info)
{
    PRECON  *precon;
    HB_DATA *data;
    
    if (bound && (bound->fe_space != fe_space)) {
	ERROR("different fe spaces ?\n");
	return(nil);
    }

    precon = MEM_CALLOC(1, PRECON);
    data   = MEM_CALLOC(1, HB_DATA);
    
    data->fe_space = fe_space;
    if (!(data->bound_dv = bound))
      data->use_get_bound = use_get_bound;

    data->info     = info;

    precon->precon_data = data;
    precon->init_precon = init_BPX_precon;
    precon->precon      = BPX_precon_s;
    precon->exit_precon = exit_BPX_precon;

    return(precon);
}

const PRECON *get_BPX_precon_d(const FE_SPACE *fe_space,
			       const DOF_SCHAR_VEC *bound,
			       int use_get_bound, int info)
{
    PRECON  *precon;
    HB_DATA *data;
    
    if (bound && (bound->fe_space != fe_space)) {
	ERROR("different fe spaces ?\n");
	return(nil);
    }

    precon = MEM_CALLOC(1, PRECON);
    data   = MEM_CALLOC(1, HB_DATA);
    
    data->fe_space = fe_space;
    if (!(data->bound_dv = bound))
      data->use_get_bound = use_get_bound;

    data->info     = info;

    precon->precon_data = data;
    precon->init_precon = init_BPX_precon;
    precon->precon      = BPX_precon_d;
    precon->exit_precon = exit_BPX_precon;

    return(precon);
}

/****************************************************************************/
