/*
 * (C) 2012 by Pablo Neira Ayuso <pablo@netfilter.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published
 * by the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This code has been sponsored by Sophos Astaro <http://www.sophos.com>
 */

#include "internal.h"

#include <stdio.h>
#include <stdint.h>
#include <string.h>	/* for memcpy */
#include <arpa/inet.h>
#include <errno.h>
#include <libmnl/libmnl.h>

#include <linux/netfilter/nf_tables.h>
#include <linux/netfilter/nf_tables_compat.h>

#include <libnftnl/expr.h>
#include <libnftnl/rule.h>

/* From include/linux/netfilter/x_tables.h */
#define XT_EXTENSION_MAXNAMELEN 29

struct nftnl_expr_match {
	char		name[XT_EXTENSION_MAXNAMELEN];
	uint32_t	rev;
	uint32_t	data_len;
	const void	*data;
};

static int
nftnl_expr_match_set(struct nftnl_expr *e, uint16_t type,
			 const void *data, uint32_t data_len)
{
	struct nftnl_expr_match *mt = nftnl_expr_data(e);

	switch(type) {
	case NFTNL_EXPR_MT_NAME:
		snprintf(mt->name, sizeof(mt->name), "%.*s", data_len,
			 (const char *)data);
		break;
	case NFTNL_EXPR_MT_REV:
		mt->rev = *((uint32_t *)data);
		break;
	case NFTNL_EXPR_MT_INFO:
		if (mt->data)
			xfree(mt->data);

		mt->data = data;
		mt->data_len = data_len;
		break;
	default:
		return -1;
	}
	return 0;
}

static const void *
nftnl_expr_match_get(const struct nftnl_expr *e, uint16_t type,
			uint32_t *data_len)
{
	struct nftnl_expr_match *mt = nftnl_expr_data(e);

	switch(type) {
	case NFTNL_EXPR_MT_NAME:
		*data_len = sizeof(mt->name);
		return mt->name;
	case NFTNL_EXPR_MT_REV:
		*data_len = sizeof(mt->rev);
		return &mt->rev;
	case NFTNL_EXPR_MT_INFO:
		*data_len = mt->data_len;
		return mt->data;
	}
	return NULL;
}

static int nftnl_expr_match_cb(const struct nlattr *attr, void *data)
{
	const struct nlattr **tb = data;
	int type = mnl_attr_get_type(attr);

	if (mnl_attr_type_valid(attr, NFTA_MATCH_MAX) < 0)
		return MNL_CB_OK;

	switch(type) {
	case NFTA_MATCH_NAME:
		if (mnl_attr_validate(attr, MNL_TYPE_NUL_STRING) < 0)
			abi_breakage();
		break;
	case NFTA_MATCH_REV:
		if (mnl_attr_validate(attr, MNL_TYPE_U32) < 0)
			abi_breakage();
		break;
	case NFTA_MATCH_INFO:
		if (mnl_attr_validate(attr, MNL_TYPE_BINARY) < 0)
			abi_breakage();
		break;
	}

	tb[type] = attr;
	return MNL_CB_OK;
}

static void
nftnl_expr_match_build(struct nlmsghdr *nlh, struct nftnl_expr *e)
{
	struct nftnl_expr_match *mt = nftnl_expr_data(e);

	if (e->flags & (1 << NFTNL_EXPR_MT_NAME))
		mnl_attr_put_strz(nlh, NFTA_MATCH_NAME, mt->name);
	if (e->flags & (1 << NFTNL_EXPR_MT_REV))
		mnl_attr_put_u32(nlh, NFTA_MATCH_REV, htonl(mt->rev));
	if (e->flags & (1 << NFTNL_EXPR_MT_INFO))
		mnl_attr_put(nlh, NFTA_MATCH_INFO, mt->data_len, mt->data);
}

static int nftnl_expr_match_parse(struct nftnl_expr *e, struct nlattr *attr)
{
	struct nftnl_expr_match *match = nftnl_expr_data(e);
	struct nlattr *tb[NFTA_MATCH_MAX+1] = {};

	if (mnl_attr_parse_nested(attr, nftnl_expr_match_cb, tb) < 0)
		return -1;

	if (tb[NFTA_MATCH_NAME]) {
		snprintf(match->name, XT_EXTENSION_MAXNAMELEN, "%s",
			 mnl_attr_get_str(tb[NFTA_MATCH_NAME]));

		match->name[XT_EXTENSION_MAXNAMELEN-1] = '\0';
		e->flags |= (1 << NFTNL_EXPR_MT_NAME);
	}

	if (tb[NFTA_MATCH_REV]) {
		match->rev = ntohl(mnl_attr_get_u32(tb[NFTA_MATCH_REV]));
		e->flags |= (1 << NFTNL_EXPR_MT_REV);
	}

	if (tb[NFTA_MATCH_INFO]) {
		uint32_t len = mnl_attr_get_payload_len(tb[NFTA_MATCH_INFO]);
		void *match_data;

		if (match->data)
			xfree(match->data);

		match_data = calloc(1, len);
		if (match_data == NULL)
			return -1;

		memcpy(match_data, mnl_attr_get_payload(tb[NFTA_MATCH_INFO]), len);

		match->data = match_data;
		match->data_len = len;

		e->flags |= (1 << NFTNL_EXPR_MT_INFO);
	}

	return 0;
}

static int nftnl_expr_match_json_parse(struct nftnl_expr *e, json_t *root,
					  struct nftnl_parse_err *err)
{
#ifdef JSON_PARSING
	const char *name;

	name = nftnl_jansson_parse_str(root, "name", err);
	if (name != NULL)
		nftnl_expr_set_str(e, NFTNL_EXPR_MT_NAME, name);

	return 0;
#else
	errno = EOPNOTSUPP;
	return -1;
#endif
}


static int nftnl_expr_match_xml_parse(struct nftnl_expr *e, mxml_node_t *tree,
					 struct nftnl_parse_err *err)
{
#ifdef XML_PARSING
	const char *name;

	name = nftnl_mxml_str_parse(tree, "name", MXML_DESCEND_FIRST,
				  NFTNL_XML_MAND, err);
	if (name != NULL)
		nftnl_expr_set_str(e, NFTNL_EXPR_MT_NAME, name);

	/* mt->info is ignored until other solution is reached */

	return 0;
#else
	errno = EOPNOTSUPP;
	return -1;
#endif
}

static int nftnl_expr_match_export(char *buf, size_t size,
				      struct nftnl_expr *e, int type)
{
	struct nftnl_expr_match *mt = nftnl_expr_data(e);
	NFTNL_BUF_INIT(b, buf, size);

	if (e->flags & (1 << NFTNL_EXPR_MT_NAME))
		nftnl_buf_str(&b, type, mt->name, NAME);

	return nftnl_buf_done(&b);
}

static int
nftnl_expr_match_snprintf(char *buf, size_t len, uint32_t type,
			     uint32_t flags, struct nftnl_expr *e)
{
	struct nftnl_expr_match *match = nftnl_expr_data(e);

	switch (type) {
	case NFTNL_OUTPUT_DEFAULT:
		return snprintf(buf, len, "name %s rev %u ",
				match->name, match->rev);
	case NFTNL_OUTPUT_XML:
	case NFTNL_OUTPUT_JSON:
		return nftnl_expr_match_export(buf, len, e, type);
	default:
		break;
	}
	return -1;
}

static void nftnl_expr_match_free(struct nftnl_expr *e)
{
	struct nftnl_expr_match *match = nftnl_expr_data(e);

	xfree(match->data);
}

struct expr_ops expr_ops_match = {
	.name		= "match",
	.alloc_len	= sizeof(struct nftnl_expr_match),
	.max_attr	= NFTA_MATCH_MAX,
	.free		= nftnl_expr_match_free,
	.set		= nftnl_expr_match_set,
	.get		= nftnl_expr_match_get,
	.parse		= nftnl_expr_match_parse,
	.build		= nftnl_expr_match_build,
	.snprintf	= nftnl_expr_match_snprintf,
	.xml_parse 	= nftnl_expr_match_xml_parse,
	.json_parse 	= nftnl_expr_match_json_parse,
};
