<!--

@license Apache-2.0

Copyright (c) 2019 The Stdlib Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

   http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.

-->

# iterPulse

> Create an iterator which generates a pulse waveform.

<!-- Section to include introductory text. Make sure to keep an empty line after the intro `section` element and another before the `/section` close. -->

<section class="intro">

A pulse waveform is represented by the following equation

<!-- <equation class="equation" label="eq:pulse_waveform" align="center" raw="f(t; T, \tau, a_{\textrm{min}}, a_{\textrm{max}}, \varphi) = \begin{cases} a_{\textrm{max}} & (t-\varphi) \mod T < \tau \\ a_{\textrm{min}} & \textrm{otherwise} \end{cases}" alt="Equation for a pulse wave."> -->

<div class="equation" align="center" data-raw-text="f(t; T, \tau, a_{\textrm{min}}, a_{\textrm{max}}, \varphi) = \begin{cases} a_{\textrm{max}} &amp; (t-\varphi) \mod T < \tau \\ a_{\textrm{min}} &amp; \textrm{otherwise} \end{cases}" data-equation="eq:pulse_waveform">
    <img src="https://cdn.jsdelivr.net/gh/stdlib-js/stdlib@36d0ed92c4707f9bb44e4a6fde78f19f6f28762e/lib/node_modules/@stdlib/simulate/iter/pulse/docs/img/equation_pulse_waveform.svg" alt="Equation for a pulse wave.">
    <br>
</div>

<!-- </equation> -->

where `T` is the pulse period, `τ` is the pulse duration, `a` is the amplitude, and `φ` is the phase offset.

<!-- TODO: add a figure showing a pulse train -->

</section>

<!-- /.intro -->

<!-- Package usage documentation. -->

<section class="usage">

## Usage

```javascript
var iterPulse = require( '@stdlib/simulate/iter/pulse' );
```

#### iterPulse( \[options] )

Returns an iterator which generates a pulse waveform.

```javascript
var it = iterPulse();
// returns <Object>

var v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

// ...
```

The returned iterator protocol-compliant object has the following properties:

-   **next**: function which returns an iterator protocol-compliant object containing the next iterated value (if one exists) assigned to a `value` property and a `done` property having a `boolean` value indicating whether the iterator is finished.
-   **return**: function which closes an iterator and returns a single (optional) argument in an iterator protocol-compliant object.

The function supports the following `options`:

-   **period**: pulse period (i.e., the number of iterations before a waveform repeats). Default: `10`.
-   **duration**: pulse duration (i.e., the number of consecutive iterations of maximum amplitude during one period). Default: `floor(options.period/2)`.
-   **min**: minimum amplitude. Default: `0.0`.
-   **max**: maximum amplitude. Default: `1.0`.
-   **offset**: phase offset (in units of iterations; zero-based). A negative offset translates a waveform to the left. A positive offset translates a waveform to the right. Default: `0`.
-   **iter**: number of iterations. Default: `1e308`.

By default, the function returns an iterator which generates a waveform that repeats every `10` iterations. To specify an alternative period, set the `period` option.

```javascript
var opts = {
    'period': 4
};

var it = iterPulse( opts );
// returns <Object>

var v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 1.0

// ...
```

By default, the function returns an iterator which generates a waveform that has a duty cycle of `50%` (i.e., the waveform spends `50%` of its time at maximum amplitude). To specify an alternative duty cycle, set the `duration` option. For example, to generate a pulse wave having a period of `4` iterations and a `25%` duty cycle,

```javascript
var opts = {
    'period': 4,
    'duration': 1 // 1/4 = 0.25 => 25%
};

var it = iterPulse( opts );
// returns <Object>

var v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 1.0

// ...
```

To adjust at what point the iterator begins in the waveform cycle, set the phase `offset` option. For example, to translate the waveform to the left,

```javascript
var opts = {
    'period': 6,
    'offset': -2
};

var it = iterPulse( opts );
// returns <Object>

var v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 1.0

// ...
```

To translate the waveform to the right,

```javascript
var opts = {
    'period': 6,
    'offset': 2
};

var it = iterPulse( opts );
// returns <Object>

var v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 0.0

v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

// ...
```

By default, the function returns an infinite iterator (i.e., an iterator which never ends). To limit the number of iterations, set the `iter` option.

```javascript
var opts = {
    'iter': 2
};
var it = iterPulse( opts );
// returns <Object>

var v = it.next().value;
// returns 1.0

v = it.next().value;
// returns 1.0

var bool = it.next().done;
// returns true
```

</section>

<!-- /.usage -->

<!-- Package usage notes. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="notes">

## Notes

-   If an environment supports `Symbol.iterator`, the returned iterator is iterable.

</section>

<!-- /.notes -->

<!-- Package usage examples. -->

<section class="examples">

## Examples

<!-- eslint no-undef: "error" -->

```javascript
var iterPulse = require( '@stdlib/simulate/iter/pulse' );

// Create an iterator:
var opts = {
    'period': 10,
    'duration': 2,
    'min': -10.0,
    'max': 10.0,
    'offset': -5,
    'iter': 100
};
var it = iterPulse( opts );

// Perform manual iteration...
var v;
while ( true ) {
    v = it.next();
    if ( v.done ) {
        break;
    }
    console.log( v.value );
}
```

</section>

<!-- /.examples -->

<!-- Section to include cited references. If references are included, add a horizontal rule *before* the section. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="references">

</section>

<!-- /.references -->

<!-- Section for all links. Make sure to keep an empty line after the `section` element and another before the `/section` close. -->

<section class="links">

</section>

<!-- /.links -->
