/* Copyright 2011 Canonical, Ltd. This software is licensed under the GNU
 * Lesser General Public License version 3 or later (see the file COPYING).
 */

#ifndef OIF_QML_ATTRIBUTES_H_
#define OIF_QML_ATTRIBUTES_H_

#include <QString>
#include <QVariant>
#include <geis/geis.h>

#include <stdexcept>
#include <string>
#include <typeinfo>

/* Template functions for getting object attributes from GEIS */

namespace geis_attributes {

class bad_object : public std::runtime_error {
 public:
  explicit bad_object(const std::string &arg) : std::runtime_error(arg) {}
};

class bad_attribute : public std::runtime_error {
 public:
  explicit bad_attribute(const std::string &arg) : std::runtime_error(arg) {}
};

class bad_pointer : public std::runtime_error {
 public:
  explicit bad_pointer(const std::string &arg) : std::runtime_error(arg) {}
};

template< typename Object >
static GeisAttr GetAttributeValue(Object object, const char* attributeName) {
  throw bad_object("Unhandled attribute value getter");
}

template< >
GeisAttr GetAttributeValue(GeisEvent event, const char* attributeName) {
  GeisAttr attribute = geis_event_attr_by_name(event, attributeName);
  if (!attribute) {
    QString msg = QString("Failed to get attribute value for '%1' from event")
      .arg(attributeName);
    throw bad_attribute(msg.toUtf8().constData());
  }

  return attribute;
}

template< >
GeisAttr GetAttributeValue(GeisFrame frame, const char* attributeName) {
  GeisAttr attribute = geis_frame_attr_by_name(frame, attributeName);
  if (!attribute) {
    QString msg = QString("Failed to get attribute value for '%1' from frame")
      .arg(attributeName);
    throw bad_attribute(msg.toUtf8().constData());
  }

  return attribute;
}

template< >
GeisAttr GetAttributeValue(GeisTouch touch, const char* attributeName) {
  GeisAttr attribute = geis_touch_attr_by_name(touch, attributeName);
  if (!attribute) {
    QString msg = QString("Failed to get attribute value for '%1' from touch")
      .arg(attributeName);
    throw bad_attribute(msg.toUtf8().constData());
  }

  return attribute;
}

template< typename Ret >
static Ret GetAttributeReturn(GeisAttr attribute) {
  Ret object = reinterpret_cast< Ret >(geis_attr_value_to_pointer(attribute));
  if (!object) {
    QString msg = QString("Failed to get valid %1 pointer from  attribute '%2'")
      .arg(typeid(Ret).name()).arg(geis_attr_name(attribute));
    throw bad_pointer(msg.toUtf8().constData());
  }

  return object;
}

template< >
GeisBoolean GetAttributeReturn(GeisAttr attribute) {
  return geis_attr_value_to_boolean(attribute);
}

template< >
GeisInteger GetAttributeReturn(GeisAttr attribute) {
  return geis_attr_value_to_integer(attribute);
}

template< >
GeisFloat GetAttributeReturn(GeisAttr attribute) {
  return geis_attr_value_to_float(attribute);
}

template< >
GeisString GetAttributeReturn(GeisAttr attribute) {
  return geis_attr_value_to_string(attribute);
}

template< typename Ret, typename Object >
Ret GetAttribute(Object object, const char* attributeName) {
  GeisAttr attribute = GetAttributeValue< Object >(object, attributeName);
  return GetAttributeReturn< Ret >(attribute);
}

namespace {

void AppendAttributeToMap(GeisAttr attr, QVariantMap* attributes) {
  GeisString name = geis_attr_name(attr);
  if (!name) {
    qCritical("Warning: Failed to get name of device attribute");
    return;
  }

  switch (geis_attr_type(attr)) {
    case GEIS_ATTR_TYPE_BOOLEAN:
      attributes->insert(
        QString(name),
        QVariant(static_cast<bool>(geis_attr_value_to_boolean(attr))));
      break;

    case GEIS_ATTR_TYPE_FLOAT:
      attributes->insert(
        QString(name),
        QVariant(static_cast<float>(geis_attr_value_to_float(attr))));
      break;

    case GEIS_ATTR_TYPE_INTEGER:
      attributes->insert(
        QString(name),
        QVariant(static_cast<int>(geis_attr_value_to_integer(attr))));
      break;

    case GEIS_ATTR_TYPE_STRING:
      attributes->insert(QString(name),
                         QVariant(QString(geis_attr_value_to_string(attr))));
      break;

    default:   // Unknown or pointer attribute
      break;
  }
}

}  // namespace

template< typename Object >
static void PopulateAttributeMap(Object object, QVariantMap* attributes) {
    throw bad_object("Unhandled object for populating attribute map");
}

template< >
void PopulateAttributeMap(GeisDevice device, QVariantMap* attributes) {
  for (GeisSize i = 0; i < geis_device_attr_count(device); ++i) {
    GeisAttr attr = geis_device_attr(device, i);
    if (!attr) {
      qCritical("Warning: Failed to get device attribute");
      continue;
    }

    AppendAttributeToMap(attr, attributes);
  }
}

template< >
void PopulateAttributeMap(GeisFrame frame, QVariantMap* attributes) {
  for (GeisSize i = 0; i < geis_frame_attr_count(frame); ++i) {
    GeisAttr attr = geis_frame_attr(frame, i);
    if (!attr) {
      qCritical("Warning: Failed to get device attribute");
      continue;
    }

    AppendAttributeToMap(attr, attributes);
  }
}

template< >
void PopulateAttributeMap(GeisTouch touch, QVariantMap* attributes) {
  for (GeisSize i = 0; i < geis_touch_attr_count(touch); ++i) {
    GeisAttr attr = geis_touch_attr(touch, i);
    if (!attr) {
      qCritical("Warning: Failed to get device attribute");
      continue;
    }

    AppendAttributeToMap(attr, attributes);
  }
}

}  // namespace geis_attributes

#endif  // OIF_QML_ATTRIBUTES_H_
