/***********************************************************************************
* Smooth Tasks
* Copyright (C) 2009 Mathias Panzenböck <grosser.meister.morti@gmx.net>
* Copyright (C) 2009-2010 Toni Dietze <smooth-tasks@derflupp.e4ward.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/
#include "SmoothTasks/Global.h"

// Qt
#include <QDebug>
#include <QSizeF>
#include <QTextLayout>
#include <QApplication>
#include <QFontMetrics>

// STD C++
#include <plasma/framesvg.h>

namespace SmoothTasks {

const QString TASK_ITEM        = QString::fromLatin1("SmoothTasks::TaskItem");
const QString NORMAL           = QString::fromLatin1("normal");
const QString ACTIVE           = QString::fromLatin1("active");
const QString MINIMIZED        = QString::fromLatin1("minimized");
const QString FOCUS            = QString::fromLatin1("focus");
const QString ATTENTION        = QString::fromLatin1("attention");
const QString HOVER            = QString::fromLatin1("hover");
const QString M                = QString::fromLatin1("M");
const int     DRAG_HOVER_DELAY = 500;

QSizeF layoutText(QTextLayout &layout, const QSizeF &constraints) {
	QFontMetrics metrics(layout.font());
	const qreal maxWidth  = constraints.width();
	const qreal maxHeight = constraints.height();
	int   leading     = metrics.leading();
	qreal height      = 0;
	qreal widthUsed   = 0;
	int   lineSpacing = metrics.lineSpacing();

	layout.beginLayout();
	forever {
		QTextLine line(layout.createLine());

		if (!line.isValid()) {
			break;
		}

		height += leading;

		// Make the last line that will fit infinitely long.
		// drawTextLayout() will handle this by fading the line out
		// if it won't fit in the constraints.
		if (height + 2 * lineSpacing > maxHeight) {
			line.setLineWidth(std::numeric_limits<qreal>::infinity());
			line.setPosition(QPointF(0, height));
			height += line.height();
			widthUsed = qMax(widthUsed, line.naturalTextWidth());
			break;
		}

		line.setLineWidth(maxWidth);
		line.setPosition(QPointF(0, height));

		height += line.height();
		widthUsed = qMax(widthUsed, line.naturalTextWidth());
	}
	layout.endLayout();

	return QSizeF(widthUsed, height);
}

QSizeF preferredTextLayoutSize(const QString& text, const QFont& font, const qreal lineWidth) {
	QTextLayout layout(text, font);

	QTextOption textOption(layout.textOption());
	textOption.setTextDirection(QApplication::layoutDirection());
	layout.setTextOption(textOption);

	QFontMetrics metrics(layout.font());
	int leading = metrics.leading();
	qreal height = 0;
	qreal widthUsed = 0;
	layout.beginLayout();
	for(;;) {
		QTextLine line = layout.createLine();
		if (!line.isValid())
			break;
		
		line.setLineWidth(lineWidth);
		height += leading;
		line.setPosition(QPointF(0, height));
		height += line.height()+1;
		widthUsed = qMax(widthUsed, line.naturalTextWidth());
	}
	layout.endLayout();
	
	return QSizeF(widthUsed + 1.0, height + 1.0);  // add 1.0 against rounding problems (e.g. font “Linux Biolinum”)
}

qreal squeezeWidths(QVector<qreal>& widths, const qreal availableWidth) {
	const qreal epsilon = 0.1;
	
	if (availableWidth <= epsilon) {
		for (QVector<qreal>::iterator w = widths.begin(); w != widths.end(); ++w)
			*w = 0.0;
		return 0.0;
	}
	
	qreal widthSum = 0.0;
	foreach (qreal w, widths)
		widthSum += w;
	// calculate fitting rdWidths
	qreal maxRdWidth = 0;
	while (widthSum > availableWidth) {
		// find biggest Button
		qreal nextSmallerRdWidth = 0;
		int maxCount = 0;
		foreach (qreal w, widths) {
			if (w == maxRdWidth) {
				++maxCount;
			} else if (w > maxRdWidth) {
				nextSmallerRdWidth = maxRdWidth;
				maxRdWidth = w;
				maxCount = 1;
			} else if (w > nextSmallerRdWidth) {
				nextSmallerRdWidth = w;
			}
		}
		
		// make biggest buttons shorter
		qreal newMaxRdWidth = maxRdWidth - (widthSum - (availableWidth - epsilon/*against numeric problems*/)) / (qreal) maxCount;
		if (newMaxRdWidth < nextSmallerRdWidth)
			newMaxRdWidth = nextSmallerRdWidth;
		
		widthSum = 0.0;
		for (QVector<qreal>::iterator w = widths.begin(); w != widths.end(); ++w) {
			if (*w == maxRdWidth)
				*w = newMaxRdWidth;
			widthSum += *w;
		}
		
		maxRdWidth = newMaxRdWidth;
	}
	return widthSum;
}

} // namespace SmoothTasks
