/*
 * Copyright (C) 2000-2004 by Oswald Buddenhagen <puf@ossi.cjb.net>
 * based on puf 0.1.x (C) 1999,2000 by Anders Gavare <gavare@hotmail.com>
 *
 * You may modify and distribute this code under the terms of the GPL.
 * There is NO WARRANTY of any kind. See COPYING for details.
 *
 * http_rst.c - receive and process http response message
 *
 */

#include "puf.h"


int economize_files;

/*  create all directories in the path  */
static void 
create_dir(char *buf)
{
    int p;

    for (p = 0; buf[p]; p++)
	if (p && buf[p] == '/') {
	    buf[p] = '\0';
	    mkdir(buf, 0777);
	    buf[p] = '/';
	}
}


/*  open a file.
    create the directory it should live in, if it's not there.
    try to free up handles, if necessary.  */
int 
mmfopen(char *name, int flags, int *f)
{
    int fi, try_free, try_mkdir;
    static int cf = -1;

    if (cf != -1) {
	close(cf);
	cf = -1;
    }
	
    try_free = try_mkdir = 0;
  retry:
    if ((fi = open(name, flags, 0666)) < 0) {
	if (errno == ENFILE || errno == EMFILE) {
	    if (!try_free && !economize_files && free_fd(1)) {
		try_free++;
		goto retry;
	    }
	    return -2;
	}
	if (errno == ENOENT && !try_mkdir) {
	    create_dir(name);
	    try_mkdir++;
	    goto retry;
	}
	return -1;
    }
    
    if (economize_files) {
	*f = -1;
	return cf = fi;
    } else
	return *f = fi;
}


/*  open file and store the handle in aurl_t structure  */
static int 
mfopen(aurl_t *au, int flags)
{
    return mmfopen(au->disposition, flags, &(au->f));
}

/*  try to "steal" a handle from an open target file  */
/*  passing != 0 allows stealing all handles, while 0 will preserve one.
    only operations able to give up handles are allowed to pass != 0.  */
int 
free_fd(int second)
{
    ls_iterate(list_urls_reply, aurl_t, au, {
        if (au->f != -1) {
	    if (second) {
        	close(au->f);
        	au->f = -1;
        	return 1;
	    } else
		second = 1;
        }
    });
    return 0;
}


/*  save data to overlap buffer. note, that we possibly are
    saving contents of the previous buffer!  */
static int 
save_buff(aurl_t *au, const char *buf, int len)
{
    char *bp;
    int siz;

    if ((siz = len < OVERLAPLEN ? OVERLAPLEN : len) > au->size) {
	au->size = siz;
	if (!(bp = mmalloc(siz))) {
	    if (au->buffer) {
		free(au->buffer);
		au->buffer = 0;
		au->size = au->offset = 0;
	    }
	    return 0;
	}
	au->offset = len;
	memcpy(bp, buf, len);
	if (au->buffer)
	    free(au->buffer);
	au->buffer = bp;
    } else {
	au->offset = len;
	memcpy(au->buffer, buf, len);
    }
    return 1;
}


static int
open_disp(aurl_t *au, int *fi)
{
    if (!au->disposition[0]) {
	*fi = 1;
	return RT_OK;
    }
    if (au->url->parm->disposition) {
	if (au->url->parm->disposition->multi)
	    au->disposition[au->displen] = 0;
	if (au->url->parm->disposition->created) {
	    if ((*fi = mfopen(au, O_WRONLY | O_APPEND | _O_BINARY)) < 0)
		return errm(au->url, "!$u: cannot open %s for appending: %s", 
				     au->disposition, strerror(errno));
	    return RT_OK;
	}
	au->url->parm->disposition->created = 1;
    }
    if ((*fi = mfopen(au, O_WRONLY | O_CREAT | O_TRUNC | _O_BINARY)) < 0)
	return errm(au->url, "!$u: cannot create %s: %s", 
			     au->disposition, strerror(errno));
    return RT_OK;
}

static int
open_pdisp(url_parm_t *parm, int *fi, int *fi2)
{
    char *disp;
    char buf[SHORTSTR];

    if (parm->disposition->devnull)
	return RT_GIVEUP;
    if (!parm->disposition->disp[0]) {
	*fi = 1;
	*fi2 = -1;
	return RT_OK;
    }
    if (parm->opt->disp_path->path[0]) {
	/* lenght checked by adden() */
	int dl = 0;
	cat_str(buf, dl, parm->opt->disp_path->path);
	cat_chr(buf, dl, '/');
	cat_str(buf, dl, parm->disposition->disp);
	disp = buf;
    } else
	disp = parm->disposition->disp;
    if (parm->disposition->created) {
	if ((*fi = mmfopen(disp, O_WRONLY | O_APPEND | _O_BINARY, fi2)) < 0) {
	    prx(ERR, "cannot open %s for appending: %s", disp, strerror(errno));
	    return RT_GIVEUP;
	}
	return RT_OK;
    }
    parm->disposition->created = 1;
    if ((*fi = mmfopen(disp, O_WRONLY | O_CREAT | O_TRUNC | _O_BINARY, fi2)) < 0) {
	prx(ERR, "cannot create %s: %s", disp, strerror(errno));
	return RT_GIVEUP;
    }
    return RT_OK;
}

static void
safe_write(int fd, const void *buf, int len)
{
    if (write(fd, buf, len) != len)
	die(1, "Write error! Disk full?\n");
}

static void
qsafe_write(int fd, const void *buf, int len)
{
    static off_t written_bytes;

    if (max_bytes && (written_bytes + len > max_bytes))
	byebye("byte quota exceeded!");
    written_bytes += len;
    safe_write(fd, buf, len);
}

static void
write_loc(int fi, url_t *u, int auth)
{
    unsigned dl;
    char buf[SHORTSTR];

    dl = 0;
    cat_str(buf, dl, "Location: ");
    dl += print_url(buf + dl, sizeof(buf) - dl, u, auth);
    lcat_chr(buf, sizeof(buf), dl, '\n');
    if (dl < sizeof(buf))
	safe_write(fi, buf, dl);
}

int
werrm(aurl_t *au, int sts, const char *msg, ...)
{
    int fi, ret, ret2;
    va_list va;
    char buf[64];

    va_start(va, msg);
    ret = verrm(au->url, msg, va);
    va_end(va);
    if (ret != RT_RETRY &&
	au->url->save_content &&
	(au->url->parm->opt->ext_dump > 1 ||
	(au->url->parm->opt->ext_dump == 1 && !au->url->referer)))
    {
	if ((ret2 = open_disp(au, &fi)) != RT_OK)
	    return ret2;
	write_loc(fi, au->url, 0);
	safe_write(fi, buf, sprintf(buf, "Status: %d\n\n", sts));
    }
    va_end(va);
    return ret;
}

void
write_usts(url_t *u, int sts)
{
    int fi, fi2;
    char buf[64];

    if ((u->parm->opt->ext_dump > 1 ||
	 (u->parm->opt->ext_dump == 1 && !u->referer)) &&
	open_pdisp(u->parm, &fi, &fi2) == RT_OK)
    {
	write_loc(fi, u, 0);
	safe_write(fi, buf, sprintf(buf, "Status: %d\n\n", sts));
      if (fi2 != -1)
	close(fi2);
    }
}

int
uwerrm(url_t *u, int sts, const char *msg, ...)
{
    int ret;
    va_list va;

    va_start(va, msg);
    ret = verrm(u, msg, va);
    va_end(va);
    if (ret != RT_RETRY)
	write_usts(u, sts);
    return ret;
}

void
write_psts(url_parm_t *parm, const char *u, int ul, int istopdir, int sts)
{
    int fi, fi2, len;
    char buf[SHORTSTR];

    if ((parm->opt->ext_dump > 1 ||
	 (parm->opt->ext_dump == 1 && istopdir)) &&
	open_pdisp(parm, &fi, &fi2) == RT_OK)
    {
	len = snprintf(buf, sizeof(buf), "Location: %.*s\nStatus: %d\n\n",
		       ul, u, sts);
	if ((unsigned)len < sizeof(buf))	
	    safe_write(fi, buf, len);
      if (fi2 != -1)
	close(fi2);
    }
}

/*  handle http reply message  */
int 
handle_reply(aurl_t *au)
{
    buffe_t *buffe;
    char *bufp, *nbuf;
    ptrarr_t *sh;
    int fi, a, e, l, o, p, len, alen, orglen, nsiz, multi, exto, ret;
    int pad, flen, wlen, pwlen;
    off_t buff_size;
    unsigned u;
    char databuf[OVERLAPLEN + MAXBUFSIZE], buf[SHORTSTR];

    multi = au->url->parm->disposition && au->url->parm->disposition->multi;
    exto = au->url->parm->opt->ext_dump;
    if (!(buff_size = au->url->parm->opt->buff_size) && (multi || exto))
	buff_size = DEFAULT_MAX_BUFFER * 1024 * 1024;

    /*  Receive some data:  */
    bufp = databuf + OVERLAPLEN;
    if ((orglen = read(au->socket, bufp, MAXBUFSIZE)) < 0)
	return werrm(au, 553, "data read for $u failed");

    len = orglen + au->offset;
    if (orglen) {
	/*  Copy overlap buffer from last read  */
	if (au->offset) {
	    if (au->offset > OVERLAPLEN) {
		if (len > MAXHEADERLEN || !(nbuf = mrealloc(au->buffer, len))) {
		    return werrm(au, 502, "reply header for $u has insane length");
		} else {
		    memcpy(nbuf + au->offset, bufp, orglen);
		    au->buffer = bufp = nbuf;
		    au->size = len;
		}
	    } else {
		bufp -= au->offset;
		memcpy(bufp, au->buffer, au->offset);
	    }
	}
    } else {
	/* note: close with size_fetched < size_total is accepted.
	   this is basically incorrect, but common practice. */
	/* handle remainig data in overlap buffer  */
	bufp = au->buffer;
    }

    /*  first the http message header  */
    if (!au->http_done_header) {
	for (p = 0;;) {
	    for (a = p;;) {
		/*  reached end-of-buffer before header end?  */
		if (p >= len) {
		    if (!orglen)
			return werrm(au, 502, "broken reply header for $u");
		    return save_buff(au, bufp + a, len - a) ? RT_OK : RT_RETRY;
		}
		if (bufp[p++] == '\n')
		    break;
	    }
	    for (e = p - 1; e > a && bufp[e - 1] <= ' '; e--);
	    l = e - a;
	    bufp[e] = '\0';
	    dbg(HDR, ("read header: %s\n", bufp + a));
	    if (!au->http_result_code) {
		/* empty lines before reply header are incorrect, 
		   but should be handled for robustness */
		if (l) {
		    /*  get result code  */
		    if (strncasecmp(bufp + a, "http/", 5))
			return werrm(au, 502, "broken reply header for $u");
		    au->http_result_code = atoi(bufp + a + 9);
		    if (bufp[a + 9] != '1')
		    switch (au->http_result_code) {
			case 200:	/*  ok  */
			    au->file_off = 0;
			case 206:	/*  partial content  */
			case 304:	/*  not modified  */
			    break;
			case 300:	/*  multiple choices  */
			case 301:	/*  moved permanently  */
			case 302:	/*  moved temporarily  */
			case 307:	/*  temporary redirect (new 302)  */
			    au->reloc = 1;
			    break;
			case 400:	/*  bad request  */
			case 505:	/*  http version not supported  */
			    if (!au->url->host->info->is_http11)
				return werrm(au, au->http_result_code,
				    "!sever failed to parse request for $u");
			    prx(NFO,
				"falling back to HTTP/1.0 for host '%s'\n",
				au->url->host->name);
			    au->url->host->info->is_http11 = 0;
			    return RT_AGAIN;
			case 401:
			  if (au->auth_chall)
			    return werrm(au, au->http_result_code,
					"!bad authorization for $u (%s)",
					 au->auth_chall);
			    break;
			/*  the following two are theoretically fatal errors,
			    but on some servers they indicate temporary
			    failure ... strange ...  */
			case 403:	/*  access denied  */
			  {
			    static const char msg[] = "!access to $u denied";
			    return werrm(au, au->http_result_code,
					au->url->parm->opt->http_err_trans ?
						msg + 1 : msg);
			  }
			case 404:	/*  not found  */
			  {
			    static const char msg[] = "!$u not found";
			    return werrm(au, au->http_result_code,
					au->url->parm->opt->http_err_trans ?
						msg + 1 : msg);
			  }
			case 406:
			    prxu(NFO, au->url, "cancelling $u (rejected)\n");
			    return RT_SKIP;
			case 407:	/*  proxy auth required  */
			    au->proxy->host = 0;	/* mark dead  */
			    return RT_AGAIN;
			case 503:	/*  service unavailable - connection refused, etc.  */
			    return RT_REFUSED;
			case 504:	/*  gateway timeout - server not responding  */
			    return RT_TIMEOUT;
			default:
			    return werrm(au, 502,
					"unrecognised HTTP status '%s' for $u",
					bufp + a);
		    }
		}
	    } else {		/*  have_result  */
		/*  end of headers?  */
		if (!l) {
		    if (au->http_result_code >= 200)
			break;
		    else {
			au->http_result_code = 0;
			dbg(HDR, ("awaiting next header after 1xx response.\n"));
			continue;
		    }
		}
		/*  continued header?  */
		if (bufp[a] <= ' ')
		    continue;
		/*  save requested  */
		sh = au->url->parm->opt->save_headers;
		for (u = 0; u < sh->nents; u++)
		    if (!strncasecmp(bufp + a, ((char **)sh->ents)[u],
				    strlen(((char **)sh->ents)[u]))) {
			if (exto &&
			    (!strncasecmp(bufp + a, "Content-Length:", 15) ||
			     !strncasecmp(bufp + a, "Location:", 9)))
			    continue;
			if (au->hdrssiz < au->hdrslen + e - a + 1) {
			    nsiz = au->hdrslen * 2 + e - a + 1;
			    if (!(nbuf = mrealloc(au->headers, nsiz)))
				break;
			    au->headers = nbuf;
			    au->hdrssiz = nsiz;
			}
			memcpy(au->headers + au->hdrslen, bufp + a, e - a);
			au->hdrslen += e - a;
			au->headers[au->hdrslen++] = '\n';
			break;
		    }
		/*  split header name and content  */
		for (o = a; o < e && bufp[o] > ' '; o++);
		bufp[o++] = '\0';
		for (; o < e && bufp[o] <= ' '; o++);
		/*  handle header  */
		if (au->reloc) {
		    if (!strcasecmp(bufp + a, "Location:")) {
			prxu(NFO, au->url, "relocation from $u to %s\n",
			    bufp + o);
			if (au->url->save_content &&
			    (exto > 1 || (exto == 1 && !au->url->referer)))
			{
			    au->url->parm->disposition->multi = 1;
			    if ((ret = open_disp(au, &fi)) != RT_OK)
				return ret;
			    write_loc(fi, au->url, 0);
			    safe_write(fi, buf,
				       snprintf(buf, sizeof(buf),
				    	        "Status: %d\n"
						"New-Location: %.*s\n\n",
						au->http_result_code,
						e - o, bufp + o));
			}
			if (au->url->relocs < 5) {
			    parse_add_url("redirect", bufp + o, e - o, au->url,
					  au->url->referer, au->url->parm,
					  au->url->is_requisite,
					  au->url->relocs + 1, 
					  au->url->link_depth,
					  0);
			} else
			    prx(ERR,
				"%s exceeds maximal redirection count!\n",
				bufp + o);
			return RT_SKIP;
		    }
		    /*  needn't check other headers when redirect encountered  */
		} else if (au->http_result_code == 401) {
		    if (!strcasecmp(bufp + a, "WWW-Authenticate:")) {
			int i, j;
			for (i = 0; o + i < e; i++)
			    if (isspace((int)bufp[o + i]))
				for (j = i + 1; o + j < e; j++)
				    if (!isspace((int)bufp[o + j]))
					goto gotau;
			return werrm(au, 502,
				    "unrecognized WWW-Authenticate for $u");
		      gotau:
			if (i == 5 && !strncasecmp(bufp + o, "Basic", 5)) {
			    if (!(au->auth_chall = mrealloc(au->auth_chall, e - j - o + 1)))
				return RT_RETRY;
			    memcpy(au->auth_chall, bufp + o + j, e - j - o);
			    au->auth_chall[e - j - o] = 0;
			}
		    }
		    /*  needn't check other headers when auth missing  */
		} else if (!strcasecmp(bufp + a, "Last-Modified:")) {
		    if (!multi &&
			(au->file_time = parseHTTPdate(bufp + o)) == BAD_DATE)
			prxu(WRN, au->url, "$u: unrecognised date format '%s'", bufp + o);
		} else if (!strcasecmp(bufp + a, "Content-Length:"))
		    sscanf(bufp + o, SOFFT, &(au->size_total));
		    if ((multi || exto) && au->size_total > buff_size)
			return errm(au->url, "!$u too big. Try a bigger -xs.");
		else if (!strcasecmp(bufp + a, "Content-Type:")) {
		    if (!strncasecmp(bufp + o, "text/", 5)) {
			au->content_is_text = 1;
		    	if (!strncasecmp(bufp + o + 5, "html", 4) &&
			    !isalpha((int)bufp[o + 9]))
			    au->content_is_html = 1;
		    } else if (multi && !exto) {
			prxu(WRN, au->url, "skipping non-text $u\n");
			return RT_SKIP;
		    }
		    if (au->url->save_content == 2) {
			for (l = 0; bufp[o + l] == '/' || isalpha((int)bufp[o + l]); l++);
			au->url->save_content = test_pat(au->url->local_part,
							 strlen(au->url->local_part),
							 au->url->path_len,
							 bufp + o, l,
							 au->url->parm);
			if (!au->url->save_content && !needs_recurse_au(au, 1)) {
			    prxu(NFO, au->url, "cancelling $u (rejected)\n");
			    return RT_SKIP;
			}
		    }
		} else if (!strcasecmp(bufp + a, "Content-Range:")) {
		    /* The Content-Range string should look somewhat like
		       this: "bytes 250260-664041471/664041472" */
		    off_t rs, re, rt;

		    if(sscanf(bufp + o, "bytes "SOFFT"-"SOFFT"/"SOFFT, 
			      &rs, &re, &rt) != 3) {
			return werrm(au, 502,
				    "unrecognized Content-Range for $u");
		    }
		}
	    }			/*  have_result  */
	}			/*  main header loop  */

	if (au->reloc)	/*  no relocation url found  */
	    return werrm(au, 502, "missing new location while redirecting $u");

	if (au->http_result_code == 304) {	/*  Not Modified  */
	    if (!multi && au->disposition[0] && needs_recurse_au(au, 0))
		recurse_file(au->url, au->disposition);
	    return RT_DONE;	/*  would HR_SKIP be more appropriate?  */
	}

	if (au->http_result_code == 401) {
	    if (!au->auth_chall)
		return werrm(au, 401,
			     "!no recognized authentication scheme for $u");
	    if (!au->url->parm->http_auth)
		return werrm(au, au->http_result_code,
			     "!need authorization for $u");
	    return RT_RESTART;
	}

	if (au->url->save_content == 2) {
	    au->url->save_content = test_pat(au->url->local_part,
					     strlen(au->url->local_part),
					     au->url->path_len,
					     "application/octet-stream", 24,
					     au->url->parm);
	    if (!au->url->save_content) {
		prxu(NFO, au->url, "cancelling $u (rejected)\n");
		return RT_SKIP;
	    }
	}

	if (au->size_total) {
	    if (au->url->parm->opt->max_bytes && 
		au->size_total > au->url->parm->opt->max_bytes)
		au->size_total = au->url->parm->opt->max_bytes;
	    total_bytes += au->size_total;	/*  update statistics  */
	}
	au->http_done_header = 1;
	bufp += p;		/*  let the header vanish  */
	len -= p;
	alen = len;
    } else	/*  done_header  */
	alen = orglen;

    /*  http message body  */

  if (!multi && !exto && au->url->save_content) {
   if (!au->disposition[0])
    fi = 1;
   else {
    if (au->file_created) {
	if (orglen || au->buff_len) {
	    if (au->f != -1)
		/*  If the file is already open, let's just write to it ...  */
		fi = au->f;
	    else {
		/*  file is switched  */
		if ((fi = mfopen(au, O_WRONLY | _O_BINARY)) < 0)
		    return errm(au->url, "!$u: cannot open %s for appending: %s",
				au->disposition, strerror(errno));
		lseek(fi, au->file_off, SEEK_SET);
	    }
	}
    } else {	/*  no attempt to open the file till now  */
	au->file_created = 1;
	if (au->file_off) {
	  if (needs_recurse_au(au, 1)) {
	    if ((fi = mfopen(au, O_RDWR | _O_BINARY)) < 0)
		return errm(au->url, 
			    "!$u: cannot open %s for reading and appending: %s", 
			    au->disposition, strerror(errno));
	    recurse_pfile(au->url, fi, &bufp, &len, au);
	  } else {
	    if ((fi = mfopen(au, O_WRONLY | _O_BINARY)) < 0)
		return errm(au->url, "!$u: cannot open %s for appending: %s", 
			    au->disposition, strerror(errno));
	  }
	  lseek(fi, au->file_off, SEEK_SET);
	} else {
	    if ((fi = mfopen(au, O_WRONLY | O_CREAT | O_TRUNC | _O_BINARY)) < 0)
		return errm(au->url, "!$u: cannot create %s: %s", 
			    au->disposition, strerror(errno));
	}
    }
   }
  }

    /*  scan the buffer for references  */
    if (needs_recurse_au(au, 1)) {
	int done = recurse_buff(au->url, bufp, len, orglen, au);
	if (orglen && !save_buff(au, bufp + done, len - done))
	    return RT_RETRY;
    }

    if (orglen) {
	int retaf;

	/*  point at "fresh" data  */
	bufp += len - alen;

	/*  hard file size limitation  */
	if (au->url->parm->opt->max_bytes && 
	    au->file_off + alen >= au->url->parm->opt->max_bytes) {
	    alen = au->url->parm->opt->max_bytes - au->file_off;
	    retaf = 0;
	} else
	    retaf = 1;

	/*  Update the counters and statistics:  */
	au->size_fetched += alen;
	if (!au->size_total)
	    total_bytes += alen;
	fetched_bytes += alen;

     if (au->url->save_content) {
      if (!multi && !exto) {
	/*  write the buffer to disk */
	if (buff_size) {
	    pad = au->file_off % buff_size;
	    flen = pad + au->buff_len + alen;
	    if (flen > buff_size) {
		wlen = flen / buff_size * buff_size - pad;
		au->file_off += wlen;
		cq_consume(au->buff, buffe_t, be, {
		    qsafe_write(fi, be->data, be->len);
		    cq_rm1st(au->buff);
		    free(be);
		});
		pwlen = wlen - au->buff_len;
		au->buff_len = 0;
		qsafe_write(fi, bufp, pwlen);
		bufp += pwlen;
		alen -= pwlen;
	    }
	    if (alen) {
		au->buff_len += alen;
		if (!(buffe = mmalloc(sizeof(*buffe) + alen)))
		    return RT_RETRY;
		buffe->len = alen;
		memcpy(buffe->data, bufp, alen);
		cq_append(au->buff, buffe);
	    }
	} else {
	    au->file_off += alen;
	    qsafe_write(fi, bufp, alen);
	}
      } else {
	au->buff_len += alen;
	if (au->buff_len > buff_size)
	    return werrm(au, 651, "!$u too big. Try a bigger -xs.");
	if (!(buffe = mmalloc(sizeof(*buffe) + alen)))
	    return RT_RETRY;
	buffe->len = alen;
	memcpy(buffe->data, bufp, alen);
	cq_append(au->buff, buffe);
      }
     }

	if (retaf)
	    return RT_OK;
    }

 if (au->url->save_content) {
  if (multi || exto) {
    if ((ret = open_disp(au, &fi)) != RT_OK)
	return ret;
    if (exto) {
	write_loc(fi, au->url, 1);
	if (au->headers)
	    safe_write(fi, au->headers, au->hdrslen);
	safe_write(fi, buf, sprintf(buf, "Content-Length: %lu\n\n",
					 (u_long)au->buff_len));
    }
  }
    cq_iterate(au->buff, buffe_t, be, qsafe_write(fi, be->data, be->len););
  if (!multi && au->disposition[0]) {
    /*  file is complete -> rename it  */
    touch(au);
    memcpy(buf, au->disposition, au->displen);
	buf[au->displen] = '\0';
    rename(au->disposition, buf);
    /*  save headers  */
    if (!exto && au->headers) {
	/* prevent unnecessary ping-pong */
	if (au->f != -1) {
	    close(au->f);
	    au->f = -1;
	}

	strcat(buf, ".hdr");
	if ((fi = mmfopen(buf, O_WRONLY | O_CREAT | O_TRUNC, &au->f)) < 0)
	    return errm(au->url, "!$u: cannot create %s: %s", 
			buf, strerror(errno));
	safe_write(fi, au->headers, au->hdrslen);
    }
  }
 } else
    prxu(NFO, au->url, "not saving $u (rejected)\n");

    return RT_DONE;
}
