% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cwi.R
\name{cwi}
\alias{cwi}
\alias{cwi.data.frame}
\title{Curvewise Intervals (CWI)}
\usage{
cwi(x, ...)

\method{cwi}{data.frame}(x, ci = 0.95, ...)
}
\arguments{
\item{x}{Vector representing a posterior distribution, or a data frame of such
vectors. Can also be a Bayesian model. \strong{bayestestR} supports a wide range
of models (see, for example, \code{methods("hdi")}) and not all of those are
documented in the 'Usage' section, because methods for other classes mostly
resemble the arguments of the \code{.numeric} or \code{.data.frame}methods.}

\item{...}{Currently not used.}

\item{ci}{Value or vector of probability of the (credible) interval - CI
(between 0 and 1) to be estimated. Default to \code{.95} (\verb{95\%}).}
}
\value{
A data frame with following columns:
\itemize{
\item \code{Parameter} The model parameter(s), if \code{x} is a model-object. If \code{x} is a vector, this column is missing.
\item \code{CI} The probability of the credible interval.
\item \code{CI_low}, \code{CI_high} The lower and upper credible interval limits for the parameters.
}
}
\description{
Compute the \strong{Curvewise interval (CWI)} (also called the "simultaneous interval" or "joint interval") of posterior distributions using \code{ggdist::curve_interval()}.
Whereas the more typical "pointwise intervals" contain xx\% of the posterior for a single parameter,
joint/curvewise intervals contain xx\% of the posterior distribution for \strong{all} parameters.
}
\details{
Applied model predictions, pointwise intervals contain xx\% of the predicted response values \strong{conditional} on specific predictor values.
In contrast, curvewise intervals contain xx\% of the predicted response values across all predictor values.
Put another way, curvewise intervals contain xx\% of the full \strong{prediction lines} from the model.

For more details, see the \href{https://mjskay.github.io/ggdist/articles/lineribbon.html#curve-boxplots-aka-lineribbons-with-joint-intervals-or-curvewise-intervals-}{\emph{ggdist} documentation on curvewise intervals}.
}
\examples{
\donttest{
library(bayestestR)

if (require("ggplot2") && require("rstanarm") && require("ggdist")) {
  # Generate data =============================================
  k <- 11 # number of curves (iterations)
  n <- 201 # number of rows
  data <- data.frame(x = seq(-15, 15, length.out = n))

  # Simulate iterations as new columns
  for (i in 1:k) {
    data[paste0("iter_", i)] <- dnorm(data$x, seq(-5, 5, length.out = k)[i], 3)
  }

  # Note: first, we need to transpose the data to have iters as rows
  iters <- datawizard::data_transpose(data[paste0("iter_", 1:k)])

  # Compute Median
  data$Median <- point_estimate(iters)[["Median"]]

  # Compute Credible Intervals ================================

  # Compute ETI (default type of CI)
  data[c("ETI_low", "ETI_high")] <- eti(iters, ci = 0.5)[c("CI_low", "CI_high")]

  # Compute CWI
  # ggdist::curve_interval(reshape_iterations(data), iter_value .width = 0.5)

  # Visualization =============================================
  ggplot(data, aes(x = x, y = Median)) +
    geom_ribbon(aes(ymin = ETI_low, ymax = ETI_high), fill = "red", alpha = 0.3) +
    geom_line(linewidth = 1) +
    geom_line(
      data = reshape_iterations(data),
      aes(y = iter_value, group = iter_group),
      alpha = 0.3
    )
}
}
}
\seealso{
Other ci: 
\code{\link{bci}()},
\code{\link{ci}()},
\code{\link{eti}()},
\code{\link{hdi}()},
\code{\link{si}()},
\code{\link{spi}()}
}
\concept{ci}
