% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/hrg.R
\name{predict_edges}
\alias{predict_edges}
\title{Predict edges based on a hierarchical random graph model}
\usage{
predict_edges(
  graph,
  hrg = NULL,
  start = FALSE,
  num.samples = 10000,
  num.bins = 25
)
}
\arguments{
\item{graph}{The graph to fit the model to. Edge directions are ignored in
directed graphs.}

\item{hrg}{A hierarchical random graph model, in the form of an
\code{igraphHRG} object. \code{predict_edges()} allow this to be
\code{NULL} as well, then a HRG is fitted to the graph first, from a
random starting point.}

\item{start}{Logical, whether to start the fitting/sampling from the
supplied \code{igraphHRG} object, or from a random starting point.}

\item{num.samples}{Number of samples to use for consensus generation or
missing edge prediction.}

\item{num.bins}{Number of bins for the edge probabilities. Give a higher
number for a more accurate prediction.}
}
\value{
A list with entries:
\item{edges}{The predicted edges, in a two-column matrix of vertex
ids.}
\item{prob}{Probabilities of these edges, according to the fitted
model.}
\item{hrg}{The (supplied or fitted) hierarchical random graph model.}
}
\description{
\code{predict_edges()} uses a hierarchical random graph model to predict
missing edges from a network. This is done by sampling hierarchical models
around the optimum model, proportionally to their likelihood. The MCMC
sampling is stated from \code{hrg()}, if it is given and the \code{start}
argument is set to \code{TRUE}. Otherwise a HRG is fitted to the graph
first.
}
\examples{
\dontshow{if (rlang::is_interactive()) withAutoprint(\{ # examplesIf}

## A graph with two dense groups
g <- sample_gnp(10, p = 1 / 2) + sample_gnp(10, p = 1 / 2)
hrg <- fit_hrg(g)
hrg

## The consensus tree for it
consensus_tree(g, hrg = hrg, start = TRUE)

## Prediction of missing edges
g2 <- make_full_graph(4) + (make_full_graph(4) - path(1, 2))
predict_edges(g2)
\dontshow{\}) # examplesIf}
}
\references{
A. Clauset, C. Moore, and M.E.J. Newman. Hierarchical structure
and the prediction of missing links in networks. \emph{Nature} 453, 98--101
(2008);

A. Clauset, C. Moore, and M.E.J. Newman. Structural Inference of Hierarchies
in Networks. In E. M. Airoldi et al. (Eds.): ICML 2006 Ws, \emph{Lecture
Notes in Computer Science} 4503, 1--13. Springer-Verlag, Berlin Heidelberg
(2007).
}
\seealso{
Other hierarchical random graph functions: 
\code{\link{consensus_tree}()},
\code{\link{fit_hrg}()},
\code{\link{hrg}()},
\code{\link{hrg-methods}},
\code{\link{hrg_tree}()},
\code{\link{print.igraphHRG}()},
\code{\link{print.igraphHRGConsensus}()},
\code{\link{sample_hrg}()}
}
\concept{hierarchical random graph functions}
