% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/games.R
\name{sample_fitness_pl}
\alias{sample_fitness_pl}
\title{Scale-free random graphs, from vertex fitness scores}
\usage{
sample_fitness_pl(
  no.of.nodes,
  no.of.edges,
  exponent.out,
  exponent.in = -1,
  loops = FALSE,
  multiple = FALSE,
  finite.size.correction = TRUE
)
}
\arguments{
\item{no.of.nodes}{The number of vertices in the generated graph.}

\item{no.of.edges}{The number of edges in the generated graph.}

\item{exponent.out}{Numeric scalar, the power law exponent of the degree
distribution. For directed graphs, this specifies the exponent of the
out-degree distribution. It must be greater than or equal to 2. If you pass
\code{Inf} here, you will get back an Erdős-Rényi random network.}

\item{exponent.in}{Numeric scalar. If negative, the generated graph will be
undirected. If greater than or equal to 2, this argument specifies the
exponent of the in-degree distribution. If non-negative but less than 2, an
error will be generated.}

\item{loops}{Logical scalar, whether to allow loop edges in the generated
graph.}

\item{multiple}{Logical scalar, whether to allow multiple edges in the
generated graph.}

\item{finite.size.correction}{Logical scalar, whether to use the proposed
finite size correction of Cho et al., see references below.}
}
\value{
An igraph graph, directed or undirected.
}
\description{
This function generates a non-growing random graph with expected power-law
degree distributions.
}
\details{
This game generates a directed or undirected random graph where the degrees
of vertices follow power-law distributions with prescribed exponents. For
directed graphs, the exponents of the in- and out-degree distributions may
be specified separately.

The game simply uses \code{\link[=sample_fitness]{sample_fitness()}} with appropriately
constructed fitness vectors. In particular, the fitness of vertex \eqn{i} is
\eqn{i^{-\alpha}}{i^(-alpha)}, where \eqn{\alpha = 1/(\gamma-1)}{alpha = 1/(gamma - 1)}
and \eqn{\gamma}{gamma} is the exponent given in the arguments.

To remove correlations between in- and out-degrees in case of directed
graphs, the in-fitness vector will be shuffled after it has been set up and
before \code{\link[=sample_fitness]{sample_fitness()}} is called.

Note that significant finite size effects may be observed for exponents
smaller than 3 in the original formulation of the game. This function
provides an argument that lets you remove the finite size effects by
assuming that the fitness of vertex \eqn{i} is
\eqn{(i+i_0-1)^{-\alpha}}{(i+i0-1)^(-alpha)} where \eqn{i_0}{i0} is a
constant chosen appropriately to ensure that the maximum degree is less than
the square root of the number of edges times the average degree; see the
paper of Chung and Lu, and Cho et al for more details.
}
\examples{

g <- sample_fitness_pl(10000, 30000, 2.2, 2.3)
plot(degree_distribution(g, cumulative = TRUE, mode = "out"), log = "xy")
}
\references{
Goh K-I, Kahng B, Kim D: Universal behaviour of load
distribution in scale-free networks. \emph{Phys Rev Lett} 87(27):278701,
2001.

Chung F and Lu L: Connected components in a random graph with given degree
sequences. \emph{Annals of Combinatorics} 6, 125-145, 2002.

Cho YS, Kim JS, Park J, Kahng B, Kim D: Percolation transitions in
scale-free networks under the Achlioptas process. \emph{Phys Rev Lett}
103:135702, 2009.
}
\seealso{
Random graph models (games)
\code{\link{erdos.renyi.game}()},
\code{\link{sample_}()},
\code{\link{sample_bipartite}()},
\code{\link{sample_chung_lu}()},
\code{\link{sample_correlated_gnp}()},
\code{\link{sample_correlated_gnp_pair}()},
\code{\link{sample_degseq}()},
\code{\link{sample_dot_product}()},
\code{\link{sample_fitness}()},
\code{\link{sample_forestfire}()},
\code{\link{sample_gnm}()},
\code{\link{sample_gnp}()},
\code{\link{sample_grg}()},
\code{\link{sample_growing}()},
\code{\link{sample_hierarchical_sbm}()},
\code{\link{sample_islands}()},
\code{\link{sample_k_regular}()},
\code{\link{sample_last_cit}()},
\code{\link{sample_pa}()},
\code{\link{sample_pa_age}()},
\code{\link{sample_pref}()},
\code{\link{sample_sbm}()},
\code{\link{sample_smallworld}()},
\code{\link{sample_traits_callaway}()},
\code{\link{sample_tree}()}
}
\author{
Tamas Nepusz \email{ntamas@gmail.com}
}
\concept{games}
\keyword{graphs}
\section{Related documentation in the C library}{\href{https://igraph.org/c/html/latest/igraph-Generators.html#igraph_static_power_law_game}{\code{igraph_static_power_law_game()}}.}

