% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/f_design_plan_survival.R
\name{getSampleSizeSurvival}
\alias{getSampleSizeSurvival}
\title{Get Sample Size Survival}
\usage{
getSampleSizeSurvival(
  design = NULL,
  ...,
  typeOfComputation = c("Schoenfeld", "Freedman", "HsiehFreedman"),
  thetaH0 = 1,
  pi1 = NA_real_,
  pi2 = NA_real_,
  lambda1 = NA_real_,
  lambda2 = NA_real_,
  median1 = NA_real_,
  median2 = NA_real_,
  kappa = 1,
  hazardRatio = NA_real_,
  piecewiseSurvivalTime = NA_real_,
  allocationRatioPlanned = NA_real_,
  eventTime = 12,
  accrualTime = c(0, 12),
  accrualIntensity = 0.1,
  accrualIntensityType = c("auto", "absolute", "relative"),
  followUpTime = NA_real_,
  maxNumberOfSubjects = NA_real_,
  dropoutRate1 = 0,
  dropoutRate2 = 0,
  dropoutTime = 12
)
}
\arguments{
\item{design}{The trial design. If no trial design is specified, a fixed sample size design is used.
In this case, Type I error rate \code{alpha}, Type II error rate \code{beta}, \code{twoSidedPower},
and \code{sided} can be directly entered as argument where necessary.}

\item{...}{Ensures that all arguments (starting from the "...") are to be named and
that a warning will be displayed if unknown arguments are passed.}

\item{typeOfComputation}{Three options are available: \code{"Schoenfeld"}, \code{"Freedman"}, \code{"HsiehFreedman"},
the default is \code{"Schoenfeld"}. For details, see Hsieh (Statistics in Medicine, 1992).
For non-inferiority testing (i.e., \code{thetaH0 != 1}), only Schoenfeld's formula can be used.}

\item{thetaH0}{The null hypothesis value,
default is \code{0} for the normal and the binary case (testing means and rates, respectively),
it is \code{1} for the survival case (testing the hazard ratio).\cr\cr
For non-inferiority designs, \code{thetaH0} is the non-inferiority bound.
That is, in case of (one-sided) testing of
\itemize{
\item \emph{means}: a value \code{!= 0}
(or a value \code{!= 1} for testing the mean ratio) can be specified.
\item \emph{rates}: a value \code{!= 0}
(or a value \code{!= 1} for testing the risk ratio \code{pi1 / pi2}) can be specified.
\item \emph{survival data}: a bound for testing H0: \code{hazard ratio = thetaH0 != 1} can be specified.
\item \emph{count data}: a bound for testing H0: \code{lambda1 / lambda2 = thetaH0 != 1} can be specified.
}
For testing a rate in one sample, a value \code{thetaH0} in (0, 1) has to be specified for
defining the null hypothesis H0: \code{pi = thetaH0}.}

\item{pi1}{A numeric value or vector that represents the assumed event rate in the treatment group,
default is \code{seq(0.2, 0.5, 0.1)} (power calculations and simulations) or
\code{seq(0.4, 0.6, 0.1)} (sample size calculations).}

\item{pi2}{A numeric value that represents the assumed event rate in the control group, default is \code{0.2}.}

\item{lambda1}{The assumed hazard rate in the treatment group, there is no default.
\code{lambda1} can also be used to define piecewise exponentially distributed survival times (see details). Must be a positive numeric of length 1.}

\item{lambda2}{The assumed hazard rate in the reference group, there is no default.
\code{lambda2} can also be used to define piecewise exponentially distributed survival times (see details). Must be a positive numeric of length 1.}

\item{median1}{The assumed median survival time in the treatment group, there is no default.}

\item{median2}{The assumed median survival time in the reference group, there is no default. Must be a positive numeric of length 1.}

\item{kappa}{A numeric value > 0. A \code{kappa != 1} will be used for the specification
of the shape of the Weibull distribution.
Default is \code{1}, i.e., the exponential survival distribution is used instead of the Weibull distribution.
Note that the Weibull distribution cannot be used for the piecewise definition of
the survival time distribution, i.e., only \code{piecewiselambda} (as a single value) and \code{kappa}
can be specified.
This function is equivalent to \code{pweibull(t, shape = kappa, scale = 1 / lambda)}
of the \code{stats} package, i.e., the scale parameter is \code{1 / 'hazard rate'}.\cr
For example,
\code{getPiecewiseExponentialDistribution(time = 130, piecewiseLambda = 0.01, kappa = 4.2)}
and \code{pweibull(q = 130, shape = 4.2, scale = 1 / 0.01)} provide the sample result.}

\item{hazardRatio}{The vector of hazard ratios under consideration.
If the event or hazard rates in both treatment groups are defined, the hazard ratio needs
not to be specified as it is calculated, there is no default. Must be a positive numeric of length 1.}

\item{piecewiseSurvivalTime}{A vector that specifies the time intervals for the piecewise
definition of the exponential survival time cumulative distribution function \cr
(for details see \code{\link[=getPiecewiseSurvivalTime]{getPiecewiseSurvivalTime()}}).}

\item{allocationRatioPlanned}{The planned allocation ratio \code{n1 / n2} for a two treatment groups
design, default is \code{1}. If \code{allocationRatioPlanned = 0} is entered,
the optimal allocation ratio yielding the smallest overall sample size is determined.}

\item{eventTime}{The assumed time under which the event rates are calculated, default is \code{12}.}

\item{accrualTime}{The assumed accrual time intervals for the study, default is
\code{c(0, 12)} (for details see \code{\link[=getAccrualTime]{getAccrualTime()}}).}

\item{accrualIntensity}{A numeric vector of accrual intensities, default is the relative
intensity \code{0.1} (for details see \code{\link[=getAccrualTime]{getAccrualTime()}}).}

\item{accrualIntensityType}{A character value specifying the accrual intensity input type.
Must be one of \code{"auto"}, \code{"absolute"}, or \code{"relative"}; default is \code{"auto"},
i.e., if all values are < 1 the type is \code{"relative"}, otherwise it is \code{"absolute"}.}

\item{followUpTime}{The assumed (additional) follow-up time for the study, default is \code{6}.
The total study duration is \code{accrualTime + followUpTime}.}

\item{maxNumberOfSubjects}{If \code{maxNumberOfSubjects > 0} is specified,
the follow-up time for the required number of events is determined.}

\item{dropoutRate1}{The assumed drop-out rate in the treatment group, default is \code{0}.}

\item{dropoutRate2}{The assumed drop-out rate in the control group, default is \code{0}.}

\item{dropoutTime}{The assumed time for drop-out rates in the control and the
treatment group, default is \code{12}.}
}
\value{
Returns a \code{\link{TrialDesignPlan}} object.
The following generics (R generic functions) are available for this result object:
\itemize{
\item \code{\link[=names.FieldSet]{names()}} to obtain the field names,
\item \code{\link[=print.FieldSet]{print()}} to print the object,
\item \code{\link[=summary.TrialDesignSet]{summary()}} to display a summary of the object,
\item \code{\link[=plot.TrialDesignPlan]{plot()}} to plot the object,
\item \code{\link[=as.data.frame.TrialDesignPlan]{as.data.frame()}} to coerce the object to a \code{\link[base]{data.frame}},
\item \code{\link[=as.matrix.FieldSet]{as.matrix()}} to coerce the object to a \code{\link[base]{matrix}}.
}
}
\description{
Returns the sample size for testing the hazard ratio in a two treatment groups survival design.
}
\details{
At given design the function calculates the number of events and an estimate for the
necessary number of subjects for testing the hazard ratio in a survival design.
It also calculates the time when the required events are expected under the given
assumptions (exponentially, piecewise exponentially, or Weibull distributed survival times
and constant or non-constant piecewise accrual).
Additionally, an allocation ratio = \code{n1 / n2} can be specified where \code{n1} and \code{n2} are the number
of subjects in the two treatment groups.

Optional argument \code{accountForObservationTimes}: if \code{accountForObservationTimes = TRUE}, the number of
subjects is calculated assuming specific accrual and follow-up time, default is \code{TRUE}.

The formula of Kim & Tsiatis (Biometrics, 1990)
is used to calculate the expected number of events under the alternative
(see also Lakatos & Lan, Statistics in Medicine, 1992). These formulas are generalized
to piecewise survival times and non-constant piecewise accrual over time.\cr

Optional argument \code{accountForObservationTimes}: if \code{accountForObservationTimes = FALSE},
only the event rates are used for the calculation of the maximum number of subjects.
}
\section{Piecewise survival time}{

The first element of the vector \code{piecewiseSurvivalTime} must be equal to \code{0}.
\code{piecewiseSurvivalTime} can also be a list that combines the definition of the
time intervals and hazard rates in the reference group.
The definition of the survival time in the treatment group is obtained by the specification
of the hazard ratio (see examples for details).
}

\section{Staggered patient entry}{

\code{accrualTime} is the time period of subjects' accrual in a study.
It can be a value that defines the end of accrual or a vector.
In this case, \code{accrualTime} can be used to define a non-constant accrual over time.
For this, \code{accrualTime} is a vector that defines the accrual intervals.
The first element of \code{accrualTime} must be equal to \code{0} and, additionally,
\code{accrualIntensity} needs to be specified.
\code{accrualIntensity} itself is a value or a vector (depending on the
length of \code{accrualTime}) that defines the intensity how subjects
enter the trial in the intervals defined through \code{accrualTime}.

\code{accrualTime} can also be a list that combines the definition of the accrual time and
accrual intensity (see below and examples for details).

If the length of \code{accrualTime} and the length of \code{accrualIntensity} are the same
(i.e., the end of accrual is undefined), \code{maxNumberOfSubjects > 0} needs to be specified
and the end of accrual is calculated.
In that case, \code{accrualIntensity} is the number of subjects per time unit, i.e., the absolute accrual intensity.

If the length of \code{accrualTime} equals the length of \code{accrualIntensity - 1}
(i.e., the end of accrual is defined), \code{maxNumberOfSubjects} is calculated if the absolute accrual intensity is given.
If all elements in \code{accrualIntensity} are smaller than 1, \code{accrualIntensity} defines
the \emph{relative} intensity how subjects enter the trial.
For example, \code{accrualIntensity = c(0.1, 0.2)} specifies that in the second accrual interval
the intensity is doubled as compared to the first accrual interval. The actual (absolute) accrual intensity
is calculated for the calculated or given \code{maxNumberOfSubjects}.
Note that the default is \code{accrualIntensity = 0.1} meaning that the \emph{absolute} accrual intensity
will be calculated.
}

\section{How to get help for generic functions}{

Click on the link of a generic in the list above to go directly to the help documentation of
the \code{rpact} specific implementation of the generic.
Note that you can use the R function \code{\link[utils]{methods}} to get all the methods of a generic and
to identify the object specific name of it, e.g.,
use \code{methods("plot")} to get all the methods for the \code{plot} generic.
There you can find, e.g., \code{plot.AnalysisResults} and
obtain the specific help documentation linked above by typing \code{?plot.AnalysisResults}.
}

\examples{
\dontrun{
# Fixed sample size trial with median survival 20 vs. 30 months in treatment and 
# reference group, respectively, alpha = 0.05 (two-sided), and power 1 - beta = 90\%.
# 20 subjects will be recruited per month up to 400 subjects, i.e., accrual time 
# is 20 months.  
getSampleSizeSurvival(alpha = 0.05, sided = 2, beta = 0.1, lambda1 = log(2) / 20, 
    lambda2 = log(2) / 30, accrualTime = c(0,20), accrualIntensity = 20)

# Fixed sample size with minimum required definitions, pi1 = c(0.4,0.5,0.6) and 
# pi2 = 0.2 at event time 12, accrual time 12 and follow-up time 6 as default, 
# only alpha = 0.01 is specified  
getSampleSizeSurvival(alpha = 0.01)

# Four stage O'Brien & Fleming group sequential design with minimum required 
# definitions, pi1 = c(0.4,0.5,0.6) and pi2 = 0.2 at event time 12, 
# accrual time 12 and follow-up time 6 as default  
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 4))

# For fixed sample design, determine necessary accrual time if 200 subjects and 
# 30 subjects per time unit can be recruited 
getSampleSizeSurvival(accrualTime = c(0), accrualIntensity = c(30), 
    maxNumberOfSubjects = 200)

# Determine necessary accrual time if 200 subjects and if the first 6 time units 
# 20 subjects per time unit can be recruited, then 30 subjects per time unit 
getSampleSizeSurvival(accrualTime = c(0, 6), accrualIntensity = c(20, 30), 
    maxNumberOfSubjects = 200)

# Determine maximum number of Subjects if the first 6 time units 20 subjects 
# per time unit can be recruited, and after 10 time units 30 subjects per time unit
getSampleSizeSurvival(accrualTime = c(0, 6, 10), accrualIntensity = c(20, 30))

# Specify accrual time as a list
at <- list(
    "0 - <6"  = 20,
    "6 - Inf" = 30)
getSampleSizeSurvival(accrualTime = at, maxNumberOfSubjects = 200)

# Specify accrual time as a list, if maximum number of subjects need to be calculated
at <- list(
    "0 - <6"   = 20,
    "6 - <=10" = 30)
getSampleSizeSurvival(accrualTime = at)

# Specify effect size for a two-stage group design with O'Brien & Fleming boundaries
# Effect size is based on event rates at specified event time 
# needs to be specified because it should be shown that hazard ratio < 1
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    pi1 = 0.2, pi2 = 0.3, eventTime = 24)

# Effect size is based on event rate at specified event 
# time for the reference group and hazard ratio 
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    hazardRatio = 0.5, pi2 = 0.3, eventTime = 24)

# Effect size is based on hazard rate for the reference group and hazard ratio
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    hazardRatio = 0.5, lambda2 = 0.02) 

# Specification of piecewise exponential survival time and hazard ratios  
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    piecewiseSurvivalTime = c(0, 5, 10), lambda2 = c(0.01, 0.02, 0.04), 
    hazardRatio = c(1.5, 1.8, 2))

# Specification of piecewise exponential survival time as a list and hazard ratios 
pws <- list(
    "0 - <5"  = 0.01,
    "5 - <10" = 0.02,
    ">=10"    = 0.04)
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    piecewiseSurvivalTime = pws, hazardRatio = c(1.5, 1.8, 2))

# Specification of piecewise exponential survival time for both treatment arms
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    piecewiseSurvivalTime = c(0, 5, 10), lambda2 = c(0.01, 0.02, 0.04), 
    lambda1 = c(0.015, 0.03, 0.06))

# Specification of piecewise exponential survival time as a list
pws <- list(
    "0 - <5"  = 0.01,
    "5 - <10" = 0.02,
    ">=10"    = 0.04)
getSampleSizeSurvival(design = getDesignGroupSequential(kMax = 2), 
    piecewiseSurvivalTime = pws, hazardRatio = c(1.5, 1.8, 2))

# Specify effect size based on median survival times
getSampleSizeSurvival(median1 = 5, median2 = 3)

# Specify effect size based on median survival times of Weibull distribtion with 
# kappa = 2
getSampleSizeSurvival(median1 = 5, median2 = 3, kappa = 2)

# Identify minimal and maximal required subjects to 
# reach the required events in spite of dropouts
getSampleSizeSurvival(accrualTime = c(0, 18), accrualIntensity = c(20, 30), 
    lambda2 = 0.4, lambda1 = 0.3, followUpTime = Inf, dropoutRate1 = 0.001, 
    dropoutRate2 = 0.005)
getSampleSizeSurvival(accrualTime = c(0, 18), accrualIntensity = c(20, 30), 
    lambda2 = 0.4, lambda1 = 0.3, followUpTime = 0, dropoutRate1 = 0.001, 
    dropoutRate2 = 0.005)
}

}
\seealso{
Other sample size functions: 
\code{\link{getSampleSizeCounts}()},
\code{\link{getSampleSizeMeans}()},
\code{\link{getSampleSizeRates}()}
}
\concept{sample size functions}
