/*
 * Schism Tracker - a cross-platform Impulse Tracker clone
 * copyright (c) 2003-2005 Storlek <storlek@rigelseven.com>
 * copyright (c) 2005-2008 Mrs. Brisby <mrs.brisby@nimh.org>
 * copyright (c) 2009 Storlek & Mrs. Brisby
 * copyright (c) 2010-2012 Storlek
 * URL: http://schismtracker.org/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "test.h"
#include "test-assertions.h"
#include "test-tempfile.h"

#include "slurp.h"
#include "fmt.h"

static const char expected_result[] =
	"abc def ghi 123 456 789\n"
	"I live in a giant bucket.\n"
	"bleugh.\n";
SCHISM_STATIC_ASSERT((ARRAY_SIZE(expected_result) - 1) % 2 == 0,
	"need the size to be a multiple of two for slurp_2memstream");

/* the padding between memory segments; used in 2mem and sf2 tests
 * 7 is a completely arbitrary value. */
#define TEST_SLURP_PADDING (7)
#define TEST_SLURP_2SIZE ((ARRAY_SIZE(expected_result) - 1) / 2)

static testresult_t test_slurp_common(slurp_t *fp)
{
	char buf[ARRAY_SIZE(expected_result) - 1];
	static const char zero[ARRAY_SIZE(buf) >> 1] = {0};
	size_t i;
	size_t j;

	ASSERT(slurp_length(fp) == sizeof(buf));

	/* go over every possible (legal) combination of reads.
	 * there's probably a simpler way to do this, but oh well. */
	for (i = 0; i < sizeof(buf); i++) {
		for (j = 0; j < (sizeof(buf) - i); j++) {
			ASSERT(slurp_seek(fp, i, SEEK_SET) == 0);
			ASSERT(slurp_read(fp, buf, sizeof(buf) - i - j) == (sizeof(buf) - i - j));

			ASSERT(slurp_tell(fp) == (sizeof(buf) - j));

			/*printf("%" PRIuSZ ", %" PRIuSZ ": buf: %.*s", i, (int)(sizeof(buf) - i - j), buf);*/

			/* data should not change */
			ASSERT(!memcmp(buf, expected_result + i, sizeof(buf) - i - j));

			/* we should never trigger EOF here */
			ASSERT(!slurp_eof(fp));
		}
	}

	/* now, we are at the end of the file.
	 * try reading a bit more, and make sure EOF stays flagged.
	 * do it many times to make sure that extra reads have no effect on EOF. */

	for (i = 0; i < 5; i++) {
		ASSERT(slurp_read(fp, buf, sizeof(buf)) == 0);
		ASSERT(slurp_tell(fp) == sizeof(buf));
		ASSERT(slurp_eof(fp));
	}

	/* TODO what should the behavior be for slurp_peek regarding EOF?
	 * TBH I think it should stay the exact same as it was... */

	/* random operations should have no effect on EOF */
	(void)slurp_tell(fp);
	ASSERT(slurp_eof(fp));
	(void)slurp_getc(fp);
	ASSERT(slurp_eof(fp));

	/* getting the length should not affect EOF
	 * (i.e., it should not call seek()) */
	(void)slurp_length(fp);
	ASSERT(slurp_eof(fp));

	/* seeking to the end should not cause EOF */
	ASSERT(slurp_seek(fp, 0, SEEK_END) == 0);
	ASSERT(!slurp_eof(fp));

	/* random operations should have no effect on EOF */
	(void)slurp_tell(fp);
	ASSERT(!slurp_eof(fp));

	/* getting the length should not affect EOF */
	(void)slurp_length(fp);
	ASSERT(!slurp_eof(fp));

	/* any reads should trigger EOF */
	(void)slurp_getc(fp);
	ASSERT(slurp_eof(fp));

	/* read should zero out remaining bytes */
	memset(buf, 0xFF, sizeof(buf));
	ASSERT(slurp_seek(fp, sizeof(buf) >> 1, SEEK_SET) == 0);
	ASSERT(slurp_read(fp, buf, sizeof(buf)) == (sizeof(buf) >> 1));
	ASSERT(memcmp(buf + sizeof(zero), zero, sizeof(zero)) == 0);

	/* verify state */
	ASSERT(slurp_tell(fp) == sizeof(buf));
	ASSERT(slurp_getc(fp) == EOF);
	ASSERT(slurp_eof(fp));

	/* peek should zero out remaining bytes */
	memset(buf, 0xFF, sizeof(buf));
	ASSERT(slurp_seek(fp, sizeof(buf) >> 1, SEEK_SET) == 0);
	ASSERT(slurp_peek(fp, buf, sizeof(buf)) == (sizeof(buf) >> 1));
	ASSERT(memcmp(buf + sizeof(zero), zero, sizeof(zero)) == 0);

	/* verify state */
	ASSERT(slurp_tell(fp) == (sizeof(buf) >> 1));
	ASSERT(slurp_getc(fp) == 101);
	ASSERT(!slurp_eof(fp));

	/* seeking should clear any EOF flag */
	ASSERT(slurp_seek(fp, 0, SEEK_SET) == 0);
	ASSERT(!slurp_eof(fp));

	/* limit should only allow reading X bytes
	 * also test read zero behavior here :) */
	slurp_limit(fp, sizeof(buf) >> 1);
	memset(buf, 0xFF, sizeof(buf));
	ASSERT(slurp_read(fp, buf, sizeof(buf)) == (sizeof(buf) >> 1));
	ASSERT(memcmp(buf + sizeof(zero), zero, sizeof(zero)) == 0);
	ASSERT(slurp_tell(fp) == (sizeof(buf) >> 1));
	/* any subsequent reads should never return any bytes */
	ASSERT(slurp_read(fp, buf, sizeof(buf)) == 0);
	ASSERT(slurp_tell(fp) == (sizeof(buf) >> 1));
	/* XXX slurp_eof should probably return 1 here */
	slurp_unlimit(fp);

	/* seek past EOF behavior */
	ASSERT(slurp_seek(fp, 0, SEEK_SET) == 0);
	ASSERT(slurp_seek(fp, sizeof(buf), SEEK_SET) == 0);
	ASSERT(slurp_tell(fp) == sizeof(buf));
	ASSERT(slurp_seek(fp, sizeof(buf) + 1, SEEK_SET) == -1);
	ASSERT(slurp_tell(fp) == sizeof(buf));

	/* slurp_available */
	ASSERT(slurp_seek(fp, 0, SEEK_SET) == 0);
	ASSERT(slurp_available(fp, sizeof(buf), SEEK_SET));
	ASSERT(!slurp_available(fp, sizeof(buf) + 1, SEEK_SET));

	/* verify state */
	ASSERT(slurp_tell(fp) == 0);
	ASSERT(!slurp_eof(fp));

	RETURN_PASS;
}

testresult_t test_slurp_memstream(void)
{
	slurp_t fp;
	uint8_t buf[ARRAY_SIZE(expected_result) - 1];
	testresult_t r;

	memcpy(buf, expected_result, sizeof(buf));

	/* should never happen */
	ASSERT(slurp_memstream(&fp, (uint8_t *)buf, sizeof(buf)) >= 0);

	r = test_slurp_common(&fp);

	/* currently not necessary for memstream, but whatever */
	unslurp(&fp);

	return r;
}

static void test_slurp_setup_padded_buf(const void *src, void *dst, size_t sz, size_t padding)
{
	/* copy the first half of the expected result */
	memcpy(dst, src, sz);
	/* set the next padding bytes to 0 */
	memset((char *)dst + sz, 0, padding);
	/* then, copy the second half of the expected result.
	 * this hopefully makes sure that if there's any buffer overrun within
	 * 2memstream, it can be detected easily. */
	memcpy((char *)dst + sz + padding, (char *)src + sz, sz);
}

testresult_t test_slurp_2memstream(void)
{
	slurp_t fp;
	uint8_t buf[(TEST_SLURP_2SIZE * 2) + TEST_SLURP_PADDING];
	testresult_t r;

	test_slurp_setup_padded_buf(expected_result, buf, TEST_SLURP_2SIZE, TEST_SLURP_PADDING);

	/* should never happen */
	ASSERT(slurp_2memstream(&fp, (uint8_t *)buf, (uint8_t *)buf + TEST_SLURP_2SIZE + TEST_SLURP_PADDING, TEST_SLURP_2SIZE) >= 0);

	r = test_slurp_common(&fp);

	/* currently not necessary for memstream, but whatever */
	unslurp(&fp);

	return r;
}

testresult_t test_slurp_sf2(void)
{
	slurp_t memfp;
	slurp_t sf2fp;
	uint8_t buf[(TEST_SLURP_2SIZE * 2) + TEST_SLURP_PADDING];
	testresult_t r;

	/* this 97 year old NYC diner still serves Coke the old fashioned way */
	test_slurp_setup_padded_buf(expected_result, buf, TEST_SLURP_2SIZE, TEST_SLURP_PADDING);

	ASSERT(slurp_memstream(&memfp, (uint8_t *)buf, sizeof(buf)) >= 0);

	slurp_sf2(&sf2fp, &memfp, 0, TEST_SLURP_2SIZE, TEST_SLURP_2SIZE + TEST_SLURP_PADDING, TEST_SLURP_2SIZE);

	r = test_slurp_common(&sf2fp);

	unslurp(&sf2fp);
	unslurp(&memfp);

	return r;
}

testresult_t test_slurp_stdio(void)
{
	slurp_t fp;
	char tmp[TEST_TEMP_FILE_NAME_LENGTH];
	FILE *stdfp;
	testresult_t r;

	stdfp = test_temp_file2(tmp, expected_result, ARRAY_SIZE(expected_result) - 1);
	REQUIRE(stdfp);

	REQUIRE(slurp_stdio(&fp, stdfp) == SLURP_OPEN_SUCCESS);

	r = test_slurp_common(&fp);

	unslurp(&fp);
	fclose(stdfp);

	return r;
}

#ifdef SCHISM_WIN32
testresult_t test_slurp_win32(void)
{
	slurp_t fp;
	char tmp[TEST_TEMP_FILE_NAME_LENGTH];
	testresult_t r;

	REQUIRE(test_temp_file(tmp, expected_result, ARRAY_SIZE(expected_result) - 1));

	REQUIRE(slurp_win32(&fp, tmp, 0) == SLURP_OPEN_SUCCESS);

	r = test_slurp_common(&fp);

	unslurp(&fp);

	return r;
}

testresult_t test_slurp_win32_mmap(void)
{
	slurp_t fp;
	char tmp[TEST_TEMP_FILE_NAME_LENGTH];
	testresult_t r;

	REQUIRE(test_temp_file(tmp, expected_result, ARRAY_SIZE(expected_result) - 1));

	REQUIRE(slurp_win32_mmap(&fp, tmp, ARRAY_SIZE(expected_result) - 1) == SLURP_OPEN_SUCCESS);

	r = test_slurp_common(&fp);

	unslurp(&fp);

	return r;
}
#endif

#ifdef HAVE_MMAP
testresult_t test_slurp_mmap(void)
{
	slurp_t fp;
	char tmp[TEST_TEMP_FILE_NAME_LENGTH];
	testresult_t r;

	REQUIRE(test_temp_file(tmp, expected_result, ARRAY_SIZE(expected_result) - 1));

	REQUIRE(slurp_mmap(&fp, tmp, ARRAY_SIZE(expected_result) - 1) == SLURP_OPEN_SUCCESS);

	r = test_slurp_common(&fp);

	unslurp(&fp);

	return r;
}
#endif

#ifdef USE_ZLIB
testresult_t test_slurp_gzip(void)
{
	testresult_t r;
	slurp_t fp;
	static const unsigned char expected_result_gz[] = {
		'\037', '\213', '\010', '\010', '\234', '\107', '\333', '\150', '\002', '\003', '\145', '\170', '\160', '\145', '\143', '\164',
		'\145', '\144', '\137', '\162', '\145', '\163', '\165', '\154', '\164', '\000', '\113', '\114', '\112', '\126', '\110', '\111',
		'\115', '\123', '\110', '\317', '\310', '\124', '\060', '\064', '\062', '\126', '\060', '\061', '\065', '\123', '\060', '\267',
		'\260', '\344', '\362', '\124', '\310', '\311', '\054', '\113', '\125', '\310', '\314', '\123', '\110', '\124', '\110', '\317',
		'\114', '\314', '\053', '\121', '\110', '\052', '\115', '\316', '\116', '\055', '\321', '\343', '\112', '\312', '\111', '\055',
		'\115', '\317', '\320', '\343', '\002', '\000', '\240', '\062', '\045', '\375', '\072', '\000', '\000', '\000'
	};

	/* should never happen */
	ASSERT(slurp_memstream(&fp, (uint8_t *)expected_result_gz, sizeof(expected_result_gz)) >= 0);

	/* -1 is an error (e.g. zlib failed to load) */
	REQUIRE(gzip_init() >= 0);
	/* -1 is an error (e.g. zlib failed to decompress) */
	REQUIRE(slurp_gzip(&fp) >= 0);

	r = test_slurp_common(&fp);

	unslurp(&fp);
	gzip_quit();

	return r;
}
#endif

/* TODO need to add slurp test functions for win32 */
