use ui;
use core:geometry;
use graphics;

class Painter extends ui:Painter {
	private ProblemData data;
	private Size size;
	private Size viewport;

	private Tabs tabs;
	private Table? table;
	private Float scroll;

	private Text loading;
	private SolidBrush scrollBrush;

	init(ProblemData data) {
		init() {
			data = data;
			tabs(data.allNames);
			loading("Loading...", defaultFont);
			scrollBrush(Color(0.4, 0.4, 0.4));
		}
		bgColor = Color(0.8, 0.8, 0.8);

		(spawn selectTab(tabs.selected)).detach();
	}

	Bool render(Size me, Graphics g) : override {
		Bool repaint = false;

		size = me;

		tabs.render(me, g);

		viewport = Size(me.w, me.h - tabs.height);

		if (table) {
			g.transform = translate(Point(0, tabs.height - scroll));
			Rect view(Point(0, scroll), viewport);
			g.push(view);

			// Scroll bar.
			Float scrollWidth = 12;
			{
				Float viewHeight = view.size.h - 6;
				Float tableHeight = table.height;
				Float scaledSize = (viewHeight / tableHeight).min(1);
				Float scaledOffset = scroll / tableHeight;
				Size sz(scrollWidth - 6, scaledSize * viewHeight);
				Point origin(view.p1.x - scrollWidth + 3, scaledOffset * viewHeight);
				origin += Point(0, scroll + 3); // To compensate for the transform. 3 is half the border around it.
				g.fill(Rect(origin, sz), Size(3), scrollBrush);
			}

			// Table.
			view.p1.x -= scrollWidth;
			repaint |= table.render(view, g);

			g.pop();
		} else {
			g.draw(loading, tableText, me / 2 - loading.size / 2);
		}

		repaint;
	}

	void onClick(Point pt) {
		if (pt.y < tabs.height) {
			if (sel = tabs.onClick(pt, size)) {
				table = null;
				(spawn selectTab(sel)).detach();
			}
			repaint();
		} else if (table) {
			pt.y -= tabs.height;
			pt.y += scroll;
			table.onClick(pt, this);
			repaint();
		}
	}

	void onVScroll(Point pt, Int delta) {
		if (table) {
			if (pt.y >= tabs.height) {
				scroll -= delta.float;

				scroll = scroll.min(table.height - viewport.h).max(0);
			}
			repaint();
		}
	}

	private void selectTab(Nat id) {
		table = Table(data.view(id));
		scroll = 0;

		repaint();
	}
}
